/*!\file Sing.c
 * \brief: implementation of the Sing object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include "./Sing.h"
#include "./SingVertexInput.h"
#include <string.h>
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../shared/shared.h"
#include "../DataSet/DataSet.h"
#include "../DataSet/Parameters.h"
#include "../DataSet/Inputs.h"
#include "../include/typedefs.h"
#include "../include/macros.h"

/*Object constructors and destructor*/
/*FUNCTION Sing::Sing(){{{1*/
Sing::Sing(){
	this->inputs=NULL;
	this->parameters=NULL;
	return;
}
/*}}}*/
/*FUNCTION Sing::Sing(int sing_id,int* sing_node_ids, int sing_matice_id, int sing_matpar_id, int sing_numpar_id) {{{1*/
Sing::Sing(int sing_id,int* sing_node_ids, int sing_matice_id, int sing_matpar_id):
	hnodes(sing_node_ids,1),
	hmatice(&sing_matice_id,1),
	hmatpar(&sing_matpar_id,1)
{

	/*all the initialization has been done by the initializer, just fill in the id: */
	this->id=sing_id;
	this->parameters=NULL;
	this->inputs=new Inputs();

	return;
}
/*}}}*/
/*FUNCTION Sing::Sing(int sing_id,Hook* sing_hnodes, Hook* sing_hmatice, Hook* sing_hmatpar, Hook* sing_hnumpar, Inputs* sing_inputs) {{{1*/
Sing::Sing(int sing_id,Hook* sing_hnodes, Hook* sing_hmatice, Hook* sing_hmatpar, Parameters* sing_parameters,Inputs* sing_inputs):
	hnodes(sing_hnodes),
	hmatice(sing_hmatice),
	hmatpar(sing_hmatpar)
{

	/*all the initialization has been done by the initializer, just fill in the id: */
	this->id=sing_id;
	if(sing_inputs){
		this->inputs=(Inputs*)sing_inputs->Copy();
	}
	else{
		this->inputs=new Inputs();
	}
	/*point parameters: */
	this->parameters=sing_parameters;
}
/*}}}*/
/*FUNCTION Sing::Sing(int sing_id, int i, IoModel* iomodel) {{{1*/
Sing::Sing(int sing_id, int i, IoModel* iomodel){

	int sing_matice_id;
	int sing_matpar_id;
	
	int sing_g;
	double sing_h;
	double sing_k;

	/*id: */
	this->id=sing_id;

	/*hooks: */
	sing_matice_id=i+1;                        //refers to the corresponding material property card
	sing_matpar_id=iomodel->numberofvertices+1;//refers to the corresponding matpar property card
	sing_g=i+1;

	this->hnodes.Init(&sing_g,1);
	this->hmatice.Init(&sing_matice_id,1);
	this->hmatpar.Init(&sing_matpar_id,1);

	//intialize inputs, and add as many inputs per element as requested: 
	this->inputs=new Inputs();

	if (iomodel->thickness) this->inputs->AddInput(new SingVertexInput(ThicknessEnum,iomodel->thickness[i]));
	if (iomodel->drag_coefficient) this->inputs->AddInput(new SingVertexInput(DragCoefficientEnum,iomodel->drag_coefficient[i]));

	//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
	this->parameters=NULL;

}
/*}}}*/
/*FUNCTION Sing::~Sing(){{{1*/
Sing::~Sing(){
	delete inputs;
	this->parameters=NULL;
}
/*}}}*/

/*Object management*/
/*FUNCTION Sing::Configure {{{1*/
void  Sing::Configure(DataSet* loadsin, DataSet* nodesin, DataSet* materialsin, Parameters* parametersin){

	int i;
	
	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hnodes.configure(nodesin);
	hmatice.configure(materialsin);
	hmatpar.configure(materialsin);

	/*point parameters to real dataset: */
	this->parameters=parametersin;

}
/*}}}*/
/*FUNCTION Sing::copy {{{1*/
Object* Sing::copy() {
	
	return new Sing(this->id,&this->hnodes,&this->hmatice,&this->hmatpar,this->parameters,this->inputs);

}
/*}}}*/
/*FUNCTION Sing::DeepEcho {{{1*/
void Sing::DeepEcho(void){

	printf("Sing:\n");
	printf("   id: %i\n",id);
	hnodes.DeepEcho();
	hmatice.DeepEcho();
	hmatpar.DeepEcho();
	printf("   parameters\n");
	parameters->DeepEcho();
	printf("   inputs\n");
	inputs->DeepEcho();

	return;
}
/*}}}*/
/*FUNCTION Sing::Demarshall {{{1*/
void  Sing::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);

	/*demarshall hooks: */
	hnodes.Demarshall(&marshalled_dataset);
	hmatice.Demarshall(&marshalled_dataset);
	hmatpar.Demarshall(&marshalled_dataset);

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Sing::Echo{{{1*/

void Sing::Echo(void){

	printf("Sing:\n");
	printf("   id: %i\n",id);
	hnodes.Echo();
	hmatice.Echo();
	hmatpar.Echo();
	printf("   parameters\n");
	parameters->Echo();
	printf("   inputs\n");
	inputs->Echo();
}
/*}}}*/
/*FUNCTION Sing::Marshall {{{1*/
void  Sing::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Sing: */
	enum_type=SingEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Sing data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	
	/*Marshall hooks: */
	hnodes.Marshall(&marshalled_dataset);
	hmatice.Marshall(&marshalled_dataset);
	hmatpar.Marshall(&marshalled_dataset);

	/*Marshall inputs: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	/*parameters: don't do anything about it. parameters are marshalled somewhere else!*/

	
	xfree((void**)&marshalled_inputs);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Sing::MashallSize {{{1*/
int   Sing::MarshallSize(){

	return sizeof(id)
		+hnodes.MarshallSize()
		+hmatice.MarshallSize()
		+hmatpar.MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Sing::UpdateInputs {{{1*/
void  Sing::UpdateInputs(double* solution, int analysis_type, int sub_analysis_type){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
		
/*Object functions*/
/*FUNCTION Sing::ComputeBasalStress {{{1*/
void  Sing::ComputeBasalStress(Vec p_g,int analysis_type,int sub_analysis_type){

	ISSMERROR("Not implemented yet");

}
/*}}}*/
/*FUNCTION Sing::ComputePressure {{{1*/
void  Sing::ComputePressure(Vec p_g,int analysis_type,int sub_analysis_type){

	int    dof;
	double pressure;
	double thickness;
	double rho_ice,g;

	/*dynamic objects pointed to by hooks: */
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*Get dof list on which we will plug the pressure values: */
	GetDofList1(&dof);

	/*recover objects from hooks: */
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	/*pressure is lithostatic: */
	rho_ice=matpar->GetRhoIce();
	g=matpar->GetG();
	inputs->GetParameterValue(&thickness,ThicknessEnum);
	pressure=rho_ice*g*thickness;
	
	/*plug local pressure values into global pressure vector: */
	VecSetValue(p_g,dof,pressure,INSERT_VALUES);

}
/*}}}*/
/*FUNCTION Sing::ComputeStrainRate {{{1*/
void  Sing::ComputeStrainRate(Vec p_g,int analysis_type,int sub_analysis_type){

	ISSMERROR("Not implemented yet");

}
/*}}}*/
/*FUNCTION Sing::CostFunction {{{1*/
double Sing::CostFunction( int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::CreateKMatrix {{{1*/

void  Sing::CreateKMatrix(Mat Kgg,int analysis_type,int sub_analysis_type){

	/*Just branch to the correct element stiffness matrix generator, according to the type of analysis we are carrying out: */
	if ((analysis_type==DiagnosticAnalysisEnum) && (sub_analysis_type==HutterAnalysisEnum)){

		CreateKMatrixDiagnosticHutter( Kgg,analysis_type,sub_analysis_type);

	}
	else{
		ISSMERROR("%s%i%s%i%s\n","analysis: ",analysis_type," and sub_analysis_type: ",sub_analysis_type," not supported yet");
	}

}
/*}}}*/
/*FUNCTION Sing::CreateKMatrixDiagnosticHutter {{{1*/

void  Sing::CreateKMatrixDiagnosticHutter(Mat Kgg,int analysis_type,int sub_analysis_type){
	
	const int numgrids=1;
	const int NDOF2=2;
	const int numdofs=numgrids*NDOF2;
	double Ke_gg[numdofs][numdofs]={{1,0},{0,1}};
	int    doflist[numdofs];
	int    numberofdofspernode;

	GetDofList(&doflist[0],&numberofdofspernode);

	MatSetValues(Kgg,numdofs,doflist,numdofs,doflist,(const double*)Ke_gg,ADD_VALUES);

}
/*}}}*/
/*FUNCTION Sing::CreatePVector {{{1*/
void  Sing::CreatePVector(Vec pg,int analysis_type,int sub_analysis_type){
	
	/*Just branch to the correct load generator, according to the type of analysis we are carrying out: */
	if ((analysis_type==DiagnosticAnalysisEnum) && (sub_analysis_type==HutterAnalysisEnum)){
	
			CreatePVectorDiagnosticHutter( pg,analysis_type,sub_analysis_type);

	}
	else{
		ISSMERROR("%s%i%s"," analysis ",analysis_type," not supported yet");
	}

}
/*}}}*/
/*FUNCTION Sing::CreatePVectorDiagnosticHutter {{{1*/

void Sing::CreatePVectorDiagnosticHutter( Vec pg,  int analysis_type,int sub_analysis_type){
	
	
	const int numgrids=1;
	const int NDOF2=2;
	const int numdofs=NDOF2*numgrids;
	int       doflist[numdofs];
	int       dofs[1]={0};
	int       found=0;
	double    slope[2];
	double    slope2;
	double    pe_g[numdofs]={0,0};
	double    ub,vb;
	double    constant_part;
	int       numberofdofspernode;
	double    rho_ice,gravity,n,B;
	double    thickness;

	/*dynamic objects pointed to by hooks: */
	Node**  node=NULL;
	Matpar* matpar=NULL;
	Matice* matice=NULL;

	/*recover objects from hooks: */
	node=(Node**)hnodes.deliverp();
	matpar=(Matpar*)hmatpar.delivers();
	matice=(Matice*)hmatice.delivers();

	inputs->GetParameterValue(&slope[0],SurfaceSlopexEnum);
	inputs->GetParameterValue(&slope[1],SurfaceSlopeyEnum);

	GetDofList(&doflist[0],&numberofdofspernode);

	//compute slope2 
	slope2=pow(slope[0],2)+pow(slope[1],2);

	//compute ub
	rho_ice=matpar->GetRhoIce();
	gravity=matpar->GetG();
	n=matice->GetN();
	B=matice->GetB();
	inputs->GetParameterValue(&thickness,ThicknessEnum);

	ub=-1.58*pow((double)10.0,(double)-10.0)*rho_ice*gravity*thickness*slope[0];
	vb=-1.58*pow((double)10.0,(double)-10.0)*rho_ice*gravity*thickness*slope[1];

	//compute constant_part
	constant_part=-2*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2));

	pe_g[0]=ub-2.0*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2.0))*pow(thickness,n)/(pow(B,n)*(n+1))*slope[0];
	pe_g[1]=vb-2.0*pow(rho_ice*gravity,n)*pow(slope2,((n-1)/2.0))*pow(thickness,n)/(pow(B,n)*(n+1))*slope[1];

	VecSetValues(pg,numdofs,doflist,(const double*)pe_g,ADD_VALUES);


}
/*}}}*/
/*FUNCTION Sing::Du {{{1*/
void  Sing::Du(Vec,int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::Enum {{{1*/
int Sing::Enum(void){

	return SingEnum;

}
/*}}}*/
/*FUNCTION Sing::GetBedList {{{1*/
void  Sing::GetBedList(double*){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GetDofList {{{1*/
void  Sing::GetDofList(int* doflist,int* pnumberofdofspernode){

	int i;
	int doflist_per_node[MAXDOFSPERNODE];
	int numberofdofspernode;
	
	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	nodes[0]->GetDofList(&doflist_per_node[0],&numberofdofspernode);
	for(i=0;i<numberofdofspernode;i++){
		doflist[i]=doflist_per_node[i];
	}

	/*Assign output pointers:*/
	*pnumberofdofspernode=numberofdofspernode;

}
/*}}}*/
/*FUNCTION Sing::GetDofList1 {{{1*/
void  Sing::GetDofList1(int* doflist){

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	int i;
	doflist[0]=nodes[0]->GetDofList1();

}
/*}}}*/
/*FUNCTION Sing::GetId {{{1*/
int    Sing::GetId(void){ return id; }
/*}}}*/
/*FUNCTION Sing::GetMatPar {{{1*/
void* Sing::GetMatPar(){

	/*dynamic objects pointed to by hooks: */
	Matpar* matpar=NULL;

	/*recover objects from hooks: */
	matpar=(Matpar*)hmatpar.delivers();

	return matpar;
}
/*}}}*/
/*FUNCTION Sing::GetName {{{1*/
char* Sing::GetName(void){
	return "sing";
}
/*}}}*/
/*FUNCTION Sing::GetNodes {{{1*/
void  Sing::GetNodes(void** vpnodes){
	
	Node** pnodes=NULL;

	/*dynamic objects pointed to by hooks: */
	Node**  nodes=NULL;
	
	/*recover nodes: */
	pnodes=(Node**)vpnodes;

	/*recover objects from hooks: */
	nodes=(Node**)hnodes.deliverp();
	
	pnodes[0]=nodes[0];
}
/*}}}*/
/*FUNCTION Sing::GetOnBed {{{1*/
bool   Sing::GetOnBed(){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GetShelf {{{1*/
bool   Sing::GetShelf(){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GetThicknessList {{{1*/
void  Sing::GetThicknessList(double* thickness_list){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::Gradj {{{1*/
void  Sing::Gradj(Vec,  int, int ,char*){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GradB {{{1*/
void  Sing::GradjB(Vec,  int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::GradjDrag {{{1*/
void  Sing::GradjDrag(Vec,  int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::MassFlux {{{1*/
double Sing::MassFlux( double* segment,double* ug){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::Misfit {{{1*/
double Sing::Misfit( int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::MyRank {{{1*/
int    Sing::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION Sing::SurfaceArea {{{1*/
double Sing::SurfaceArea( int,int){
	ISSMERROR(" not supported yet!");
}
/*}}}*/
/*FUNCTION Sing::SetClone {{{1*/
void  Sing::SetClone(int* minranks){

	ISSMERROR("not implemented yet");
}
/*}}}1*/
