function md=solve(md,varargin)
%SOLVE - apply solution sequence for this model
%
%   Usage:
%      md=solve(md,varargin)
%      where varargin is a lit of paired arguments. 
%      arguments can be: 'analysis_type': 'diagnostic','thermal','prognostic','transient'
%      arguments can be: 'sub_analysis_type': 'transient','steady','' (default if empty = 'steady')
%
%   Examples:
%      md=solve(md,'analysis_type','diagnostic');
%      md=solve(md,'analysis_type','thermal','sub_analysis_type','transient');
%      md=solve(md,'analysis_type','thermal','sub_analysis_type','steady');
%      md=solve(md,'analysis_type','thermal');

%some checks on list of arguments
global ISSM_DIR

%recover options
options=pairoptions(varargin{:});

%add default options
options=process_solve_options(options);

%recover some fields
md.analysis_type=options.analysis_type;
md.sub_analysis_type=options.sub_analysis_type;

%check model consistency
displaystring(md.verbose,'\n%s\n','checking model consistency');
ismodelselfconsistent(md),

%preprocesses model before solving
md=presolve(md);

%if running qmu analysis, some preprocessing of dakota files using 
%models fields needs to be carried out. 
if md.qmu_analysis,
	md=preqmu(md,options);
end

%Save model as is (in case of a crash)
assignin('base',inputname(1),md);

displaystring(md.verbose,'\n%s\n','launching solution sequence');

%If running in parallel, we have a different way of launching the solution
%sequences. 
if ~strcmpi(md.cluster,'none'),
	md=solveparallel(md,options);
	return;
end

%Launch correct solution sequence
if md.analysis_type==DiagnosticAnalysisEnum,
	md=diagnostic(md);

elseif md.analysis_type==TransientAnalysisEnum,
	md=transient(md);

elseif md.analysis_type==PrognosticAnalysisEnum,
	md=prognostic(md);

elseif md.analysis_type==Prognostic2AnalysisEnum,
	md=prognostic2(md);

elseif md.analysis_type==ThermalAnalysisEnum,
	md=thermal(md);

elseif md.analysis_type==ParametersAnalysisEnum,
	md=parameters(md);

elseif md.analysis_type==SteadystateAnalysisEnum,
	md=steadystate(md);

elseif md.analysis_type==BalancedthicknessAnalysisEnum,
	md=balancedthickness(md);

elseif md.analysis_type==Balancedthickness2AnalysisEnum,
	md=balancedthickness2(md);

elseif md.analysis_type==BalancedvelocitiesAnalysisEnum,
	md=balancedvelocities(md);

elseif md.analysis_type==SlopecomputeAnalysisEnum,
	md=slopecompute(md);

else
	error('solution type not supported for this model configuration.');
end

%post processes qmu results if necessary
if md.qmu_analysis,
	md=postqmu(md);
	cd ..
end

%Check result is consistent
displaystring(md.verbose,'%s\n','checking result consistency');
if ~isresultconsistent(md,options.analysis_type),
	disp('!! results not consistent correct the model !!') %it would be very cruel to put an error, it would kill the computed results (even if not consistent...)
end

%convert analysis type to string finally
md.analysis_type=AnalysisTypeFromEnum(options.analysis_type);
md.sub_analysis_type=AnalysisTypeFromEnum(options.sub_analysis_type);
