/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

// Class:        NIDRProblemDescDB
//- Description: Implementation code for the New IDR ProblemDescDB class.
//-              It defines the keyword handlers that yacc calls to populate
//-              the database based on the parsed input.
//- Owner:       David M. Gay
//- Checked by:

#include "system_defs.h"
#include <functional>
#include <stdarg.h>
#include "NIDRProblemDescDB.H"
#include "pecos_stat_util.hpp"


extern "C" int not_executable(const char*);

char nidr_please_refer[] =
	"Please refer to the DAKOTA Reference and/or User's Manual or the\n"
	"dakota.input.txt file distributed with this executable.";

namespace Dakota {

extern "C" FILE *nidrin;
extern "C" int nidr_parse(const char*, FILE*);

int NIDRProblemDescDB::nerr = 0;
NIDRProblemDescDB* NIDRProblemDescDB::pDDBInstance(NULL);

 void NIDRProblemDescDB::
botch(const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	fprintf(stderr, "\nError: ");
	vfprintf(stderr, fmt, ap);
	fputs(".\n", stderr);
	va_end(ap);
	abort_handler(-1);
	}

 void NIDRProblemDescDB::
squawk(const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	fprintf(stderr, "\nError: ");
	vfprintf(stderr, fmt, ap);
	fputs(".\n", stderr);
	va_end(ap);
	++nerr;
	}

 void NIDRProblemDescDB::
warn(const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	fprintf(stderr, "\nWarning: ");
	vfprintf(stderr, fmt, ap);
	fputs(".\n", stderr);
	va_end(ap);
	}


/** Parse the input file using the Input Deck Reader (IDR) parsing system.
    IDR populates the IDRProblemDescDB object with the input file data. */
void NIDRProblemDescDB::derived_manage_inputs(const char* dakota_input_file)
{
  // set the pDDBInstance
  pDDBInstance = this;

  // Open the dakota input file passed in and "attach" it to stdin
  // (required by nidr_parse)
  if (dakota_input_file) {
    if (dakota_input_file[0] == '-' && !dakota_input_file[1])
      nidrin = stdin;
    else if( !(nidrin = fopen(dakota_input_file, "r")) )
      botch("cannot open \"%s\"", dakota_input_file);
  }

  // nidr_parse parses the input file and invokes the keyword handlers
  if (nidr_parse(parser,0)) {
    //Cerr << "\nErrors parsing input file." << endl;
    abort_handler(-1); // allows proper abort in parallel case
  }
  if (nerr)
    abort_handler(-1);
}


NIDRProblemDescDB::
NIDRProblemDescDB(ParallelLibrary& parallel_lib, const char *Parser):
  ProblemDescDB(BaseConstructor(), parallel_lib), parser(Parser)
{}

#define Squawk NIDRProblemDescDB::squawk
#define Warn NIDRProblemDescDB::warn

 typedef unsigned long UL;

 struct
Iface_Info {
	DataInterfaceRep *di;
	DataInterface *di0;
	};

 struct
Iface_mp_Rlit {
	String DataInterfaceRep::* sp;
	RealDenseVector DataInterfaceRep::* rv;
	const char *lit;
	};

 struct
Iface_mp_ilit {
	String DataInterfaceRep::* sp;
	int DataInterfaceRep::* iv;
	const char *lit;
	};

 struct
Iface_mp_lit {
	String DataInterfaceRep::* sp;
	const char *lit;
	};

 struct
Meth_Info {
	DataMethodRep *dme;
	DataMethod *dme0;
	};

 struct
Method_mp_Ii {
	int DataMethodRep::* vp;
	short DataMethodRep::* ip;
	int ival;
	};

 struct
Method_mp_Ri {
	Real DataMethodRep::* vp;
	short DataMethodRep::* ip;
	int ival;
	};

 struct
Method_mp_ilit2 {
	String DataMethodRep::* sp;
	int DataMethodRep::* ip;
	const char *lit;
	};

 struct
Method_mp_ilit2z {
	String DataMethodRep::* sp;
	size_t DataMethodRep::* ip;
	const char *lit;
	};

 struct
Method_mp_lit {
	String DataMethodRep::* sp;
	const char *lit;
	};

 struct
Method_mp_lit2 {
	String DataMethodRep::* sp;
	String DataMethodRep::* sp2;
	const char *lit;
	const char *lit2;
	};

 struct
Method_mp_litc {
	String DataMethodRep::* sp;
	Real DataMethodRep::* rp;
	const char *lit;
	};

 struct
Method_mp_liti {
	String DataMethodRep::* sp;
	short DataMethodRep::* ip;
	const char *lit;
	int ival;
	};
 struct
Method_mp_litrv {
	String DataMethodRep::* sp;
	RealVector DataMethodRep::* rp;
	const char *lit;
	};

 struct
Method_mp_slit {
	short DataMethodRep::* sp;
	int lit;
	};

 struct
Method_mp_slit2 {
	String DataMethodRep::* sp;
	String DataMethodRep::* sp2;
	const char *lit;
	};

 struct
Mod_Info {
	DataModelRep *dmo;
	DataModel *dmo0;
	};

 struct
Model_mp_lit {
	String DataModelRep::* sp;
	const char *lit;
	};

 struct
Model_mp_ord {
	short DataModelRep::* sp;
	int ord;
	};

//  struct
// Model_mp_slit2 {
// 	String DataModelRep::* sp;
// 	String DataModelRep::* sp2;
// 	const char *lit;
// 	};

 struct
Resp_Info {
	DataResponsesRep *dr;
	DataResponses *dr0;
	};

 struct
Resp_mp_lit {
	String DataResponsesRep::* sp;
	const char *lit;
	};

 struct
Strategy_mp_lit {
	String DataStrategyRep::* sp;
	const char *lit;
	};

 struct
Strategy_mp_slit {
	short DataStrategyRep::* sp;
	int lit;
	};

 enum { // kinds of uncertain variables
	UncVar_normal = 0,
	UncVar_lognormal = 1,
	UncVar_uniform = 2,
	UncVar_loguniform = 3,
	UncVar_triangular = 4,
	UncVar_exponential = 5,
	UncVar_beta = 6,
	UncVar_gamma = 7,
	UncVar_gumbel = 8,
	UncVar_frechet = 9,
	UncVar_weibull = 10,
	UncVar_histogram = 11,
	UncVar_interval = 12,
	UncVar_Nkinds = 13	// number of kinds of uncertain variables
	};

 struct
VarLabel {
	size_t n;
	const char **s;
	};

 struct
Var_Info {
	DataVariablesRep *dv;
	DataVariables *dv0;
	VarLabel Uv[UncVar_Nkinds];
	IntVector *nIv, *nbp, *npp;
	RealDenseVector *Ivb, *Ivp, *bp, *pp, *ucm;
	};

 struct
Var_bchk
{
	const char *name;
	size_t DataVariablesRep::* n;
	RealDenseVector DataVariablesRep::* L;
	RealDenseVector DataVariablesRep::* U;
	RealDenseVector DataVariablesRep::* V;	// initial guess, possibly null
	StringArray DataVariablesRep::* Lbl;	// possibly null
	};

 struct
Var_ibchk
{
	const char *name;
	size_t DataVariablesRep::* n;
	IntDenseVector DataVariablesRep::* L;
	IntDenseVector DataVariablesRep::* U;
	IntDenseVector DataVariablesRep::* V;
	StringArray DataVariablesRep::* Lbl;
	};

 struct
Var_uinfo {
	const char *lbl;
	const char *vkind;
	size_t DataVariablesRep::* n;
	void(*vadj)(DataVariablesRep*, size_t, Var_Info*);
	};

 struct
Var_brv {
	RealDenseVector DataVariablesRep::* rv;
	Real b;
	};

 void NIDRProblemDescDB::
iface_Rlit(const char *keyname, Values *val, void **g, void *v)
{
	DataInterfaceRep *di = (*(Iface_Info**)g)->di;
	Iface_mp_Rlit *R = (Iface_mp_Rlit*)v;
	Real *r;
	RealDenseVector *rv;
	size_t i, n;

	di->*R->sp = R->lit;
	rv = &(di->*R->rv);
	n = val->n;
	rv->sizeUninitialized(n);
	r = val->r;
	for(i = 0; i < n; ++i)
		(*rv)[i] = r[i];
	}

 void NIDRProblemDescDB::
iface_false(const char *keyname, Values *val, void **g, void *v)
{
	(*(Iface_Info**)g)->di->**(bool DataInterfaceRep::**)v = false;
	}

 void NIDRProblemDescDB::
iface_ilit(const char *keyname, Values *val, void **g, void *v)
{
	DataInterfaceRep *di = (*(Iface_Info**)g)->di;
	Iface_mp_ilit *I = (Iface_mp_ilit*)v;

	di->*I->sp = I->lit;
	di->*I->iv = *val->i;
	}

 void NIDRProblemDescDB::
iface_int(const char *keyname, Values *val, void **g, void *v)
{
	(*(Iface_Info**)g)->di->**(int DataInterfaceRep::**)v = *val->i;
	}

 void NIDRProblemDescDB::
iface_lit(const char *keyname, Values *val, void **g, void *v)
{
	(*(Iface_Info**)g)->di->*((Iface_mp_lit*)v)->sp = ((Iface_mp_lit*)v)->lit;
	}

 void NIDRProblemDescDB::
iface_start(const char *keyname, Values *val, void **g, void *v)
{
	Iface_Info *ii;

	if (!(ii = new Iface_Info))
 Botch:		botch("new failure in iface_start");
	if (!(ii->di0 = new DataInterface))
		goto Botch;
	ii->di = ii->di0->dataIfaceRep;
	*g = (void*)ii;
	}

 void NIDRProblemDescDB::
iface_true(const char *keyname, Values *val, void **g, void *v)
{
	(*(Iface_Info**)g)->di->**(bool DataInterfaceRep::**)v = true;
	}

 void NIDRProblemDescDB::
iface_stop(const char *keyname, Values *val, void **g, void *v)
{
	Iface_Info *ii = *(Iface_Info**)g;
	DataInterfaceRep *di = ii->di;
	StringArray *sa;
	const char *s;
	int ac, ec, j, nd;
	size_t i, n;

	nd = di->analysisDrivers.length();
	ac = di->asynchLocalAnalysisConcurrency;
	ec = di->asynchLocalEvalConcurrency;

	if (di->algebraicMappings == "" && nd == 0)
		squawk("interface specification must provide algebraic_mappings,\n\t"
			"analysis_drivers, or both");
	if (nd > 0 && di->asynchLocalAnalysisConcurrency > nd) {
		warn("analysis_concurrency specification greater than length of\n\t"
			"analysis_drivers list.  Truncating analysis_concurrency to %d",nd);
		di->asynchLocalAnalysisConcurrency = nd;
		}
	if (ec && ec < 2 && ac && ac < 2) {
		warn("asynchronous option not required for evaluation and analysis.\n\t"
			"Concurrency limited to %d and %d.\n\t"
			"Synchronous operations will be used", ec, ac);
		di->interfaceSynchronization = "synchronous";
		}
	if (di->interfaceType == "system" || di->interfaceType == "fork") {
		sa = &di->analysisDrivers;
		n = sa->length();
		for(i = 0; i < n; ++i)
			if ((j = not_executable(s = (*sa)[i])))
				squawk("analysis driver \"%s\" %s", s,
					j == 1	? "not found"
						: "exists but is not executable");
		}
	pDDBInstance->dataInterfaceList.insert(*ii->di0);
	delete ii->di0;
	delete ii;
	}

 void NIDRProblemDescDB::
iface_str(const char *keyname, Values *val, void **g, void *v)
{
	(*(Iface_Info**)g)->di->**(String DataInterfaceRep::**)v = *val->s;
	}

 void NIDRProblemDescDB::
iface_str2D(const char *keyname, Values *val, void **g, void *v)
{
	DataInterfaceRep *di = (*(Iface_Info**)g)->di;
	String2DArray *sa2 = &(di->**(String2DArray DataInterfaceRep::**)v);
	StringArray *sa;
	const char **s = val->s;
	size_t i, j, k, n, nc, nd;

	// This is just for analysisComponents -- the only String2DArray in a DataInterfaceRep.

	nd = di->analysisDrivers.length();
	n = val->n;
	if (nd <= 0)
		botch("num_drivers = %d in iface_str2D", (int)nd);
	if (n % nd) {
		squawk("number of analysis_components not evenly divisible "
			"by number of analysis_drivers");
		return;
		}
	nc = n / nd;
	sa2->reshape(nd);
	for(i = k = 0; i < nd; i++) {
		sa = &((*sa2)[i]);
		sa->reshape(nc);
		for(j = 0; j < nc; ++j, ++k)
			(*sa)[j] = s[k];
		}
	}

 void NIDRProblemDescDB::
iface_strL(const char *keyname, Values *val, void **g, void *v)
{
	StringArray *sa = &((*(Iface_Info**)g)->di->**(StringArray DataInterfaceRep::**)v);
	const char **s = val->s;
	size_t i, n = val->n;

	sa->reshape(n);
	for(i = 0; i < n; i++)
		(*sa)[i] = s[i];
	}

 void NIDRProblemDescDB::
method_Ii(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	dm->*((Method_mp_Ii*)v)->vp = *val->i;
	dm->*((Method_mp_Ii*)v)->ip = ((Method_mp_Ii*)v)->ival;
	}

 void NIDRProblemDescDB::
method_Real(const char *keyname, Values *val, void **g, void *v)
{
	(*(Meth_Info**)g)->dme->**(Real DataMethodRep::**)v = *val->r;
	}

 void NIDRProblemDescDB::
method_Real01(const char *keyname, Values *val, void **g, void *v)
{
	Real t = *val->r;
	if (t < 0. || t > 1.)
		botch("%s must be in [0, 1]", keyname);
	(*(Meth_Info**)g)->dme->**(Real DataMethodRep::**)v = t;
	}

 void NIDRProblemDescDB::
method_RealL(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	RealVector DataMethodRep::* sa = *(RealVector DataMethodRep::**)v;
	Real *r = val->r;
	size_t i, n = val->n;

	(dm->*sa).reshape(n);
	for(i = 0; i < n; i++)
		(dm->*sa)[i] = r[i];
	}

 void NIDRProblemDescDB::
method_RealDL(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	RealDenseVector DataMethodRep::* sa = *(RealDenseVector DataMethodRep::**)v;
	Real *r = val->r;
	size_t i, n = val->n;

	(dm->*sa).sizeUninitialized(n);
	for(i = 0; i < n; i++)
		(dm->*sa)[i] = r[i];
	}

 void NIDRProblemDescDB::
method_RealLlit(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	Real *r;
	RealVector *rv = &(dm->*((Method_mp_litrv*)v)->rp);
	size_t i, n;
	dm->*((Method_mp_litrv*)v)->sp = ((Method_mp_litrv*)v)->lit;
	r = val->r;
	n = val->n;
	rv->reshape(n);
	for(i = 0; i < n; i++)
		(*rv)[i] = r[i];
	}

 void NIDRProblemDescDB::
method_Realp(const char *keyname, Values *val, void **g, void *v)
{
	Real t = *val->r;
	if (t <= 0.)
		botch("%s must be positive", keyname);
	(*(Meth_Info**)g)->dme->**(Real DataMethodRep::**)v = t;
	}

 void NIDRProblemDescDB::
method_Realz(const char *keyname, Values *val, void **g, void *v)
{
	Real t = *val->r;
	if (t < 0.)
		botch("%s must be nonnegative", keyname);
	(*(Meth_Info**)g)->dme->**(Real DataMethodRep::**)v = t;
	}

 void NIDRProblemDescDB::
method_Ri(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	dm->*((Method_mp_Ri*)v)->vp = *val->r;
	dm->*((Method_mp_Ri*)v)->ip = ((Method_mp_Ri*)v)->ival;
	}

 void NIDRProblemDescDB::
method_coliny_ea(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	dm->methodName = "coliny_ea";
	dm->fitnessType = "linear_rank";
	dm->replacementType = "elitist";
	dm->numberRetained = 1;
	}

 void NIDRProblemDescDB::
method_false(const char *keyname, Values *val, void **g, void *v)
{
	(*(Meth_Info**)g)->dme->**(bool DataMethodRep::**)v = false;
	}

 void NIDRProblemDescDB::
method_int(const char *keyname, Values *val, void **g, void *v)
{
	(*(Meth_Info**)g)->dme->**(int DataMethodRep::**)v = *val->i;
	}

 void NIDRProblemDescDB::
method_intL(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	int *z = val->i;
	IntVector *iv = &(dm->**(IntVector DataMethodRep::**)v);
	size_t i, n = val->n;

	iv->reshape(n);
	for(i = 0; i < n; i++)
		(*iv)[i] = z[i];
	}

 void NIDRProblemDescDB::
method_ilit2(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	dm->*((Method_mp_ilit2*)v)->sp = ((Method_mp_ilit2*)v)->lit;
	dm->*((Method_mp_ilit2*)v)->ip = *val->i;
	}

 void NIDRProblemDescDB::
method_ilit2p(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	dm->*((Method_mp_ilit2z*)v)->sp = ((Method_mp_ilit2z*)v)->lit;
	if ((dm->*((Method_mp_ilit2z*)v)->ip = *val->i) <= 0)
		botch("%s must be positive", keyname);
	}

 void NIDRProblemDescDB::
method_lit(const char *keyname, Values *val, void **g, void *v)
{
	(*(Meth_Info**)g)->dme->*((Method_mp_lit*)v)->sp = ((Method_mp_lit*)v)->lit;
	}

 void NIDRProblemDescDB::
method_lit2(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	dm->*((Method_mp_lit2*)v)->sp  = ((Method_mp_lit2*)v)->lit;
	dm->*((Method_mp_lit2*)v)->sp2 = ((Method_mp_lit2*)v)->lit2;
	}

 void NIDRProblemDescDB::
method_litc(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	if (dm->*((Method_mp_litc*)v)->rp > 0.)
		dm->*((Method_mp_litc*)v)->sp = ((Method_mp_litc*)v)->lit;
	}

 void NIDRProblemDescDB::
method_liti(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	Method_mp_liti *Li = (Method_mp_liti*)v;
	dm->*Li->sp = Li->lit;
	dm->*Li->ip = (short)Li->ival;
	}

 void NIDRProblemDescDB::
method_litp(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	Real t = *val->r;
	if (t <= 0.)
		botch("%s must be positive",keyname);
	dm->*((Method_mp_litc*)v)->sp = ((Method_mp_litc*)v)->lit;
	dm->*((Method_mp_litc*)v)->rp = t;
	}

 void NIDRProblemDescDB::
method_litpp(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	dm->*((Method_mp_lit*)v)->sp = ((Method_mp_lit*)v)->lit;
	dm->stepLenToBoundary = dm->centeringParam = -1.;
	}

 void NIDRProblemDescDB::
method_litz(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	Real t = *val->r;
	if (t < 0.)
		botch("%s must be nonnegative",keyname);
	if ((dm->*((Method_mp_litc*)v)->rp = t) == 0.)
		dm->*((Method_mp_litc*)v)->sp = ((Method_mp_litc*)v)->lit;
	}

 void NIDRProblemDescDB::
method_litpp_final(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;

	if (dm->stepLenToBoundary == -1.) {
		if (dm->meritFn == "el_bakry")
			dm->stepLenToBoundary = 0.8;
		else if (dm->meritFn == "argaez_tapia")
			dm->stepLenToBoundary = 0.99995;
		else if (dm->meritFn == "van_shanno")
			dm->stepLenToBoundary = 0.95;
		else
 Botch:			botch("Unexpected meritFn in method_litpp_final");
		}
	if (dm->centeringParam == -1.) {
		if (dm->meritFn == "el_bakry")
			dm->centeringParam = 0.2;
		else if (dm->meritFn == "argaez_tapia")
			dm->centeringParam = 0.2;
		else if (dm->meritFn == "van_shanno")
			dm->centeringParam = 0.1;
		else
			goto Botch;
		}
	}

 void NIDRProblemDescDB::
method_meritFn(const char *keyname, Values *val, void **g, void *v)
{
	const char *s;

	(*(Meth_Info**)g)->dme->**(String DataMethodRep::**)v = s = *val->s;
	if (strcmp(s,"el_bakry")
	 && strcmp(s,"argaez_tapia")
	 && strcmp(s,"van_shanno"))
		botch("merit_function %s not supported", s);
	}

 void NIDRProblemDescDB::
method_moga_begin(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	dm->methodName = "moga";
	dm->randomSeed = -1;	// MOGA should "randomly" choose a seed.
	dm->crossoverRate = 0.75;
	dm->mutationType = "";
	}

 void NIDRProblemDescDB::
method_moga_final(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;

	if (dm->mutationType == "") {
		dm->mutationType = "replace_uniform";
		dm->mutationRate = 0.1;
		}
	}

 void NIDRProblemDescDB::
method_nnint(const char *keyname, Values *val, void **g, void *v)
{
	int n = *val->i;
	if (n < 0)
		botch("%s must be non-negative", keyname);
	(*(Meth_Info**)g)->dme->**(int DataMethodRep::**)v = n;
	}

 void NIDRProblemDescDB::
method_nnintz(const char *keyname, Values *val, void **g, void *v)
{
	int n = *val->i;
#if 0
	if (n < 0)
		botch("%s must be non-negative", keyname);
#endif
	(*(Meth_Info**)g)->dme->**(size_t DataMethodRep::**)v = n;
	}

 void NIDRProblemDescDB::
method_num_resplevs(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	Real *r;
	RealVectorArray *rva = &(dm->**(RealVectorArray DataMethodRep::**)v);
	RealVector *ra = &(*rva)[0];
	int m, *z = val->i;
	size_t i, i1, j, je, k, n = val->n;

	for(i = k = 0; i < n; ++i)
		k += z[i];
	if (ra->length() != k)
		botch("number of %s = %u does not match\n"
			"%s specification of %u response levels",
			keyname+4, (Uint)ra->length(), keyname, (Uint)k);
	r = new Real[k];
	for(i = 0; i < k; i++)
		r[i] = (*ra)[i];
	(*rva).reshape(n);
	for(i = j = je = 0; i < n; i++) {
		m = z[i];
		(*rva)[i].reshape(m);
		ra = &(*rva)[i];
		for(i1 = 0, je += m; j < je; ++i1, ++j)
			(*ra)[i1] = r[j];
		}
	delete[] r;
	}

 void NIDRProblemDescDB::
method_pint(const char *keyname, Values *val, void **g, void *v)
{
	int n = *val->i;
	if (n <= 0)
		botch("%s must be positive", keyname);
	(*(Meth_Info**)g)->dme->**(int DataMethodRep::**)v = n;
	}

 void NIDRProblemDescDB::
method_pintz(const char *keyname, Values *val, void **g, void *v)
{
	int n = *val->i;
#if 0
	if (n <= 0)
		botch("%s must be positive", keyname);
#endif
	(*(Meth_Info**)g)->dme->**(size_t DataMethodRep::**)v = n;
	}

 void NIDRProblemDescDB::
method_resplevs(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	Real *r = val->r;
	RealVector *ra;
	RealVectorArray *rva = &(dm->**(RealVectorArray DataMethodRep::**)v);
	size_t i, n = val->n;

	(*rva).reshape(1);
	ra = &(*rva)[0];
	ra->reshape(n);
	for(i = 0; i < n; ++i)
		(*ra)[i] = r[i];
	}

 void NIDRProblemDescDB::
method_resplevs01(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	Real *r = val->r, t;
	RealVector *ra;
	RealVectorArray *rva = &(dm->**(RealVectorArray DataMethodRep::**)v);
	size_t i, n = val->n;

	(*rva).reshape(1);
	ra = &(*rva)[0];
	ra->reshape(n);
	for(i = 0; i < n; ++i) {
		(*ra)[i] = t = r[i];
		if (t < 0. || t > 1.)
			botch("%s must be between 0 and 1", keyname);
		}
	}

 void NIDRProblemDescDB::
method_shint(const char *keyname, Values *val, void **g, void *v)
{
	(*(Meth_Info**)g)->dme->**(short DataMethodRep::**)v = *val->i;
	}

 void NIDRProblemDescDB::
method_ushint(const char *keyname, Values *val, void **g, void *v)
{
	(*(Meth_Info**)g)->dme->**(unsigned short DataMethodRep::**)v = *val->i;
	}

 void NIDRProblemDescDB::
method_ushintL(const char *keyname, Values *val, void **g, void *v)
{
	UShortArray *usa
	  = &((*(Meth_Info**)g)->dme->**(UShortArray DataMethodRep::**)v);
	int *z = val->i;
	size_t i, n = val->n;

	usa->reshape(n);
	for(i = 0; i < n; i++)
		(*usa)[i] = z[i];
	}

 void NIDRProblemDescDB::
method_slit(const char *keyname, Values *val, void **g, void *v)
{
	(*(Meth_Info**)g)->dme->*((Method_mp_slit*)v)->sp = ((Method_mp_slit*)v)->lit;
	}

 void NIDRProblemDescDB::
method_slit2(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;

	dm->*((Method_mp_slit2*)v)->sp  = ((Method_mp_slit2*)v)->lit;
	dm->*((Method_mp_slit2*)v)->sp2 = *val->s;
	}

 void NIDRProblemDescDB::
method_soga_begin(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;
	dm->methodName = "soga";
	dm->constraintPenalty = 1.;
	dm->crossoverType = "multi_point_parameterized_binary";
	dm->fitnessType = "";
	dm->mutationType = "";
	dm->randomSeed = -1;	// SOGA should "randomly" choose a seed.
	}

 void NIDRProblemDescDB::
method_soga_final(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm = (*(Meth_Info**)g)->dme;

	if (dm->fitnessType == "") {
		dm->fitnessType = "merit_function";
		dm->constraintPenalty = 1.0;
		}
	if (dm->mutationType == "") {
		dm->mutationType == "replace_uniform";
		dm->mutationRate = 0.1;
		}
	}

 void NIDRProblemDescDB::
method_start(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep *dm;
	Meth_Info *mi;

	if (!(mi = new Meth_Info))
 Botch:		botch("new failure in method_start");
	if (!(mi->dme0 = new DataMethod))
		goto Botch;
	dm = mi->dme = mi->dme0->dataMethodRep;
	*g = (void*)mi;
	dm->maxIterations = 100;
	dm->maxFunctionEvaluations = 1000;
	}

 static void
scale_chk(StringArray &ST, RealVector &S, const char *what, const char **univ)
{
	const char *s, **u;
	size_t i, n, nbad, vseen;

	n = ST.length();
	for(i = nbad = vseen = 0; i < n; ++i) {
		s = ST[i].data();
		if (!strcmp(s,"value")) {
			++vseen;
			goto break2;
			}
		for(u = univ; *u; ++u)
			if (!strcmp(s,*u)) {
				goto break2;
				}
		NIDRProblemDescDB::squawk("\"%s\" cannot appear in %s_scale_types", s, what);
		++nbad;
 break2:	;
		}
	if (vseen && S.length() <= 0)
		NIDRProblemDescDB::squawk(
			"\"value\" in %s_scale_types requires at least one value for %s_scales",
			what, what);
	}

 static const char *aln_scaletypes[] = { "auto", "log", "none", 0 };

 void NIDRProblemDescDB::
method_stop(const char *keyname, Values *val, void **g, void *v)
{
	static const char *mr_scaletypes[] = { "auto", "none", 0 };
	Meth_Info *mi = *(Meth_Info**)g;
	DataMethodRep *dm = mi->dme;
	scale_chk(dm->linearIneqScaleTypes, dm->linearIneqScales,
		"linear_inequality", mr_scaletypes);
	scale_chk(dm->linearEqScaleTypes, dm->linearEqScales,
		"linear_equality", mr_scaletypes);
	pDDBInstance->dataMethodList.insert(*mi->dme0);
	delete mi->dme0;
	delete mi;
	}

 void NIDRProblemDescDB::
method_str(const char *keyname, Values *val, void **g, void *v)
{
	(*(Meth_Info**)g)->dme->**(String DataMethodRep::**)v = *val->s;
	}

 void NIDRProblemDescDB::
method_strL(const char *keyname, Values *val, void **g, void *v)
{
	StringArray *sa = &((*(Meth_Info**)g)->dme->**(StringArray DataMethodRep::**)v);
	const char **s = val->s;
	size_t i, n = val->n;

	sa->reshape(n);
	for(i = 0; i < n; i++)
		(*sa)[i] = s[i];
	}

 void NIDRProblemDescDB::
method_true(const char *keyname, Values *val, void **g, void *v)
{
	(*(Meth_Info**)g)->dme->**(bool DataMethodRep::**)v = true;
	}

 void NIDRProblemDescDB::
model_Real(const char *keyname, Values *val, void **g, void *v)
{
	(*(Mod_Info**)g)->dmo->**(Real DataModelRep::**)v = *val->r;
	}

 void NIDRProblemDescDB::
model_RealL(const char *keyname, Values *val, void **g, void *v)
{
	Real *r = val->r;
	RealVector *rv = &((*(Mod_Info**)g)->dmo->**(RealVector DataModelRep::**)v);
	size_t i, n = val->n;

	rv->reshape(n);
	for(i = 0; i < n; i++)
		(*rv)[i] = r[i];
	}

 void NIDRProblemDescDB::
model_RealDL(const char *keyname, Values *val, void **g, void *v)
{
	Real *r = val->r;
	RealDenseVector *rdv = &((*(Mod_Info**)g)->dmo->**(RealDenseVector DataModelRep::**)v);
	size_t i, n = val->n;

	rdv->sizeUninitialized(n);
	for(i = 0; i < n; i++)
		(*rdv)[i] = r[i];
	}

 void NIDRProblemDescDB::
model_intset(const char *keyname, Values *val, void **g, void *v)
{
	IntSet *is = &((*(Mod_Info**)g)->dmo->**(IntSet DataModelRep::**)v);
	int *z = val->i;
	size_t i, n = val->n;

	for(i = 0; i < n; i++)
		is->insert(z[i]-1);
	}

 void NIDRProblemDescDB::
model_lit(const char *keyname, Values *val, void **g, void *v)
{
	(*(Mod_Info**)g)->dmo->*((Model_mp_lit*)v)->sp = ((Model_mp_lit*)v)->lit;
	}

 void NIDRProblemDescDB::
model_order(const char *keyname, Values *val, void **g, void *v)
{
	(*(Mod_Info**)g)->dmo->*((Model_mp_ord*)v)->sp = ((Model_mp_ord*)v)->ord;
	}

 void NIDRProblemDescDB::
model_shint(const char *keyname, Values *val, void **g, void *v)
{
	(*(Mod_Info**)g)->dmo->**(short DataModelRep::**)v = (short)*val->i;
	}

//  void NIDRProblemDescDB::
// model_slit2(const char *keyname, Values *val, void **g, void *v)
// {
// 	DataModelRep *dm = (*(Mod_Info**)g)->dmo;
//
// 	dm->*((Model_mp_slit2*)v)->sp  = ((Model_mp_slit2*)v)->lit;
// 	dm->*((Model_mp_slit2*)v)->sp2 = *val->s;
// 	}

 void NIDRProblemDescDB::
model_start(const char *keyname, Values *val, void **g, void *v)
{
	DataModelRep *dm;
	Mod_Info *mi;

	if (!(mi = new Mod_Info))
 Botch:		botch("new failure in model_start");
	if (!(mi->dmo0 = new DataModel))
		goto Botch;
	dm = mi->dmo = mi->dmo0->dataModelRep;
	*g = (void*)mi;
	}

 void NIDRProblemDescDB::
model_stop(const char *keyname, Values *val, void **g, void *v)
{
	Mod_Info *mi = *(Mod_Info**)g;
	pDDBInstance->dataModelList.insert(*mi->dmo0);
	delete mi->dmo0;
	delete mi;
	}

 void NIDRProblemDescDB::
model_str(const char *keyname, Values *val, void **g, void *v)
{
	(*(Mod_Info**)g)->dmo->**(String DataModelRep::**)v = *val->s;
	}

 void NIDRProblemDescDB::
model_strL(const char *keyname, Values *val, void **g, void *v)
{
	DataModelRep *dm = (*(Mod_Info**)g)->dmo;
	StringArray DataModelRep::* sa = *(StringArray DataModelRep::**)v;
	const char **s = val->s;
	size_t i, n = val->n;

	(dm->*sa).reshape(n);
	for(i = 0; i < n; i++)
		(dm->*sa)[i] = s[i];
	}

 void NIDRProblemDescDB::
model_true(const char *keyname, Values *val, void **g, void *v)
{
	(*(Mod_Info**)g)->dmo->**(bool DataModelRep::**)v = true;
	}

 void NIDRProblemDescDB::
resp_RealDL(const char *keyname, Values *val, void **g, void *v)
{
	Real *r = val->r;
	RealDenseVector *rv = &((*(Resp_Info**)g)->dr->**(RealDenseVector DataResponsesRep::**)v);
	size_t i, n = val->n;

	rv->sizeUninitialized(n);
	for(i = 0; i < n; i++)
		(*rv)[i] = r[i];
	}

 void NIDRProblemDescDB::
resp_RealL(const char *keyname, Values *val, void **g, void *v)
{
	Real *r = val->r;
	RealVector *rv = &((*(Resp_Info**)g)->dr->**(RealVector DataResponsesRep::**)v);
	size_t i, n = val->n;

	rv->reshape(n);
	for(i = 0; i < n; i++)
		(*rv)[i] = r[i];
	}

 void NIDRProblemDescDB::
resp_intL(const char *keyname, Values *val, void **g, void *v)
{
	DataResponsesRep *dr = (*(Resp_Info**)g)->dr;
	int *z = val->i;
	IntList *il = &(dr->**(IntList DataResponsesRep::**)v);
	size_t i, n = val->n;

	for(i = 0; i < n; i++)
		il->insert(z[i]);
	}

 void NIDRProblemDescDB::
resp_lit(const char *keyname, Values *val, void **g, void *v)
{
	(*(Resp_Info**)g)->dr->*((Resp_mp_lit*)v)->sp = ((Resp_mp_lit*)v)->lit;
	}

 void NIDRProblemDescDB::
resp_nnintz(const char *keyname, Values *val, void **g, void *v)
{
	int n = *val->i;
#if 0
	if (n < 0)
		botch("%s must be non-negative", keyname);
#endif
	(*(Resp_Info**)g)->dr->**(size_t DataResponsesRep::**)v = n;
	}

 void NIDRProblemDescDB::
resp_start(const char *keyname, Values *val, void **g, void *v)
{
	Resp_Info *ri;

	if (!(ri = new Resp_Info))
 Botch:		botch("new failure in resp_start");
	if (!(ri->dr0 = new DataResponses))
		goto Botch;
	ri->dr = ri->dr0->dataRespRep;
	*g = (void*)ri;
	}

 static void
BuildLabels(StringArray *sa, size_t nsa, size_t n1, size_t n2, const char *stub)
{
	char buf[64];
	size_t i, n0;
	if (nsa)
		sa->reshape(nsa);
	i = n0 = n1;
	while(n1 < n2) {
		sprintf(buf, "%s%lu", stub, (UL)(++n1 - n0));
		(*sa)[i++] = buf;
		}
	}

 static int
flist_check(IntList *L, int n, IntVector *iv, const char *what)
{
	int nbad, j;

	nbad = 0;
	for(ILCIter it = L->begin(), ite = L->end(); it != ite; ++it) {
		j = *it;
		if (j < 1 || j > n) {
			if (!nbad++)
			 Squawk("%s values must be between 1 and %d", what, n);
			}
		else
			++(*iv)[j-1];
		}
	return nbad;
	}

 static void
flist_check2(size_t n, IntVector *iv, const char *what)
{
	int j;
	size_t i;

	for(i = 0; i < n; ) {
		j = (*iv)[i++];
		if (j == 0)
			Squawk("Function %lu missing from mixed %s lists", (UL)i);
		else if (j > 1)
			Squawk("Function %lu replicated in mixed %s lists", (UL)i);
		}
	}

 void NIDRProblemDescDB::
resp_stop(const char *keyname, Values *val, void **g, void *v)
{
	static const char *osc[] = { "log", "none", 0 };
	Resp_Info *ri = *(Resp_Info**)g;
	DataResponsesRep *dr = ri->dr;
	scale_chk(dr->primaryRespFnScaleTypes, dr->primaryRespFnScales,
		dr->numLeastSqTerms ? "least_squares_term" : "objective_function", osc);
	scale_chk(dr->nonlinearIneqScaleTypes, dr->nonlinearIneqScales,
		"nonlinear_inequality", aln_scaletypes);
	scale_chk(dr->nonlinearEqScaleTypes, dr->nonlinearEqScales,
		"nonlinear_equality", aln_scaletypes);
	pDDBInstance->dataResponsesList.insert(*ri->dr0);
	delete ri->dr0;
	delete ri;
	}

 void NIDRProblemDescDB::
post_process_response_data(DataResponses *dr0)
{
	DataResponsesRep *dr = dr0->dataRespRep;
	IntVector *iv;
	StringArray *rl;
	int ni;
	size_t i, n, n1, nf, no, nrl;
	struct RespStr_chk {
		const char *what;
		size_t DataResponsesRep::* n;
		StringArray DataResponsesRep::* sa;
		} *sc;
	struct RespDVec_chk {
		const char *what;
		size_t DataResponsesRep::* n;
		RealDenseVector DataResponsesRep::* rv;
		} *rdvc;
	struct RespVec_chk {
		const char *what;
		size_t DataResponsesRep::* n;
		RealVector DataResponsesRep::* rv;
		} *rvc;
	#define Schk(a,b,c) {#a,&DataResponsesRep::b, &DataResponsesRep::c}
	static RespStr_chk Str_chk[] = {  // for StringArray checking
		Schk(least_squares_term_scale_types,numLeastSqTerms,primaryRespFnScaleTypes),
		Schk(nonlinear_equality_scale_types,numNonlinearEqConstraints,nonlinearEqScaleTypes),
		Schk(nonlinear_inequality_scale_types,numNonlinearIneqConstraints,nonlinearIneqScaleTypes),
		Schk(objective_function_scale_types,numObjectiveFunctions,primaryRespFnScaleTypes)
		};
	static RespDVec_chk RespVec_chk_Bound[] = {// Bounds:  length must be right
    	        Schk(least_squares_weights,numLeastSqTerms,primaryRespFnWeights),
		Schk(multi_objective_weights,numObjectiveFunctions,primaryRespFnWeights),
		Schk(nonlinear_equality_targets,numNonlinearEqConstraints,nonlinearEqTargets),
		Schk(nonlinear_inequality_lower_bounds,numNonlinearIneqConstraints,nonlinearIneqLowerBnds),
		Schk(nonlinear_inequality_upper_bounds,numNonlinearIneqConstraints,nonlinearIneqUpperBnds)
		};
	static RespVec_chk RespVec_chk_Scale[] = {// Scales:  length must be 1 or the right length
		Schk(least_squares_term_scales,numLeastSqTerms,primaryRespFnScales),
		Schk(nonlinear_equality_scales,numNonlinearEqConstraints,nonlinearEqScales),
		Schk(nonlinear_inequality_scales,numNonlinearIneqConstraints,nonlinearIneqScales),
		Schk(objective_function_scales,numObjectiveFunctions,primaryRespFnScales)
		};
	#undef Schk
	#define Numberof(x) sizeof(x)/sizeof(x[0])

	for(sc = Str_chk, i = 0; i < Numberof(Str_chk); ++sc, ++i)
		if ((n1 = dr->*sc->n) && (n = (dr->*sc->sa).length()) > 0
			&& n != n1 && n != 1)
				squawk("%s must have length 1 or %lu, not %lu",
					sc->what, (UL)n1, (UL)n);
	rl = &dr->responseLabels;
	nrl = rl->length();
	no = dr->numObjectiveFunctions;
	if ((n = dr->numLeastSqTerms)) {
		nf = n + dr->numNonlinearEqConstraints + dr->numNonlinearIneqConstraints;
		if (!nrl) {
			BuildLabels(rl, nf, 0, n, "least_sq_term_");
			n1 = n + dr->numNonlinearIneqConstraints;
			BuildLabels(rl, 0, n, n1, "nln_ineq_con_");
			BuildLabels(rl, 0, n1, nf, "nln_eq_con_");
			}
		}
	else if ((nf = no + dr->numNonlinearEqConstraints + dr->numNonlinearIneqConstraints)) {
		if (!nrl) {
			rl->reshape(nf);
			if (no == 1)
				(*rl)[0] = "obj_fn";
			else
				BuildLabels(rl, 0, 0, no, "obj_fn_");
			n1 = no + dr->numNonlinearIneqConstraints;
			BuildLabels(rl, 0, no, n1, "nln_ineq_con_");
			BuildLabels(rl, 0, n1, nf, "nln_eq_con_");
			}
		}
	else if ((nf = dr->numResponseFunctions)) {
		if (!nrl)
			BuildLabels(rl, nf, 0, nf, "response_fn_");
		}
	for(rdvc = RespVec_chk_Bound, i = 0; i < Numberof(RespVec_chk_Bound); ++rdvc, ++i)
		if ((n1 = dr->*rdvc->n) && (n = (dr->*rdvc->rv).length()) && n != n1)
			squawk("%s needs %lu elements, not %lu",
				rdvc->what, (UL)(dr->*rdvc->n), (UL)n);
	for(rvc = RespVec_chk_Scale, i = 0; i < Numberof(RespVec_chk_Scale); ++rvc, ++i)
		if ((n1 = dr->*rvc->n) && (n = (dr->*rvc->rv).length())
		 && n != n1 && n != 1)
			squawk("%s needs %lu elements (or just one), not %lu",
				rvc->what, (UL)n1, (UL)n);
	if (dr->methodSource == "vendor" && dr->fdGradStepSize.length() > 1)
		squawk("vendor numerical gradients only support a single fd_gradient_step_size");

	ni = (int)nf;
	if (dr->gradientType == "mixed") {
		iv = new IntVector;
		iv->reshape(nf);
		*iv = 0;
		if (!(flist_check(&dr->idAnalyticGrads, ni, iv, "id_analytic_gradients")
		    + flist_check(&dr->idNumericalGrads, ni, iv, "id_numerical_gradients")))
			flist_check2(nf, iv, "gradient");
		delete iv;
		}
	if ((n = dr->numNonlinearEqConstraints) > 0
	 && dr->nonlinearEqTargets.length() == 0) {
		dr->nonlinearEqTargets.sizeUninitialized(n);
		dr->nonlinearEqTargets = 0.;
		}
	if ((n = dr->numNonlinearIneqConstraints) > 0) {
		if (dr->nonlinearIneqLowerBnds.length() == 0) {
			dr->nonlinearIneqLowerBnds.sizeUninitialized(n);
			dr->nonlinearIneqLowerBnds = -DBL_MAX;
			}
		if (dr->nonlinearIneqUpperBnds.length() == 0) {
			dr->nonlinearIneqUpperBnds.sizeUninitialized(n);
			dr->nonlinearIneqUpperBnds = 0.;
			// default is a one-sided inequality <= 0.0
			}
		}
	if (dr->hessianType == "mixed") {
		iv = new IntVector;
		iv->reshape(nf);
		*iv = 0;
		if (!(flist_check(&dr->idAnalyticHessians, ni, iv, "id_analytic_hessians")
		    + flist_check(&dr->idNumericalHessians, ni, iv, "id_numerical_hessians")
		    + flist_check(&dr->idQuasiHessians, ni, iv, "id_quasi_hessians")))
			flist_check2(nf, iv, "Hessian");
		delete iv;
		}
	if (nerr)
		abort_handler(-1);
	}

 void NIDRProblemDescDB::
resp_str(const char *keyname, Values *val, void **g, void *v)
{
	(*(Resp_Info**)g)->dr->**(String DataResponsesRep::**)v = *val->s;
	}

 void NIDRProblemDescDB::
resp_strL(const char *keyname, Values *val, void **g, void *v)
{
	StringArray *sa = &((*(Resp_Info**)g)->dr->**(StringArray DataResponsesRep::**)v);
	const char **s = val->s;
	size_t i, n = val->n;

	sa->reshape(n);
	for(i = 0; i < n; i++)
		(*sa)[i] = s[i];
	}

 void NIDRProblemDescDB::
resp_false(const char *keyname, Values *val, void **g, void *v)
{
	(*(Resp_Info**)g)->dr->**(bool DataResponsesRep::**)v = false;
	}

 void NIDRProblemDescDB::
resp_true(const char *keyname, Values *val, void **g, void *v)
{
	(*(Resp_Info**)g)->dr->**(bool DataResponsesRep::**)v = true;
	}

 void NIDRProblemDescDB::
strategy_Real(const char *keyname, Values *val, void **g, void *v)
{
	(*(DataStrategyRep**)g)->**(Real DataStrategyRep::**)v = *val->r;
	}

 void NIDRProblemDescDB::
strategy_RealL(const char *keyname, Values *val, void **g, void *v)
{
	Real *r = val->r;
	RealDenseVector *rdv = &((*(DataStrategyRep**)g)->**(RealDenseVector DataStrategyRep::**)v);
	size_t i, n = val->n;

	rdv->sizeUninitialized(n);
	for(i = 0; i < n; i++)
		(*rdv)[i] = r[i];
	}

 void NIDRProblemDescDB::
strategy_true(const char *keyname, Values *val, void **g, void *v)
{
	(*(DataStrategyRep**)g)->**(bool DataStrategyRep::**)v = true;
	}

 void NIDRProblemDescDB::
strategy_int(const char *keyname, Values *val, void **g, void *v)
{
	(*(DataStrategyRep**)g)->**(int DataStrategyRep::**)v = *val->i;
	}

 void NIDRProblemDescDB::
strategy_lit(const char *keyname, Values *val, void **g, void *v)
{
	(*(DataStrategyRep**)g)->*((Strategy_mp_lit*)v)->sp = ((Strategy_mp_lit*)v)->lit;
	}

 void NIDRProblemDescDB::
strategy_slit(const char *keyname, Values *val, void **g, void *v)
{
	(*(DataStrategyRep**)g)->*((Strategy_mp_slit*)v)->sp = ((Strategy_mp_slit*)v)->lit;
	}

 void NIDRProblemDescDB::
strategy_start(const char *keyname, Values *val, void **g, void *v)
{
	*g = (void*)pDDBInstance->strategySpec.dataStratRep;
	}

 void NIDRProblemDescDB::
strategy_str(const char *keyname, Values *val, void **g, void *v)
{
	(*(DataStrategyRep**)g)->**(String DataStrategyRep::**)v = *val->s;
	}

 void NIDRProblemDescDB::
strategy_strL(const char *keyname, Values *val, void **g, void *v)
{
	StringArray *sa = &((*(DataStrategyRep**)g)->**(StringArray DataStrategyRep::**)v);
	const char **s = val->s;
	size_t i, n = val->n;

	sa->reshape(n);
	for(i = 0; i < n; i++)
		(*sa)[i] = s[i];
	}

 void NIDRProblemDescDB::
method_tr_final(const char *keyname, Values *val, void **g, void *v)
{
	DataMethodRep &data_method = *(*(Meth_Info**)g)->dme;

      // sanity checks on trust region user-defined values
      if ( data_method.surrBasedLocalTRInitSize <= 0.0 ||
	   data_method.surrBasedLocalTRInitSize >  1.0 )
		botch("initial_size must be in (0,1]");
      if ( data_method.surrBasedLocalTRMinSize <= 0.0 ||
	   data_method.surrBasedLocalTRMinSize >  1.0 ||
	   data_method.surrBasedLocalTRMinSize >
	   data_method.surrBasedLocalTRInitSize )
		botch("minimum_size must be in (0,1]");
      if( data_method.surrBasedLocalTRContractTrigger <= 0.0 ||
	  data_method.surrBasedLocalTRContractTrigger >
	    data_method.surrBasedLocalTRExpandTrigger        ||
	  data_method.surrBasedLocalTRExpandTrigger   >  1.0 )
		botch("expand/contract threshold values must satisfy\n\t"
			"0 < contract_threshold <= expand_threshold < 1");
      if ( data_method.surrBasedLocalTRContract == 1.0 )
		warn("contraction_factor = 1.0 is valid, but should be < 1\n\t"
			"to assure convergence of the surrrogate_based_opt method");
      if ( data_method.surrBasedLocalTRContract <= 0.0 ||
	   data_method.surrBasedLocalTRContract >  1.0 )
		botch("contraction_factor must be in (0,1]");
      if (data_method.surrBasedLocalTRExpand < 1.0)
		botch("expansion_factor must be >= 1");
    }


 void NIDRProblemDescDB::
var_RealL(const char *keyname, Values *val, void **g, void *v)
{
	Real *r = val->r;
	RealVector *rv = &((*(Var_Info**)g)->dv->**(RealVector DataVariablesRep::**)v);
	size_t i, n = val->n;

	rv->reshape(n);
	for(i = 0; i < n; i++)
		(*rv)[i] = r[i];
	}


 void NIDRProblemDescDB::
var_RealLd(const char *keyname, Values *val, void **g, void *v)
{
	Real *r = val->r;
	RealDenseVector *rv = &((*(Var_Info**)g)->dv->**(RealDenseVector DataVariablesRep::**)v);
	size_t i, n = val->n;

	rv->sizeUninitialized(n);
	for(i = 0; i < n; i++)
		(*rv)[i] = r[i];
	}

 void NIDRProblemDescDB::
var_RealLb(const char *keyname, Values *val, void **g, void *v)
{
	Real b, *r;
	RealDenseVector *rv;
	Var_brv *V;
	size_t i, n;

	V = (Var_brv*)v;
	rv = &((*(Var_Info**)g)->dv->*V->rv);
	b = V->b;
	r = val->r;
	n = val->n;
	for(i = 0; i < n; i++)
		if (r[i] <= b) {
			squawk("%s values must be > %g", keyname, b);
			break;
			}
	rv->sizeUninitialized(n);
	for(i = 0; i < n; i++)
		(*rv)[i] = r[i];
	}

 void NIDRProblemDescDB::
var_RealUb(const char *keyname, Values *val, void **g, void *v)
{
	Real b, *r;
	RealDenseVector *rv;
	Var_brv *V;
	size_t i, n;

	V = (Var_brv*)v;
	rv = &((*(Var_Info**)g)->dv->*V->rv);
	b = V->b;
	r = val->r;
	n = val->n;
	for(i = 0; i < n; i++)
		if (r[i] >= b) {
			squawk("%s values must be < %g", keyname, b);
			break;
			}
	rv->sizeUninitialized(n);
	for(i = 0; i < n; i++)
		(*rv)[i] = r[i];
	}

 void NIDRProblemDescDB::
var_hbp(const char *keyname, Values *val, void **g, void *v)
{
	Var_Info *vi = *(Var_Info**)g;
	Real *r;
	RealDenseVector *rv;
	size_t i, n = val->n;

	if (!(rv = new RealDenseVector))
		botch("new failure in var_hbp");
	vi->**(RealDenseVector *Var_Info::**)v = rv;
	rv->sizeUninitialized(n);
	r = val->r;
	for(i = 0; i < n; i++)
		(*rv)[i] = r[i];
	}

 void NIDRProblemDescDB::
var_intDL(const char *keyname, Values *val, void **g, void *v)
{
	IntDenseVector *iv = &((*(Var_Info**)g)->dv->**(IntDenseVector DataVariablesRep::**)v);
	int *z = val->i;
	size_t i, n = val->n;

	iv->sizeUninitialized(n);
	for(i = 0; i < n; i++)
		(*iv)[i] = z[i];
	}

 void NIDRProblemDescDB::
var_intL(const char *keyname, Values *val, void **g, void *v)
{
	IntVector *iv = &((*(Var_Info**)g)->dv->**(IntVector DataVariablesRep::**)v);
	int *z = val->i;
	size_t i, n = val->n;

	iv->reshape(n);
	for(i = 0; i < n; i++)
		(*iv)[i] = z[i];
	}

 void NIDRProblemDescDB::
var_intz(const char *keyname, Values *val, void **g, void *v)
{
	int n = *val->i;
#if 0		/* now handled in the .nspec file */
	if (n <= 0)
		botch("%s must be positive", keyname);
#endif
	(*(Var_Info**)g)->dv->**(size_t DataVariablesRep::**)v = n;
	}

 void NIDRProblemDescDB::
var_nbp(const char *keyname, Values *val, void **g, void *v)
{
	Var_Info *vi = *(Var_Info**)g;
	Real *r;
	RealVector *rv;
	size_t i, n = val->n;

	if (!(rv = new RealVector))
		botch("new failure in var_nbp");
	vi->**(RealVector *Var_Info::**)v = rv;
	rv->reshape(n);
	r = val->r;
	for(i = 0; i < n; i++)
		(*rv)[i] = r[i];
	}

 void NIDRProblemDescDB::
var_nhbp(const char *keyname, Values *val, void **g, void *v)
{
	Var_Info *vi = *(Var_Info**)g;
	IntVector *iv;
	int *z;
	size_t i, n = val->n;

	if (!(iv = new IntVector))
		botch("new failure in var_nhbp");
	vi->**(IntVector *Var_Info::**)v = iv;
	iv->reshape(n);
	z = val->i;
	for(i = 0; i < n; i++)
		(*iv)[i] = z[i];
	}

 void NIDRProblemDescDB::
var_start(const char *keyname, Values *val, void **g, void *v)
{
	Var_Info *vi;

	if (!(vi = new Var_Info))
 Botch:		botch("new failure in var_start");
	memset(vi, 0, sizeof(Var_Info));
	if (!(vi->dv0 = new DataVariables))
		goto Botch;
	vi->dv = vi->dv0->dataVarsRep;
	*g = (void*)vi;
	}

 void NIDRProblemDescDB::
var_true(const char *keyname, Values *val, void **g, void *v)
{
	(*(Var_Info**)g)->dv->**(bool DataVariablesRep::**)v = true;
	}

 static int
wronglen(size_t n, RealDenseVector *V, const char *what)
{
	size_t n1 = V->length();
	if (n != n1) {
		Squawk("Expected %d numbers for %s, but got %d", (int)n, what, (int)n1);
		return 1;
		}
	return 0;
	}

 static void
Vcopyup(RealDenseVector *V, RealDenseVector *M, size_t i, size_t n)
{
	size_t j;
	for(j = 0; j < n; ++i, ++j)
		(*V)[i] = (*M)[j];
	}

 static void
Set_rdv(RealDenseVector *V, double d, size_t n)
{
	size_t i;
	V->sizeUninitialized(n);
	for(i = 0; i < n; ++i)
		(*V)[i] = d;
	}

 static void
Vadj_Normal(DataVariablesRep *dv, size_t i0, Var_Info *vi)
{
	size_t j, n;
	RealDenseVector *B, *M, *Sd, *V;

	n = dv->numNormalUncVars;
	if (wronglen(n, M = &dv->normalUncMeans, "nuv_means")
	 || wronglen(n, Sd = &dv->normalUncStdDevs, "nuv_std_deviations"))
		return;
	Vcopyup(&dv->uncertainVars, M, i0, n);
	B = &dv->normalUncLowerBnds;
	if (!B->length()) {
		Set_rdv(B, -DBL_MAX, n);
		V = &dv->uncertainLowerBnds;
		for(j = 0; j < n; ++j)
			(*V)[i0+j] = (*M)[j] - 3.*(*Sd)[j];
		}
	else if (wronglen(n, B, "nuv_lower_bounds"))
		return;
	else
		Vcopyup(&dv->uncertainLowerBnds, B, i0, n);
	B = &dv->normalUncUpperBnds;
	if (!B->length()) {
		Set_rdv(B, DBL_MAX, n);
		V = &dv->uncertainUpperBnds;
		for(j = 0; j < n; ++j)
			(*V)[i0+j] = (*M)[j] + 3.*(*Sd)[j];
		}
	else if (wronglen(n, B, "nuv_upper_bounds"))
		return;
	else
		Vcopyup(&dv->uncertainUpperBnds, B, i0, n);
	}

 static void
Vadj_Lognormal(DataVariablesRep *dv, size_t i0, Var_Info *vi)
{
	size_t i, j, n, nsd;
	Real mean, stdev, zeta;
	RealDenseVector *B, *Ef, *M, *Sd, *U;

	n = dv->numLognormalUncVars;
	if (wronglen(n, M = &dv->lognormalUncMeans, "lnuv_means"))
		return;
	Sd = &dv->lognormalUncStdDevs;
	nsd = Sd->length();
	if (nsd > 0) {
		if (wronglen(n, Sd, "lnuv_std_deviations"))
			return;
		}
	else if (wronglen(n, Ef = &dv->lognormalUncErrFacts, "lnuv_error_factors"))
		return;
	Vcopyup(&dv->uncertainVars, M, i0, n);
	B = &dv->lognormalUncLowerBnds;
	if (!B->length())
		B->size(n);	// sets B to all zeros
	else if (wronglen(n, B, "lnuv_lower_bounds"))
		return;
	Vcopyup(&dv->uncertainLowerBnds, B, i0, n);
	B = &dv->lognormalUncUpperBnds;
	U = &dv->uncertainUpperBnds;
	if (!B->length()) {
		Set_rdv(B, DBL_MAX, n);
		if (nsd > 0)
			for(i = i0, j = 0; j < n; ++i, ++j)
				(*U)[i] = (*M)[j] + 3.*(*Sd)[j];
		else
			for(i = i0, j = 0; j < n; ++i, ++j) {
				zeta = log((*Ef)[i])/1.645;
				mean = (*M)[j];
				stdev = mean*sqrt(exp(zeta*zeta)-1.);
				(*U)[i] = mean + 3.*stdev;
				}
		}
	else if (wronglen(n, B, "lnuv_upper_bounds"))
		return;
	else
		Vcopyup(U, B, i0, n);
	}

 static void
Vadj_Uniform(DataVariablesRep *dv, size_t i0, Var_Info *vi)
{
	size_t i, j, n;
	RealDenseVector *L, *U, *V;

	n = dv->numUniformUncVars;
	L = &dv->uniformUncLowerBnds;
	U = &dv->uniformUncUpperBnds;
	if (wronglen(n, L, "uuv_lower_bounds")
	 || wronglen(n, U, "uuv_upper_bounds"))
		return;
	Vcopyup(&dv->uncertainLowerBnds, L, i0, n);
	Vcopyup(&dv->uncertainUpperBnds, U, i0, n);
	V = &dv->uncertainVars;
	for(i = i0, j = 0; j < n; ++i, ++j)
		(*V)[i] = 0.5 * ((*L)[j] + (*U)[j]);
	}

 static void
Vadj_Loguniform(DataVariablesRep *dv, size_t i0, Var_Info *vi)
{
	size_t i, j, n;
	Real Lj, Uj, t;
	RealDenseVector *L, *U, *V;

	n = dv->numLoguniformUncVars;
	L = &dv->loguniformUncLowerBnds;
	U = &dv->loguniformUncUpperBnds;
	if (wronglen(n, L, "luuv_lower_bounds")
	 || wronglen(n, U, "luuv_upper_bounds"))
		return;
	Vcopyup(&dv->uncertainLowerBnds, L, i0, n);
	Vcopyup(&dv->uncertainUpperBnds, U, i0, n);
	V = &dv->uncertainVars;
	for(i = i0, j = 0; j < n; ++i, ++j) {
		t = Lj = (*L)[j];
		Uj = (*U)[j];
		if (Uj > Lj) {
			if (Uj >= DBL_MAX) {
				Squawk("luuv_upper_bounds must be finite");
				return;
				}
			t = (Uj - Lj)/(log(Uj) - log(Lj));
			}
		(*V)[i] = t;
		}
	}

 static void
Vadj_Triangular(DataVariablesRep *dv, size_t i0, Var_Info *vi)
{
	size_t i, j, n;
	Real Lj, Mj, Uj;
	RealDenseVector *L, *M, *U, *V;

	n = dv->numTriangularUncVars;
	L = &dv->triangularUncLowerBnds;
	M = &dv->triangularUncModes;
	U = &dv->triangularUncUpperBnds;
	if (wronglen(n, L, "tuv_lower_bounds")
	 || wronglen(n, M, "tuv_modes")
	 || wronglen(n, U, "tuv_upper_bounds"))
		return;
	Vcopyup(&dv->uncertainLowerBnds, L, i0, n);
	Vcopyup(&dv->uncertainUpperBnds, U, i0, n);
	V = &dv->uncertainVars;
	for(i = i0, j = 0; j < n; ++i, ++j) {
		Lj = (*L)[j];
		Mj = (*M)[j];
		Uj = (*U)[j];
		if (Lj > Mj || Mj > Uj) {
			Squawk("triangular uncertain variables must have\n\t"
				"tuv_lower_bounds <= tuv_modes <= tuv_upper_bounds");
			return;
			}
		(*V)[i] = (Lj + Mj + Uj)/3.;
		}
	}

 static void
Vadj_Exponential(DataVariablesRep *dv, size_t i0, Var_Info *vi)
{
	size_t i, j, n;
	Real mean, stdev;
	RealDenseVector *B, *L, *U, *V;

	n = dv->numExponentialUncVars;
	B = &dv->exponentialUncBetas;
	if (wronglen(n, B, "euv_betas"))
		return;
	L = &dv->uncertainLowerBnds;
	U = &dv->uncertainUpperBnds;
	V = &dv->uncertainVars;
	for(i = i0, j = 0; j < n; ++i, ++j) {
		mean = stdev = (*B)[j];
		(*L)[i] = 0.;
		(*U)[i] = mean + 3.*stdev;
		(*V)[i] = mean;
		}
	}

 static void
Vadj_Beta(DataVariablesRep *dv, size_t i0, Var_Info *vi)
{
	size_t i, j, n;
	Real Aj, Bj, Lj, Uj;
	RealDenseVector *A, *B, *L, *U, *V;

	n = dv->numBetaUncVars;
	A = &dv->betaUncAlphas;
	B = &dv->betaUncBetas;
	L = &dv->betaUncLowerBnds;
	U = &dv->betaUncUpperBnds;
	if (wronglen(n, A, "buv_alphas")
	 || wronglen(n, B, "buv_betas")
	 || wronglen(n, L, "buv_lower_bounds")
	 || wronglen(n, U, "buv_upper_bounds"))
		return;
	V = &dv->uncertainVars;
	Vcopyup(&dv->uncertainLowerBnds, L, i0, n);
	Vcopyup(&dv->uncertainUpperBnds, U, i0, n);
	for(i = i0, j = 0; j < n; ++i, ++j) {
		Aj = (*A)[j];
		Bj = (*B)[j];
		Lj = (*L)[j];
		Uj = (*U)[j];
		(*V)[i] = Lj + (Uj-Lj)*Aj/(Aj + Bj);
		}
	}

 static void
Vadj_Gamma(DataVariablesRep *dv, size_t i0, Var_Info *vi)
{
	size_t i, j, n;
	Real alpha, beta, mean, stdev;
	RealDenseVector *A, *B, *L, *U, *V;

	n = dv->numGammaUncVars;
	A = &dv->gammaUncAlphas;
	B = &dv->gammaUncBetas;
	if (wronglen(n, A, "gauv_alphas")
	 || wronglen(n, B, "gauv_betas"))
		return;
	L = &dv->uncertainLowerBnds;
	U = &dv->uncertainUpperBnds;
	V = &dv->uncertainVars;
	for(i = i0, j = 0; j < n; ++i, ++j) {
		alpha = (*A)[j];
		beta  = (*B)[j];
		mean  = alpha * beta;
		stdev = sqrt(alpha)*beta;
		(*L)[i] = 0.;
		(*U)[i] = mean + 3.*stdev;
		(*V)[i] = mean;
		}
	}

 static void
Vadj_Gumbel(DataVariablesRep *dv, size_t i0, Var_Info *vi)
{
	size_t i, j, n;
	Real alpha, beta, mean, stdev;
	RealDenseVector *A, *B, *L, *U, *V;
	static Real c = 0., pi;

	if (c == 0.) {
		pi = 4. * atan(1.);
		c = pi / sqrt(6.);
		}

	n = dv->numGumbelUncVars;
	A = &dv->gumbelUncAlphas;
	B = &dv->gumbelUncBetas;
	if (wronglen(n, A, "guuv_alphas")
	 || wronglen(n, B, "guuv_betas"))
		return;
	L = &dv->uncertainLowerBnds;
	U = &dv->uncertainUpperBnds;
	V = &dv->uncertainVars;
	for(i = i0, j = 0; j < n; ++i, ++j) {
		alpha = (*A)[j];
		beta  = (*B)[j];
		mean  = beta + 0.5772/alpha;
		stdev = c / alpha;
		(*L)[i] = mean - 3.*stdev;
		(*U)[i] = mean + 3.*stdev;
		(*V)[i] = mean;
		}
	}

 static void
Vadj_Frechet(DataVariablesRep *dv, size_t i0, Var_Info *vi)
{
	size_t i, j, n;
	Real alpha, beta, mean, stdev;
	RealDenseVector *A, *B, *L, *U, *V;

	n = dv->numFrechetUncVars;
	A = &dv->frechetUncAlphas;
	B = &dv->frechetUncBetas;
	if (wronglen(n, A, "fuv_alphas")
	 || wronglen(n, B, "fuv_betas"))
		return;
	L = &dv->uncertainLowerBnds;
	U = &dv->uncertainUpperBnds;
	V = &dv->uncertainVars;
	for(i = i0, j = 0; j < n; ++i, ++j) {
		alpha = (*A)[j];
		beta  = (*B)[j];
		Pecos::moments_from_frechet_params(alpha, beta, mean, stdev);
		(*L)[i] = 0;
		(*U)[i] = mean + 3.*stdev;
		(*V)[i] = mean;
		}
	}

 static void
Vadj_Weibull(DataVariablesRep *dv, size_t i0, Var_Info *vi)
{
	size_t i, j, n;
	Real alpha, beta, mean, stdev;
	RealDenseVector *A, *B, *L, *U, *V;

	n = dv->numWeibullUncVars;
	A = &dv->weibullUncAlphas;
	B = &dv->weibullUncBetas;
	if (wronglen(n, A, "wuv_alphas")
	 || wronglen(n, B, "wuv_betas"))
		return;
	L = &dv->uncertainLowerBnds;
	U = &dv->uncertainUpperBnds;
	V = &dv->uncertainVars;
	for(i = i0, j = 0; j < n; ++i, ++j) {
		alpha = (*A)[j];
		beta  = (*B)[j];
		Pecos::moments_from_weibull_params(alpha, beta, mean, stdev);
		(*L)[i] = 0;
		(*U)[i] = mean + 3.*stdev;
		(*V)[i] = mean;
		}
	}

 static void
Vadj_Histogram(DataVariablesRep *dv, size_t i0, Var_Info *vi)
{
	IntVector *nbp, *npp;
	RealDenseVector *bp, *bpi, *pp, *ppi;
	RealDenseVector *L, *U, *V;
	RealDenseVectorArray *hbp, *hpp;
	int ik, totbp2, totpp2;
	size_t i, j, j2, je, k, m, n, nseen, totbp, totpp;

	n = dv->numHistogramUncVars;
	L = &dv->uncertainLowerBnds;
	U = &dv->uncertainUpperBnds;
	V = &dv->uncertainVars;
	nseen = 0;
	if ((nbp = vi->nbp)) {
		nseen += m = nbp->length();
		for(i = totbp = 0; i < m; i++) {
			j = (*nbp)[i];
			if (j < 2) {
				Squawk("huv_num_bin_pairs must be >= 2");
				return;
				}
			totbp += j;
			}
		bp = vi->bp;
		k = bp->length();
		totbp2 = (int)(2*totbp);
		ik = (int)k;
		if (ik != totbp2) {
			Squawk("Expected %d huv_bin_pairs, not %d", totbp2, ik);
			return;
			}
		hbp = &dv->histogramUncBinPairs;
		hbp->reshape(m);
		for(i = k = 0; i < m; i++) {
			je = 2*((*nbp)[i]);
			bpi = &((*hbp)[i]);
			bpi->sizeUninitialized(je);
			j2 = je - 2;
			for(j = 0; j < je; ++j, ++k)	 //can we just say "(*bpi) = (*bp)?"
				(*bpi)[j] = (*bp)[k];
			for(j = 0; j < j2; j += 2) {
				if ((*bpi)[j] >= (*bpi)[j+2]) {
					Squawk("histogram bin x values must increase");
					return;
					}
				if ((*bpi)[j+1] <= 0.) {
					Squawk("Nonpositive intermediate histogram y value");
					return;
					}
				}
			if ((*bpi)[je-1] != 0.) {
				Squawk("histogram bin y values must end with 0");
				return;
				}
			(*V)[i0] = (*L)[i0] = (*bpi)[0];
			(*U)[i0] = (*bpi)[j2];
			++i0;
			}
		}
	if ((npp = vi->npp)) {
		nseen += m = npp->length();
		for(i = totpp = 0; i < m; i++) {
			totpp += j = (*npp)[i];
			if (j < 1) {
				Squawk("huv_num_point_pairs must be >= 1");
				return;
				}
			}
		totpp2 = (int)(2*totpp);
		pp = vi->pp;
		k = pp->length();
		ik = (int)k;
		if (ik != totpp2) {
			Squawk("Expected %d huv_point_pairs, not %d", totbp2, ik);
			return;
			}
		hpp = &dv->histogramUncPointPairs;
		hpp->reshape(m);
		for(i = k = 0; i < m; ++i) {
			ppi = &((*hpp)[i]);
			je = 2*(*npp)[i];
			ppi->sizeUninitialized(je);
			for(j = 0; j < je; ++j, ++k)
				(*ppi)[j] = (*pp)[k];
			j2 = je - 2;
			for(j = 0; j < j2; j += 2) {
				if ((*ppi)[j] >= (*ppi)[j+2]) {
					Squawk("histogram point x values must increase");
					return;
					}
				}
			for(j = 1; j < je; j += 2) {
				if ((*ppi)[j] <= 0) {
					Squawk("histogram point y values must be > 0");
					return;
					}
				}
			(*V)[i0] = (*L)[i0] = (*ppi)[0];
			(*U)[i0] = (*ppi)[j2];
			++i0;
			}
		}
	if (nseen != n && NIDRProblemDescDB::pDDBInstance)
		Squawk("The number %d of bin and point histogram specifications differs\n\t"
			"from the specified number %d histogram uncertain variables",
			(int)nseen, (int)n);
	}

 static void
Vadj_Interval(DataVariablesRep *dv, size_t i0, Var_Info *vi)
{
	size_t i, j, j2, k, k2, m, n, ni;
	IntVector *nIv;
	Real lb, lbj, ub, ubj;
	RealDenseVector *Bi, *Ivb, *Ivp, *L, *Pi, *U;
	RealDenseVectorArray *B, *P;

	n = dv->numIntervalUncVars;
	nIv = vi->nIv;
	Ivb = vi->Ivb;
	Ivp = vi->Ivp;
	L = &dv->uncertainLowerBnds;
	U = &dv->uncertainUpperBnds;
	m = nIv->length();
	if (m != n) {
		Squawk("Expected %d numbers for iuv_num_intervals, but got %d", (int)n, (int)m);
		return;
		}
	for(i = ni = 0; i < m; i++) {
		ni += j = (*nIv)[i];
		if (j < 1) {
			Squawk("iuv_num_intervals values should be positive");
			return;
			}
		}
	if (wronglen(ni, Ivp, "iuv_interval_probs")
	 || wronglen(2*ni, Ivb, "iuv_interval_bounds"))
		return;
	B = &dv->intervalUncBounds;
	P = &dv->intervalUncBasicProbs;
	B->reshape(n);
	P->reshape(n);
	for(i = k = k2 = 0; i < n; ++i0, ++i) {
		Bi = &((*B)[i]);
		Pi = &((*P)[i]);
		m = (*nIv)[i];
		Pi->sizeUninitialized(m);
		Bi->sizeUninitialized(2*m);
		ub = -(lb = DBL_MAX);
		for(j = j2 = 0; j < m; ++j, ++k, j2 += 2, k2 += 2) {
			(*Pi)[j] = (*Ivp)[k];
			(*Bi)[j2] = lbj = (*Ivb)[k2];
			(*Bi)[j2+1] = ubj = (*Ivb)[k2+1];
			if (lb > lbj)
				lb = lbj;
			if (ub < ubj)
				ub = ubj;
			}
		if (lb > ub)
			Squawk("Inconsistent interval uncertain bounds: %g > %g",
				lb, ub);
		(*L)[i0] = lb;
		(*U)[i0] = ub;
		}
	}

#define UncInfo(a,b)     { #a, #b, &DataVariablesRep::num##b##UncVars, Vadj_##b }
 static Var_uinfo UncLbl[UncVar_Nkinds] = {
	UncInfo(nuv_, Normal),
	UncInfo(lnuv_, Lognormal),
	UncInfo(uuv_, Uniform),
	UncInfo(luuv_, Loguniform),
	UncInfo(tuv_, Triangular),
	UncInfo(euv_, Exponential),
	UncInfo(buv_, Beta),
	UncInfo(gauv_, Gamma),
	UncInfo(guuv_, Gumbel),
	UncInfo(fuv_, Frechet),
	UncInfo(wuv_, Weibull),
	UncInfo(huv_, Histogram),
	UncInfo(iuv_, Interval)
	};
#undef UncInfo

 void NIDRProblemDescDB::
var_stop(const char *keyname, Values *val, void **g, void *v)
{
	Var_Info *vi = *(Var_Info**)g;
	DataVariablesRep *dv = vi->dv;
	scale_chk(dv->continuousDesignScaleTypes, dv->continuousDesignScales, "cdv", aln_scaletypes);
	pDDBInstance->VIL.push_back(vi);
	pDDBInstance->dataVariablesList.insert(*vi->dv0);
	delete vi->dv0;
	}

struct VarLabelChk {
	size_t DataVariablesRep::* n;
	StringArray DataVariablesRep::* sa;
	const char *stub;
	const char *name;
	};

#define AVI &DataVariablesRep::
	static VarLabelChk Vlch[] = {
	 { AVI numContinuousDesVars,	AVI continuousDesignLabels, "cdv_", "cdv_descriptors" },
	 { AVI numDiscreteDesVars,	AVI discreteDesignLabels,   "ddv_", "ddv_descriptors" },
	 { AVI numContinuousStateVars,	AVI continuousStateLabels,  "csv_", "csv_descriptors" },
	 { AVI numDiscreteStateVars,	AVI discreteStateLabels,    "dsv_", "dsv_descriptors" },
	 { AVI numContinuousDesVars,	AVI continuousDesignScaleTypes, 0,  "cdv_scale_types" }
	 };
#undef  AVI

 void NIDRProblemDescDB::
make_variable_labels(List<DataVariables>* vl)
{
	DataVariables *dv0;
	DataVariablesRep *dv;
	List<DataVariables>::iterator It, Ite;
	StringArray *sa;
	VarLabelChk *vlc, *vlce;
	size_t n;

	It = vl->begin();
	for(Ite = vl->end(); It != Ite; ++It) {
		dv0 = &*It;
		dv = dv0->dataVarsRep;
		for(vlc = Vlch, vlce = vlc + sizeof(Vlch)/sizeof(Vlch[0]); vlc < vlce; ++vlc)
			if (vlc->stub && (n = dv->*vlc->n)) {
				sa = &(dv->*vlc->sa);
				if (sa->length() == 0)
					BuildLabels(sa, n, 0, n, vlc->stub);
				}
		}
	}

 void NIDRProblemDescDB::
var_stop1(void *v)
{
	DataVariablesRep *dv;
	IntVector *Iv;
	RealSymDenseMatrix *Rm;
	RealDenseVector *L, *Rv, *U;
	StringArray *sa;
	VarLabel *vl;
	Var_Info *vi;
	Var_uinfo *vui, *vuie;
	VarLabelChk *vlc, *vlce;
	char buf[32];
	const char **sp;
	int havelabels;
	size_t i, j, k, n, nu, nv;
#define AVI &Var_Info::
	static IntVector *Var_Info::* Ivzap[] = { AVI nIv, AVI nbp, AVI npp };
	static RealDenseVector *Var_Info::* Rvzap[] = { AVI Ivb, AVI Ivp, AVI bp, AVI pp, AVI ucm };
#undef  AVI
	static char ucmerr[] = "Got %lu entries for the uncertain_correlation_matrix\n\
	but needed %lu for %lu uncertain variables";

	vi = (Var_Info*)v;
	dv = vi->dv;

	/* check and generate labels for design and state variables */

	nv = 0;
	for(vlc = Vlch, vlce = vlc + sizeof(Vlch)/sizeof(Vlch[0]); vlc < vlce; ++vlc)
		if ((n = dv->*vlc->n)) {
			if (vlc->stub)
				nv += n;
			sa = &(dv->*vlc->sa);
			if ((nu = sa->length())) {
				if (nu != n) {
					if (nu == 1 && !vlc->stub)
						continue;
					squawk("Found %lu rather than %lu %s values",
						(UL)nu, (UL)n, vlc->name);
					}
				}
			}

	/* now deal with uncertain variables */

	vuie = UncLbl + UncVar_Nkinds;
	for(nu = 0, vui = UncLbl; vui < vuie; ++vui)
		nu += dv->*vui->n;
	if (nu > 0) {
		dv->uncertainVars.sizeUninitialized(nu);
		L = &dv->uncertainLowerBnds;
		L->sizeUninitialized(nu);
		U = &dv->uncertainUpperBnds;
		U->sizeUninitialized(nu);
		havelabels = 0;
		if ((j = dv->uncertainLabels.length())) {
			if (j == nu)
				havelabels = 1;	// parallel slave process
			else
				squawk("Inconsistent incoming uncertainLabels:"
					" expected %lu, got %lu",
					(unsigned long) nu, (unsigned long)j);
			}
		else
			dv->uncertainLabels.reshape(nu);
		i = 0;
		for(vl = vi->Uv, vui = UncLbl; vui < vuie; ++vl, ++vui) {
			if ((n = dv->*vui->n) == 0)
				continue;
			vui->vadj(dv,i,vi);
			for(j = 0; j < n; ++j)
				if ((*L)[i+j] > (*U)[i+j]) {
					squawk("Inconsistent bounds on %s uncertain variables",
						vui->vkind);
					break;
					}
			if ((sp = vl->s) && vl->n != n) {
				squawk("Expected %d %s_descriptors, but got %d",
					n, vui->lbl, vl->n);
				free(sp);
				sp = 0;
				}
			if (sp) {
				for(j = 0; j < n; ++i, ++j)
					dv->uncertainLabels[i] = sp[j];
				free(sp);
				}
			else if (!havelabels)
				for(j = 1; j <= n; ++j) {
					sprintf(buf, "%s%d", vui->lbl, (int)j);
					dv->uncertainLabels[i++] = buf;
					}
			}
		if ((Rv = vi->ucm)) { // uncertain_correlation_matrix
			n = Rv->length();
			if (n != nu*nu)
				squawk(ucmerr, (UL)n, (UL)nu*nu, (UL)nu);
			else {
				Rm = &dv->uncertainCorrelations;
				Rm->reshape(nu);
				for(i = k = 0; i < nu; ++i) {
					for(j = 0; j < nu;)
						(*Rm)(i,j++) = (*Rv)[k++];
					}
				}
			}
		}

	if (nv + nu == 0)
		squawk("at least one variable must be specified");

	/* check bounds */

	Var_boundchk(dv);
	Var_iboundchk(dv);

	/* finish up and clean up */

	n = sizeof(Ivzap)/sizeof(Ivzap[0]);
	for(i = 0; i < n; i++) {
		if ((Iv = vi->*Ivzap[i]))
			delete Iv;
		}
	n = sizeof(Rvzap)/sizeof(Rvzap[0]);
	for(i = 0; i < n; i++) {
		if ((Rv = vi->*Rvzap[i]))
			delete Rv;
		}
	delete vi;
	if (nerr)
		abort_handler(-1);
	}

 static void
Rdv_copy(RealDenseVector **prdv, RealDenseVectorArray *rdva)
{
	size_t i, j, k, m, n;
	RealDenseVector *rdv, *rdv1;

	*prdv = rdv = new RealDenseVector();
	m = rdva->length();
	for(i = n = 0; i < m; ++i)
		n += (*rdva)[i].length();
	rdv->sizeUninitialized(n);
	for(i = k = 0; i < m; ++i) {
		rdv1 = &(*rdva)[i];
		n = rdv1->length();
		for(j = 0; j < n; ++j)
			(*rdv)[k++] = (*rdv1)[j];
		}
	}

 void NIDRProblemDescDB::
finish_variables(List<DataVariables>* vl)
{
	if (pDDBInstance) {
		List<void*>::iterator It  = pDDBInstance->VIL.begin();
		List<void*>::iterator Ite = pDDBInstance->VIL.end();
		while(It != Ite) {
			var_stop1(*It);
			It++;
			}
		pDDBInstance->VIL.clear();
		}
	else {
		// library mode with manual provision of everything
		DataVariables *dv0;
		DataVariablesRep *dv;
		IntVector *iv;
		List<DataVariables>::iterator It, Ite;
		RealSymDenseMatrix *rdsm;
		RealDenseVector *rdv;
		RealDenseVectorArray *rdva;
		Var_Info *vi;
		size_t i, j, m, n;

		It = vl->begin();
		for(Ite = vl->end(); It != Ite; ++It) {
			vi = new Var_Info;
			memset(vi, 0, sizeof(Var_Info));
			vi->dv0 = dv0 = &*It;
			vi->dv = dv = dv0->dataVarsRep;
			if ((n = dv->numIntervalUncVars)) {
				rdva = &dv->intervalUncBasicProbs;
				m = rdva->length();
				vi->nIv = iv = new IntVector(m);
				for(i = 0; i < m; ++i)
					(*iv)[i] = (*rdva)[i].length();
				Rdv_copy(&vi->Ivb, &dv->intervalUncBounds);
				Rdv_copy(&vi->Ivp, rdva);
				}
			rdva = &dv->histogramUncBinPairs;
			if ((m = rdva->length())) {
				vi->nbp = iv = new IntVector(m);
				for(i = 0; i < m; ++i)
					(*iv)[i] = (*rdva)[i].length() >> 1;
				Rdv_copy(&vi->bp, rdva);
				}
			rdva = &dv->histogramUncPointPairs;
			if ((m = rdva->length())) {
				vi->npp = iv = new IntVector(m);
				for(i = 0; i < m; ++i)
					(*iv)[i] = (*rdva)[i].length() >> 1;
				Rdv_copy(&vi->pp, rdva);
				}
			rdsm = &dv->uncertainCorrelations;
			if ((m = rdsm->numRows())) {
				vi->ucm = rdv = new RealDenseVector();
				rdv->sizeUninitialized((m*m+m) >> 1);
				for(i = n = 0; i < m; ++i)
					for(j = 0; j <= i; ++j, ++n)
						(*rdv)[n] = (*rdsm)(i,j);
				}
			var_stop1((void*)vi);
			}
		}
	}

 void NIDRProblemDescDB::
var_str(const char *keyname, Values *val, void **g, void *v)
{
	(*(Var_Info**)g)->dv->**(String DataVariablesRep::**)v = *val->s;
	}

 void NIDRProblemDescDB::
var_strL(const char *keyname, Values *val, void **g, void *v)
{
	StringArray *sa = &((*(Var_Info**)g)->dv->**(StringArray DataVariablesRep::**)v);
	const char **s = val->s;
	size_t i, n = val->n;

	sa->reshape(n);
	for(i = 0; i < n; i++)
		(*sa)[i] = s[i];
	}

 void NIDRProblemDescDB::
var_ulbl(const char *keyname, Values *val, void **g, void *v)
{
	// IRIX disallows (int)v, so we use the circumlocution (char*)v - (char*)0.
	VarLabel *vl = &(*(Var_Info**)g)->Uv[(char*)v - (char*)0];
	char *t;
	const char **s, **sl;
	size_t i, L, n;

	L = n = val->n;	// n for null
	s = val->s;
	for(i = 0; i < n; i++)
		L += strlen(s[i]);
	vl->s = sl = (const char **)malloc(n*sizeof(char*) + L);
	if (!sl)
		botch("malloc failure in var_ulbl");
	vl->n = n;
	t = (char*)(sl + n);
	for(i = 0; i < n; i++) {
		strcpy(t, s[i]);
		sl[i] = (const char*)t;
		t += strlen(t) + 1;
		}
	}

#define MP_(x) DataInterfaceRep::* iface_mp_##x = &DataInterfaceRep::x
#define MP2(x,y) iface_mp_##x##_##y = {&DataInterfaceRep::x,#y}
#define MP3(x,y,z) iface_mp_3##x##_##z = {&DataInterfaceRep::x,&DataInterfaceRep::y,#z}

static Iface_mp_Rlit
	MP3(failAction,recoveryFnVals,recover);

static Iface_mp_ilit
	MP3(failAction,retryLimit,retry);

static Iface_mp_lit
	MP2(analysisScheduling,self),
	MP2(analysisScheduling,static),
	MP2(evalScheduling,self),
	MP2(evalScheduling,static),
	MP2(failAction,abort),
	MP2(failAction,continuation),
	MP2(interfaceSynchronization,asynchronous),
	MP2(interfaceType,direct),
	MP2(interfaceType,fork),
	MP2(interfaceType,grid),
	MP2(interfaceType,system);

static String
	MP_(algebraicMappings),
	MP_(idInterface),
	MP_(inputFilter),
	MP_(outputFilter),
	MP_(parametersFile),
	MP_(resultsFile);

static String2DArray
	MP_(analysisComponents);

static StringArray
	MP_(analysisDrivers);

static bool
	MP_(activeSetVectorFlag),
	MP_(apreproFlag),
	MP_(verbatimFlag),
	MP_(evalCacheFlag),
	MP_(fileSaveFlag),
	MP_(fileTagFlag),
	MP_(restartFileFlag);

static int
	MP_(analysisServers),
	MP_(asynchLocalAnalysisConcurrency),
	MP_(asynchLocalEvalConcurrency),
	MP_(evalServers),
	MP_(procsPerAnalysis);

#undef MP3
#undef MP2
#undef MP_

#define MP_(x) DataMethodRep::* method_mp_##x = &DataMethodRep::x
#define MP2(x,y) method_mp_##x##_##y = {&DataMethodRep::x,#y}
#define MP2a(x,y) method_mp_##x##_##y = {&DataMethodRep::x,y}
#define MP2i(x,y) method_mp_##x = {&DataMethodRep::x,#y}
#define MP3(x,y,z) method_mp_3##x##_##z = {&DataMethodRep::x,&DataMethodRep::y,#z}
#define MP3x(x,y,z) method_mp_##x##_##z = {&DataMethodRep::x,&DataMethodRep::y,z}
#define MP4(w,x,y,z) method_mp_##w##_##y = {&DataMethodRep::w,&DataMethodRep::x,#y,z}

static IntVector
	MP_(primeBase),
	MP_(sequenceLeap),
	MP_(sequenceStart);

static Method_mp_Ii
	MP3x(numSteps,paramStudyType,2),
	MP3x(numSteps,paramStudyType,3);

static Method_mp_Ri
	MP3x(stepLength,paramStudyType,1);

static Method_mp_ilit2
	MP3(replacementType,numberRetained,chc),
	MP3(replacementType,numberRetained,elitist),
	MP3(replacementType,numberRetained,random);

static Method_mp_ilit2z
	MP3(crossoverType,numCrossPoints,multi_point_binary),
	MP3(crossoverType,numCrossPoints,multi_point_parameterized_binary),
	MP3(crossoverType,numCrossPoints,multi_point_real);

static Method_mp_lit
	MP2(boxDivision,all_dimensions),
	MP2(boxDivision,major_dimension),
	MP2(collocSampleReuse,all),
	MP2(convergenceType,average_fitness_tracker),
	MP2(convergenceType,best_fitness_tracker),
	MP2(convergenceType,metric_tracker),
	MP2(crossoverType,blend),
	MP2(crossoverType,two_point),
	MP2(crossoverType,uniform),
	MP2(daceMethod,box_behnken),
	MP2(daceMethod,central_composite),
	MP2(daceMethod,grid),
	MP2(daceMethod,lhs),
	MP2(daceMethod,oa_lhs),
	MP2(daceMethod,oas),
	MP2(daceMethod,random),
	MP2(distributionType,complementary),
	MP2(distributionType,cumulative),
	MP2(evalSynchronization,blocking),
	MP2(evalSynchronization,nonblocking),
	MP2(evalSynchronize,blocking),
	MP2(evalSynchronize,nonblocking),
	MP2(expansionSampleType,incremental_lhs),
	MP2(exploratoryMoves,adaptive),
	MP2(exploratoryMoves,multi_step),
	MP2(exploratoryMoves,simple),
	MP2(fitnessType,domination_count),
	MP2(fitnessType,layer_rank),
	MP2(fitnessType,linear_rank),
	MP2(fitnessType,merit_function),
	MP2(fitnessType,proportional),
	MP2(initializationType,random),
	MP2(initializationType,unique_random),
        MP2(meritFunction,merit_max),
        MP2(meritFunction,merit_max_smooth),
        MP2(meritFunction,merit1),
        MP2(meritFunction,merit1_smooth),
        MP2(meritFunction,merit2),
        MP2(meritFunction,merit2_smooth),
        MP2(meritFunction,merit2_squared),
	MP2(methodName,asynch_pattern_search),
	MP2(methodName,coliny_cobyla),
	MP2(methodName,coliny_direct),
	MP2(methodName,coliny_pattern_search),
	MP2(methodName,coliny_solis_wets),
	MP2(methodName,conmin_frcg),
	MP2(methodName,conmin_mfd),
	MP2(methodName,dace),
	MP2(methodName,dot_bfgs),
	MP2(methodName,dot_frcg),
	MP2(methodName,dot_mmfd),
	MP2(methodName,dot_slp),
	MP2(methodName,dot_sqp),
	MP2(methodName,efficient_global),
	MP2(methodName,fsu_cvt),
	MP2(methodName,fsu_halton),
	MP2(methodName,fsu_hammersley),
	MP2(methodName,ncsu_direct),
	MP2(methodName,nl2sol),
	MP2(methodName,nlpql_sqp),
	MP2(methodName,nlssol_sqp),
	MP2(methodName,nond_evidence),
	MP2(methodName,nond_global_reliability),
        MP2(methodName,nond_interval_est),
	MP2(methodName,nond_polynomial_chaos),
	MP2(methodName,nond_sampling),
	MP2(methodName,nond_stoch_collocation),
	MP2(methodName,nonlinear_cg),
	MP2(methodName,npsol_sqp),
	MP2(methodName,optpp_cg),
	MP2(methodName,optpp_fd_newton),
	MP2(methodName,optpp_g_newton),
	MP2(methodName,optpp_newton),
	MP2(methodName,optpp_pds),
	MP2(methodName,optpp_q_newton),
	MP2(methodName,psuade_moat),
	MP2(methodName,surrogate_based_global),
	MP2(methodName,surrogate_based_local),
	MP2(methodName,vector_parameter_study),
	MP2(minMaxType,maximize),
	MP2(minMaxType,minimize),
	MP2(mutationType,bit_random),
	MP2(mutationType,offset_cauchy),
	MP2(mutationType,offset_normal),
	MP2(mutationType,offset_uniform),
	MP2(mutationType,replace_uniform),
	MP2(patternBasis,coordinate),
	MP2(patternBasis,simplex),
	MP2(reliabilityIntegration,first_order),
	MP2(reliabilityIntegration,second_order),
	MP2(reliabilityIntegrationRefine,ais),
	MP2(reliabilityIntegrationRefine,is),
	MP2(reliabilityIntegrationRefine,mmais),
	MP2(reliabilitySearchAlgorithm,nip),
	MP2(reliabilitySearchAlgorithm,sqp),
	MP2(reliabilitySearchType,amv_plus_u),
	MP2(reliabilitySearchType,amv_plus_x),
	MP2(reliabilitySearchType,amv_u),
	MP2(reliabilitySearchType,amv_x),
	MP2(reliabilitySearchType,egra_u),
	MP2(reliabilitySearchType,egra_x),
	MP2(reliabilitySearchType,no_approx),
	MP2(reliabilitySearchType,tana_u),
	MP2(reliabilitySearchType,tana_x),
	MP2(replacementType,elitist),
	MP2(replacementType,favor_feasible),
	MP2(replacementType,roulette_wheel),
	MP2(replacementType,unique_roulette_wheel),
	MP2(responseLevelMappingType,gen_reliabilities),
	MP2(responseLevelMappingType,probabilities),
	MP2(responseLevelMappingType,reliabilities),
	MP2(sampleType,incremental_lhs),
	MP2(sampleType,incremental_random),
	MP2(sampleType,lhs),
	MP2(sampleType,random),
	MP2(searchMethod,gradient_based_line_search),
	MP2(searchMethod,tr_pds),
	MP2(searchMethod,trust_region),
	MP2(searchMethod,value_based_line_search),
	MP2(trialType,grid),
	MP2(trialType,halton),
	MP2(trialType,random);

static Method_mp_lit2
	MP4(methodName,reliabilitySearchType,nond_local_reliability,"mv");

static Method_mp_litc
	MP3(crossoverType,crossoverRate,shuffle_random),
	MP3(crossoverType,crossoverRate,null_crossover),
	MP3(mutationType,mutationRate,null_mutation),
	MP3(mutationType,mutationRate,offset_cauchy),
	MP3(mutationType,mutationRate,offset_normal),
	MP3(mutationType,mutationRate,offset_uniform),
	MP3(replacementType,fitnessLimit,below_limit);

static Method_mp_liti
	MP4(methodName,paramStudyType,list_parameter_study,-1),
	MP4(methodName,paramStudyType,centered_parameter_study,4),
	MP4(methodName,paramStudyType,multidim_parameter_study,5);

static Method_mp_litrv
	MP3(nichingType,nicheVector,distance),
	MP3(nichingType,nicheVector,radial),
	MP3(postProcessorType,distanceVector,distance_postprocessor);

static Method_mp_slit
	MP2a(methodOutput,DEBUG_OUTPUT),
	MP2a(methodOutput,QUIET_OUTPUT),
	MP2a(methodOutput,SILENT_OUTPUT),
	MP2a(methodOutput,VERBOSE_OUTPUT),
	MP2a(surrBasedLocalAcceptLogic,FILTER),
	MP2a(surrBasedLocalAcceptLogic,TR_RATIO),
	MP2a(surrBasedLocalConstrRelax,HOMOTOPY),
	MP2a(surrBasedLocalMeritFn,ADAPTIVE_PENALTY_MERIT),
	MP2a(surrBasedLocalMeritFn,AUGMENTED_LAGRANGIAN_MERIT),
	MP2a(surrBasedLocalMeritFn,LAGRANGIAN_MERIT),
	MP2a(surrBasedLocalMeritFn,PENALTY_MERIT),
	MP2a(surrBasedLocalSubProbCon,LINEARIZED_CONSTRAINTS),
	MP2a(surrBasedLocalSubProbCon,NO_CONSTRAINTS),
	MP2a(surrBasedLocalSubProbCon,ORIGINAL_CONSTRAINTS),
	MP2a(surrBasedLocalSubProbObj,AUGMENTED_LAGRANGIAN_OBJECTIVE),
	MP2a(surrBasedLocalSubProbObj,LAGRANGIAN_OBJECTIVE),
	MP2a(surrBasedLocalSubProbObj,ORIGINAL_PRIMARY),
	MP2a(surrBasedLocalSubProbObj,SINGLE_OBJECTIVE);

static Method_mp_slit2
	MP3(initializationType,flatFile,flat_file),
	MP3(methodName,dlDetails,dl_solver);

static Real
	MP_(absConvTol),
	MP_(centeringParam),
	MP_(collocationRatio),
	MP_(constraintPenalty),
	MP_(constrPenalty),
	MP_(constraintTolerance),
	MP_(contractFactor),
	MP_(contractStepLength),
	MP_(convergenceTolerance),
	MP_(crossoverRate),
	MP_(falseConvTol),
	MP_(functionPrecision),
	MP_(globalBalanceParam),
	MP_(gradientTolerance),
	MP_(initDelta),
	MP_(initStepLength),
	MP_(initTRRadius),
	MP_(lineSearchTolerance),
	MP_(localBalanceParam),
	MP_(maxBoxSize),
	MP_(maxStep),
	MP_(minBoxSize),
	MP_(mutationRate),
	MP_(mutationScale),
	MP_(percentDelta),
	MP_(shrinkagePercent),	// should be called shrinkageFraction
	MP_(singConvTol),
	MP_(singRadius),
        MP_(smoothFactor),
 	MP_(solnAccuracy),
 	MP_(solnTarget),
	MP_(stepLenToBoundary),
	MP_(surrBasedLocalTRContract),
	MP_(surrBasedLocalTRContractTrigger),
	MP_(surrBasedLocalTRExpand),
	MP_(surrBasedLocalTRExpandTrigger),
	MP_(surrBasedLocalTRInitSize),
	MP_(surrBasedLocalTRMinSize),
	MP_(threshDelta),
	MP_(threshStepLength),
	MP_(volBoxSize),
	MP_(xConvTol);

static RealVector
	MP_(linearEqScales),
	MP_(linearIneqScales);

static RealDenseVector
	MP_(finalPoint),
	MP_(linearEqConstraintCoeffs),
	MP_(linearEqTargets),
	MP_(linearIneqConstraintCoeffs),
	MP_(linearIneqLowerBnds),
	MP_(linearIneqUpperBnds),
	MP_(listOfPoints),
	MP_(stepVector);

static RealVectorArray
	MP_(genReliabilityLevels),
	MP_(probabilityLevels),
	MP_(reliabilityLevels),
	MP_(responseLevels);

static UShortArray
        MP_(expansionOrder),
	MP_(sparseGridLevel),
        MP_(quadratureOrder),
	MP_(varPartitions);

static String
	MP_(centralPath),
	MP_(expansionImportFile),
	MP_(idMethod),
	MP_(logFile),
	MP_(meritFn),
	MP_(modelPointer),
	MP_(subMethodName),
	MP_(subMethodPointer);

static StringArray
	MP_(linearEqScaleTypes),
	MP_(linearIneqScaleTypes),
        MP_(miscOptions);

static bool
	MP_(allVarsFlag),
	MP_(constantPenalty),
	MP_(expansionFlag),
	MP_(fixedSeedFlag),
	MP_(fixedSequenceFlag),
	MP_(latinizeFlag),
	MP_(mainEffectsFlag),
	MP_(methodScaling),
	MP_(mutationAdaptive),
	MP_(printPopFlag),
	MP_(randomizeOrderFlag),
	MP_(regressDiag),
	MP_(showMiscOptions),
	MP_(speculativeFlag),
	MP_(surrBasedGlobalReplacePts),
	MP_(surrBasedLocalLayerBypass),
	MP_(varBasedDecompFlag),
	MP_(volQualityFlag);

static int
	MP_(collocationPoints),
	MP_(contractAfterFail),
	MP_(covarianceType),
	MP_(deltasPerVariable),
	MP_(expandAfterSuccess),
	MP_(expansionSamples),
	MP_(expansionTerms),
	MP_(maxFunctionEvaluations),
	MP_(maxIterations),
	MP_(mutationRange),
	MP_(newSolnsGenerated),
	MP_(numSamples),
	MP_(numSymbols),
	MP_(numTrials),
	MP_(populationSize),
	MP_(previousSamples),
	MP_(randomSeed),
	MP_(searchSchemeSize),
	MP_(surrBasedLocalSoftConvLimit),
	MP_(totalPatternSize),
	MP_(verifyLevel);

static size_t
	MP_(numGenerations),
	MP_(numOffspring),
	MP_(numParents);

#undef MP4
#undef MP3x
#undef MP3
#undef MP2i
#undef MP2a
#undef MP2
#undef MP_

#define MP_(x) DataModelRep::* model_mp_##x = &DataModelRep::x
#define MP2(x,y) model_mp_##x##_##y = {&DataModelRep::x,#y}
#define MP2s(x,y) model_mp_##x##_##y = {&DataModelRep::x,y}
//#define MP3(x,y,z) model_mp_3##x##_##z = {&DataModelRep::x,&DataModelRep::y,#z}

static IntSet
	MP_(surrogateFnIndices);

static Model_mp_lit
	MP2(approxCorrectionType,additive),
	MP2(approxCorrectionType,combined),
	MP2(approxCorrectionType,multiplicative),
	MP2(approxSampleReuse,all),
	MP2(approxSampleReuse,none),
	MP2(approxSampleReuse,region),
	MP2(marsInterpolation,linear),
	MP2(marsInterpolation,cubic),
	MP2(modelType,nested),
	MP2(modelType,single),
	MP2(modelType,surrogate),
	MP2(surrogateType,hierarchical),
	MP2(surrogateType,global_gaussian),
	MP2(surrogateType,global_kriging),
	MP2(surrogateType,global_mars),
	MP2(surrogateType,global_moving_least_squares),
	MP2(surrogateType,global_neural_network),
	MP2(surrogateType,global_polynomial),
	MP2(surrogateType,global_radial_basis),
	MP2(surrogateType,local_taylor),
	MP2(surrogateType,multipoint_tana);

static Model_mp_ord
	MP2s(approxCorrectionOrder,0),
	MP2s(approxCorrectionOrder,1),
	MP2s(approxCorrectionOrder,2),
	MP2s(polynomialOrder,1),
	MP2s(polynomialOrder,2),
	MP2s(polynomialOrder,3),
	MP2s(trendOrder,0),
	MP2s(trendOrder,1),
	MP2s(trendOrder,2);

//static Model_mp_slit2
//	MP3(approxSampleReuse,approxSampleReuseFile,file);

static Real
	MP_(annRange);

static RealDenseVector
	MP_(primaryRespCoeffs),
	MP_(secondaryRespCoeffs);

static RealVector
	MP_(krigingConminSeed),
	MP_(krigingCorrelations),
	MP_(krigingMaxCorrelations),
	MP_(krigingMinCorrelations);
	//MP_(primaryRespCoeffs),
	//MP_(secondaryRespCoeffs);

static String
	MP_(approxSampleReuseFile),
	MP_(idModel),
	MP_(interfacePointer),
	MP_(lowFidelityModelPointer),
	MP_(optionalInterfRespPointer),
	MP_(responsesPointer),
	MP_(subMethodPointer),
	MP_(truthModelPointer),
	MP_(variablesPointer);

static StringArray
	MP_(primaryVarMaps),
        MP_(secondaryVarMaps),
        MP_(diagMetrics);

bool
	MP_(approxGradUsageFlag),
	MP_(pointSelection);

short
	MP_(annNodes),
	MP_(annRandomWeight),
	MP_(krigingMaxTrials),
	MP_(marsMaxBases),
	MP_(mlsPolyOrder),
	MP_(mlsWeightFunction),
	MP_(rbfBases),
	MP_(rbfMaxPts),
	MP_(rbfMaxSubsets),
	MP_(rbfMinPartition);

//#undef MP3
#undef MP2s
#undef MP2
#undef MP_

#define MP_(x) DataResponsesRep::* resp_mp_##x = &DataResponsesRep::x
#define MP2(x,y) resp_mp_##x##_##y = {&DataResponsesRep::x,#y}

static IntList
	MP_(idAnalyticGrads),
	MP_(idAnalyticHessians),
	MP_(idNumericalGrads),
	MP_(idNumericalHessians),
	MP_(idQuasiHessians);

static RealVector
	MP_(fdGradStepSize),
	MP_(fdHessStepSize),
	MP_(primaryRespFnScales),
	MP_(nonlinearEqScales),
	MP_(nonlinearIneqScales);

static RealDenseVector
	MP_(primaryRespFnWeights),
	MP_(nonlinearEqTargets),
	MP_(nonlinearIneqLowerBnds),
	MP_(nonlinearIneqUpperBnds);

static Resp_mp_lit
	MP2(gradientType,analytic),
	MP2(gradientType,mixed),
	MP2(gradientType,none),
	MP2(gradientType,numerical),
	MP2(hessianType,analytic),
	MP2(hessianType,mixed),
	MP2(hessianType,none),
	MP2(hessianType,numerical),
	MP2(hessianType,quasi),
	MP2(intervalType,central),
	MP2(intervalType,forward),
	MP2(methodSource,dakota),
	MP2(methodSource,vendor),
	MP2(quasiHessianType,bfgs),
	MP2(quasiHessianType,damped_bfgs),
	MP2(quasiHessianType,sr1);

static String
	MP_(idResponses),
	MP_(leastSqDataFile);

static StringArray
	MP_(primaryRespFnScaleTypes),
	MP_(nonlinearEqScaleTypes),
	MP_(nonlinearIneqScaleTypes),
	MP_(responseLabels);

static bool
	MP_(centralHess),
	MP_(ignoreBounds);

static size_t
	MP_(numLeastSqTerms),
	MP_(numNonlinearEqConstraints),
	MP_(numNonlinearIneqConstraints),
	MP_(numObjectiveFunctions),
	MP_(numResponseFunctions);

#undef MP2
#undef MP_

#define MP_(x) DataStrategyRep::* strategy_mp_##x = &DataStrategyRep::x
#define MP2(x,y) strategy_mp_##x##_##y = {&DataStrategyRep::x,#y}

static Real
      //MP_(hybridProgThresh),
	MP_(hybridLSProb);

static RealDenseVector
	MP_(concurrentParameterSets);

static Strategy_mp_lit
      //MP2(hybridCollabType,abo),
      //MP2(hybridCollabType,hops),
	MP2(hybridType,collaborative),
	MP2(hybridType,embedded),
	MP2(hybridType,sequential),
      //MP2(hybridType,sequential_adaptive),
	MP2(iteratorScheduling,self),
	MP2(iteratorScheduling,static),
	MP2(strategyType,hybrid),
	MP2(strategyType,multi_start),
	MP2(strategyType,pareto_set),
	MP2(strategyType,single_method);

static String
	MP_(hybridGlobalMethodPointer),
	MP_(hybridLocalMethodPointer),
	MP_(methodPointer),
	MP_(tabularDataFile);

static StringArray
	MP_(hybridMethodList);

static bool
	MP_(graphicsFlag),
	MP_(tabularDataFlag);

static int
	MP_(concurrentRandomJobs),
	MP_(concurrentSeed),
	MP_(iteratorServers);

static size_t
	MP_(hybridNumSolnsTrans);

#undef MP2
#undef MP_

#define MP_(x) DataVariablesRep::* var_mp_##x = &DataVariablesRep::x
#define MP2s(x,y) var_mp_##x = {&DataVariablesRep::x,y}
#define Vchu(x,y,z) {#x,&DataVariablesRep::y,&DataVariablesRep::z##LowerBnds,&DataVariablesRep::z##UpperBnds}
#define Vchv(x,y,z) {#x,&DataVariablesRep::y,&DataVariablesRep::z##LowerBnds,&DataVariablesRep::z##UpperBnds,&DataVariablesRep::z##Vars,&DataVariablesRep::z##Labels}
#define VP_(x) *Var_Info::* var_mp_Var_Info_##x = &Var_Info::x

static bool
        MP_(histogramUncSkylineFlag);

static size_t
	MP_(numBetaUncVars),
	MP_(numContinuousDesVars),
	MP_(numContinuousStateVars),
	MP_(numDiscreteDesVars),
	MP_(numDiscreteStateVars),
	MP_(numExponentialUncVars),
	MP_(numFrechetUncVars),
	MP_(numGammaUncVars),
	MP_(numGumbelUncVars),
	MP_(numHistogramUncVars),
	MP_(numIntervalUncVars),
	MP_(numLognormalUncVars),
	MP_(numLoguniformUncVars),
	MP_(numNormalUncVars),
	MP_(numTriangularUncVars),
	MP_(numUniformUncVars),
	MP_(numWeibullUncVars);

static IntDenseVector
	MP_(discreteDesignLowerBnds),
	MP_(discreteDesignUpperBnds),
	MP_(discreteDesignVars),
	MP_(discreteStateLowerBnds),
	MP_(discreteStateUpperBnds),
	MP_(discreteStateVars);

static IntVector
	VP_(nIv),
	VP_(nbp),
	VP_(npp);

static RealDenseVector
	MP_(betaUncLowerBnds),
	MP_(betaUncUpperBnds),
	MP_(continuousDesignLowerBnds),
	MP_(continuousDesignUpperBnds),
	MP_(continuousDesignVars),
	MP_(continuousStateLowerBnds),
	MP_(continuousStateUpperBnds),
	MP_(continuousStateVars),
	MP_(frechetUncBetas),
	MP_(gumbelUncBetas),
	MP_(normalUncLowerBnds),
	MP_(normalUncMeans),
	MP_(normalUncUpperBnds),
	MP_(triangularUncModes),
	VP_(Ivb),
	VP_(Ivp),
	VP_(bp),
	VP_(pp),
	VP_(ucm);

static RealVector
	MP_(continuousDesignScales);

static String
	MP_(idVariables);

static StringArray
	MP_(continuousDesignLabels),
	MP_(continuousDesignScaleTypes),
	MP_(continuousStateLabels),
	MP_(discreteDesignLabels),
	MP_(discreteStateLabels);

static Var_bchk var_mp_bndchk[] = {
	Vchv(continuous_design,numContinuousDesVars,continuousDesign),
	Vchu(normal_uncertain,numNormalUncVars,normalUnc),
	Vchu(lognormal_uncertain,numLognormalUncVars,lognormalUnc),
	Vchu(uniform_uncertain,numUniformUncVars,uniformUnc),
	Vchu(loguniform_uncertain,numLoguniformUncVars,loguniformUnc),
	Vchu(triangular_uncertain,numTriangularUncVars,triangularUnc),
	Vchu(exponential_uncertain,numExponentialUncVars,triangularUnc),
	Vchu(beta_uncertain,numBetaUncVars,betaUnc),
	Vchv(continuous_state,numContinuousStateVars,continuousState)
	};

static Var_ibchk var_mp_ibndchk[] = {
	Vchv(discrete_design,numDiscreteDesVars,discreteDesign),
	Vchv(discrete_state,numDiscreteStateVars,discreteState)
	};

static Var_brv //* change _ to 2s
	MP2s(betaUncAlphas,0.),
	MP2s(betaUncBetas,0.),
	MP2s(exponentialUncBetas,0.),
	MP2s(frechetUncAlphas,2.),
	MP2s(gammaUncAlphas,0.),
	MP2s(gammaUncBetas,0.),
	MP2s(gumbelUncAlphas,0.),
	MP2s(lognormalUncErrFacts,1.),
	MP2s(lognormalUncLowerBnds,0.),
	MP2s(lognormalUncMeans,0.),
	MP2s(lognormalUncStdDevs,0.),
	MP2s(lognormalUncUpperBnds,DBL_MAX),
	MP2s(loguniformUncLowerBnds,0.),
	MP2s(loguniformUncUpperBnds,DBL_MAX),
	MP2s(normalUncStdDevs,0.),
	MP2s(triangularUncLowerBnds, -DBL_MAX),
	MP2s(triangularUncUpperBnds, DBL_MAX),
	MP2s(uniformUncLowerBnds, -DBL_MAX),
	MP2s(uniformUncUpperBnds, DBL_MAX),
	MP2s(weibullUncAlphas,0.),
	MP2s(weibullUncBetas,0.);

#undef Vchv
#undef Vchu
#undef VP_
#undef MP2s
#undef MP_

 static const char *
Var_Name(StringArray *sa, char *buf, size_t i)
{
	if (sa)
		return (*sa)[i].data();
	sprintf(buf,"%lu", (UL)(i+1));
	return (const char*)buf;
	}

 void NIDRProblemDescDB::
Var_boundchk(DataVariablesRep *dv)
{
	Var_bchk *b, *be;
	Real t;
	RealDenseVector *L, *U, *V;
	StringArray *sa;
	char namebuf[32];
	int ndflt;
	size_t i, n;

	b = var_mp_bndchk;
	be = b + sizeof(var_mp_bndchk)/sizeof(var_mp_bndchk[0]);
	for(; b < be; ++b) {
		if ((n = dv->*b->n) == 0)
			continue;
		L = &(dv->*b->L);
		ndflt = 0;
		if (L->length() == 0) {
			++ndflt;
			Set_rdv(L, -DBL_MAX, n);
			}
		else if (L->length() != n) {
			squawk("%s_lower_bounds needs %lu elements, not %lu",
				b->name, (UL)n, (UL)L->length());
			continue;
			}
		U = &(dv->*b->U);
		if (U->length() == 0) {
			++ndflt;
			Set_rdv(U, DBL_MAX, n);
			}
		else if (U->length() != n) {
			squawk("%s_upper_bounds needs %lu elements, not %lu",
				b->name, (UL)n, (UL)L->length());
			continue;
			}
		sa = 0;
		if (b->Lbl) {
			sa = &(dv->*b->Lbl);
			if (sa->length() == 0)
				sa = 0;
			}
		if (ndflt == 0)
		     for(i = 0; i < n; i++) {
			if ((*L)[i] > (*U)[i])
			  squawk("lower bound for %s variable %s exceeds upper bound",
					b->name, Var_Name(sa,namebuf,i));
			}
		if (b->V == 0)
			continue;
		V = &(dv->*b->V);
		if (V->length() == 0) {
			V->sizeUninitialized(n);
			for(i = 0; i < n; i++) {
				t = 0.;
				if ((*L)[i] > t)
					t = (*L)[i];
				else if ((*U)[i] < t)
					t = (*U)[i];
				(*V)[i] = t;
				}
			continue;
			}
		if (V->length() != n) {
			squawk("initial point for %s needs %lu elements, not %lu",
				b->name, (UL)n, (UL)V->length());
			continue;
			}
		for(i = 0; i < n; i++) {
			t = (*V)[i];
			if (t < (*L)[i]) {
				Warn("Setting initial_value for %s variable %s to its lower bound",
					b->name, Var_Name(sa,namebuf,i));
				(*V)[i] = (*L)[i];
				}
			else if (t > (*U)[i]) {
				Warn("Setting initial_value for %s variable %s to its upper bound",
					b->name, Var_Name(sa,namebuf,i));
				(*V)[i] = (*U)[i];
				}
			}
		}
	}

 void NIDRProblemDescDB::
Var_iboundchk(DataVariablesRep *dv)
{
	Var_ibchk *b, *be;
	IntDenseVector *L, *U, *V;
	StringArray *sa;
	char namebuf[32];
	int ndflt, t;
	size_t i, n;

	b = var_mp_ibndchk;
	be = b + sizeof(var_mp_ibndchk)/sizeof(var_mp_ibndchk[0]);
	for(; b < be; ++b) {
		if ((n = dv->*b->n) == 0)
			continue;
		L = &(dv->*b->L);
		ndflt = 0;
		if (L->length() == 0) {
			++ndflt;
			L->sizeUninitialized(n);
			*L = INT_MIN;
			}
		else if (L->length() != n) {
			squawk("%s_lower_bounds needs %lu elements, not %lu",
				b->name, (UL)n, (UL)L->length());
			continue;
			}
		U = &(dv->*b->U);
		if (U->length() == 0) {
			++ndflt;
			U->sizeUninitialized(n);
			*U = INT_MAX;
			}
		else if (U->length() != n) {
			squawk("%s_upper_bounds needs %lu elements, not %lu",
				b->name, (UL)n, (UL)L->length());
			continue;
			}
		sa = 0;
		if (b->Lbl) {
			sa = &(dv->*b->Lbl);
			if (sa->length() == 0)
				sa = 0;
			}
		if (ndflt == 0)
		     for(i = 0; i < n; i++) {
			if ((*L)[i] > (*U)[i])
			  squawk("lower bound for %s variable %s exceeds upper bound",
					b->name, Var_Name(sa,namebuf,i));
			}
		if (b->V == 0)
			continue;	// won't happen for discrete variables
		V = &(dv->*b->V);
		if (V->length() == 0) {
			V->sizeUninitialized(n);
			for(i = 0; i < n; i++) {
				t = 0;
				if ((*L)[i] > t)
					t = (*L)[i];
				else if ((*U)[i] < t)
					t = (*U)[i];
				(*V)[i] = t;
				}
			continue;
			}
		if (V->length() != n) {
			squawk("initial point for %s needs %lu elements, not %lu",
				b->name, (UL)n, (UL)V->length());
			continue;
			}
		for(i = 0; i < n; i++) {
			t = (*V)[i];
			if (t < (*L)[i]) {
				warn("Setting initial_value for %s variable %s to its lower bound",
					b->name, Var_Name(sa,namebuf,i));
				(*V)[i] = (*L)[i];
				}
			else if (t > (*U)[i]) {
				warn("Setting initial_value for %s variable %s to its upper bound",
					b->name, Var_Name(sa,namebuf,i));
				(*V)[i] = (*U)[i];
				}
			}
		}
	}

} // namespace Dakota

#undef Warn
#undef Squawk

#define N_ifm(x,y)	NIDRProblemDescDB::iface_##x,&iface_mp_##y
#define N_ifm3(x,y,z)	NIDRProblemDescDB::iface_##x,y,NIDRProblemDescDB::iface_##z
#define N_mdm(x,y)	NIDRProblemDescDB::method_##x,&method_mp_##y
#define N_mdf(x,y)	N_mdm(x,y),NIDRProblemDescDB::method_##x##_final
#define N_mdm3(x,y,z)	NIDRProblemDescDB::method_##x,y,NIDRProblemDescDB::method_##z
#define N_mom(x,y)	NIDRProblemDescDB::model_##x,&model_mp_##y
#define N_mof(x,y)	N_mom(x,y),NIDRProblemDescDB::model_##x##_final
#define N_mom3(x,y,z)	NIDRProblemDescDB::model_##x,y,NIDRProblemDescDB::model_##z
#define N_rem(x,y)	NIDRProblemDescDB::resp_##x,&resp_mp_##y
#define N_rem3(x,y,z)	NIDRProblemDescDB::resp_##x,y,NIDRProblemDescDB::resp_##z
#define N_stm(x,y)	NIDRProblemDescDB::strategy_##x,&strategy_mp_##y
#define N_vae(x,y)	NIDRProblemDescDB::var_##x,(void*)y
#define N_vam(x,y)	NIDRProblemDescDB::var_##x,&var_mp_##y
#define N_vaf(x,y)	N_vam(x,y),NIDRProblemDescDB::var_##x##_final
#define N_vam3(x,y,z)	NIDRProblemDescDB::var_##x,y,NIDRProblemDescDB::var_##z

#include "NIDR_keywds.H"
