/*********************************************************************
Copyright 2008 Sandia Corporation.  Under the terms of Contract
DE-AC04-94AL85000 with Sandia Corporation, the U.S. Government
retains certain rights in this software.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

* Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

* Neither the name of Sandia Corporation nor the names of its
contributors may be used to endorse or promote products derived from
this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
***********************************************************************/

/* nidr.c */

#include "nidr.h"
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include "avltree.h"

#ifndef NIDR_SQUAWKMAX
#define NIDR_SQUAWKMAX 10
#endif

 static int nsquawk, nparse_errors, primary, strict;

 extern KeyWord Dakota_Keyword_Top;
 extern int nidrLineNumber;
 void *KW_g;
 void (*nidr_comment)(const char*);
 static void nidr_keyword_finish(void);
 static Comment *OutsideComment;
 static void kw_finish2(void), kw_finish3(void);
 static void kw_setup1(KeyWord *);
 static FILE *dumpfile;
 static KeyWord **ToClear, **ToClear0, **ToClearEnd;
 static int dumplev;

 enum {n_KWStack0 = 64};

 static KWinfo KWStack0[n_KWStack0];

 static Uint n_KWStack = n_KWStack0;

 static KeyWord *curid, *curkw;
 static KWinfo	*KWStack = KWStack0,
		*KWStackBot = KWStack0,
		*KWStackEnd = KWStack0 + n_KWStack0;

 static Values KWval, KWvalmax;
 static Real *KWvalbuf;
 static Uint nKWvalbuf;

 typedef struct Sbuf Sbuf;
 enum { n_KWsbuf = 8192 };
 struct Sbuf {
	char buf[n_KWsbuf];
	Sbuf *next;
	};

 typedef struct KWseen KWseen;

 struct
KWseen {
	const char *name;
	KeyWord *kw;
	KWseen *mnext, *mprev;	/* for identifiers so far unrequited when kw == 0 */
				/* kw != 0 ==> mprev = first child, and mnext = next sibling */
	KWseen *parent;
	KWseen **lcn;		/* &mprev field of last child; lcn == 0 when this */
				/* keyword was entered into the AVL tree because */
				/* its parent was seen. */
	Comment *comment;
	char **svals;
	Real *rvals;
	size_t nvals;
	};

 static Sbuf KWsbuf0, *KWsbuf = &KWsbuf0;
 static char *KWsbuf1 = KWsbuf0.buf, *KWsbufe = KWsbuf0.buf + n_KWsbuf;
 static KWseen *curkws;
 static char *valkind[3] = {"integer","numeric","string"};

 int
nidr_parse_error(void)
{
	int n;
	if ((n = nsquawk - NIDR_SQUAWKMAX) > 0)
		fprintf(stderr, "\n%d error message%s suppressed.\n",
			n, "s" + (n == 1));
	return nsquawk + nparse_errors;
	}

 void
nidr_signal_parse_error(void)
{ ++nparse_errors; }

 void
nidr_tolower(char *s)
{
	for(; *s; ++s)
		*s = tolower(*s);
	}

 static void
botch(const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	fprintf(stderr, "\nBotch:  ");
	vfprintf(stderr, fmt, ap);
	fputs(".\n", stderr);
	va_end(ap);
	exit(1);
	}

 static void
squawk(const char *fmt, ...)
{
	va_list ap;
	if (++nsquawk <= NIDR_SQUAWKMAX) {
		fprintf(stderr, "Input line %d: ", nidrLineNumber);
		va_start(ap, fmt);
		vfprintf(stderr, fmt, ap);
		fputs(".\n", stderr);
		va_end(ap);
		}
	}

#ifdef NIDR_MALLOC_DEBUG
 typedef struct MallocDebug MallocDebug;
 struct MallocDebug
{
	MallocDebug *next, *prev;
	char *where;
	int nalloc;
	};

 static MallocDebug MDtop = {&MDtop, &MDtop, 0, 0};
 int MallocDebugCount, MallocDebugCount1;

 static void*
Alloc(const char *where, size_t len)
{
	MallocDebug *md = malloc(len + sizeof(MallocDebug));
	if (!md) {
		fprintf(stderr, "malloc(%lu) failure in %s\n", (unsigned long)len, where);
		exit(1);
		}
	(md->next = MDtop.next)->prev = md;
	(md->prev = &MDtop)->next = md;
	md->where = where;
	if ((md->nalloc = ++MallocDebugCount) == MallocDebugCount1)
		printf("Hit %d\n", md->nalloc);
	return (void*)(md + 1);
	}

 static void
MallocDebugFree(void *v)
{
	MallocDebug *md = (MallocDebug *)v - 1;
	md->next->prev = md->prev;
	md->prev->next = md->next;
	free(md);
	}
#define free MallocDebugFree

#else //!NIDR_MALLOC_DEBUG
 static void*
Alloc(const char *where, size_t len)
{
	void *rv = malloc(len);
	if (!rv) {
		fprintf(stderr, "malloc(%lu) failure in %s\n", (unsigned long)len, where);
		exit(1);
		}
	return rv;
	}
#endif //NIDR_MALLOC_DEBUG

 struct
Comment {
	int k;		/* subscript for comfree */
	size_t avail;	/* bytes left (from tnext) */
	char *text;	/* text of comment */
	char *tnext;	/* null byte at end of comment */
	Comment *fnext;	/* next free Comment */
	};

 enum { Comment_kmax = 7 };

 static Comment *comfree[Comment_kmax+1];
 static size_t Comment_maxlen[Comment_kmax+1];

 static void
comment_free(Comment *c)
{
	int k = c->k;

	if (k > Comment_kmax)
		free(c);
	else {
		c->fnext = comfree[k];
		comfree[k] = c;
		}
	}

 static Comment*
alloc_comment(int k, size_t L)
{
	Comment *c;

	for(; k <= Comment_kmax; ++k) {
		if (L <= Comment_maxlen[k]) {
			L = Comment_maxlen[k];
			if ((c = comfree[k])) {
				comfree[k] = c->fnext;
				goto have_c;
				}
			break;
			}
		}
	c = (Comment*)Alloc("save_comment", L + sizeof(Comment) + 1);
	c->k = k;
	c->text = (char*)(c+1);
 have_c:
	c->avail = L;
	c->tnext = c->text;
	return c;
	}

 static void
save_comment(const char *s)
{
	Comment *c, *c1, **cp;
	size_t L, L1;

	L = strlen(s);
	cp = curid ? &curid->comment : curkws ? &curkws->comment : &OutsideComment;
	if ((c = *cp)) {
		if (c->avail >= L)
			goto cupdate;
		L1 = c->tnext - c->text;
		c1 = alloc_comment(c->k + 1, L + L1);
		memcpy(c1->text, c->text, L1);
		c1->tnext = c1->text + L1;
		c1->avail -= L1;
		comment_free(c);
		c = c1;
		}
 	else
		c = alloc_comment(0, L);
 cupdate:
	memcpy(c->tnext, s, L+1);
	c->tnext += L;
	c->avail -= L;
	*cp = c;
	}

 static void
comment_setup(void)
{
	int i;
	size_t L;
	nidr_comment = save_comment;
	/* "- 1" to allow for terminating '\0' */
	for(L = 64; L <= sizeof(Comment) - 1; L <<= 1);
	for(i = 0; i <= Comment_kmax; ++i, L <<= 1)
		Comment_maxlen[i] = L - sizeof(Comment) - 1;
	}

 static void
comment_reset(void)
{
	Comment *c, *c1;
	int i;

	for(i = 0; i <= Comment_kmax; ++i) {
		c1 = comfree[i];
		comfree[i] = 0;
		while((c = c1)) {
			c1 = c->fnext;
			free(c);
			}
		}
	nidr_comment = 0;
	}

 static void
dumpcomment(Comment **cp)
{
	Comment *c = *cp;
	*cp = 0;
	fprintf(dumpfile, "%s", c->text);
	comment_free(c);
	}

 static void
dumpname(int hasval, KeyWord *kw)
{
	const char *fmt[2] = { "%s", "%s =" };
	int i;
	if (OutsideComment)
		dumpcomment(&OutsideComment);
	if (primary)
		kw += kw->paoff;
	for(i = 0; i < dumplev; ++i)
		putc(' ', dumpfile);
	fprintf(dumpfile,fmt[hasval],kw->name);
	if (!hasval) {
		if (kw->comment)
			dumpcomment(&kw->comment);
		else if (kw != curkw)
			putc('\n', dumpfile);
		}
	}

 static void
dumpvals0(KeyWord *kw)
{
	Real *r;
	const char **sp;
	int i, *ip, indent, j, n;

	if (!(r = KWval.r) && !(ip = KWval.i) && !(sp = KWval.s))
		return;
	n = KWval.n;
	putc((indent = n > 1) ? '\n' : ' ', dumpfile);
	for(i = 0;;) {
		if (indent) {
			putc('\t', dumpfile);
			for(j = 0; j < dumplev; ++j)
				putc(' ', dumpfile);
			}
		if (r)
			fprintf(dumpfile, "%.15g", r[i]);
		else if (ip)
			fprintf(dumpfile, "%d", ip[i]);
		else
			fprintf(dumpfile, "'%s'", sp[i]);
		if (++i >= n)
			break;
		indent = 1;
		putc('\n', dumpfile);
		}
	if (kw->comment)
		dumpcomment(&kw->comment);
	else
		putc('\n', dumpfile);
	}

 static void (*dumpvals)(KeyWord *kw) = dumpvals0;

 static void
dumpvals1(KeyWord *kw)
{
	Real *r;
	const char **sp;
	int i, *ip, n;

	if ((r = KWval.r) || (ip = KWval.i) || (sp = KWval.s)) {
		n = KWval.n;
		for(i = 0; i < n; ++i) {
			if (r)
				fprintf(dumpfile, " %.15g", r[i]);
			else if (ip)
				fprintf(dumpfile, " %d", ip[i]);
			else
				fprintf(dumpfile, " '%s'", sp[i]);
			}
		}
	if (kw->comment)
		dumpcomment(&kw->comment);
	else
		putc('\n', dumpfile);
	}

 char *
KWscopy(const char *s)
{
	Sbuf *sb;
	char *rv;

	size_t L = strlen(s) + 1;
	if (L >= n_KWsbuf)
		botch("String too long in KWscopy");
	if (KWsbufe - KWsbuf1 < L) {
		if (!KWsbuf->next) {
			KWsbuf->next = sb = (Sbuf*)Alloc("KWscopy", sizeof(Sbuf));
			sb->next = 0;
			}
		KWsbuf = KWsbuf->next;
		KWsbuf1 = KWsbuf->buf;
		KWsbufe = KWsbuf1 + n_KWsbuf;
		}
	strcpy(KWsbuf1, s);
	rv = KWsbuf1;
	KWsbuf1 += L;
	return rv;
	}

 static void
KWvalbuf_inc(void)
{
	Real *r;
	Uint n;

	n = nKWvalbuf << 1;
	r = (Real*)Alloc("KWvalbuf", n*sizeof(Real));
	memcpy(r, KWvalbuf, nKWvalbuf*sizeof(Real));
	free(KWvalbuf);
	KWvalbuf = r;
	nKWvalbuf = n;
	KWvalmax.n <<= 1;
	if (KWval.r) {
		KWval.r = r;
		KWvalmax.r = r + n;
		}
	else if (KWval.i) {
		KWval.i = (int*) r;
		KWvalmax.i = (int*)(r + n);
		}
	else if (KWval.s) {
		KWval.s = (const char**)r;
		KWvalmax.s = (const char**)(r + n);
		}
	else
		botch("Unexpected case in KWvalbuf_inc");
	}

 static void
nidr_bufr_strict(Real r)
{
	int k, n;

	if (KWval.s) {
		squawk("expected a quoted string, but found a number");
		return;
		}
	if (!KWval.r && !KWval.i) {
		squawk("No values may be specified for %s", KWStack->kw->name);
		return;
		}
	if ((n = KWval.n) >= KWvalmax.n)
		KWvalbuf_inc();
	if (KWval.r)
		KWval.r[n] = r;
	else {
		k = (int)r;
		if (k != r)
			squawk("truncating %.17g to %d", r, k);
		KWval.i[n] = k;
		}
	++KWval.n;
	}

 static void
nidr_bufs_strict(const char *s)
{
	if (!KWval.s) {
		squawk("expected a number, but found a quoted string");
		return;
		}
	if (KWval.n >= KWvalmax.n)
		KWvalbuf_inc();
	KWval.s[KWval.n++] = s;
	}

 void
nidr_reset(void)
{
	/* Originally did this in case KWKind_Str of kw_setup(), */
	/* but this leads to confusion with erroneous input. */
	KWsbuf = &KWsbuf0;
	KWsbuf1 = KWsbuf0.buf;
	KWsbufe = KWsbuf0.buf + n_KWsbuf;
	}

 static void
kw_setup(KeyWord *kw, void *g, const char *name)
{
	KWinfo *kwi;
	KeyWord **alt, *kw1, *kwe, **req;
	Uint k, nalt, nn, nreq;
	int *altct;
	size_t len;

	if ((kw1 = kw->kw)) {
		while(!kw1->name) {
			if (!(kw1->kind & KWKind_Stacked)) {
				kw1->kind |= KWKind_Stacked;
				kw_setup(kw1, g, name);
				}
			++kw1;
			}
		if (!kw->nkw)
			kw1 = 0;
		}
	if (!curkw) {
		KWStack = KWStackBot = KWStack0;
		KWStackEnd = KWStack0 + n_KWStack0;
		curkw = kw;
		}
	else if (++KWStack >= KWStackEnd) {
		nn = n_KWStack << 1;
		kwi = (KWinfo*)Alloc("kw_setup", len = nn*sizeof(KWinfo));
		memcpy(kwi, KWStackBot, len >> 1);
		if (KWStackBot != KWStack0)
			free(KWStackBot);
		KWStackBot = kwi;
		KWStackEnd = kwi + nn;
		KWStack = kwi + n_KWStack;
		n_KWStack = nn;
		}
	kwi = KWStack;
	kwi->name = name;
	kwi->kw = kw;
	kwi->kw1 = kw1;
	nalt = nreq = 0;
	if (kw1)
		for(kwe = kw1 + kw->nkw; kw1 < kwe; ++kw1) {
			if (nalt < kw1->alt)
				nalt = kw1->alt;
			if (nreq < kw1->req)
				nreq = kw1->req;
			}
	kwi->nalt = nalt;
	kwi->nreq = nreq;
	alt = req = 0;
	altct = 0;
	if ((nn = nalt + nreq) > 0) {
		nn += 2;
		alt = (KeyWord**)Alloc("kw_setup(alt)",
				len = nn*sizeof(KeyWord*) + (nalt+1)*sizeof(int));
		memset(alt, 0, len);
		req = alt + nalt + 1;
		altct = (int*)(req + nreq + 1);
		/* altct[0], alt[0] and req[0] = "don't care" slots */
		}
	kwi->alt = alt;
	kwi->req = req;
	kwi->altct = altct;
	if (nreq)
		for(kw1 = kwi->kw1; kw1 < kwe; kw1++)	/* kwe was set above */
			req[kw1->req] = kw1;
	if (nalt)
		for(kw1 = kwi->kw1; kw1 < kwe; kw1++)
			if (kw1->kind & KWKind_primary)
				++altct[kw1->alt];
	kwi->g = g;
	KWval.n = 0;
	KWval.i = 0;
	KWval.r = 0;
	KWval.s = 0;
	if ((k = kw->kind & KWKind_Mask)) {
		if (!KWvalmax.r)
			KWvalbuf = (Real *)Alloc("kw_setup(KWValbuf)",
						(nKWvalbuf = 128)*sizeof(Real));
		switch(k) {

		  case KWKind_Int:
			KWval.i = (int*)KWvalbuf;
			KWvalmax.n = (nKWvalbuf*sizeof(Real))/sizeof(int);
			KWvalmax.i = KWval.i + KWvalmax.n;
			break;

		  case KWKind_Real:
			KWval.r = KWvalbuf;
			KWvalmax.r = KWvalbuf + (KWvalmax.n = nKWvalbuf);
			break;

		  case KWKind_Str:
			KWval.s = (const char**)KWvalbuf;
			KWvalmax.n = (nKWvalbuf*sizeof(Real))/sizeof(char*);
			KWvalmax.s = KWval.s + KWvalmax.n;
		  }
		}
	if (!(kwi->needstart = kw->kind & KWKind_Mask)) {
		if (kw->name) {
			if (dumpfile)
				dumpname(0, kw);
			++dumplev;
			}
		if (kw->start)
			(*kw->start)(kw->name, 0, &KWStack->g, kw->vs);
		}
	else if (!kw->start)
		botch("No start routine for %s", kw->name);
	}

 static KeyWord *
kwfind(const char *name, KeyWord *keywds, int n, int *nmatch)
{
	KeyWord *kn, *kn1;
	int k, n0, n1, n2, nn;
	size_t L;

	*nmatch = 0;
	L = strlen(name);
	n0 = 0;
	nn = n;
	while(n > 0) {
		n1 = n >> 1;
		kn = keywds + n1;
		k = strncmp(name, kn->name, L);
		if (k < 0)
			n = n1;
		else if (k > 0) {
			n -= ++n1;
			n0 += n1;
			keywds = kn + 1;
			}
		else {
			/* Found -- check for range of matches. */
			/* Here we use linear search, as we expect */
			/* the range to be small. */
			n = n1 + n0;
			n2 = n + 1;
			if (kn->name[L]) {
				for(kn1 = kn; n2 < nn; ++n2) {
					++kn1;
					if (strncmp(name, kn1->name, L))
						break;
					if (!kn1->name[L])
						goto found1;
					}
				kn1 = kn;
				while(n > 0) {
					--kn1;
					if (strncmp(name, kn1->name, L))
						break;
					if (!kn1->name[L]) {
 found1:
						*nmatch = 1;
						return kn1;
						}
					--n;
					kn = kn1;
					}
				}
			*nmatch = n2 - n;
			return kn;
			}
		}
	return 0;	/* not found */
	}

 static KeyWord *
toomany(const char *name, KeyWord *kw, int nmatch)
{
	int i;
	squawk("\"%s\" is ambiguous; possible matches..", name);
	if (nsquawk <=  NIDR_SQUAWKMAX)
		for(i = 0; i < nmatch; i++, kw++)
			fprintf(stderr, "\t%s\n", kw->name);
	return 0;
	}

 KeyWord *
nidr_keyword(const char *name)
{
	int nmatch;
	KeyWord *kw, *kw1;

	kw = kwfind(name, Dakota_Keyword_Top.kw, Dakota_Keyword_Top.nkw, &nmatch);
	if (nmatch > 1)
		return toomany(name, kw, nmatch);
	else if (kw) {
		if (!(kw1 = curkw))
			kw_setup(kw, KW_g, name);
		if (!strict) {
			if (kw1)
				nidr_keyword_finish();
			kw_setup1(kw);
			}
		}
	return kw;
	}

 static void
valcheck(KeyWord *kw)
{
	Real *r;
	int *z;
	int i, k, n;

	n = KWval.n;
	switch(k = kw->kind & KWKind_Mask) {
	  case KWKind_Int:
		z = KWval.i;
		if (kw->kind & KWKind_strictLb) {
			for(i = 0; i < n; ++i)
				if (z[i] <= kw->Lb) {
					squawk("%s must be > %.0f", kw->name, kw->Lb);
					break;
					}
				}
		else if (kw->kind & KWKind_caneqLb) {
			for(i = 0; i < n; ++i)
				if (z[i] < kw->Lb) {
					squawk("%s must be >= %.0f", kw->name, kw->Lb);
					break;
					}
				}
		if (kw->kind & KWKind_strictUb) {
			for(i = 0; i < n; ++i)
				if (z[i] >= kw->Ub) {
					squawk("%s must be < %.0f", kw->name, kw->Ub);
					break;
					}
				}
		else if (kw->kind & KWKind_caneqUb) {
			for(i = 0; i < n; ++i)
				if (z[i] > kw->Ub) {
					squawk("%s must be >= %.0f", kw->name, kw->Ub);
					break;
					}
				}
		break;
	  case KWKind_Real:
		r = KWval.r;
		if (kw->kind & KWKind_strictLb) {
			for(i = 0; i < n; ++i)
				if (r[i] <= kw->Lb) {
					squawk("%s must be > %g", kw->name, kw->Lb);
					break;
					}
				}
		else if (kw->kind & KWKind_caneqLb) {
			for(i = 0; i < n; ++i)
				if (r[i] < kw->Lb) {
					squawk("%s must be >= %g", kw->name, kw->Lb);
					break;
					}
				}
		if (kw->kind & KWKind_strictUb) {
			for(i = 0; i < n; ++i)
				if (r[i] >= kw->Ub) {
					squawk("%s must be < %g", kw->name, kw->Ub);
					break;
					}
				}
		else if (kw->kind & KWKind_caneqUb) {
			for(i = 0; i < n; ++i)
				if (r[i] > kw->Ub) {
					squawk("%s must be >= %g", kw->name, kw->Ub);
					break;
					}
				}
		break;
	  default:
		botch("Bug: unexpected (kw->kind & KWKind_Mask) = %d in valcheck",n);
	  }
	}

 static void
dispatch_val(KWinfo *kwi)
{
	KeyWord *kw = kwi->kw;

	if (KWval.n) {
		if (dumpfile) {
			dumpname(1, kw);
			dumpvals(kw);
			}
		if (kw->kind & (KWKind_Lb|KWKind_Ub))
			valcheck(kw);
		(*kw->start)(kw->name, &KWval, &kwi->g, kw->vs);
		KWval.n = 0;
		}
	else
		squawk("expected %sone %s value for %s",
			kw->kind & KWKind_List ? "at least " : "",
			valkind[(kw->kind & KWKind_Mask)-1], kw->name);
	kwi->needstart = 0;
	++dumplev;
	}

 static void
oneof(KeyWord *kw, int alt, int n)
{
	KeyWord *kw1, *kwe;

	squawk("One of the following %d entities\nmust be specified for %s..",
		n, kw->name);
	for(kw1 = kw->kw; !kw1->name; ++kw1);
	for(kwe = kw1 + kw->nkw; kw1 < kwe; ++kw1)
		if (kw1->alt == alt && kw1->kind & KWKind_primary)
			fprintf(stderr, "\t%s\n", kw1->name);
	}

 static void
missing_chk(KeyWord *kw1, KeyWord *kwe, KWinfo *kwi)
{
	KeyWord *kw0, *kw2, **req;
	Uint a;
	char seen0[1024], *seen;
	const char *kwname;
	int n;
	size_t nreq;

	/* only issue one error message per missing keyword */

	nreq = 0;
	for(kw0 = kw1; kw1 < kwe; ++kw1)
		if (nreq < kw1->req)
			nreq = kw1->req;
	seen = seen0;
	if (++nreq > sizeof(seen0))
		seen = (char*)Alloc("missing_chk", nreq);
	memset(seen, 0, nreq);
	req = kwi->req;
	for(kw1 = kw0; kw1 < kwe; ++kw1) {
		if (kw1->req && req[kw1->req] && !seen[kw1->req] && kw1->kind & KWKind_primary) {
			seen[kw1->req] = 1;
			a = -1;
			if (!kw1->alt || (n = kwi->altct[a = kw1->alt]) <= 1) {
				if (!(kwname = kwi->name))
					kwname = "<NIDRBUG>";
				for(kw2 = kw1;; ++kw2) {
					if (kw2->alt == a && kw2->kind & KWKind_primary)
						break;
					if (++kw2 >= kwe)
						botch("Bug in missing_chk");
					}
				squawk("%s must be specified for %s",
					kw2->name, kwname);
				}
			else
				oneof(kwi->kw, kw1->alt, n);
			}
		}
	if (seen != seen0)
		free(seen);
	}

 static void
finalize(KWinfo *kwi)
{
	KeyWord *kw, *kw1, *kwe, **req;

	kw = kwi->kw;
	kw->kind &= ~KWKind_Stacked;
	if (kwi->needstart)
		dispatch_val(kwi);
	if (kw->name)
		--dumplev;
	if (kw->final)
		(*kw->final)(kw->name, 0, &kwi->g, kw->vf);
	if (kwi->alt) {
		if (kwi->nreq) {
			req = kwi->req;
			for(kw1 = kwi->kw1, kwe = kw1 + kw->nkw; kw1 < kwe; ++kw1)
				if (kw1->req && req[kw1->req]) {
					missing_chk(kw1, kwe, kwi);
					break;
					}
			}
		free(kwi->alt);
		}
	}

 static KeyWord *
nidr_identifier_strict(const char *name)
{
	KWinfo *kwi, *kwi1;
	KeyWord *kw, *kw1;
	int n, nmatch;
	size_t height;

	if (!curkw)
		botch("curkw = 0 in nidr_identifier");
	kwi = KWStack;
	if (kwi->needstart)
		dispatch_val(kwi);
	for(kwi1 = kwi;;) {
		kw1 = kwi->kw;
		if ((kw = kwfind(name, kwi->kw1, kw1->nkw, &nmatch)))
			break;
		if (kwi == KWStackBot)
			return 0;
		if ((--kwi)->kw->name)
			kwi1 = kwi;
		}
	if (nmatch > 1)
		return toomany(name, kw, nmatch);
	while(KWStack > kwi1)
		finalize(KWStack--);
	height = kwi - KWStackBot;
	kw_setup(kw, kwi->g, name);
	kwi = KWStackBot + height;	/* in case kw_setup reallocated KWStack */
	if (kw->alt) {
		if ((kw1 = kwi->alt[n = kw->alt])) {
			if (strcmp(kw1->name, name))
				squawk("%s and %s are mutually exclusive",
					kw1->name, name);
			else
				squawk("%s was already specified", name);
			}
		else
			kwi->alt[n] = kw;
		}
	if (kw->req) {
		if (kwi->req[n = kw->req])
			kwi->req[n] = 0;
		else if (!kw->alt)
			squawk("%s specified more than once", name);
		}
	return kw;
	}

 static void
nidr_keyword_finish(void)
{
	if (!strict)
		kw_finish2();
	for(;;--KWStack) {
		finalize(KWStack);
		if (KWStack == KWStackBot)
			break;
		}
	if (!strict)
		kw_finish3();
	curid = curkw = 0;
	}

 const char*
nidr_keyword_name(void)
{ return curkw ? curkw->name : "<none>"; }

/* Some of the above assumes strict nesting according to dakota.input.nspec. */
/* Code here is meant to relax this assumption, allowing more flexibility in */
/* the order of identifiers within a DAKOTA "keyword". */

 typedef struct KWpair KWpair;
 typedef struct KWmblk KWmblk;

 struct
KWmblk {
	KWmblk *next;
	size_t len;
	/* memory of length len immediately follows */
	};

 struct
KWpair {
	KeyWord *kw;
	KWseen *kws;
	};

 enum{ KWmblk_gulp = 32000 };

 static AVL_Tree *AVLT, *AVLKWP;
 static KWseen *KW_cur, **KW_p, **KW_pe, KWmissing, *KWs0;
 static KWmblk *KWmblk0, *KWmblk1;
 static const char *KWmem0, *KWmem1;

 typedef struct
AVLCmpInfo {
	KWseen **found[2];
	int nfound;
	int inexact;
	} AVLCmpInfo;

 static int
avlcmp(void *v, KWseen **a, KWseen **b)
{
	AVLCmpInfo *AI = (AVLCmpInfo*)v;
	KWseen *ksa, *ksb;
	const char *s, *t;

	s = (ksa = *a)->name;
	t = (ksb = *b)->name;
	for(; *s == *t; ++s, ++t)
		if (!*s)
			return 0;
	if ((!*s && !ksa->kw && ksb->kw)
	  ||(!*t && !ksb->kw && ksa->kw)) {
		/* inexact match */
		if (AI->nfound == 0
		|| (AI->nfound == 1 && AI->found[0] != b))
			AI->found[AI->nfound++] = b;
		return AI->inexact;
		}
	return *s - *t;
	}

 static int
kwpcmp(void *v, KWpair *a, KWpair *b)
{
	if (a->kw == b->kw)
		return 0;
	return a->kw > b->kw ? 1 : -1;
	}

 static void
KWmeminit(void)
{
	KWmblk0 = KWmblk1 = (KWmblk*)Alloc("KWmeminit",
			sizeof(KWmblk) + KWmblk_gulp);
	KWmem0 = (char*)(KWmblk0 + 1);
	KWmem1 = KWmem0 + KWmblk_gulp;
	KWmblk0->len = KWmblk_gulp;
	KWmblk0->next = 0;
	KWmissing.mnext = KWmissing.mprev = &KWmissing;
	KW_cur = 0;
	memset(&KWval, 0, sizeof(KWval));
	KWvalbuf = (Real *)Alloc("kw_setup(KWValbuf)", (nKWvalbuf = 128)*sizeof(Real));
	ToClear = ToClear0 = (KeyWord**)Alloc("kw_setup(ToClear)", 256*sizeof(KeyWord*));
	ToClearEnd = ToClear0 + 256;
	}

 static void
KWmembump(size_t L)
{
	KWmblk *mb, *mb1;
	size_t L1;

	for(L1 = KWmblk_gulp; L1 < L; L1 <<= 1);
	if ((mb = mb1 = KWmblk1->next) && L1 <= mb->len)
		L1 = mb->len;
	else {
		KWmblk1->next = mb = (KWmblk*)Alloc("KWmembump", L1 + sizeof(KWmblk));
		mb->len = L1;
		mb->next = mb1;
		}
	KWmblk1 = mb;
	KWmem0 = (char*)(mb+1);
	KWmem1 = KWmem0 + L1;
	}

 static void *
KWgetmem(size_t L)	/* for aligned memory */
{
	void *rv;

	L = (L + sizeof(Real) - 1) & ~(sizeof(Real) - 1);
	if (KWmem1 - KWmem0 < L)
		KWmembump(L);
	rv = (void*)KWmem0;
	KWmem0 += L;
	return rv;
	}

 static KWseen **
KWhash(const char *s, KeyWord *kw)
{
	AVLCmpInfo AI;
	KWseen KW0, *KW0p, *kws, **kwsp;
	const char *sa, *sb;

	AI.nfound = 0;
	AI.inexact = -1;
	AVL_setv(AVLT, &AI);
	KW0.name = s;
	KW0.kw = kw;
	KW0p = &KW0;
	curkws = 0;
	if ((kwsp = (KWseen**)AVL_find((const Element*)&KW0p, AVLT)))
		return kwsp;
	if (AI.nfound) {
		if (AI.nfound == 1) {
			AI.inexact = 1;
			AVL_find((const Element*)&KW0p, AVLT);
			if (AI.nfound == 1)
				return AI.found[0];
			}
		sa = (*AI.found[0])->name;
		sb = (*AI.found[1])->name;
		if (kw)
			squawk("Both '%s' and '%s' match '%s'",
				sa, sb, s);
		else
			squawk("'%s' is ambiguous:\n\tit matches both '%s' and '%s'",
				s, sa, sb);
		}
	kws = (KWseen*)KWgetmem(sizeof(KWseen));
	memset(kws, 0, sizeof(KWseen));
	if ((kws->kw = kw))
		s = kw->name;
	else {
		curkws = kws;
		kws->mnext = &KWmissing;
		KWmissing.mprev = (kws->mprev = KWmissing.mprev)->mnext = kws;
		s = KWscopy(s);
		}
	kws->name = s;
	if (KW_p >= KW_pe) {
		KW_p = (KWseen**)KWgetmem(32*sizeof(KWseen*));
		KW_pe = KW_p + 32;
		}
	*(kwsp = KW_p++) = kws;
	AVL_insert((const Element*)kwsp, AVLT);
	return kwsp;
	}

 static void
mixed_squawk(void)
{
	squawk("values for %s cannot be both strings and numbers",
		KW_cur->name);
	}

 static void
nidr_bufr_relaxed(Real r)
{
	int n;

	if (!(n = KWval.n)) {
		KWval.r = KWvalbuf;
		KWvalmax.r = KWvalbuf + (KWvalmax.n = nKWvalbuf);
		}
	else if (KWval.s) {
		mixed_squawk();
		return;
		}
	if (n >= KWvalmax.n)
		KWvalbuf_inc();
	KWval.r[KWval.n++] = r;
	}

 static void
nidr_bufs_relaxed(const char *s)
{
	int n;

	if (!(n = KWval.n)) {
		KWval.s = (const char**)KWvalbuf;
		KWvalmax.n = (nKWvalbuf*sizeof(Real))/sizeof(char*);
		KWvalmax.s = KWval.s + KWvalmax.n;
		}
	else if (KWval.r) {
		mixed_squawk();
		return;
		}
	if (n >= KWvalmax.n)
		KWvalbuf_inc();
	KWval.s[KWval.n++] = s;
	}

 static void
kw_finish1(KWseen *kws)
{
	int n;
	size_t L;

	kws->nvals = n = KWval.n;
	KWval.n = 0;
	if (KWval.r) {
		L = n*sizeof(Real);
		memcpy(kws->rvals = (Real*)KWgetmem(L), KWval.r, L);
		KWval.r = 0;
		}
	else if (KWval.s) {
		L = n*sizeof(char*);
		memcpy(kws->svals = (char**)KWgetmem(L), KWval.s, L);
		KWval.s = 0;
		}
	}

 static void*
Alloc1(size_t len)
{
	void *rv = malloc(len);
	if (!rv) {
		fprintf(stderr, "malloc(%lu) failure in Alloc1\n", (unsigned long)len);
		exit(1);
		}
	return rv;
	}

 static void
AVL_Clear(void)
{
	while(ToClear > ToClear0)
		(*--ToClear)->kind &= ~KWKind_Hashed;
	AVL_Tree_free(&AVLT);
	if (AVLKWP)
		AVL_Tree_free(&AVLKWP);
	}

 static void
kw_setup1(KeyWord *kw)
{
	KWseen *kws, *kws1;
	KeyWord *kw1, *kwe;

	if (!KWmblk0)
		KWmeminit();
	if (AVLT)
		AVL_Clear();
	AVLT = AVL_Tree_alloc(0, (AVL_Elcomp)avlcmp, Alloc1);
	KW_cur = KWs0 = kws = (KWseen*)KWgetmem(sizeof(KWseen));
	memset(kws, 0, sizeof(KWseen));
	kws->name = kw->name;
	kws->kw = kw;
	kws->lcn = &kws->mprev;
	if ((kw1 = kw->kw)) {
		while(!kw1->name)
			++kw1;
		for(kwe = kw1 + kw->nkw; kw1 < kwe; ++kw1) {
			kws1 = *KWhash(kw1->name, kw1);
			kws1->parent = kws;
			}
		}
	}

 static KWseen**
kw_setup3(KWseen **kwtodo1, KWseen *kws, KeyWord *kw, int n)
{
	KWseen *kws1, **kwsp;
	KeyWord *kwe;

	for(kwe = kw + n; kw < kwe; ++kw) {
		kwsp = KWhash(kw->name, kw);
		kws1 = *kwsp;
		if (kws1->comment) {
			kw->comment = kws1->comment;
			kws1->comment = 0;
			}
		if (kws1->parent) {
			kws1 = (KWseen*)KWgetmem(sizeof(KWseen));
			memset(kws1, 0, sizeof(KWseen));
			kws1->kw = kw;
			kws1->name = kw->name;
			*kwsp = kws1;
			}
		kws1->parent = kws;
		if (!kws1->kw) {
			kws1->mprev->mnext = kws1->mnext;
			kws1->mnext->mprev = kws1->mprev;
			*kwtodo1 = kws1;
			kwtodo1 = kws1->lcn = &kws1->mprev;
			*kws->lcn = kws1;
			kws->lcn = &kws1->mnext;
			}
		kws1->kw = kw;
		}
	return kwtodo1;
	}

 static void
bumpToClear(void)
{
	KeyWord **ntc;
	size_t L, L1;

	L = ToClearEnd - ToClear0;
	L1 = L << 1;
	ntc = (KeyWord**)Alloc("bumpToClear", L1*sizeof(KeyWord*));
	memcpy(ntc, ToClear0, L*sizeof(KeyWord*));
	free(ToClear0);
	ToClear0 = ntc;
	ToClear  = ntc + L;
	ToClearEnd = ntc + L1;
	}

 static void
kw_setup2(KWseen *kws)
{
	KWpair kwp, *pkwp;
	KWseen *kws1, *kws2, *kws3, *kwtodo, **kwtodo1, **pkws;
	KeyWord *kw, *kw1;

	kwtodo1 = &kwtodo;
	for(;;) {
		kw = kws->kw;
		if ((kw1 = kw->kw)) {
			kws2 = kws;
			while(!kw1->name) {
				if (!AVLKWP)
					AVLKWP = AVL_Tree_alloc(0, (AVL_Elcomp)kwpcmp, Alloc1);
				if (kw1->kind & KWKind_Hashed) {
					kwp.kw = kw1->kw;
					kwp.kws = 0;
					pkwp = (KWpair*)AVL_find((const Element*)&kwp, AVLKWP);
					kws2 = pkwp->kws;
					}
				else {
					if (ToClear >= ToClearEnd)
						bumpToClear();
					*ToClear++ = kw1;
					kw1->kind |= KWKind_Hashed;
					pkwp = (KWpair*)KWgetmem(sizeof(KWpair) + sizeof(KWseen));
					kws1 = (KWseen*)(pkwp + 1);
					pkwp->kw = kw1->kw;
					pkwp->kws = kws1;
					memset(kws1, 0, sizeof(KWseen));
					kws1->kw = kw1;
					kws1->name = kws->name;
					kws1->lcn = &kws1->mprev;
					kws1->parent = kws2;
					*kws2->lcn = 0;
					for(pkws = &kws2->mprev;
						(kws3 = *pkws) && !kws3->name;
						pkws = &kws3->mnext);
					kws1->mnext = *pkws;
					if (pkws == kws2->lcn)
						kws2->lcn = &kws1->mnext;
					kws2 = *pkws = kws1;
					kwtodo1 = kw_setup3(kwtodo1, kws1, kw1->kw, kw1->nkw);
					AVL_insert((const Element*)pkwp, AVLKWP);
					}
				++kw1;
				}
			if (kw->nkw)
				kwtodo1 = kw_setup3(kwtodo1, kws2, kw1, kw->nkw);
			}
		*kwtodo1 = 0;
		if (!kwtodo)
			break;
		kws = kwtodo;
		kw = kws->kw;
		if (!(kwtodo = kwtodo->mprev))
			kwtodo1 = &kwtodo;
		}
	}

 static KeyWord *
nidr_identifier_relaxed(const char *name)
{
	KWseen *kws, *kws1;
	KeyWord *kw;

	kw_finish1(KW_cur);
	KW_cur = kws = *KWhash(name, 0);
	if ((kw = kws->kw)) {
		curid = kw;
		if (kws->lcn)
			squawk("'%s' already seen", kw->name);
		else {
			if (kws->comment) {
				kw->comment = kws->comment;
				kws->comment = 0;
				}
			kws->lcn = &kws->mprev;
			kws1 = kws->parent;
			*kws1->lcn = kws;
			kws1->lcn = &kws->mnext;
			if (kw->kw)
				kw_setup2(kws);
			}
		}
	return (KeyWord*)kws;	/* just needs to be nonzero; won't be dereferenced */
	}

 static void
num_expected(KeyWord *kw, int n)
{
	squawk("expected numerical value%s for %s, not quoted strings",
		"s" + (n == 1), kw->name);
	}

 static void
kw_process(KWseen *kws)
{
	KWseen *kws1;
	KeyWord *kw;
	Real *r;
	Uint k;
	int i, n;

	kw = kws->kw;
	if (kw->name) {
		if (kws != KWs0 && !nidr_identifier_strict(kw->name))
			botch("nidr_identifier_strict did not find \"%s\"", kw->name);
		if ((n = KWval.n = kws->nvals)) {
			KWval.i = 0;
			KWval.r = 0;
			KWval.s = 0;
			switch(k = kw->kind & KWKind_Mask) {
			  case 0:
				squawk("No values may be specified for %s", kw->name);
				break;

			  case KWKind_Int:
				if (!(r = kws->rvals)) {
					num_expected(kw,n);
					break;
					}
				KWval.i = (int*)KWvalbuf;
				for(i = 0; i < n; i++)
					KWval.i[i] = (int)r[i];
				break;

			  case KWKind_Real:
				if (!(r = kws->rvals)) {
					num_expected(kw,n);
					break;
					}
				KWval.r = r;
				break;

			  case KWKind_Str:
				if (!(KWval.s = (const char **)kws->svals))
					squawk("expected string value%s for %s",
						"s" + (n == 1), kw->name);
			  }
			}
		}
	*kws->lcn = 0;
	for(kws1 = kws->mprev; kws1; kws1 = kws1->mnext)
		kw_process(kws1);
	}

 static void
kw_finish2(void)
{
	KWseen *kws, *kwe;

	kw_finish1(KW_cur);
	kwe = &KWmissing;
	for(kws = KWmissing.mnext; kws != kwe; kws = kws->mnext) {
		squawk("unrecognized identifier '%s'", kws->name);
		}
	KWmissing.mnext = KWmissing.mprev = &KWmissing;
	kw_process(KWs0);
	KWs0 = 0;
	AVL_Clear();
	}

 static void
kw_finish3(void)
{
	KWmblk1 = KWmblk0;
	KWmem0 = (char*)(KWmblk0 + 1);
	KWmem1 = KWmem0 + KWmblk_gulp;
	KW_p = KW_pe = 0;
	}

 void (*nidr_bufr)(Real) = nidr_bufr_relaxed;
 void (*nidr_bufs)(const char*) = nidr_bufs_relaxed;
 KeyWord *(*nidr_identifier)(const char*) = nidr_identifier_relaxed;

 void
nidr_set_strict(int n)
{
	if ((strict = n)) {
		nidr_bufr = nidr_bufr_strict;
		nidr_bufs = nidr_bufs_strict;
		nidr_identifier = nidr_identifier_strict;
		}
	else {
		nidr_bufr = nidr_bufr_relaxed;
		nidr_bufs = nidr_bufs_relaxed;
		nidr_identifier = nidr_identifier_relaxed;
		}
	}

 int
nidr_cleanup(void)
{
	KWmblk *mb, *mb1;
	Sbuf *sb, *sb1;

	if (curkw)
		nidr_keyword_finish();
	if (dumpfile) {
		if (OutsideComment)
			dumpcomment(&OutsideComment);
		fclose(dumpfile);
		dumpfile = 0;
		if (nidr_comment)
			comment_reset();
		}
	if (ToClear0) {
		free(ToClear0);
		ToClear = ToClear0 = 0;
		}
	if ((mb1 = KWmblk0)) {
		KWmblk0 = 0;
		do {
			mb = mb1;
			mb1 = mb->next;
			free(mb);
			} while(mb1);
		}
	if (KWvalbuf) {
		free(KWvalbuf);
		KWvalbuf = 0;
		}
	if ((sb1 = KWsbuf0.next)) {
		KWsbuf0.next = 0;
		do {
			sb = sb1;
			sb1 = sb->next;
			free(sb);
			} while(sb1);
		}
	if (AVLT)
		AVL_Clear();
	return nidr_parse_error();
	}

 void
nidr_setup(const char *parser, FILE *df)
{
	const char *s;
	int comkeep, oneline;

	if (!(s = parser))
		return;
	if (!strncmp(s,"nidr",4))
		s += 4;
	if (!strncmp(parser,"strict",6)) {
		nidr_set_strict(1);
		s += 6;
		}
	comkeep = oneline = 0;
	if (*s == '-') for(;;) {
		switch(*++s) {
		  case '1':
			++oneline;
			continue;
		  case 'p':
			++primary;
			continue;
		  case 'c':
			++comkeep;
			continue;
		  }
		break;
		}
	if (df)
		dumpfile = df;
	else if (s[0] == ':' && s[1]) {
		dumpfile = df = fopen(++s,"w");
		if (!dumpfile) {
			fprintf(stderr, "Cannot open \"%s\"\n", s);
			exit(1);
			}
		}
	if (df) {
		if (oneline)
			dumpvals = dumpvals1;
		if (comkeep)
			comment_setup();
		}
	}
