/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

#ifndef DATA_IO_H
#define DATA_IO_H

#include "data_types.h"


namespace Dakota {

// -----------------------
// templated I/O functions
// -----------------------

/// global ostream insertion operator for std::set
template <class T>
ostream& operator<<(ostream& s, const std::set<T>& data)
{
  for (typename std::set<T>::const_iterator cit = data.begin();
       cit != data.end(); ++cit)
    s << "                     " << *cit << '\n';
  return s;
}

/// global MPIUnpackBuffer extraction operator for std::set
template <class T>
MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s, std::set<T>& data)
{
  data.clear();
  size_t len;
  s >> len;
  T val;
  for (size_t i=0; i<len; ++i){
    s >> val; 
    data.insert(val);
  }
  return s;
}

/// global MPIPackBuffer insertion operator for std::set
template <class T>
MPIPackBuffer& operator<<(MPIPackBuffer& s, const std::set<T>& data)
{
  size_t len = data.size();
  s << len;
  for (typename std::set<T>::const_iterator cit = data.begin();
       cit != data.end(); ++cit)
    s << *cit;
  return s;
}

/// global MPIPackBuffer insertion operator for Teuchos::SerialDenseVector
template <typename OrdinalType, typename ScalarType> 
MPIPackBuffer& operator<<(MPIPackBuffer& s,
  const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& data)
{
  OrdinalType i, n = data.length();
  s << n;
  for (i=0; i<n; ++i)
    s << data[i];
  return s;
}

/// global MPIPackBuffer insertion operator for Teuchos::SerialDenseMatrix
template <typename OrdinalType, typename ScalarType> 
MPIPackBuffer& operator<<(MPIPackBuffer& s,
  const Teuchos::SerialDenseMatrix<OrdinalType, ScalarType>& data)
{
  OrdinalType i, j, n = data.numRows(), m = data.numCols();
  s << n << m;
  for (i=0; i<n; ++i)
    for (j=0; j<m; ++j)
      s << data(i,j);
  return s;
}

/// global MPIPackBuffer insertion operator for Teuchos::SerialSymDenseMatrix
template <typename OrdinalType, typename ScalarType> 
MPIPackBuffer& operator<<(MPIPackBuffer& s,
  const Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType>& data)
{
  OrdinalType i, j, n = data.numRows();
  s << n;
  for (i=0; i<n; ++i)
    for (j=0; j<=i; ++j)
      s << data(i,j);
  return s;
}

/// global MPIUnpackBuffer extraction operator for Teuchos::SerialDenseVector
template <typename OrdinalType, typename ScalarType> 
MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s,
  Teuchos::SerialDenseVector<OrdinalType, ScalarType>& data)
{
  OrdinalType i, n;
  s >> n;
  data.sizeUninitialized(n);
  for(i=0; i<n; ++i)
    s >> data[i];
  return s;
}

/// global MPIUnpackBuffer extraction operator for Teuchos::SerialDenseMatrix
template <typename OrdinalType, typename ScalarType> 
MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s,
  Teuchos::SerialDenseMatrix<OrdinalType, ScalarType>& data)
{
  OrdinalType i, j, n, m;
  s >> n >> m;
  data.shapeUninitialized(n, m);
  for (i=0; i<n; ++i)
    for (j=0; j<m; ++j)
      s >> data(i,j);
  return s;
}

/// global MPIUnpackBuffer extraction operator for Teuchos::SerialSymDenseMatrix
template <typename OrdinalType, typename ScalarType> 
MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s,
  Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType>& data)
{
  OrdinalType i, j, n;
  s >> n;
  data.shapeUninitialized(n);
  for (i=0; i<n; ++i)
    for (j=0; j<=i; ++j)
      s >> data(i,j);
  return s;
}


template <typename OrdinalType, typename ScalarType>
void read_data(istream& s,
	       Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv)
{
  OrdinalType len = sdv.length();
  for (OrdinalType i=0; i<len; i++)
    s >> sdv[i];
}


template <typename OrdinalType, typename ScalarType>
void read_data(istream& s,
	       Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv,
	       StringMultiArray& label_array)
{
  OrdinalType len = sdv.length();
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in read_data(istream) does not "
	 << "equal length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  for (OrdinalType i=0; i<len; i++)
    s >> sdv[i] >> label_array[i];
}


template <typename OrdinalType, typename ScalarType>
void read_data_partial(istream& s, size_t start_index, size_t num_items,
		       Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv)
{
  size_t end = start_index + num_items;
  if (end > sdv.length()) { // start_index >= 0 since size_t
    Cerr << "Error: indexing in Vector<T>::read_data_partial(istream) exceeds "
	 << "length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  for (OrdinalType i=start_index; i<end; i++)
    s >> sdv[i];
}


template <typename OrdinalType, typename ScalarType>
void read_data_partial(istream& s, size_t start_index, size_t num_items,
		       Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv,
		       StringMultiArray& label_array)
{
  size_t end = start_index + num_items;
  OrdinalType len = sdv.length();
  if (end > len) { // start_index >= 0 since size_t
    Cerr << "Error: indexing in read_data_partial(istream) exceeds "
	 << "length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in read_data_partial(istream) "
	 << "does not equal length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  for (OrdinalType i=start_index; i<end; i++)
    s >> sdv[i] >> label_array[i];
}


template <typename OrdinalType, typename ScalarType>
void read_data_tabular(istream& s,
		       Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv)
{
  // differs from read_data(istream& s) only in exception handling
  OrdinalType len = sdv.length();
  for (OrdinalType i=0; i<len; i++) {
    if (s)
      s >> sdv[i];
    else {
      char err[80];
      sprintf(err,
	      "At EOF: insufficient tabular data for SerialDenseVector[%d]", i);
      throw String(err);
    }
  }
}


template <typename OrdinalType, typename ScalarType>
void read_data_annotated(istream& s,
  Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv,
  StringMultiArray& label_array)
{
  OrdinalType len;
  s >> len;
  if( len != sdv.length() )
    sdv.sizeUninitialized(len);
  if( len != label_array.size() )
    label_array.resize(boost::extents[len]);
  for (OrdinalType i=0; i<len; i++)
    s >> sdv[i] >> label_array[i];
}


template <typename OrdinalType, typename ScalarType>
void write_data(ostream& s,
		const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv)
{
  OrdinalType len = sdv.length();
  s.setf(ios::scientific);
  s << setprecision(write_precision);
  for (OrdinalType i=0; i<len; i++)
    s << "                     " << setw(write_precision+7) << sdv[i] << '\n';
}


template <typename OrdinalType, typename ScalarType>
void write_data(ostream& s,
		const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv,
		const StringMultiArray& label_array)
{
  OrdinalType len = sdv.length();
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in write_data(ostream) does not "
	 << "equal length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  s.setf(ios::scientific);
  s << setprecision(write_precision);
  for (OrdinalType i=0; i<len; i++)
    s << "                     " << setw(write_precision+7) << sdv[i] << ' '
      << label_array[i] << '\n';
}


template <typename OrdinalType, typename ScalarType>
void write_data(ostream& s,
		const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv,
		const StringArray& label_array)
{
  OrdinalType len = sdv.length();
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in write_data(ostream) does not "
	 << "equal length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  s.setf(ios::scientific);
  s << setprecision(write_precision);
  for (OrdinalType i=0; i<len; i++)
    s << "                     " << setw(write_precision+7) << sdv[i] << ' '
      << label_array[i] << '\n';
}


template <typename OrdinalType, typename ScalarType>
void write_data_aprepro(ostream& s,
  const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv,
  const StringMultiArray& label_array)
{
  OrdinalType len = sdv.length();
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in write_data_aprepro(ostream) "
	 << "does not equal length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  s.setf(ios::scientific);
  s << setprecision(write_precision);
  for (OrdinalType i=0; i<len; i++)
    s << "                    { " << setw(15) << setiosflags(ios::left) 
      << label_array[i].data() << resetiosflags(ios::adjustfield) << " = " 
      << setw(write_precision+7) << sdv[i] <<" }\n";
}


template <typename OrdinalType, typename ScalarType>
void write_data_partial(ostream& s, size_t start_index, size_t num_items,
  const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv)
{
  size_t end = start_index + num_items;
  if (end > sdv.length()) { // start_index >= 0 since size_t
    Cerr << "Error: indexing in write_data_partial(ostream) exceeds "
	 << "length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  s.setf(ios::scientific);
  s << setprecision(write_precision);
  for (OrdinalType i=start_index; i<end; i++)
    s << "                     " << setw(write_precision+7) << sdv[i] << '\n';
}


template <typename OrdinalType, typename ScalarType>
void write_data_partial(ostream& s, size_t start_index, size_t num_items,
  const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv, 
  const StringMultiArray& label_array)
{
  size_t end = start_index + num_items;
  OrdinalType len = sdv.length();
  if (end > len) { // start_index >= 0 since size_t
    Cerr << "Error: indexing in write_data_partial(ostream) exceeds "
	 << "length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in write_data_partial(ostream) "
	 << "does not equal length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  s.setf(ios::scientific);
  s << setprecision(write_precision);
  for (OrdinalType i=start_index; i<end; i++)
    s << "                     " << setw(write_precision+7) << sdv[i] << ' '
      << label_array[i] << '\n';
}


template <typename OrdinalType, typename ScalarType>
void write_data_partial_aprepro(ostream& s, size_t start_index,
  size_t num_items,
  const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv, 
  const StringMultiArray& label_array)
{
  size_t end = start_index + num_items;
  OrdinalType len = sdv.length();
  if (end > len) { // start_index >= 0 since size_t
    Cerr << "Error: indexing in write_data_partial_aprepro(ostream) "
	 << "exceeds length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in write_data_partial_aprepro"
	 << "(ostream) does not equal length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  s.setf(ios::scientific);
  s << setprecision(write_precision);
  for (OrdinalType i=start_index; i<end; i++)
    s << "                    { " << setw(15) << setiosflags(ios::left) 
      << label_array[i].data() << resetiosflags(ios::adjustfield) << " = " 
      << setw(write_precision+7) << sdv[i] <<" }\n";
}


template <typename OrdinalType, typename ScalarType>
void write_data_annotated(ostream& s,
  const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv,
  const StringMultiArray& label_array)
{
  OrdinalType len = sdv.length();
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in write_data_annotated(ostream) "
	 << "does not equal length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  s.setf(ios::scientific);
  s << len << ' ' << setprecision(write_precision);
  for (OrdinalType i=0; i<len; i++)
    s << sdv[i] << ' ' << label_array[i] << ' ';
}


template <typename OrdinalType, typename ScalarType>
void write_data_tabular(ostream& s,
  const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv)
{
  OrdinalType len = sdv.length();
  s << setprecision(10) << resetiosflags(ios::floatfield);
  for (OrdinalType i=0; i<len; i++)
    s << setw(14) << sdv[i] << ' ';
}


template <typename OrdinalType, typename ScalarType>
void write_data_partial_tabular(ostream& s, size_t start_index,
  size_t num_items,
  const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv)
{
  size_t end = start_index + num_items;
  if (end > sdv.length()) { // start_index >= 0 since size_t
    Cerr << "Error: indexing in write_data_partial_tabular(ostream) "
	 << "exceeds length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  s << setprecision(10) << resetiosflags(ios::floatfield);
  for (OrdinalType i=start_index; i<end; i++)
    s << setw(14) << sdv[i] << ' ';
}


inline void write_data_tabular(ostream& s, StringMultiArrayConstView ma)
{
  s << setprecision(10) << resetiosflags(ios::floatfield);
  size_t size_ma = ma.size();
  for (size_t i=0; i<size_ma; ++i)
    s << setw(14) << ma[i] << ' ';
}


/// global istream extraction operator for Vector
template <typename OrdinalType, typename ScalarType>
inline istream& operator>>(istream& s,
  Teuchos::SerialDenseVector<OrdinalType, ScalarType>& data)
{ read_data(s, data); return s; }


/// global ostream insertion operator for Vector
template <typename OrdinalType, typename ScalarType>
inline ostream& operator<<(ostream& s,
  const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& data)
{ write_data(s, data); return s; }


template <typename OrdinalType, typename ScalarType>
void read_data(BiStream& s,
	       Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv,
	       StringMultiArray& label_array)
{
  OrdinalType len;
  s >> len;
  if( len != sdv.length() )
    sdv.sizeUninitialized(len);
  if( len != label_array.size() )
    label_array.resize(boost::extents[len]);
  for (OrdinalType i=0; i<len; i++)
    s >> sdv[i] >> label_array[i];
}


template <typename OrdinalType, typename ScalarType>
void write_data(BoStream& s,
		const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv,
		const StringMultiArray& label_array)
{
  OrdinalType len = sdv.length();
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in write_data(BoStream) does not "
	 << "equal length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  s << len;
  for (OrdinalType i=0; i<len; i++)
    s << sdv[i] << label_array[i];
}


template <typename OrdinalType, typename ScalarType>
void read_data(MPIUnpackBuffer& s,
	       Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv,
	       StringMultiArray& label_array)
{
  OrdinalType len;
  s >> len;
  if( len != sdv.length() )
    sdv.sizeUninitialized(len);
  if( len != label_array.size() )
    label_array.resize(boost::extents[len]);
  for (OrdinalType i=0; i<len; i++)
    s >> sdv[i] >> label_array[i];
}


template <typename OrdinalType, typename ScalarType>
void write_data(MPIPackBuffer& s,
		const Teuchos::SerialDenseVector<OrdinalType, ScalarType>& sdv,
		const StringMultiArray& label_array)
{
  OrdinalType len = sdv.length();
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in write_data(MPIPackBuffer) "
	 << "does not equal length of SerialDenseVector." << endl;
    abort_handler(-1);
  }
  s << len;
  for (OrdinalType i=0; i<len; i++)
    s << sdv[i] << label_array[i];
}

template <typename OrdinalType, typename ScalarType>
void read_data(std::istream& s,
               Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType>& m)
{
  for (OrdinalType i=0; i<m.numRows(); ++i)
    for (OrdinalType j=0; j<m.numCols(); ++j)
      s >> m(i,j);
}

template <typename OrdinalType, typename ScalarType>
void read_data(BiStream& s,
               Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType>& m)
{
  for (OrdinalType i=0; i<m.numRows(); ++i)
    for (OrdinalType j=0; j<m.numCols(); ++j)
      s >> m(i,j);
}

template <typename OrdinalType, typename ScalarType>
void read_data(MPIUnpackBuffer& s,
               Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType>& m)
{
  for (OrdinalType i=0; i<m.numRows(); ++i)
    for (OrdinalType j=0; j<m.numCols(); ++j)
      s >> m(i,j);
}

template <typename OrdinalType, typename ScalarType>
void write_data(std::ostream& s,
                const Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType>& m,
                bool brackets, bool row_rtn,
                bool final_rtn)
{
  s.setf(std::ios::scientific); // formatting optimized for T = double
  s << std::setprecision(Dakota::write_precision);
  if (brackets)
    s << "[[ ";
  for (OrdinalType i=0; i<m.numRows(); ++i) {
    for (OrdinalType j=0; j<m.numCols(); ++j)
      s << std::setw(Dakota::write_precision+7) << m(i,j) << ' ';
    // NOTE: newlines on every 4th component (as in the row vector case)
    // could lead to ambiguity in the matrix case.
    if (row_rtn && i!=m.numRows()-1)
      s << "\n   ";
  }
  if (brackets)
    s << "]] ";
  if (final_rtn)
    s << '\n';
}

template <typename OrdinalType, typename ScalarType>
void write_data(BoStream& s,
                const Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType>& m)
{
  for (OrdinalType i=0; i<m.numRows(); ++i)
    for (OrdinalType j=0; j<m.numCols(); ++j)
      s << m(i,j);
}

// WJB: BoStream and MPIPackBuffer common BaseClass? need to duplicate func??
template <typename OrdinalType, typename ScalarType>
void write_data(MPIPackBuffer& s,
                const Teuchos::SerialSymDenseMatrix<OrdinalType, ScalarType>& m)
{
  for (OrdinalType i=0; i<m.numRows(); ++i)
    for (OrdinalType j=0; j<m.numCols(); ++j)
      s << m(i,j);
}

} // namespace Dakota

#endif // DATA_IO_H
