/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2001, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 NonDSparseGrid
//- Description: Implementation code for NonDSparseGrid class
//- Owner:       Mike Eldred
//- Revised by:  
//- Version:

#include "data_types.h"
#include "system_defs.h"
#include "NonDSparseGrid.H"
#include "sandia_rules.H"
#include "sparse_grid_mixed_growth.H"
#include "DakotaModel.H"
#include "ProblemDescDB.H"
#include "BasisPolynomial.H"

static const char rcsId[]="@(#) $Id: NonDSparseGrid.C,v 1.57 2004/06/21 19:57:32 mseldre Exp $";

//#define DEBUG

namespace Dakota {


/** This constructor is called for a standard letter-envelope iterator
    instantiation.  In this case, set_db_list_nodes has been called
    and probDescDB can be queried for settings from the method
    specification.  It is not currently used, as there is not yet a
    separate nond_sparse_grid method specification. */
NonDSparseGrid::NonDSparseGrid(Model& model): NonDIntegration(model),
  sparseGridLevelSpec(probDescDB.get_dusa("method.nond.sparse_grid_level")),
  sparseGridLevel(sparseGridLevelSpec), duplicateTol(1.e-15)
{
  // pure virtual fn cannot be called from NonDIntegration ctor
  check_input();
}


/** This alternate constructor is used for on-the-fly generation and
    evaluation of sparse grids. */
NonDSparseGrid::NonDSparseGrid(Model& model, const UShortArray& levels): 
  NonDIntegration(NoDBBaseConstructor(), model), sparseGridLevelSpec(levels),
  sparseGridLevel(levels), duplicateTol(1.e-15)
{
  // ProbabilityTransformation::ranVarTypesX not yet available: defer until
  // quantify_uncertainty()
  //check_input();
}


NonDSparseGrid::~NonDSparseGrid()
{ }


/** Called from probDescDB-based constructors and from
    NonDIntegration::quantify_uncertainty() */
void NonDSparseGrid::check_input()
{
  bool err_flag = false;

  const Pecos::ShortArray& x_types = natafTransform.x_types();
  numDesignVars = std::count(x_types.begin(), x_types.end(), (short)DESIGN);
  numStateVars  = std::count(x_types.begin(), x_types.end(), (short)STATE);
  numContinuousVars = numDesignVars + numUncertainVars + numStateVars;

  if (x_types.size() != numContinuousVars) {
    Cerr << "Error: Bad variable counts in NonDSparseGrid::check_input()." 
	 << endl;
    err_flag = true;
  }

  // Promote sparseGridLevel to full length, if needed.
  // (Leave sparseGridLevelSpec as original user specification.)
  if (sparseGridLevel.empty()) {
    Cerr << "Error: sparse grid level specification required in NonDSparseGrid."
	 << endl;
    err_flag = true;
  }
  else if (sparseGridLevel.length() != numContinuousVars) {
    if (sparseGridLevel.length() == 1) {
      unsigned short sp_grid_level = sparseGridLevel[0];
      sparseGridLevel.reshape(numContinuousVars);
      sparseGridLevel = sp_grid_level;
    }
    else {
      Cerr << "Error: length of sparse grid level specification does not equal "
	   << "number of active variables." << endl;
      err_flag = true;
    }
  }

  if (err_flag)
    abort_handler(-1);

  const Pecos::ShortArray& u_types = natafTransform.u_types();
  initialize_rules(u_types, integrationRules, compute1DPoints, compute1DWeights,
		   alphaPolyParams, betaPolyParams);
}


void NonDSparseGrid::
initialize_rules(const Pecos::ShortArray& u_types, IntArray& rules,
		 Array<FPType>& compute_1d_pts, Array<FPType>& compute_1d_wts,
		 RealArray& poly_alphas, RealArray& poly_betas)
{
  size_t i, num_u_vars = u_types.size();
  rules.reshape(num_u_vars);
  compute_1d_pts.reshape(num_u_vars);
  compute_1d_wts.reshape(num_u_vars);
  poly_alphas.reshape(num_u_vars); poly_alphas = 0.;
  poly_betas.reshape(num_u_vars);  poly_betas  = 0.;

  const RealDenseVector& beta_alphas  = iteratedModel.beta_alphas();
  const RealDenseVector& beta_betas   = iteratedModel.beta_betas();
  const RealDenseVector& gamma_alphas = iteratedModel.gamma_alphas();

  // For now, this logic is more restrictive than it may ultimately need to be.
  // If Gauss-Patterson can be extended to Hermite/Lag/Gen Lag/Jacobi, then a
  // reasonable approach would be to enforce consistency among nested approaches
  // with nonlinear growth rules, and non-nested approaches with (future) linear
  // growth rules.
  bool fully_nested = true;
  for (i=0; i<num_u_vars; ++i)
    if (u_types[i] != UNIFORM)
      { fully_nested = false; break; }

  size_t b_cntr = 0, g_cntr = 0;
  for (i=0; i<num_u_vars; i++) {
    switch (u_types[i]) {
    case NORMAL:
      compute_1d_pts[i] = webbur::hermite_compute_points;
      compute_1d_wts[i] = webbur::hermite_compute_weights;
      rules[i] = 5;
      break; // Gauss-Hermite open weakly nested
    case UNIFORM:
      // For NonDQuadrature, Gauss-Legendre is used due to greater polynomial
      // exactness since nesting is not a concern.  For NonDSparseGrid, nesting
      // is a concern and Clenshaw-Curtis or Gauss-Patterson can be better
      // selections.  However, sparse grids that are isotropic in level but
      // anisotropic in rule become skewed when mixing Gauss rules with CC.
      // For this reason, CC is selected only if isotropic in rule (for now).
      if (fully_nested) {
	compute_1d_pts[i] = webbur::clenshaw_curtis_compute_points;
	compute_1d_wts[i] = webbur::clenshaw_curtis_compute_weights;
	rules[i] = 1; // Clenshaw-Curtis closed fully nested
      }
      else {
	compute_1d_pts[i] = webbur::legendre_compute_points;
	compute_1d_wts[i] = webbur::legendre_compute_weights;
	rules[i] = 4; // Gauss-Legendre open weakly nested
      }
      break;
    case EXPONENTIAL:
      compute_1d_pts[i] = webbur::laguerre_compute_points;
      compute_1d_wts[i] = webbur::laguerre_compute_weights;
      rules[i] = 7;
      break; // Gauss-Laguerre open non-nested
    case BETA:
      compute_1d_pts[i] = webbur::jacobi_compute_points;
      compute_1d_wts[i] = webbur::jacobi_compute_weights;
      rules[i] = 9;         // Gauss-Jacobi open non-nested
      // convert stat to poly and assume consistent ordering via b_cntr:
      poly_alphas[i] = beta_betas[b_cntr]  - 1.;
      poly_betas[i]  = beta_alphas[b_cntr] - 1.;
      b_cntr++;
      break;
    case GAMMA:
      compute_1d_pts[i] = webbur::gen_laguerre_compute_points;
      compute_1d_wts[i] = webbur::gen_laguerre_compute_weights;
      rules[i] = 8;         // Gen. Gauss-Laguerre open non-nested
      // convert stat to poly and assume consistent ordering via g_cntr:
      poly_alphas[i] = gamma_alphas[g_cntr++] - 1.;
      break;
    default:
    //case BOUNDED_NORMAL: case LOGNORMAL: case BOUNDED_LOGNORMAL:
    //case LOGUNIFORM: case TRIANGULAR: case GUMBEL: case FRECHET: case WEIBULL:
      Cerr << "Error: Golub-Welsch not yet supported in NonDSparseGrid."<< endl;
      abort_handler(-1);
      //compute_1d_pts[i] = GolubWelschOrthogPolynomial::compute_points;
      //compute_1d_wts[i] = GolubWelschOrthogPolynomial::compute_weights;
      rules[i] = 10;
      break; // Golub-Welsch open non-nested
    }
  }
}


void NonDSparseGrid::get_parameter_sets(const Model& model)
{
  BasisPolynomial hermite_poly, legendre_poly, laguerre_poly,
    jacobi_poly, gen_laguerre_poly;
  if (numNormalVars)
    hermite_poly  = BasisPolynomial(HERMITE);
  if ( (numDesignVars || numUniformVars || numStateVars) &&
       integrationRules.contains(4) )
    legendre_poly = BasisPolynomial(LEGENDRE);
  if (numExponentialVars)
    laguerre_poly = BasisPolynomial(LAGUERRE);
  if (numBetaVars)
    jacobi_poly = BasisPolynomial(JACOBI);
  if (numGammaVars)
    gen_laguerre_poly = BasisPolynomial(GENERALIZED_LAGUERRE);

  // --------------------------------
  // Get number of collocation points
  // --------------------------------
  unsigned short sp_grid_level = sparseGridLevel[0]; // isotropic for now
  int num_total_pts  =
    webbur::sparse_grid_mixed_growth_size_total(numContinuousVars,
    sp_grid_level, integrationRules, webbur::level_to_order_default);
  int num_colloc_pts =
    webbur::sparse_grid_mixed_growth_size(numContinuousVars, sp_grid_level,
    integrationRules, alphaPolyParams, betaPolyParams, compute1DPoints,
    duplicateTol, webbur::level_to_order_default);
  Cout << "Total number of sparse grid integration points: "
       << num_colloc_pts << '\n';

  // ----------------------------------------------
  // Get collocation points and integration weights
  // ----------------------------------------------
  uniqueIndexMapping.reshape(num_total_pts);
  int*    sparse_order  = new int    [num_colloc_pts*numContinuousVars];
  int*    sparse_index  = new int    [num_colloc_pts*numContinuousVars];
  double* weight_sets   = new double [num_colloc_pts];
  double* variable_sets = new double [num_colloc_pts*numContinuousVars];

  webbur::sparse_grid_mixed_growth_unique_index(numContinuousVars,
    sp_grid_level, integrationRules, alphaPolyParams, betaPolyParams,
    compute1DPoints, duplicateTol, num_colloc_pts, num_total_pts,
    webbur::level_to_order_default, uniqueIndexMapping);
  webbur::sparse_grid_mixed_growth_index(numContinuousVars, sp_grid_level,
    integrationRules, num_colloc_pts, num_total_pts, uniqueIndexMapping,
    webbur::level_to_order_default, sparse_order, sparse_index);
  webbur::sparse_grid_mixed_growth_weight(numContinuousVars, sp_grid_level,
    integrationRules, alphaPolyParams, betaPolyParams, compute1DWeights,
    num_colloc_pts, num_total_pts, uniqueIndexMapping,
    webbur::level_to_order_default, weight_sets);
  webbur::sparse_grid_mixed_growth_point(numContinuousVars, sp_grid_level,
    integrationRules, alphaPolyParams, betaPolyParams, compute1DPoints,
    num_colloc_pts, sparse_order, sparse_index, webbur::level_to_order_default,
    variable_sets);

  size_t i, j, index = 0;
  Real wt_factor = 1.;
  RealVector pt_factor(numContinuousVars, 1.);
  for (i=0; i<numContinuousVars; i++) {
    switch (integrationRules[i]) {
    case 5: // Gauss-Hermite open weakly nested
      wt_factor   *= hermite_poly.weight_factor();
      pt_factor[i] = hermite_poly.point_factor();
      break;
    case 1: // Clenshaw-Curtis closed fully nested
      wt_factor /= 2.;
      break;
    case 4: // Gauss-Legendre open weakly nested
      wt_factor *= legendre_poly.weight_factor();
      break;
    //case 7: // Gauss-Laguerre scaling is OK
    case 9: // Gauss-Jacobi open non-nested
      jacobi_poly.alpha_polynomial(alphaPolyParams[i]);
      jacobi_poly.beta_polynomial(betaPolyParams[i]);
      wt_factor *= jacobi_poly.weight_factor();
      break;
    case 8: // Gen. Gauss-Laguerre open non-nested
      gen_laguerre_poly.alpha_polynomial(alphaPolyParams[i]);
      wt_factor *= gen_laguerre_poly.weight_factor();
      break;
    }
  }
  if (weightProducts.length() != num_colloc_pts)
    weightProducts.reshape(num_colloc_pts);
  for (i=0; i<num_colloc_pts; i++)
    weightProducts[i] = weight_sets[i] * wt_factor;

  if (allVariables.length() != num_colloc_pts)
    allVariables.reshape(num_colloc_pts);
  const Variables& vars = iteratedModel.current_variables();
  RealDenseVector c_vars(numContinuousVars, false);
  for (i=0; i<num_colloc_pts; i++) {
    if (allVariables[i].is_null())
      allVariables[i] = Variables(vars.view(), vars.variables_components());
    for (j=0; j<numContinuousVars; j++)
      c_vars[j] = variable_sets[index++] * pt_factor[j];
    allVariables[i].continuous_variables(c_vars);
#ifdef DEBUG
    Cout << "Point " << i+1 << ":\n" << c_vars
	 << "Weight " << i+1 << ": " << weightProducts[i] << '\n';
#endif
  }
#ifdef DEBUG
  Cout << "uniqueIndexMapping:\n" << uniqueIndexMapping << '\n';
#endif

  delete [] sparse_order;
  delete [] sparse_index;
  delete [] weight_sets;
  delete [] variable_sets;

  // ----------------------------
  // Define 1-D point/weight sets
  // ----------------------------
  if (gaussPts1D.empty() || gaussWts1D.empty()) {
    gaussPts1D.reshape(numContinuousVars);
    gaussWts1D.reshape(numContinuousVars);
  }
  size_t num_levels_per_var = sp_grid_level + 1;
  // level_index (j indexing) range is 0:w, level (i indexing) range is 1:w+1
  unsigned short level_index, order;
  for (i=0; i<numContinuousVars; i++) {
    gaussPts1D[i].reshape(num_levels_per_var);
    gaussWts1D[i].reshape(num_levels_per_var);
    switch (integrationRules[i]) {
    case 5: // Gauss-Hermite
      for (level_index=0; level_index<num_levels_per_var; level_index++) {
	level_to_order_open_linear(level_index, order); // j = i - 1 -> order
	gaussPts1D[i][level_index] = hermite_poly.gauss_points(order);
	gaussWts1D[i][level_index] = hermite_poly.gauss_weights(order);
      }
      break;
    case 1: // Clenshaw-Curtis
      for (level_index=0; level_index<num_levels_per_var; level_index++) {
	// Clenshaw-Curtis pts are extrema (not roots) of Chebyshev polynomials
	level_to_order_closed_exponential(level_index, order); // j=i-1 -> order
	gaussPts1D[i][level_index].reshape(order);
	gaussWts1D[i][level_index].reshape(order);
	double* cc_pts = new double [order];
	double* cc_wts = new double [order];
	webbur::clenshaw_curtis_compute(order, 0., 0., cc_pts, cc_wts);
	for (j=0; j<order; j++) {
	  gaussPts1D[i][level_index][j] = cc_pts[j];
	  gaussWts1D[i][level_index][j] = cc_wts[j]/2.;// scale to PDF weighting
	}
	delete [] cc_pts;
	delete [] cc_wts;
      }
      break;
    case 4: // Gauss-Legendre
      for (level_index=0; level_index<num_levels_per_var; level_index++) {
	level_to_order_open_linear(level_index, order); // j = i - 1 -> order
	gaussPts1D[i][level_index] = legendre_poly.gauss_points(order);
	gaussWts1D[i][level_index] = legendre_poly.gauss_weights(order);
      }
      break;
    case 7: // Gauss-Laguerre
      for (level_index=0; level_index<num_levels_per_var; level_index++) {
	level_to_order_open_linear(level_index, order); // j = i - 1 -> order
	gaussPts1D[i][level_index] = laguerre_poly.gauss_points(order);
	gaussWts1D[i][level_index] = laguerre_poly.gauss_weights(order);
      }
      break;
    case 9: // Gauss-Jacobi
      jacobi_poly.alpha_polynomial(alphaPolyParams[i]);
      jacobi_poly.beta_polynomial(betaPolyParams[i]);
      for (level_index=0; level_index<num_levels_per_var; level_index++) {
	level_to_order_open_linear(level_index, order); // j = i - 1 -> order
	gaussPts1D[i][level_index] = jacobi_poly.gauss_points(order);
	gaussWts1D[i][level_index] = jacobi_poly.gauss_weights(order);
      }
      break;
    case 8: // Gen. Gauss-Laguerre
      gen_laguerre_poly.alpha_polynomial(alphaPolyParams[i]);
      for (level_index=0; level_index<num_levels_per_var; level_index++) {
	level_to_order_open_linear(level_index, order); // j = i - 1 -> order
	gaussPts1D[i][level_index] = gen_laguerre_poly.gauss_points(order);
	gaussWts1D[i][level_index] = gen_laguerre_poly.gauss_weights(order);
      }
      break;
    }
  }

  /*
  // -----------------------------------
  // Get sparse grid index/base mappings
  // -----------------------------------
  size_t size = numContinuousVars*num_colloc_pts, cntr = 0;
  int* indices = new int [size];
  int* bases   = new int [size];

  webbur::sparse_grid_mixed_growth_index(numContinuousVars, sp_grid_level,
    integrationRules, num_colloc_pts, num_total_pts, uniqueIndexMapping,
    webbur::level_to_order_default, bases, indices);

  IntArray key(2*numContinuousVars);
  unsigned short closed_order_max;
  level_to_order_closed_exponential(sp_grid_level, closed_order_max);
  for (i=0; i<num_colloc_pts; i++) {
    for (j=0; j<numContinuousVars; j++, cntr++) {
      switch (integrationRules[j]) {
      case 5: // Gauss-Hermite
      case 4: // Gauss-Legendre
	key[j] = 2 * bases[cntr] + 1;             // map to quad order
	key[j+numContinuousVars] = indices[cntr] + bases[cntr]; // 0-based index
	break;
      case 1: // Clenshaw-Curtis
	key[j] = closed_order_max;                // promotion to highest grid
	key[j+numContinuousVars] = indices[cntr]; // already 0-based
	break;
      case 7: // Gauss-Laguerre
	key[j] = bases[cntr];                         // already quad order
	key[j+numContinuousVars] = indices[cntr] - 1; // map to 0-based
	break;
      }
    }
    sparseGridIndexMap[key] = i;
#ifdef DEBUG
    Cout << "i = " << i << " key =\n" << key << endl;
#endif // DEBUG
  }
  delete [] indices;
  delete [] bases;
  */
}


/** used by DataFitSurrModel::build_global() to publish the minimum
    number of points needed from the sparse grid routine in order to
    build a particular global approximation. */
void NonDSparseGrid::
sampling_reset(int min_samples,  int rec_samples, bool all_data_flag,
	       bool stats_flag)
{
  // determine the minimum sparse grid level that provides at least min_samples
  unsigned short min_level = 0;
  int sparse_grid_samples = 1;
  while (sparse_grid_samples < min_samples)
    sparse_grid_samples =
      webbur::sparse_grid_mixed_growth_size(numContinuousVars, ++min_level,
      integrationRules, alphaPolyParams, betaPolyParams, compute1DPoints,
      duplicateTol, webbur::level_to_order_default);

  // sparse grid level may be increased or decreased to provide at least
  // min_samples, but the original user specification (sparseGridLevelSpec) is
  // a hard lower bound.  maxConcurrency must not be updated since parallel
  // config management depends on having the same value at ctor/run/dtor times.
  sparseGridLevel[0] = (min_level > sparseGridLevelSpec[0]) ?
    min_level : sparseGridLevelSpec[0]; // isotropic for now

  // not currently used by this class:
  //allDataFlag = all_data_flag;
  //statsFlag   = stats_flag;
}

} // namespace Dakota
