/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2001, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 NonDQuadrature
//- Description: Implementation code for NonDQuadrature class
//- Owner:       Mike Eldred
//- Revised by:  
//- Version:

#include "data_types.h"
#include "system_defs.h"
#include "NonDQuadrature.H"
#include "DakotaModel.H"
#include "ProblemDescDB.H"
#include "BasisPolynomial.H"
#include "GolubWelschOrthogPolynomial.H"
#include "BasisPolyApproximation.H"

static const char rcsId[]="@(#) $Id: NonDQuadrature.C,v 1.57 2004/06/21 19:57:32 mseldre Exp $";


namespace Dakota {

/** This constructor is called for a standard letter-envelope iterator
    instantiation.  In this case, set_db_list_nodes has been called
    and probDescDB can be queried for settings from the method
    specification.  It is not currently used, as there is not yet a
    separate nond_quadrature method specification. */
NonDQuadrature::NonDQuadrature(Model& model): NonDIntegration(model), 
  quadOrderSpec(probDescDB.get_dusa("method.nond.quadrature_order")),
  quadOrder(quadOrderSpec)
{
  // pure virtual fn cannot be called from NonDIntegration ctor
  check_input();
}


/** This alternate constructor is used for on-the-fly generation and
    evaluation of numerical quadrature points. */
NonDQuadrature::NonDQuadrature(Model& model, const UShortArray& order): 
  NonDIntegration(NoDBBaseConstructor(), model), quadOrderSpec(order),
  quadOrder(order)
{
  // pure virtual fn cannot be called from NonDIntegration ctor
  //check_input();
}


NonDQuadrature::~NonDQuadrature()
{ }


void NonDQuadrature::check_input()
{
  bool err_flag = false;

  const Pecos::ShortArray& x_types = natafTransform.x_types();
  numDesignVars     = std::count(x_types.begin(), x_types.end(), (short)DESIGN);
  numStateVars      = std::count(x_types.begin(), x_types.end(), (short)STATE);
  numContinuousVars = numDesignVars + numUncertainVars + numStateVars;

  // Gauss-Hermite, Gauss-Legendre, Gauss-Laguerre, generalized Gauss-Laguerre,
  // and Gauss-Jacobi are currently available
  if ( numUncertainVars != numNormalVars + numUniformVars + numExponentialVars +
                           numBetaVars   + numGammaVars ) {
    Cerr << "Error: All variables in NonDQuadrature must be standard forms of "
	 << "normal, uniform,\n       exponential, beta, or gamma "
	 << "distributions since only Gauss-Hermite,\n       Gauss-Legendre, "
	 << "Gauss-Laguerre, generalized Gauss-Laguerre, and Gauss-Jacobi"
	 << "\n       numerical integration are supported." << endl;
    err_flag = true;
  }

  // Promote quadOrder to full length, if needed.
  // (Leave quadOrderSpec as original user specification.)
  if (quadOrder.empty()) {
    Cerr << "Error: quadrature order specification required in NonDQuadrature."
	 << endl;
    err_flag = true;
  }
  else if (quadOrder.length() != numContinuousVars) {
    if (quadOrder.length() == 1) {
      unsigned short quad_order = quadOrder[0];
      quadOrder.reshape(numContinuousVars);
      quadOrder = quad_order;
    }
    else {
      Cerr << "Error: length of quadrature order specification does not equal "
	   << "number of active variables." << endl;
      err_flag = true;
    }
  }

  if (err_flag)
    abort_handler(-1);
}


void NonDQuadrature::get_parameter_sets(const Model& model)
{
  size_t i, j, num_quad_points = 1;
  Cout << "\nNumber of Gauss points per variable: { ";
  for (i=0; i<numContinuousVars; i++) {
    num_quad_points *= quadOrder[i];
    Cout << quadOrder[i] << ' ';
  }
  Cout << "}\nTotal number of integration points:  " << num_quad_points << '\n';

  BasisPolynomial hermite_poly, legendre_poly, laguerre_poly, jacobi_poly,
    gen_laguerre_poly, golub_welsch_poly;
  BasisPolynomial* poly_ptr = NULL;
  if (numNormalVars)
    hermite_poly  = BasisPolynomial(HERMITE);
  if (numDesignVars || numUniformVars || numStateVars)
    legendre_poly = BasisPolynomial(LEGENDRE);
  if (numExponentialVars)
    laguerre_poly = BasisPolynomial(LAGUERRE);
  if (numBetaVars)
    jacobi_poly   = BasisPolynomial(JACOBI);
  if (numGammaVars)
    gen_laguerre_poly = BasisPolynomial(GENERALIZED_LAGUERRE);
  if (numLognormalVars || numLoguniformVars || numTriangularVars ||
      numGumbelVars    || numFrechetVars    || numWeibullVars)
    golub_welsch_poly = BasisPolynomial(GOLUB_WELSCH);
  if (gaussPts1D.empty() || gaussWts1D.empty()) {
    gaussPts1D.reshape(numContinuousVars);
    gaussWts1D.reshape(numContinuousVars);
  }
  const RealDenseVector& ln_means    = iteratedModel.lognormal_means();
  const RealDenseVector& ln_std_devs = iteratedModel.lognormal_std_deviations();
  const RealDenseVector& lu_l_bnds   = iteratedModel.loguniform_lower_bounds();
  const RealDenseVector& lu_u_bnds   = iteratedModel.loguniform_upper_bounds();
  const RealDenseVector& tri_modes   = iteratedModel.triangular_modes();
  const RealDenseVector& tri_l_bnds  = iteratedModel.triangular_lower_bounds();
  const RealDenseVector& tri_u_bnds  = iteratedModel.triangular_upper_bounds();
  const RealDenseVector& beta_alphas = iteratedModel.beta_alphas();
  const RealDenseVector& beta_betas  = iteratedModel.beta_betas();
  const RealDenseVector& gamma_alphas   = iteratedModel.gamma_alphas();
  const RealDenseVector& gumbel_alphas  = iteratedModel.gumbel_alphas();
  const RealDenseVector& gumbel_betas   = iteratedModel.gumbel_betas();
  const RealDenseVector& frechet_alphas = iteratedModel.frechet_alphas();
  const RealDenseVector& frechet_betas  = iteratedModel.frechet_betas();
  const RealDenseVector& weibull_alphas = iteratedModel.weibull_alphas();
  const RealDenseVector& weibull_betas  = iteratedModel.weibull_betas();
  const Pecos::ShortArray& u_types = natafTransform.u_types();
  size_t beta_cntr = 0, gamma_cntr = 0, ln_cntr, lu_cntr, tri_cntr,
    gumbel_cntr, frechet_cntr, weibull_cntr;
  for (i=0; i<numContinuousVars; i++) {
    switch (u_types[i]) {
    case NORMAL:
      poly_ptr = hermite_poly.polynomial_rep(); break;
    case LOGNORMAL:
      poly_ptr = golub_welsch_poly.polynomial_rep();
      ((GolubWelschOrthogPolynomial*)poly_ptr)->
	lognormal_distribution(ln_means[ln_cntr], ln_std_devs[ln_cntr]);
      ln_cntr++; break;
    case UNIFORM:
      // Gauss-Legendre used for NonDQuadrature, CC used for NonDSparseGrid
      poly_ptr = legendre_poly.polynomial_rep(); break;
    case LOGUNIFORM:
      poly_ptr = golub_welsch_poly.polynomial_rep();
      ((GolubWelschOrthogPolynomial*)poly_ptr)->
	loguniform_distribution(lu_l_bnds[lu_cntr], lu_u_bnds[lu_cntr]);
      lu_cntr++; break;
    case TRIANGULAR:
      poly_ptr = golub_welsch_poly.polynomial_rep();
      ((GolubWelschOrthogPolynomial*)poly_ptr)->
	triangular_distribution(tri_modes[tri_cntr], tri_l_bnds[tri_cntr],
				tri_u_bnds[tri_cntr]);
      tri_cntr++; break;
    case EXPONENTIAL:
      poly_ptr = laguerre_poly.polynomial_rep(); break;
    case BETA:
      poly_ptr = jacobi_poly.polynomial_rep();
      poly_ptr->alpha_stat(beta_alphas[beta_cntr]);
      poly_ptr->beta_stat(beta_betas[beta_cntr]);
      poly_ptr->reset_gauss();
      beta_cntr++; break;
    case GAMMA:
      poly_ptr = gen_laguerre_poly.polynomial_rep();
      poly_ptr->alpha_stat(gamma_alphas[gamma_cntr]);
      poly_ptr->reset_gauss();
      gamma_cntr++; break;
    case GUMBEL:
      poly_ptr = golub_welsch_poly.polynomial_rep();
      ((GolubWelschOrthogPolynomial*)poly_ptr)->
	gumbel_distribution(gumbel_alphas[gumbel_cntr],
			    gumbel_betas[gumbel_cntr]);
      gumbel_cntr++; break;
    case FRECHET:
      poly_ptr = golub_welsch_poly.polynomial_rep();
      ((GolubWelschOrthogPolynomial*)poly_ptr)->
	frechet_distribution(frechet_alphas[frechet_cntr],
			     frechet_betas[frechet_cntr]);
      frechet_cntr++; break;
    case WEIBULL:
      poly_ptr = golub_welsch_poly.polynomial_rep();
      ((GolubWelschOrthogPolynomial*)poly_ptr)->
	weibull_distribution(weibull_alphas[weibull_cntr],
			     weibull_betas[weibull_cntr]);
      weibull_cntr++; break;
    }
    gaussPts1D[i].reshape(1);
    gaussWts1D[i].reshape(1);
    gaussPts1D[i][0] = poly_ptr->gauss_points(quadOrder[i]);
    gaussWts1D[i][0] = poly_ptr->gauss_weights(quadOrder[i]);
  }

  // Tensor-product quadrature: Integral of f approximated by
  // Sum_i1 Sum_i2 ... Sum_in (w_i1 w_i2 ... w_in) f(x_i1, x_i2, ..., x_in)
  // > project 1-D gauss point arrays (of potentially different type and order)
  //   into an n-dimensional stencil.
  // > compute and store products of 1-D Gauss weights at each point in stencil.
  UShortArray gauss_indices(numContinuousVars, 0);
  if (allVariables.length() != num_quad_points)
    allVariables.reshape(num_quad_points);
  if (weightProducts.length() != num_quad_points)
    weightProducts.reshape(num_quad_points);
  const Variables& vars = iteratedModel.current_variables();
  RealDenseVector c_vars(numContinuousVars, false);
  for (i=0; i<num_quad_points; i++) {
    if (allVariables[i].is_null())
      allVariables[i] = Variables(vars.view(), vars.variables_components());
    Real& wt_prod_i = weightProducts[i];
    wt_prod_i = 1.0;
    for (j=0; j<numContinuousVars; j++) {
      c_vars[j]  = gaussPts1D[j][0][gauss_indices[j]];
      wt_prod_i *= gaussWts1D[j][0][gauss_indices[j]];
    }
    allVariables[i].continuous_variables(c_vars);
    // increment the n-dimensional gauss point index set
    BasisPolyApproximation::increment_indices(gauss_indices, quadOrder, true);
  }
}

} // namespace Dakota
