/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 NonDEvidence
//- Description: Implementation code for NonDEvidence class
//- Owner:       Laura Swiler
//- Checked by:
//- Version:

#include "data_types.h"
#include "system_defs.h"
//#include "DakotaResponse.H"
#include "ProblemDescDB.H"
#include "NonDEvidence.H"
#include "NonDLHSSampling.H"
#include "time.h"
#include "pecos_stat_util.hpp"

//#define DEBUG

static const char rcsId[] = "@(#) $Id: NonDEvidence.C 5852 2009-05-01 22:56:36Z mseldre $";

// Prototype for a direct call to the F77 DDS code
#define CBPIIC_F77 F77_FUNC(cbpiic,CBPIIC)
#define MINMAX_F77 F77_FUNC(minmax,MINMAX)
#define CCBFPF_F77 F77_FUNC(ccbfpf,CCBFPF)

extern "C" {

void CBPIIC_F77(int&, int&, int&, int*, int*, double*, double*);

void MINMAX_F77(int&, int&, int&, int*, int*, double*, double*,	double*,
		double*, double*, double&);

void CCBFPF_F77(int&, int*, int*, double*, double*, double*);

}


namespace Dakota {

NonDEvidence::NonDEvidence(Model& model): NonD(model),
   numSamples(probDescDB.get_int("method.samples")),
   originalSeed(probDescDB.get_int("method.random_seed"))
{
  // Check for suitable distribution types
  if (numUncertainVars != numIntervalVars) {
    Cerr << "Error: only interval distributions are currently supported in "
	 << "nond_evidence." << endl;
    abort_handler(-1);
  }

  // reliability_levels not currently supported, but could be
  if (!probDescDB.get_drva("method.nond.reliability_levels").empty()) {
    Cerr << "Error: reliability_levels not supported in nond_evidence." << endl;
    abort_handler(-1);
  }
  // size output arrays.  Note that for each response or probability level
  // request, we get two values from the belief and plausibility functions.
  computedRespLevels.reshape(numFunctions);
  computedProbLevels.reshape(numFunctions);
  computedGenRelLevels.reshape(numFunctions);
  size_t i, j;
  for (i=0; i<numFunctions; i++) {
    size_t rl_len = requestedRespLevels[i].length(), pl_gl_len
      = requestedProbLevels[i].length() + requestedGenRelLevels[i].length();
    computedRespLevels[i].reshape(2*pl_gl_len);
    if (respLevelTarget == PROBABILITIES)
      computedProbLevels[i].reshape(2*rl_len);
    else
      computedGenRelLevels[i].reshape(2*rl_len);
  }

  // initialize finalStatistics using non-default definition (there is no mean
  // or standard deviation and each level mapping involves lower/upper bounds).
  initialize_final_statistics();

  const RealDenseVectorArray& interval_bpa    = model.interval_probabilities();
  const RealDenseVectorArray& interval_bounds = model.interval_bounds();
  NV = numIntervalVars;
  NI = new int[NV];
  NCMB = 1;
  MAXINTVLS = 1;
  for (i=0; i<NV; i++) {
    size_t num_intervals_i = interval_bpa[i].length();
    NI[i] = num_intervals_i;
    NCMB *= num_intervals_i;
    if (num_intervals_i > MAXINTVLS)
      MAXINTVLS = num_intervals_i;
  }

  IP    = new int[NCMB];
  IPBEL = new int[NCMB];
  IPPLA = new int[NCMB];

  // Test
  BPA  = new Real[NV * MAXINTVLS];
  VMIN = new Real[NV * MAXINTVLS];
  VMAX = new Real[NV * MAXINTVLS];
 
  BPAC = new Real[NCMB];
  CMIN = new Real[NCMB];
  CMAX = new Real[NCMB];
  X    = new Real[NV];
  double Y;

  //initialize BPA, VMIN, and VMAX
  for (i=0; i<NV; i++) {
    int NI_i = NI[i];
    for (j=0; j<NI_i; j++) {
      int Jindex = i*MAXINTVLS+j;
      BPA[Jindex]  = interval_bpa[i][j];
      VMIN[Jindex] = interval_bounds[i][2*j];
      VMAX[Jindex] = interval_bounds[i][2*j+1];
    }
  }

  //initialize CMIN and CMAX and BPAC
  for (i=0; i<NCMB; i++) {
    CMIN[i] = DBL_MAX;
    CMAX[i] = DBL_MIN;
    BPAC[i] = 0.0;
  }

  lhsSampler.assign_rep(
    new NonDLHSSampling(iteratedModel, numSamples, originalSeed), false);
  iteratedModel.init_communicators(lhsSampler.maximum_concurrency());
}


NonDEvidence::~NonDEvidence()
{  
  delete [] NI;
  delete [] IP;
  delete [] IPBEL;
  delete [] IPPLA;
  delete [] BPA;
  delete [] VMIN;
  delete [] VMAX;
  delete [] BPAC;
  delete [] CMIN;
  delete [] CMAX;
  delete [] X;

  iteratedModel.free_communicators(lhsSampler.maximum_concurrency());
}


void NonDEvidence::initialize_final_statistics()
{
  size_t num_final_stats = 2*totalLevelRequests;
  ActiveSet stats_set(num_final_stats, numUncertainVars);
  finalStatistics = Response(stats_set);

  // Assign meaningful fn labels to final stats (appear in NestedModel output)
  size_t i, j, num_levels, cntr = 0;
  char tag_string[10], lev_string[15];
  StringArray stats_labels(num_final_stats);
  for (i=0; i<numFunctions; i++) {
    sprintf(tag_string, "_r%i", i+1);
    num_levels = requestedRespLevels[i].length();
    for (j=0; j<num_levels; j++) {
      stats_labels[cntr] = (cdfFlag) ? String("cdf") : String("ccdf");
      if (respLevelTarget == PROBABILITIES)
	sprintf(lev_string, "_plev%i", j+1);
      else
	sprintf(lev_string, "_b*lev%i", j+1);
      stats_labels[cntr] +=
	String("_bel") + String(lev_string) + String(tag_string);
      cntr++;
      stats_labels[cntr] +=
	String("_pls") + String(lev_string) + String(tag_string);
      cntr++;
    }
    num_levels = requestedProbLevels[i].length() +
      requestedGenRelLevels[i].length();
    for (j=0; j<num_levels; j++) {
      stats_labels[cntr] = (cdfFlag) ? String("cdf") : String("ccdf");
      sprintf(lev_string, "_zlev%i", j+1);
      stats_labels[cntr] +=
	String("_bel") + String(lev_string) + String(tag_string);
      cntr++;
      stats_labels[cntr] +=
	String("_pls") + String(lev_string) + String(tag_string);
      cntr++;
    }
  }
  finalStatistics.function_labels(stats_labels);
}


void NonDEvidence::quantify_uncertainty()
{
  size_t i;
  // Call Johnson/Helton routine to calculate the basic probabilities for 
  // input interval calculations
  calculate_basic_prob_intervals();

  // Construct a surrogate based on a set of LHS samples, and evaluate that 
  // sample a large number of times (default: 1 million) to sufficiently 
  // populate the belief intervals
  // Use the sample set generated above to determine the maximum and minimum 
  // of each function within each input interval combination

  //if the user does not specify the number of samples, 
  //perform at least 10000 to get accurate assessments of belief and plaus.

  if (numSamples == 0)
    numSamples = 10000;

  lhsSampler.run_iterator();
  all_vars      = lhsSampler.all_variables();
  all_responses = lhsSampler.all_responses();
  
  cc_bel_fn.reshape(numFunctions);
  cc_plaus_fn.reshape(numFunctions);
  cc_bel_val.reshape(numFunctions);
  cc_plaus_val.reshape(numFunctions);
  for (i=0; i<numFunctions; i++) {
    cc_bel_fn[i].reshape(NCMB);
    cc_plaus_fn[i].reshape(NCMB);
    cc_bel_val[i].reshape(NCMB);
    cc_plaus_val[i].reshape(NCMB);
  }

  for (i=0; i<numFunctions; i++) {
    calculate_maxmin_per_interval(i);
    // Use the max and mins to determine the cumulative distributions
    // of plausibility and belief
    calculate_cum_belief_plaus(i);
  }

  // compute statistics 
  compute_statistics();
}


void NonDEvidence::calculate_basic_prob_intervals()
{
  Cout << "\n>>>>> Calculating basic probability intervals\n";

#ifdef DEBUG
  size_t i, j;
  for (i=0; i<NV; i++)
    for (j=0; j<MAXINTVLS; j++)
      Cout << "BPA " << i << j << " is " << BPA[(i*MAXINTVLS)+j] << '\n';
#endif //DEBUG

  CBPIIC_F77(MAXINTVLS,NCMB,NV,NI,IP,BPA,BPAC);

#ifdef DEBUG
  Cout << "NV is " << NV << "\nMAXINTVLS is " << MAXINTVLS << "\n" ;
  for (i=0; i<NV; i++)
    for (j=0; j<MAXINTVLS; j++) {
      Cout << "BPA " << i<<j<< " is " << BPA[(i*MAXINTVLS)+j] << "\n" ;
      Cout << "VMIN " << i<<j<< " is " << VMIN[(i*MAXINTVLS)+j] << "\n" ;
     }
  for (j=0; j<NCMB; j++)
    Cout << "BPAC " <<j<< " is " << BPAC[j] << "\n" ;
#endif //DEBUG
}


void NonDEvidence::calculate_maxmin_per_interval(const size_t& func_num)
{
  
  Cout << "\n>>>>> Calculating maxmin for response function " << func_num+1
       << '\n';
  size_t i;
  for (i=0; i<numSamples; i++) {
    copy_data(all_vars[i].continuous_variables(), X, (int)numUncertainVars);

    // need to generalize to multiple response fns:
    //for(j=0; j<numResponseFunctions; j++)
    //  Y[j] = all_responses[i].function_values()[j];
    Y = all_responses[i].function_values()[func_num];

    // We put the vars and response in X and Y to call minmax.
    // Note that minmax wants them one at a time.
    MINMAX_F77(MAXINTVLS,NCMB,NV,NI,IP,VMIN,VMAX,CMIN,CMAX,X,Y);
    //Cout << "x1 " << x1 << "x2 " << x2 << "Y "<< Y << endl;
  }
#ifdef DEBUG
  for (i=0; i<NCMB; i++) {
    Cout << "CMAX " <<i<< " is " << CMAX[i] << "\n" ;
    Cout << "CMIN " <<i<< " is " << CMIN[i] << "\n" ;
  }
#endif //DEBUG 
}


void NonDEvidence::calculate_cum_belief_plaus(const size_t& func_num)
{
  Cout << "\n>>>>> Calculating cumulative belief and plaus for response "
       << "function " << func_num+1 << '\n';
#ifdef DEBUG
  for (size_t i=0; i<NCMB; i++) {
    Cout << "CMAX " <<i<< " is " << CMAX[i] << "\n" ;
    Cout << "CMIN " <<i<< " is " << CMIN[i] << "\n" ;
    Cout << "BPAC " <<i<< " is " << BPAC[i] << "\n" ;
  }
#endif //DEBUG

  CCBFPF_F77(NCMB,IPBEL,IPPLA,BPAC,CMIN,CMAX); 
#ifdef DEBUG
  for (size_t i=0; i<NCMB; i++) {
    Cout << "IPBEL " <<i<< " is " << IPBEL[i] << "\n" ;
    Cout << "IPPLA " <<i<< " is " << IPPLA[i] << "\n" ;
    Cout << "CMAX " <<i<< " is " << CMAX[i] << "\n" ;
    Cout << "CMIN " <<i<< " is " << CMIN[i] << "\n" ;
    Cout << "BPAC " <<i<< " is " << BPAC[i] << "\n" ;
  }
#endif //DEBUG

  //Get the CCBF and CCPF as calculated by the evidence code
  //Note that the evidence code produces complementary distributions 
  Real cc_bel_total = 0.0, cc_plaus_total = 0.0;
  int bel_index, plaus_index;
  for (int i=NCMB-1; i>=0; i--){
    bel_index       = IPBEL[i]-1;
    plaus_index     = IPPLA[i]-1;
    cc_bel_total   += BPAC[bel_index];
    cc_plaus_total += BPAC[plaus_index]; 
    cc_bel_fn[func_num][i]    = cc_bel_total;
    cc_plaus_fn[func_num][i]  = cc_plaus_total;
    cc_bel_val[func_num][i]   = CMIN[bel_index];
    cc_plaus_val[func_num][i] = CMAX[plaus_index];
  }
}
 
void NonDEvidence::compute_statistics()
{
  // > CDF/CCDF mappings of probability/reliability levels to response levels

  int i, j, bel_index, plaus_index, cntr = 0;
#ifdef DEBUG 
  for (i=0; i<numFunctions; i++) {
    Cout << "Function number " << i << '\n';
  for (j=0; j< NCMB; j++)
    Cout << "CCBF " << cc_bel_fn[i][j] << "value "
	 << cc_bel_val[i][j] << "CCPF "
	 << cc_plaus_fn[i][j] << "value " << cc_plaus_val[i][j] << '\n';
  }
#endif //DEBUG

  // CDF/CCDF mappings of response levels to belief and plaus: z -> belief/plaus
  for (i=0; i<numFunctions; i++) {
    size_t rl_len = requestedRespLevels[i].length(),
           pl_len = requestedProbLevels[i].length(),
           gl_len = requestedGenRelLevels[i].length(),
        pl_gl_len = pl_len + gl_len;

    for (j=0; j<rl_len; j++) { // z -> belief/plausibility
      Real z = requestedRespLevels[i][j];
      // z -> belief (based on CCBF)
      bel_index = 0;
      Real prob;
      if (cdfFlag) {
        while (z > cc_plaus_val[i][bel_index] && bel_index < NCMB-1)
	  bel_index++;
        prob = (z > cc_plaus_val[i][NCMB-1]) ? 1. : 
	  1.-cc_plaus_fn[i][bel_index];
      }
      else {
        while (z > cc_bel_val[i][bel_index] && bel_index < NCMB-1)
	  bel_index++;
	prob = (z > cc_bel_val[i][NCMB-1]) ? 0. : cc_bel_fn[i][bel_index];
      }
      if (respLevelTarget == PROBABILITIES) {
	computedProbLevels[i][j] = prob;
	finalStatistics.function_value(prob, cntr++);
      }
      else {
	Real beta;
	if (prob < 1.e-50) // numerical safeguards for p=0,p=1 assignments above
	  beta =  1.e50; // Phi(-Inf) = 0
	else if (1.- prob < 1.e-50)
	  beta = -1.e50; // Phi(+Inf) = 1
	else
	  beta = -Pecos::Phi_inverse(prob);
	computedGenRelLevels[i][j] = beta;
	finalStatistics.function_value(beta, cntr++);
      }
      // z -> plausibility (based on CCPF)
      plaus_index = 0;
      if (cdfFlag) {
        while (z > cc_bel_val[i][plaus_index] && plaus_index < NCMB-1)
	  plaus_index++;
        prob = (z > cc_bel_val[i][NCMB-1]) ? 1. : 1.-cc_bel_fn[i][plaus_index];
      }
      else { 
        while (z > cc_plaus_val[i][plaus_index] && plaus_index < NCMB-1)
	  plaus_index++;
	prob = (z > cc_plaus_val[i][NCMB-1]) ? 0. : cc_plaus_fn[i][plaus_index];
      }
      if (respLevelTarget == PROBABILITIES) {
	computedProbLevels[i][j+rl_len] = prob;
	finalStatistics.function_value(prob, cntr++);
      }
      else {
	Real beta;
	if (prob < 1.e-50) // numerical safeguards for p=0,p=1 assignments above
	  beta =  1.e50; // Phi(-Inf) = 0
	else if (1.- prob < 1.e-50)
	  beta = -1.e50; // Phi(+Inf) = 1
	else
	  beta = -Pecos::Phi_inverse(prob);
	computedGenRelLevels[i][j+rl_len] = beta;
	finalStatistics.function_value(beta, cntr++);
      }
    }

    for (j=0; j<pl_gl_len; j++) { // belief/plausibility -> z
      Real cc_prob;
      if (j<pl_len)
	cc_prob = (cdfFlag) ? 1. - requestedProbLevels[i][j] :
	  requestedProbLevels[i][j];
      else
	cc_prob = (cdfFlag) ? Pecos::Phi(requestedGenRelLevels[i][j]) :
	  Pecos::Phi(-requestedGenRelLevels[i][j]);
      // belief -> z (based on CCBF)
      bel_index = 0;
      if (cdfFlag) {
        while (cc_prob < cc_plaus_fn[i][bel_index] && bel_index < NCMB-1)
	  bel_index++;
        const Real& z = computedRespLevels[i][j] = cc_plaus_val[i][bel_index]; 
	finalStatistics.function_value(z, cntr++);
      } 
      else {
        while (cc_prob < cc_bel_fn[i][bel_index] && bel_index < NCMB-1)
	  bel_index++;
        const Real& z = computedRespLevels[i][j] = cc_bel_val[i][bel_index]; 
	finalStatistics.function_value(z, cntr++);
      }
      // plausibility -> z (based on CCPF)
      plaus_index = 0;
      if (cdfFlag) {
        while (cc_prob <cc_bel_fn[i][plaus_index] && plaus_index < NCMB-1)
	  plaus_index++;
        const Real& z = computedRespLevels[i][j+pl_gl_len]
	  = cc_bel_val[i][plaus_index];
	finalStatistics.function_value(z, cntr++);
      }
      else {
        while (cc_prob < cc_plaus_fn[i][plaus_index] && plaus_index < NCMB-1)
	  plaus_index++;
        const Real& z = computedRespLevels[i][j+pl_gl_len]
	  = cc_plaus_val[i][plaus_index];
	finalStatistics.function_value(z, cntr++);
      }
    }
  }
}


void NonDEvidence::print_results(ostream& s)
{
  const StringArray& fn_labels = iteratedModel.response_labels();
  size_t i, j;

  s << "------------------------------------------------------------------\n";

  // output CBF/CCBF and CPF/CCPF response/probabilities pairs
  s.setf(ios::scientific);
  s << setprecision(write_precision)
    << "\nBelief and Plausibility for each response function:\n";
  for (i=0; i<numFunctions; i++) {
    size_t rl_len = requestedRespLevels[i].length(),
           pl_len = requestedProbLevels[i].length(),
           gl_len = requestedGenRelLevels[i].length(),
        pl_gl_len = pl_len + gl_len;
    if (cdfFlag)
      s << "Cumulative Belief/Plausibility Functions (CBF/CPF) for ";
    else
      s << "Complementary Cumulative Belief/Plausibility Functions "
	<< "(CCBF/CCPF) for ";
    s << fn_labels[i] << ":\n";
    if (rl_len) {
      if (respLevelTarget == PROBABILITIES) {
	s << "     Response Level  Belief Prob Level   Plaus Prob Level\n"
	  << "     --------------  -----------------   ----------------\n";
	for (j=0; j<rl_len; j++)
	  s << "  " << setw(17) << requestedRespLevels[i][j] << "  "
	    << setw(17) << computedProbLevels[i][j] << "  " 
	    << setw(17) << computedProbLevels[i][j+rl_len] << '\n';
      }
      else {
	s << "     Response Level Belief Gen Rel Lev  Plaus Gen Rel Lev\n"
	  << "     -------------- ------------------  -----------------\n";
	for (j=0; j<rl_len; j++)
	  s << "  " << setw(17) << requestedRespLevels[i][j] << "  "
	    << setw(17) << computedGenRelLevels[i][j] << "  " 
	    << setw(17) << computedGenRelLevels[i][j+rl_len] << '\n';
      }
    }
    if (pl_len) {
      s << "  Probability Level  Belief Resp Level   Plaus Resp Level\n"
        << "  -----------------  -----------------   ----------------\n";
      for (j=0; j<pl_len; j++)
        s << "  " << setw(17) << requestedProbLevels[i][j] << "  "
	  << setw(17) << computedRespLevels[i][j] << "  " 
	  << setw(17) << computedRespLevels[i][j+pl_gl_len] << '\n';
    }
    if (gl_len) {
      s << "  General Rel Level  Belief Resp Level   Plaus Resp Level\n"
        << "  -----------------  -----------------   ----------------\n";
      for (j=0; j<gl_len; j++)
        s << "  " << setw(17) << requestedGenRelLevels[i][j] << "  "
	  << setw(17) << computedRespLevels[i][j+pl_len] << "  " 
	  << setw(17) << computedRespLevels[i][j+pl_len+pl_gl_len] << '\n';
    }
  }

  s << "-----------------------------------------------------------------"
    << endl;
#ifdef DEBUG
  s << "Final statistics:\n" << finalStatistics;
#endif //DEBUG	
}

} // namespace Dakota
