/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       MergedVariables
//- Description: Container class of variables employing the "merged" approach
//-              to variables usage.
//- Owner:       Mike Eldred
//- Version: $Id: MergedVariables.H 5793 2009-03-31 03:15:00Z mseldre $

#ifndef MERGED_VARIABLES_H
#define MERGED_VARIABLES_H

#include "DakotaVariables.H"
#include "DataVariables.H"


namespace Dakota {

/// Derived class within the Variables hierarchy which employs the
/// merged data view.

/** Derived variables classes take different views of the design,
    uncertain, and state variable types and the continuous and
    discrete domain types.  The MergedVariables derived class combines
    continuous and discrete domain types but separates design,
    uncertain, and state variable types.  The result is a single
    continuous array of design variables (mergedDesignVars), a single
    continuous array of uncertain variables (uncertainVars), and a
    single continuous array of state variables (mergedStateVars).  The
    branch and bound strategy uses this approach (see
    Variables::get_variables(problem_db)). */

class MergedVariables: public Variables
{
public:

  //
  //- Heading: Constructors and destructor
  //

  /// default constructor
  MergedVariables();
  /// standard constructor
  MergedVariables(const ProblemDescDB& problem_db,
		  const pair<short,short>& view);
  /// destructor
  ~MergedVariables();

  //
  //- Heading: Virtual function redefinitions
  //

  const UIntArray& merged_discrete_ids() const;

  void read(istream& s);
  void write(ostream& s) const;
  void write_aprepro(ostream& s) const;

  void read_annotated(istream& s);
  void write_annotated(ostream& s) const;

  void write_tabular(ostream& s) const;

  void read(BiStream& s);
  void write(BoStream& s) const;

  void read(MPIUnpackBuffer& s);
  void write(MPIPackBuffer& s) const;

protected:

  void copy_rep(const Variables* vars_rep);
  void reshape_rep(const Sizet2DArray& vars_comps);

  void build_active_views();
  void build_inactive_views();

private:

  //
  //- Heading: Private data views
  //

  /// the active continuous variables array view
  RealDenseVector continuousVars;
  /// the inactive continuous variables array view
  RealDenseVector inactiveContinuousVars;

  //
  //- Heading: Private data members
  //

  /// array of discrete variable identifiers for which the discrete
  /// requirement is relaxed by merging them into a continuous array
  UIntArray mergedDiscreteIds;
};


inline MergedVariables::MergedVariables()
{ }


inline MergedVariables::~MergedVariables()
{ }


inline const UIntArray& MergedVariables::merged_discrete_ids() const
{ return mergedDiscreteIds; }


inline void MergedVariables::copy_rep(const Variables* vars_rep)
{
  build_views(); // mergedDiscreteIds is rebuilt
  //mergedDiscreteIds = ((MergedVariables*)vars_rep)->mergedDiscreteIds;
}

} // namespace Dakota

#endif
