/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        GolubWelschOrthogPolynomial
//- Description:  Implementation code for GolubWelschOrthogPolynomial class
//-               
//- Owner:        Mike Eldred, Sandia National Laboratories

#include "GolubWelschOrthogPolynomial.H"
#include "Teuchos_LAPACK.hpp"
#include "data_util.h"
#include "pecos_stat_util.hpp"
#include "DataModel.H"


namespace Dakota {


void GolubWelschOrthogPolynomial::solve_eigenproblem(unsigned short n)
{
  // size class-scope arrays
  orthogPolyNormsSq.sizeUninitialized(n+1);;
  polyCoeffs.reshape(n+1);
  polyCoeffs[0].sizeUninitialized(1);
  polyCoeffs[0][0] = 1.; // constant term

  // set up the coefficients
  int i, j, ldz = std::max((unsigned short)1,n);
  RealDenseVector D(n, false), E(n-1, false), xi_poly_coeffs_i;
  RealDenseMatrix Z(ldz, n, false);
  Real beta_0 = 1.,//inner_product(polyCoeffs[0], polyCoeffs[0]),
    alpha_i, beta_i, inner_prod_pi_i, inner_prod_xi_pi_i, prev_inner_prod_pi_i;
  for (i=0; i<n; ++i) {
    const RealDenseVector& poly_coeffs_i = polyCoeffs[i];
    int i_len = poly_coeffs_i.length();
    // multiply poly_coeffs_i by \xi to form xi_poly_coeffs_i
    xi_poly_coeffs_i.sizeUninitialized(i_len+1);
    xi_poly_coeffs_i[0] = 0.;
    for (j=0; j<i_len; ++j)
      xi_poly_coeffs_i[j+1] = poly_coeffs_i[j];
    // compute inner products needed for recursion coefficients
    orthogPolyNormsSq[i] = inner_prod_pi_i
      = inner_product(poly_coeffs_i, poly_coeffs_i);
    inner_prod_xi_pi_i = inner_product(xi_poly_coeffs_i, poly_coeffs_i);
    // update matrix diagonal D, off-diagonal E, and next polynomial coeffs
    D[i] = alpha_i = inner_prod_xi_pi_i / inner_prod_pi_i;
    RealDenseVector& poly_coeffs_ip1 = polyCoeffs[i+1];
    poly_coeffs_ip1.size(i_len+1); // initialize to zero
    for (j=0; j<i_len; ++j) {
      poly_coeffs_ip1[j]   -= alpha_i*poly_coeffs_i[j]; // -alpha_i * pi_i
      poly_coeffs_ip1[j+1] += poly_coeffs_i[j];         // xi * pi_i
    }
    if (i) {
      beta_i = inner_prod_pi_i / prev_inner_prod_pi_i;
      E[i-1] = std::sqrt(beta_i);
      const RealDenseVector& poly_coeffs_im1 = polyCoeffs[i-1];
      int im1_len = poly_coeffs_im1.length();
      for (j=0; j<im1_len; ++j)
	poly_coeffs_ip1[j] -= beta_i*poly_coeffs_im1[j];// -beta_i * pi_{i-1}
    }
    if (i == n-1)
      orthogPolyNormsSq[i+1] = inner_product(poly_coeffs_ip1, poly_coeffs_ip1);
    else
      prev_inner_prod_pi_i = inner_prod_pi_i;
  }

  // solve the symmetric tridiagonal eigenvalue problem using LAPACK
  Teuchos::LAPACK<int, Real> la;
  int info = 0;
  double* work = new double [std::max(1,2*n-2)]; // temporary work array
  la.STEQR('I', n, D.values(), E.values(), Z.values(), ldz, work, &info);
  if (info) {
    Cerr << "Error: nonzero return code from LAPACK STEQR (symmetric "
	 << "tridiagonal eigensolution)\n       in GolubWelschOrthogPolynomial"
	 << "::solve_eigenproblem()" << endl;
    abort_handler(-1);
  }
  delete [] work;

  // compute the Guass points and weights from the eigenvalues/eigenvectors
  copy_data(D, gaussPoints); // eigenvalues are Gauss points
  gaussWeights.reshape(n);
  for (i=0; i<n; ++i)
    gaussWeights[i] = std::pow(Z(0, i), 2.) * beta_0;
}


Real GolubWelschOrthogPolynomial::
inner_product(const RealDenseVector& poly_coeffs1,
	      const RealDenseVector& poly_coeffs2)
{
  // Preferred approach: use classical Gaussian quadratures (Legendre/Jacobi,
  // Laguerre/Gen Laguerre, and Hermite) for bounded, semi-bounded, and
  // unbounded integration domains corresponding to supported PDFs.
  switch (distributionType) {
  //case BOUNDED_NORMAL:    // bounded -> Legendre
  //  return legendre_bounded_integral(poly_coeffs1, poly_coeffs2,
  //				     Pecos::bounded_normal_pdf);
  //  break;
  case LOGNORMAL:    // semi-bounded -> Laguerre
    return laguerre_semibounded_integral(poly_coeffs1, poly_coeffs2,
					 Pecos::lognormal_pdf);
    break;
  //case BOUNDED_LOGNORMAL: // bounded -> Legendre
  //  return legendre_bounded_integral(poly_coeffs1, poly_coeffs2,
  //				     Pecos::bounded_lognormal_pdf);
  //  break;
  case LOGUNIFORM:        // bounded -> Legendre
    return legendre_bounded_integral(poly_coeffs1, poly_coeffs2,
				     Pecos::loguniform_pdf);
    break;
  //case TRIANGULAR:        // bounded -> Legendre
  //  return legendre_bounded_integral(poly_coeffs1, poly_coeffs2,
  //				     Pecos::triangular_pdf);
  //  break;
  case GUMBEL:          // unbounded -> Hermite
    return hermite_unbounded_integral(poly_coeffs1, poly_coeffs2,
				      Pecos::gumbel_pdf);
    break;
  case FRECHET:      // semi-bounded -> Laguerre
    return laguerre_semibounded_integral(poly_coeffs1, poly_coeffs2,
					 Pecos::frechet_pdf);
    break;
  case WEIBULL:      // semi-bounded -> Laguerre
    return laguerre_semibounded_integral(poly_coeffs1, poly_coeffs2,
					 Pecos::weibull_pdf);
    break;
  //case HISTOGRAM: // TO DO
    //return legendre_bounded_integral(poly_coeffs1, poly_coeffs2,
    //				       Pecos::histogram_pdf);
    //break;
  default:
    Cerr << "Error: unsupported distribution type in GolubWelschOrthog"
	 << "Polynomial::inner_product()." << endl;
    abort_handler(-1);
    break;
  }

  // Alternate approach: follow simple discrete sum approach.  Need rules of
  // thumb for the semi-bounded/unbounded integration domains.
}


Real GolubWelschOrthogPolynomial::
laguerre_semibounded_integral(const RealDenseVector& poly_coeffs1,
			      const RealDenseVector& poly_coeffs2,
			      GWFPType weight_fn)
{
  BasisPolynomial laguerre_poly(LAGUERRE);
  unsigned short quad_order = 10; // hardwire for initial cut
  const RealVector& gauss_pts = laguerre_poly.gauss_points(quad_order);
  const RealVector& gauss_wts = laguerre_poly.gauss_weights(quad_order);

  Real prod = 1.;
  for (size_t i=0; i<quad_order; ++i)
    prod += gauss_wts[i] * get_value(gauss_pts[i], poly_coeffs1)
         *  get_value(gauss_pts[i], poly_coeffs2)
         *  weight_fn(gauss_pts[i], distParam1, distParam2)//, distParam3, distParam4)
         /  Pecos::std_exponential_pdf(gauss_pts[i]);
  return prod;
}


Real GolubWelschOrthogPolynomial::
legendre_bounded_integral(const RealDenseVector& poly_coeffs1,
			  const RealDenseVector& poly_coeffs2,
			  GWFPType weight_fn)
{
  BasisPolynomial legendre_poly(LEGENDRE);
  unsigned short quad_order = 10; // hardwire for initial cut
  const RealVector& gauss_pts = legendre_poly.gauss_points(quad_order);
  const RealVector& gauss_wts = legendre_poly.gauss_weights(quad_order);

  Real prod = 1.;
  for (size_t i=0; i<quad_order; ++i)
    prod += gauss_wts[i] * get_value(gauss_pts[i], poly_coeffs1)
         *  get_value(gauss_pts[i], poly_coeffs2)
         *  weight_fn(gauss_pts[i], distParam1, distParam2)//, distParam3, distParam4)
         /  Pecos::std_uniform_pdf(gauss_pts[i]);
  return prod;
}


Real GolubWelschOrthogPolynomial::
hermite_unbounded_integral(const RealDenseVector& poly_coeffs1,
			   const RealDenseVector& poly_coeffs2,
			   GWFPType weight_fn)
{
  BasisPolynomial hermite_poly(HERMITE);
  unsigned short quad_order = 10; // hardwire for initial cut
  const RealVector& gauss_pts = hermite_poly.gauss_points(quad_order);
  const RealVector& gauss_wts = hermite_poly.gauss_weights(quad_order);

  Real prod = 1.;
  for (size_t i=0; i<quad_order; ++i)
    prod += gauss_wts[i] * get_value(gauss_pts[i], poly_coeffs1)
         *  get_value(gauss_pts[i], poly_coeffs2)
         *  weight_fn(gauss_pts[i], distParam1, distParam2)//, distParam3, distParam4)
         /  Pecos::phi(gauss_pts[i]);
  return prod;
}


const Real& GolubWelschOrthogPolynomial::
get_value(const Real& x, unsigned short order)
{ return get_value(x, polyCoeffs[order]); }


const Real& GolubWelschOrthogPolynomial::
get_value(const Real& x, const RealDenseVector& poly_coeffs)
{
  size_t num_terms = poly_coeffs.length();
  basisPolyValue = poly_coeffs[0];
  for (int i=1; i<num_terms; ++i)
    basisPolyValue += poly_coeffs[i]*std::pow(x,i);
  return basisPolyValue;
}


const Real& GolubWelschOrthogPolynomial::
get_gradient(const Real& x, unsigned short order)
{ 
  //basisPolyGradient = (order) ? order*get_value(x, order-1): 0;
  return basisPolyGradient;
}


const Real& GolubWelschOrthogPolynomial::norm_squared(unsigned short order)
{ return orthogPolyNormsSq[order]; }


const RealVector& GolubWelschOrthogPolynomial::
gauss_points(unsigned short order)
{
  // pull this out from default below since order=0 is initial gauss pts length
  if (order < 1) {
    Cerr << "Error: underflow in minimum quadrature order (1) in "
	 << "GolubWelschOrthogPolynomial::gauss_points()." << endl;
    abort_handler(-1);
  }

  if (gaussPoints.length() != order) // if not already computed
    solve_eigenproblem(order);

  return gaussPoints;
}


const RealVector& GolubWelschOrthogPolynomial::
gauss_weights(unsigned short order)
{
  // pull this out from default below since order=0 is initial gauss pts length
  if (order < 1) {
    Cerr << "Error: underflow in minimum quadrature order (1) in "
	 << "GolubWelschOrthogPolynomial::gauss_weights()." << endl;
    abort_handler(-1);
  }

  if (gaussWeights.length() != order) // if not already computed
    solve_eigenproblem(order);

  return gaussWeights;
}


/*
// For NonDSparseGrid fn pointers, something like the following (?):
void gumbel_gauss_points(int order, double alpha, double beta, double* data)
{
  // avoid recalculating existing Gauss pts
  //if (distributionType != GUMBEL || order != gaussPoints.length()) {
  gumbel_distribution(alpha, beta);
  gauss_points(order);
  data = gaussPoints.values();
  //}
}


void gumbel_gauss_weights(int order, double alpha, double beta, double* data)
{
  // avoid recalculating existing Gauss wts
  //if (distributionType != GUMBEL || order != gaussWeights.length()) {
  gumbel_distribution(alpha, beta);
  gauss_weights(order);
  data = gaussWeights.values();
  //}
}
*/

} // namespace Dakota
