/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        DistinctVariables
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "DistinctVariables.H"
#include "ProblemDescDB.H"
#include "DakotaBinStream.H"
#include "data_io.h"
#include "data_util.h"

static const char rcsId[]="@(#) $Id";


namespace Dakota {

/** In this class, the distinct approach is used (design, uncertain, and
    state variable types and continuous and discrete domain types are 
    distinct).  Most iterators/strategies use this approach. */
DistinctVariables::
DistinctVariables(const ProblemDescDB& problem_db,
		  const pair<short,short>& view):
  Variables(BaseConstructor(), problem_db, view)
{
  aggregate_data(
    problem_db.get_rdv("variables.continuous_design.initial_point"),
    problem_db.get_rdv("variables.uncertain.initial_point"),
    problem_db.get_rdv("variables.continuous_state.initial_state"),
    allContinuousVars);
  aggregate_data(
    problem_db.get_idv("variables.discrete_design.initial_point"),
    problem_db.get_idv("variables.discrete_state.initial_state"),
    allDiscreteVars);

  aggregate_data(
    problem_db.get_dsa("variables.continuous_design.labels"),
    problem_db.get_dsa("variables.uncertain.labels"),
    problem_db.get_dsa("variables.continuous_state.labels"),
    allContinuousLabels);
  aggregate_data(
    problem_db.get_dsa("variables.discrete_design.labels"),
    problem_db.get_dsa("variables.discrete_state.labels"),
    allDiscreteLabels);

  initialize_all_continuous_var_types_ids(false);
  initialize_all_discrete_var_types_ids();

  // construct active/inactive views of all arrays
  build_views();

#ifdef REFCOUNT_DEBUG
  Cout << "Letter instantiated: variablesView active = " << variablesView.first
       << " inactive = " << variablesView.second << endl;
#endif
}


void DistinctVariables::reshape_rep(const Sizet2DArray& vars_comps)
{
  size_t num_acv = vars_comps[0][1] + vars_comps[1][0] + vars_comps[2][0] +
    vars_comps[3][1], num_adv = vars_comps[0][2] + vars_comps[3][2];

  allContinuousVars.resize(num_acv);
  allDiscreteVars.resize(num_adv);

  allContinuousLabels.resize(boost::extents[num_acv]);
  allDiscreteLabels.resize(boost::extents[num_adv]);

  initialize_all_continuous_var_types_ids(false);
  initialize_all_discrete_var_types_ids();

  // construct active/inactive views of all arrays
  build_views();
}


void DistinctVariables::build_active_views()
{
  // Initialize continuousVarTypes/discreteVarTypes/continuousVarIds.
  // Don't bleed over any logic about supported view combinations; rather,
  // keep this class general and encapsulated.
  size_t num_cdv = variablesComponents[0][1],
    num_ddv = variablesComponents[0][2], num_auv = variablesComponents[1][0],
    num_euv = variablesComponents[2][0], num_csv = variablesComponents[3][1],
    num_dsv = variablesComponents[3][2];

  // Initialize active views
  //cvStart = dvStart = numCV = numDV = 0;
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    numCV   = num_cdv;                     numDV = num_ddv;           break;
  case MIXED_DISTINCT_ALEATORY_UNCERTAIN:
    cvStart = num_cdv;                     numCV = num_auv;           break;
  case MIXED_DISTINCT_EPISTEMIC_UNCERTAIN:
    cvStart = num_cdv+num_auv;             numCV = num_euv;           break;
  case MIXED_DISTINCT_UNCERTAIN:
    cvStart = num_cdv;                     numCV = num_auv + num_euv; break;
  case MIXED_DISTINCT_STATE:
    cvStart = num_cdv + num_auv + num_euv; numCV = num_csv;
    dvStart = num_ddv;                     numDV = num_dsv;           break;
  }
  if (numCV)
    continuousVars
      = RealDenseVector(Teuchos::View, &allContinuousVars[cvStart], numCV);
  if (numDV)
    discreteVars
      = IntDenseVector(Teuchos::View,  &allDiscreteVars[dvStart],   numDV);
}


void DistinctVariables::build_inactive_views()
{
  // Initialize continuousVarTypes/discreteVarTypes/continuousVarIds.
  // Don't bleed over any logic about supported view combinations; rather,
  // keep this class general and encapsulated.
  size_t num_cdv = variablesComponents[0][1],
    num_ddv = variablesComponents[0][2], num_auv = variablesComponents[1][0],
    num_euv = variablesComponents[2][0], num_csv = variablesComponents[3][1],
    num_dsv = variablesComponents[3][2];

  // Initialize inactive views
  //icvStart = idvStart = numICV = numIDV = 0;
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    numICV   = num_cdv;                     numIDV = num_ddv;           break;
  case MIXED_DISTINCT_ALEATORY_UNCERTAIN:
    icvStart = num_cdv;                     numICV = num_auv;           break;
  case MIXED_DISTINCT_EPISTEMIC_UNCERTAIN:
    icvStart = num_cdv+num_auv;             numICV = num_euv;           break;
  case MIXED_DISTINCT_UNCERTAIN:
    icvStart = num_cdv;                     numICV = num_auv + num_euv; break;
  case MIXED_DISTINCT_STATE:
    icvStart = num_cdv + num_auv + num_euv; numICV = num_csv;
    idvStart = num_ddv;                     numIDV = num_dsv;           break;
  }
  if (numICV)
    inactiveContinuousVars
      = RealDenseVector(Teuchos::View, &allContinuousVars[icvStart], numICV);
  if (numIDV)
    inactiveDiscreteVars
      = IntDenseVector(Teuchos::View,  &allDiscreteVars[idvStart],   numIDV);
}


// Reordering is required in all read/write cases that will be visible to the
// user since all derived vars classes should use the same CDV/DDV/UV/CSV/DSV
// ordering for clarity.  Neutral file I/O, binary streams, and packed buffers
// do not need to reorder (so long as read/write are consistent) since this data
// is not intended for public consumption.
void DistinctVariables::read(istream& s)
{
  // ASCII version.
  size_t num_cdv    = variablesComponents[0][1],
         num_ddv    = variablesComponents[0][2],
         num_uv_csv = allContinuousVars.length() - num_cdv,
         num_dsv    = allDiscreteVars.length()   - num_ddv;
  read_data_partial(s, 0, num_cdv, allContinuousVars, allContinuousLabels);
  read_data_partial(s, 0, num_ddv, allDiscreteVars, allDiscreteLabels);
  read_data_partial(s, num_cdv, num_uv_csv, allContinuousVars,
		    allContinuousLabels);
  read_data_partial(s, num_ddv, num_dsv, allDiscreteVars, allDiscreteLabels);
}


void DistinctVariables::write(ostream& s) const
{
  // ASCII version.
  size_t num_cdv    = variablesComponents[0][1],
         num_ddv    = variablesComponents[0][2],
         num_uv_csv = allContinuousVars.length() - num_cdv,
         num_dsv    = allDiscreteVars.length()   - num_ddv;
  write_data_partial(s, 0, num_cdv, allContinuousVars, allContinuousLabels);
  write_data_partial(s, 0, num_ddv, allDiscreteVars, allDiscreteLabels);
  write_data_partial(s, num_cdv, num_uv_csv, allContinuousVars,
		     allContinuousLabels);
  write_data_partial(s, num_ddv, num_dsv, allDiscreteVars, allDiscreteLabels);
}


void DistinctVariables::write_aprepro(ostream& s) const
{
  // ASCII version in APREPRO/DPREPRO format.
  size_t num_cdv    = variablesComponents[0][1],
         num_ddv    = variablesComponents[0][2],
         num_uv_csv = allContinuousVars.length() - num_cdv,
         num_dsv    = allDiscreteVars.length()   - num_ddv;
  write_data_partial_aprepro(s, 0, num_cdv, allContinuousVars,
			     allContinuousLabels);
  write_data_partial_aprepro(s, 0, num_ddv, allDiscreteVars, allDiscreteLabels);
  write_data_partial_aprepro(s, num_cdv, num_uv_csv, allContinuousVars,
			     allContinuousLabels);
  write_data_partial_aprepro(s, num_ddv, num_dsv, allDiscreteVars,
			     allDiscreteLabels);
}


void DistinctVariables::read_annotated(istream& s)
{
  // ASCII version for neutral file I/O.
  if (variablesComponents.empty()) {
    variablesComponents.reshape(4);
    variablesComponents[0].reshape(3); // no sum
    variablesComponents[1].reshape(1); // sum only
    variablesComponents[2].reshape(1); // sum only
    variablesComponents[3].reshape(3); // no sum
  }
  s >> variablesComponents[0][1] >> variablesComponents[0][2]
    >> variablesComponents[1][0] >> variablesComponents[2][0]
    >> variablesComponents[3][1] >> variablesComponents[3][2];
  read_data_annotated(s, allContinuousVars, allContinuousLabels);
  read_data_annotated(s, allDiscreteVars, allDiscreteLabels);
  build_views();
}


void DistinctVariables::write_annotated(ostream& s) const
{
  // ASCII version for neutral file I/O.
  s << variablesComponents[0][1] << ' ' << variablesComponents[0][2] << ' '
    << variablesComponents[1][0] << ' ' << variablesComponents[2][0] << ' '
    << variablesComponents[3][1] << ' ' << variablesComponents[3][2] << ' ';
  write_data_annotated(s, allContinuousVars, allContinuousLabels);
  write_data_annotated(s, allDiscreteVars, allDiscreteLabels);
}


void DistinctVariables::write_tabular(ostream& s) const
{
  // ASCII version for tabular file I/O.
  write_data_tabular(s, allContinuousVars);
  write_data_tabular(s, allDiscreteVars);
}


void DistinctVariables::read(BiStream& s)
{
  // Binary version.
  if (variablesComponents.empty()) {
    variablesComponents.reshape(4);
    variablesComponents[0].reshape(3); // no sum
    variablesComponents[1].reshape(1); // sum only
    variablesComponents[2].reshape(1); // sum only
    variablesComponents[3].reshape(3); // no sum
  }
  s >> variablesComponents[0][1] >> variablesComponents[0][2]
    >> variablesComponents[1][0] >> variablesComponents[2][0]
    >> variablesComponents[3][1] >> variablesComponents[3][2];
  read_data(s, allContinuousVars, allContinuousLabels);
  read_data(s, allDiscreteVars, allDiscreteLabels);
  build_views();
}


void DistinctVariables::write(BoStream& s) const
{
  // Binary version.
  s << variablesComponents[0][1] << variablesComponents[0][2]
    << variablesComponents[1][0] << variablesComponents[2][0]
    << variablesComponents[3][1] << variablesComponents[3][2];
  write_data(s, allContinuousVars, allContinuousLabels);
  write_data(s, allDiscreteVars, allDiscreteLabels);
}


void DistinctVariables::read(MPIUnpackBuffer& s)
{
  // MPI buffer version.
  if (variablesComponents.empty()) {
    variablesComponents.reshape(4);
    variablesComponents[0].reshape(3); // no sum
    variablesComponents[1].reshape(1); // sum only
    variablesComponents[2].reshape(1); // sum only
    variablesComponents[3].reshape(3); // no sum
  }
  s >> variablesComponents[0][1] >> variablesComponents[0][2]
    >> variablesComponents[1][0] >> variablesComponents[2][0]
    >> variablesComponents[3][1] >> variablesComponents[3][2];
  read_data(s, allContinuousVars, allContinuousLabels);
  read_data(s, allDiscreteVars, allDiscreteLabels);
  build_views();
}


void DistinctVariables::write(MPIPackBuffer& s) const
{
  // MPI buffer version.
  s << variablesComponents[0][1] << variablesComponents[0][2]
    << variablesComponents[1][0] << variablesComponents[2][0]
    << variablesComponents[3][1] << variablesComponents[3][2];
  write_data(s, allContinuousVars, allContinuousLabels);
  write_data(s, allDiscreteVars, allDiscreteLabels);
}

} // namespace Dakota
