/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       Variables
//- Description: Base class for variables.  Using the foundational
//-              cdv/ddv/uv/csv/dsv arrays from an input specification,
//-              derived classes use different views (All, Distinct, Merged)
//-              to define active and inactive variable sets.
//- Owner:       Mike Eldred
//- Version: $Id: DakotaVariables.H 5796 2009-04-02 17:49:52Z mseldre $

#ifndef DAKOTA_VARIABLES_H
#define DAKOTA_VARIABLES_H

#include "data_types.h"


namespace Dakota {

class ProblemDescDB;


/// Base class for the variables class hierarchy.

/** The Variables class is the base class for the class hierarchy
    providing design, uncertain, and state variables for continuous
    and discrete domains within a Model.  Using the fundamental
    arrays from the input specification, different derived classes
    define different views of the data.  For memory efficiency and
    enhanced polymorphism, the variables hierarchy employs the
    "letter/envelope idiom" (see Coplien "Advanced C++", p. 133), for
    which the base class (Variables) serves as the envelope and one of
    the derived classes (selected in Variables::get_variables())
    serves as the letter. */

class Variables
{
  //
  //- Heading: Friends
  //

  /// equality operator
  friend bool operator==(const Variables& vars1, const Variables& vars2);
  /// inequality operator
  friend bool operator!=(const Variables& vars1, const Variables& vars2);

#ifdef HAVE_BOOST
  /// hash_value
  friend std::size_t hash_value(const Variables& vars);

  /// binary_equal_to (since 'operator==' is not suitable for boost/hash_set)
  friend bool binary_equal_to(const Variables& vars1, const Variables& vars2);
#endif

public:

  //
  //- Heading: Constructors, destructor, assignment operator
  //

  /// default constructor
  Variables();
  /// standard constructor
  Variables(const ProblemDescDB& problem_db);
  // alternate constructor for minimal instantiations on the fly (reshape reqd)
  //Variables(const pair<short,short>& view);
  /// alternate constructor for instantiations on the fly
  Variables(const pair<short,short>& view, const Sizet2DArray& vars_comps);
  /// copy constructor
  Variables(const Variables& vars);

  /// destructor
  virtual ~Variables();

  /// assignment operator
  Variables operator=(const Variables& vars);

  //
  //- Heading: Virtual functions
  //

  /// returns the list of discrete variables merged into a continuous array
  virtual const UIntArray& merged_discrete_ids() const;

  // INPUT/OUTPUT

  /// read a variables object from an istream
  virtual void read(istream& s);
  /// write a variables object to an ostream
  virtual void write(ostream& s) const;
  /// write a variables object to an ostream in aprepro format
  virtual void write_aprepro(ostream& s) const;

  // For neutral file I/O:
  /// read a variables object in annotated format from an istream
  virtual void read_annotated(istream& s);
  /// write a variables object in annotated format to an ostream
  virtual void write_annotated(ostream& s) const;

  /// write a variables object in tabular format to an ostream
  virtual void write_tabular(ostream& s) const;

  /// read a variables object from the binary restart stream
  virtual void read(BiStream& s);
  /// write a variables object to the binary restart stream
  virtual void write(BoStream& s) const;

  /// read a variables object from a packed MPI buffer
  virtual void read(MPIUnpackBuffer& s);
  /// write a variables object to a packed MPI buffer
  virtual void write(MPIPackBuffer& s) const;

  //
  //- Heading: Member functions
  //

  size_t tv()  const; ///< Returns total number of vars
  size_t cv()  const; ///< Returns number of active continuous vars
  size_t dv()  const; ///< Returns number of active discrete vars
  size_t icv() const; ///< returns number of inactive continuous vars
  size_t idv() const; ///< returns number of inactive discrete vars
  size_t acv() const; ///< returns total number of continuous vars
  size_t adv() const; ///< returns total number of discrete vars

  // ACTIVE VARIABLES

  /// return an active continuous variable
  const Real& continuous_variable(const size_t& i) const;
  /// return the active continuous variables
  const RealDenseVector& continuous_variables() const;
  /// set an active continuous variable
  void continuous_variable(const Real& c_var, const size_t& i);
  /// set the active continuous variables
  void continuous_variables(const RealDenseVector& c_vars);
  /// return an active discrete variable
  const int& discrete_variable(const size_t& i) const;
  /// return the active discrete variables
  const IntDenseVector& discrete_variables() const;
  /// set an active discrete variable
  void discrete_variable(const int& d_var, const size_t& i);
  /// set the active discrete variables
  void discrete_variables(const IntDenseVector& d_vars);

  /// return the active continuous variable labels
  StringMultiArrayConstView continuous_variable_labels() const;
  /// set the active continuous variable labels
  void continuous_variable_labels(StringMultiArrayConstView cv_labels);
  /// set an active continuous variable label
  void continuous_variable_label(const String& cv_label, const size_t& i);
  /// return the active discrete variable labels
  StringMultiArrayConstView discrete_variable_labels() const;
  /// set the active discrete variable labels
  void discrete_variable_labels(StringMultiArrayConstView dv_labels);
  /// set an active discrete variable label
  void discrete_variable_label(const String& dv_label, const size_t& i);

  /// return the active continuous variable types
  StringMultiArrayConstView continuous_variable_types() const;
  /// return the active discrete variable types
  StringMultiArrayConstView discrete_variable_types() const;

  /// return the active continuous variable position identifiers
  UIntMultiArrayConstView continuous_variable_ids() const;

  // INACTIVE VARIABLES

  /// return the inactive continuous variables
  const RealDenseVector& inactive_continuous_variables() const;
  /// set the inactive continuous variables
  void inactive_continuous_variables(const RealDenseVector& i_c_vars);
  /// return the inactive discrete variables
  const IntDenseVector& inactive_discrete_variables() const;
  /// set the inactive discrete variables
  void inactive_discrete_variables(const IntDenseVector& i_d_vars);

  /// return the inactive continuous variable labels
  StringMultiArrayConstView inactive_continuous_variable_labels() const;
  /// set the inactive continuous variable labels
  void inactive_continuous_variable_labels(StringMultiArrayConstView i_c_vars);
  /// return the inactive discrete variable labels
  StringMultiArrayConstView inactive_discrete_variable_labels() const;
  /// set the inactive discrete variable labels
  void inactive_discrete_variable_labels(StringMultiArrayConstView i_d_vars);

  /// return the inactive continuous variable position identifiers
  UIntMultiArrayConstView inactive_continuous_variable_ids() const;

  // ALL VARIABLES

  /// returns a single array with all continuous variables
  const RealDenseVector& all_continuous_variables() const;
  /// sets all continuous variables using a single array
  void all_continuous_variables(const RealDenseVector& a_c_vars);
  /// set a variable within the all continuous array
  void all_continuous_variable(const Real& a_c_var, const size_t& i);
  /// returns a single array with all discrete variables
  const IntDenseVector& all_discrete_variables() const;
  /// sets all discrete variables using a single array
  void all_discrete_variables(const IntDenseVector& a_d_vars);
  /// set a variable within the all discrete array
  void all_discrete_variable(const int& a_d_var, const size_t& i);

  /// returns a single array with all continuous variable labels
  StringMultiArrayConstView all_continuous_variable_labels() const;
  /// sets all continuous variable labels using a single array
  void all_continuous_variable_labels(StringMultiArrayConstView a_c_v_labels);
  /// set a label within the all continuous label array
  void all_continuous_variable_label(const String& a_c_v_label,const size_t& i);
  /// returns a single array with all discrete variable labels
  StringMultiArrayConstView all_discrete_variable_labels() const;
  /// sets all discrete variable labels using a single array
  void all_discrete_variable_labels(StringMultiArrayConstView a_d_v_labels);
  /// set a label within the all discrete label array
  void all_discrete_variable_label(const String& a_d_v_label, const size_t& i);

  /// return all continuous variable types
  StringMultiArrayConstView all_continuous_variable_types() const;
  /// return all discrete variable types
  StringMultiArrayConstView all_discrete_variable_types() const;

  /// return all continuous variable position identifiers
  UIntMultiArrayConstView all_continuous_variable_ids() const;

  /// for use when a deep copy is needed (the representation is _not_ shared)
  Variables copy() const;

  /// reshapes an existing Variables object based on the incoming
  /// variablesComponents
  void reshape(const Sizet2DArray& vars_comps);

  /// returns variablesView
  const pair<short,short>& view() const;
  /// defines variablesView from problem_db attributes
  pair<short,short> get_view(const ProblemDescDB& problem_db) const;
  /// sets the inactive view based on higher level (nested) context
  void inactive_view(short view2);

  /// returns the variables identifier string
  const String& variables_id() const;

  /// returns the number of variables for each of the constitutive components
  const Sizet2DArray& variables_components() const;

  /// function to check variablesRep (does this envelope contain a letter)
  bool is_null() const;

protected:

  //
  //- Heading: Constructors
  //

  /// constructor initializes the base class part of letter classes
  /// (BaseConstructor overloading avoids infinite recursion in the
  /// derived class constructors - Coplien, p. 139)
  Variables(BaseConstructor, const ProblemDescDB& problem_db,
	    const pair<short,short>& view);

  //
  //- Heading: Virtual functions
  //

  /// Used by copy() to copy the contents of a letter class
  virtual void copy_rep(const Variables* vars_rep);
  /// Used by reshape() to reshape the contents of a letter class
  virtual void reshape_rep(const Sizet2DArray& vars_comps);

  /// construct active views of all variables arrays
  virtual void build_active_views();
  /// construct inactive views of all variables arrays
  virtual void build_inactive_views();

  //
  //- Heading: Member functions
  //

  /// construct active/inactive views of all variables arrays
  void build_views();

  /// Convenience function for initializing allContinuousVarTypes and
  /// allContinuousVarIds
  void initialize_all_continuous_var_types_ids(bool relax);
  /// Convenience function for initializing allDiscreteVarTypes
  void initialize_all_discrete_var_types_ids();

  //
  //- Heading: Data
  //

  /// the variables view pair containing active (first) and inactive (second)
  /// view enumerations
  pair<short,short> variablesView;
  /// two-dimensional array containing variable type counts for
  /// design/uncertain/state (first index) by sub-type (second index)
  Sizet2DArray variablesComponents;

  /// array combining all of the continuous variables (design, uncertain, state)
  RealDenseVector allContinuousVars;
  /// array combining all of the discrete variables (design, state)
  IntDenseVector allDiscreteVars;

  /// array combining all of the continuous variable labels
  /// (design, uncertain, state)
  StringMultiArray allContinuousLabels;
  /// array combining all of the discrete variable labels (design, state)
  StringMultiArray allDiscreteLabels;

  /// array of variable types for all of the continuous variables
  StringMultiArray allContinuousVarTypes;
  /// array of variable types for all of the discrete variables
  StringMultiArray allDiscreteVarTypes;

  /// array of position identifiers for the all continuous variables array
  /** These identifiers define positions of the all continuous variables array
      within the total variable sequence. */
  UIntMultiArray allContinuousVarIds;

  //
  //- Heading: Private data views
  //

  /// start index of active continuous variables within allContinuousVars
  size_t cvStart;
  /// start index of active discrete variables within allDiscreteVars
  size_t dvStart;
  /// start index of inactive continuous variables within allContinuousVars
  size_t icvStart;
  /// start index of inactive discrete variables within allDiscreteVars
  size_t idvStart;
  size_t numCV;  ///< number of active continuous variables
  size_t numDV;  ///< number of active discrete variables
  size_t numICV; ///< number of inactive continuous variables
  size_t numIDV; ///< number of inactive discrete variables

  /// the active continuous variables array view
  RealDenseVector continuousVars;
  /// the active discrete variables array view
  IntDenseVector discreteVars;
  /// the inactive continuous variables array view
  RealDenseVector inactiveContinuousVars;
  /// the inactive discrete variables array view
  IntDenseVector inactiveDiscreteVars;

private:

  //
  //- Heading: Member functions
  //

  /// Used by the standard envelope constructor to instantiate the
  /// correct letter class
  Variables* get_variables(const ProblemDescDB& problem_db);
  /// Used by the alternate envelope constructors, by read functions,
  /// and by copy() to instantiate a new letter class
  Variables* get_variables(const pair<short,short>& view) const;

  /// perform sanity checks on view.first and view.second after update
  void check_view_compatibility();

  //
  //- Heading: Data
  //

  /// variables identifier string from the input file
  String idVariables;
  /// pointer to the letter (initialized only for the envelope)
  Variables* variablesRep;
  /// number of objects sharing variablesRep
  int referenceCount;
};


// nonvirtual functions can access letter attributes directly (only need to fwd
// member function call when the function could be redefined).
inline size_t Variables::tv() const
{
  return (variablesRep) ? variablesRep->allContinuousVars.length() +
    variablesRep->allDiscreteVars.length() :
    allContinuousVars.length() + allDiscreteVars.length();
}


inline size_t Variables::cv() const
{ return (variablesRep) ? variablesRep->numCV : numCV; }


inline size_t Variables::dv() const
{ return (variablesRep) ? variablesRep->numDV : numDV; }


inline size_t Variables::icv() const
{ return (variablesRep) ? variablesRep->numICV : numICV; }


inline size_t Variables::idv() const
{ return (variablesRep) ? variablesRep->numIDV : numIDV; }


inline size_t Variables::acv() const
{
  return (variablesRep) ?
    variablesRep->allContinuousVars.length() : allContinuousVars.length();
}


inline size_t Variables::adv() const
{
  return (variablesRep) ?
    variablesRep->allDiscreteVars.length() : allDiscreteVars.length();
}


inline void Variables::continuous_variable(const Real& c_var, const size_t& i)
{
  if (variablesRep)
    variablesRep->continuousVars[i] = c_var;
  else
    continuousVars[i] = c_var;
}


inline const Real& Variables::continuous_variable(const size_t& i) const
{
  if (variablesRep)
    return variablesRep->continuousVars[i];
  else
    return continuousVars[i];
}


inline const RealDenseVector& Variables::continuous_variables() const
{ return (variablesRep) ? variablesRep->continuousVars : continuousVars; }


inline void Variables::
continuous_variables(const RealDenseVector& c_vars)
{
  if (variablesRep)
    variablesRep->continuousVars.assign(c_vars);
  else
    continuousVars.assign(c_vars);
}


inline void Variables::discrete_variable(const int& d_var, const size_t& i)
{
  if (variablesRep)
    variablesRep->discreteVars[i] = d_var;
  else
    discreteVars[i] = d_var;
}


inline const int& Variables::discrete_variable(const size_t& i) const
{
  if (variablesRep)
    return variablesRep->discreteVars[i];
  else
    return discreteVars[i];
}


inline const IntDenseVector& Variables::discrete_variables() const
{ return (variablesRep) ? variablesRep->discreteVars : discreteVars; }


inline void Variables::discrete_variables(const IntDenseVector& d_vars)
{
  if (variablesRep)
    variablesRep->discreteVars.assign(d_vars);
  else
    discreteVars.assign(d_vars);
}


inline StringMultiArrayConstView Variables::continuous_variable_labels() const
{
  return (variablesRep) ? variablesRep->
    allContinuousLabels[boost::indices[
      idx_range(variablesRep->cvStart,
		variablesRep->cvStart+variablesRep->numCV)]] :
    allContinuousLabels[boost::indices[idx_range(cvStart, cvStart+numCV)]];
}


inline void Variables::
continuous_variable_labels(StringMultiArrayConstView c_v_labels)
{
  if (variablesRep)
    variablesRep->allContinuousLabels[boost::indices[
      idx_range(variablesRep->cvStart,
		variablesRep->cvStart+variablesRep->numCV)]] = c_v_labels;
  else
    allContinuousLabels[boost::indices[idx_range(cvStart, cvStart+numCV)]]
      = c_v_labels;
}


inline void Variables::
continuous_variable_label(const String& cv_label, const size_t& i)
{
  if (variablesRep)
    variablesRep->allContinuousLabels[variablesRep->cvStart+i] = cv_label;
  else
    allContinuousLabels[cvStart+i] = cv_label;
}


inline StringMultiArrayConstView Variables::discrete_variable_labels() const
{
  return (variablesRep) ? variablesRep->
    allDiscreteLabels[boost::indices[
      idx_range(variablesRep->dvStart,
		variablesRep->dvStart+variablesRep->numDV)]] :
    allDiscreteLabels[boost::indices[idx_range(dvStart, dvStart+numDV)]];
}


inline void Variables::
discrete_variable_labels(StringMultiArrayConstView d_v_labels)
{
  if (variablesRep)
    variablesRep->allDiscreteLabels[boost::indices[
      idx_range(variablesRep->dvStart,
		variablesRep->dvStart+variablesRep->numDV)]] = d_v_labels;
  else
    allDiscreteLabels[boost::indices[idx_range(dvStart, dvStart+numDV)]]
      = d_v_labels;
}


inline void Variables::
discrete_variable_label(const String& dv_label, const size_t& i)
{
  if (variablesRep)
    variablesRep->allDiscreteLabels[variablesRep->dvStart+i] = dv_label;
  else
    allDiscreteLabels[dvStart+i] = dv_label;
}


inline StringMultiArrayConstView Variables::continuous_variable_types() const
{
  return (variablesRep) ? variablesRep->
    allContinuousVarTypes[boost::indices[
      idx_range(variablesRep->cvStart,
		variablesRep->cvStart+variablesRep->numCV)]] :
    allContinuousVarTypes[boost::indices[idx_range(cvStart, cvStart+numCV)]];
}


inline StringMultiArrayConstView Variables::discrete_variable_types() const
{
  return (variablesRep) ? variablesRep->
    allDiscreteVarTypes[boost::indices[
      idx_range(variablesRep->dvStart,
		variablesRep->dvStart+variablesRep->numDV)]] :
    allDiscreteVarTypes[boost::indices[idx_range(dvStart, dvStart+numDV)]];
}


inline UIntMultiArrayConstView Variables::continuous_variable_ids() const
{
  return (variablesRep) ? variablesRep->
    allContinuousVarIds[boost::indices[
      idx_range(variablesRep->cvStart,
		variablesRep->cvStart+variablesRep->numCV)]] :
    allContinuousVarIds[boost::indices[idx_range(cvStart, cvStart+numCV)]];
}


inline const RealDenseVector& Variables::inactive_continuous_variables() const
{
  return (variablesRep) ?
    variablesRep->inactiveContinuousVars : inactiveContinuousVars;
}


inline void Variables::
inactive_continuous_variables(const RealDenseVector& i_c_vars)
{
  if (variablesRep)
    variablesRep->inactiveContinuousVars.assign(i_c_vars);
  else
    inactiveContinuousVars.assign(i_c_vars);
}


inline const IntDenseVector& Variables::inactive_discrete_variables() const
{
  return (variablesRep) ?
    variablesRep->inactiveDiscreteVars : inactiveDiscreteVars;
}


inline void Variables::
inactive_discrete_variables(const IntDenseVector& i_d_vars)
{
  if (variablesRep)
    variablesRep->inactiveDiscreteVars.assign(i_d_vars);
  else
    inactiveDiscreteVars.assign(i_d_vars);
}


inline StringMultiArrayConstView Variables::
inactive_continuous_variable_labels() const
{
  return (variablesRep) ? variablesRep->
    allContinuousLabels[boost::indices[
      idx_range(variablesRep->icvStart,
		variablesRep->icvStart+variablesRep->numICV)]] :
    allContinuousLabels[boost::indices[idx_range(icvStart, icvStart+numICV)]];
}


inline void Variables::
inactive_continuous_variable_labels(StringMultiArrayConstView i_c_v_labels)
{
  if (variablesRep)
    variablesRep->allContinuousLabels[boost::indices[
      idx_range(variablesRep->icvStart,
		variablesRep->icvStart+variablesRep->numICV)]] = i_c_v_labels;
  else
    allContinuousLabels[boost::indices[idx_range(icvStart, icvStart+numICV)]]
      = i_c_v_labels;
}


inline StringMultiArrayConstView Variables::
inactive_discrete_variable_labels() const
{
  return (variablesRep) ? variablesRep->
    allDiscreteLabels[boost::indices[
      idx_range(variablesRep->idvStart,
		variablesRep->idvStart+variablesRep->numIDV)]] :
    allDiscreteLabels[boost::indices[idx_range(idvStart, idvStart+numIDV)]];
}


inline void Variables::
inactive_discrete_variable_labels(StringMultiArrayConstView i_d_v_labels)
{
  if (variablesRep)
    variablesRep->allDiscreteLabels[boost::indices[
      idx_range(variablesRep->idvStart,
		variablesRep->idvStart+variablesRep->numIDV)]] = i_d_v_labels;
  else
    allDiscreteLabels[boost::indices[idx_range(idvStart, idvStart+numIDV)]]
      = i_d_v_labels;
}


inline UIntMultiArrayConstView Variables::
inactive_continuous_variable_ids() const
{
  return (variablesRep) ? variablesRep->
    allContinuousVarIds[boost::indices[
      idx_range(variablesRep->icvStart,
		variablesRep->icvStart+variablesRep->numICV)]] :
    allContinuousVarIds[boost::indices[idx_range(icvStart, icvStart+numICV)]];
}


inline const RealDenseVector& Variables::all_continuous_variables() const
{ return (variablesRep) ? variablesRep->allContinuousVars : allContinuousVars; }


inline void Variables::all_continuous_variables(const RealDenseVector& a_c_vars)
{
  if (variablesRep)
    variablesRep->allContinuousVars.assign(a_c_vars);
  else
    allContinuousVars.assign(a_c_vars);
}


inline void Variables::
all_continuous_variable(const Real& a_c_var, const size_t& i)
{
  if (variablesRep)
    variablesRep->allContinuousVars[i] = a_c_var;
  else
    allContinuousVars[i] = a_c_var;
}


inline const IntDenseVector& Variables::all_discrete_variables() const
{ return (variablesRep) ? variablesRep->allDiscreteVars : allDiscreteVars; }


inline void Variables::all_discrete_variables(const IntDenseVector& a_d_vars)
{
  if (variablesRep)
    variablesRep->allDiscreteVars.assign(a_d_vars);
  else
    allDiscreteVars.assign(a_d_vars);
}


inline void Variables::
all_discrete_variable(const int& a_d_var, const size_t& i)
{
  if (variablesRep)
    variablesRep->allDiscreteVars[i] = a_d_var;
  else
    allDiscreteVars[i] = a_d_var;
}


inline StringMultiArrayConstView Variables::
all_continuous_variable_labels() const
{
  return (variablesRep) ? variablesRep->
    allContinuousLabels[boost::indices[
      idx_range(0, variablesRep->allContinuousLabels.size())]] :
    allContinuousLabels[boost::indices[
      idx_range(0, allContinuousLabels.size())]];
}


inline void Variables::
all_continuous_variable_labels(StringMultiArrayConstView a_c_v_labels)
{
  if (variablesRep)
    variablesRep->allContinuousLabels = a_c_v_labels;
  else
    allContinuousLabels = a_c_v_labels;
}


inline void Variables::
all_continuous_variable_label(const String& a_c_v_label, const size_t& i)
{
  if (variablesRep)
    variablesRep->allContinuousLabels[i] = a_c_v_label;
  else
    allContinuousLabels[i] = a_c_v_label;
}


inline StringMultiArrayConstView Variables::all_discrete_variable_labels() const
{
  return (variablesRep) ? variablesRep->
    allDiscreteLabels[boost::indices[
      idx_range(0, variablesRep->allDiscreteLabels.size())]] : 
    allDiscreteLabels[boost::indices[idx_range(0, allDiscreteLabels.size())]];
}


inline void Variables::
all_discrete_variable_labels(StringMultiArrayConstView a_d_v_labels)
{
  if (variablesRep)
    variablesRep->allDiscreteLabels = a_d_v_labels;
  else
    allDiscreteLabels = a_d_v_labels;
}


inline void Variables::
all_discrete_variable_label(const String& a_d_v_label, const size_t& i)
{
  if (variablesRep)
    variablesRep->allDiscreteLabels[i] = a_d_v_label;
  else
    allDiscreteLabels[i] = a_d_v_label;
}


inline StringMultiArrayConstView Variables::
all_continuous_variable_types() const
{
  return (variablesRep) ? variablesRep->
    allContinuousVarTypes[boost::indices[
      idx_range(0, variablesRep->allContinuousVarTypes.size())]] :
    allContinuousVarTypes[boost::indices[
      idx_range(0, allContinuousVarTypes.size())]];
}


inline StringMultiArrayConstView Variables::all_discrete_variable_types() const
{
  return (variablesRep) ? variablesRep->
    allDiscreteVarTypes[boost::indices[
      idx_range(0, variablesRep->allDiscreteVarTypes.size())]] :
    allDiscreteVarTypes[boost::indices[
      idx_range(0, allDiscreteVarTypes.size())]];
}


inline UIntMultiArrayConstView Variables::all_continuous_variable_ids() const
{
  return (variablesRep) ? variablesRep->
    allContinuousVarIds[boost::indices[
      idx_range(0, variablesRep->allContinuousVarIds.size())]] :
    allContinuousVarIds[boost::indices[
      idx_range(0, allContinuousVarIds.size())]];
}


inline const pair<short,short>& Variables::view() const
{ return (variablesRep) ? variablesRep->variablesView : variablesView; }


inline const String& Variables::variables_id() const
{ return (variablesRep) ? variablesRep->idVariables : idVariables; }


inline const Sizet2DArray& Variables::variables_components() const
{
  return (variablesRep) ? variablesRep->variablesComponents
                        : variablesComponents;
}


inline bool Variables::is_null() const
{ return (variablesRep) ? false : true; }


inline void Variables::build_views()
{
  // called only from letters
  if (variablesView.first)  //!= EMPTY)
    build_active_views();
  if (variablesView.second) //!= EMPTY)
    build_inactive_views();
}


/// global comparison function for Variables
inline bool variables_id_compare(const Variables& vars, const void* id)
{ return ( *(const String*)id == vars.variables_id() ); }


// Having overloaded operators call read/write means that the operators need 
// not be a friend to private data because read/write functions are public.

/// istream extraction operator for Variables.
inline istream& operator>>(istream& s, Variables& vars)
{ vars.read(s); return s; }


/// ostream insertion operator for Variables.
inline ostream& operator<<(ostream& s, const Variables& vars)
{ vars.write(s); return s; }


/// BiStream extraction operator for Variables.
inline BiStream& operator>>(BiStream& s, Variables& vars)
{ vars.read(s); return s; }


/// BoStream insertion operator for Variables.
inline BoStream& operator<<(BoStream& s, const Variables& vars)
{ vars.write(s); return s; }


/// MPIUnpackBuffer extraction operator for Variables.
inline MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s, Variables& vars)
{ vars.read(s); return s; }


/// MPIPackBuffer insertion operator for Variables.
inline MPIPackBuffer& operator<<(MPIPackBuffer& s, const Variables& vars)
{ vars.write(s); return s; }

/// inequality operator for Variables
inline bool operator!=(const Variables& vars1, const Variables& vars2)
{ return !(vars1 == vars2); }

} // namespace Dakota

#endif
