/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        Variables
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "ProblemDescDB.H"
#include "DakotaVariables.H"
#include "AllVariables.H"
#include "MergedVariables.H"
#include "DistinctVariables.H"
#include "DakotaBinStream.H"
#include "data_util.h"
#ifdef HAVE_BOOST
#include "boost/functional/hash/hash.hpp"
#endif

//#define REFCOUNT_DEBUG

static const char rcsId[]="@(#) $Id";

namespace Dakota {


/** This constructor is the one which must build the base class data for all
    derived classes.  get_variables() instantiates a derived class letter
    and the derived constructor selects this base class constructor in its 
    initialization list (to avoid the recursion of the base class constructor
    calling get_variables() again).  Since the letter IS the representation, 
    its representation pointer is set to NULL (an uninitialized pointer causes
    problems in ~Variables). */
Variables::
Variables(BaseConstructor, const ProblemDescDB& problem_db,
	  const pair<short,short>& view):
  variablesView(view), cvStart(0), dvStart(0), icvStart(0), idvStart(0),
  numCV(0), numDV(0), numICV(0), numIDV(0),
  idVariables(problem_db.get_string("variables.id")), variablesRep(NULL),
  referenceCount(1)
{
  variablesComponents.reshape(4);
  // design
  variablesComponents[0].reshape(3);
  variablesComponents[0][1]
    = problem_db.get_sizet("variables.continuous_design");
  variablesComponents[0][2] = problem_db.get_sizet("variables.discrete_design");
  // aleatory uncertain
  variablesComponents[1].reshape(13);
  variablesComponents[1][1]
    = problem_db.get_sizet("variables.normal_uncertain");
  variablesComponents[1][2]
    = problem_db.get_sizet("variables.lognormal_uncertain");
  variablesComponents[1][3]
    = problem_db.get_sizet("variables.uniform_uncertain");
  variablesComponents[1][4]
    = problem_db.get_sizet("variables.loguniform_uncertain");
  variablesComponents[1][5]
    = problem_db.get_sizet("variables.triangular_uncertain");
  variablesComponents[1][6]
    = problem_db.get_sizet("variables.exponential_uncertain");
  variablesComponents[1][7] = problem_db.get_sizet("variables.beta_uncertain");
  variablesComponents[1][8] = problem_db.get_sizet("variables.gamma_uncertain");
  variablesComponents[1][9]
    = problem_db.get_sizet("variables.gumbel_uncertain");
  variablesComponents[1][10]
    = problem_db.get_sizet("variables.frechet_uncertain");
  variablesComponents[1][11]
    = problem_db.get_sizet("variables.weibull_uncertain");
  variablesComponents[1][12]
    = problem_db.get_sizet("variables.histogram_uncertain");
  // epistemic uncertain
  variablesComponents[2].reshape(2);
  variablesComponents[2][1]
    = problem_db.get_sizet("variables.interval_uncertain");
  // state
  variablesComponents[3].reshape(3);
  variablesComponents[3][1]
    = problem_db.get_sizet("variables.continuous_state");
  variablesComponents[3][2] = problem_db.get_sizet("variables.discrete_state");
  // sum total number of design, uncertain, state and store as first entry
  for (size_t i=0; i<4; i++) {
    size_t sum = 0, len = variablesComponents[i].length();
    for (size_t j=1; j<len; j++)
      sum += variablesComponents[i][j];
    variablesComponents[i][0] = sum;
  }

#ifdef REFCOUNT_DEBUG
  Cout << "Variables::Variables(BaseConstructor) called to build base class "
       << "data for letter object." << endl;
#endif
}


/** The default constructor: variablesRep is NULL in this case (a populated
    problem_db is needed to build a meaningful Variables object).  This
    makes it necessary to check for NULL in the copy constructor, assignment
    operator, and destructor. */
Variables::Variables():
  cvStart(0), dvStart(0), icvStart(0), idvStart(0), numCV(0), numDV(0),
  numICV(0), numIDV(0), variablesRep(NULL), referenceCount(1)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Variables::Variables() called to build empty variables object."
       << endl;
#endif
}


/** This is the primary envelope constructor which uses problem_db to
    build a fully populated variables object.  It only needs to
    extract enough data to properly execute get_variables(problem_db),
    since the constructor overloaded with BaseConstructor builds the
    actual base class data inherited by the derived classes. */
Variables::Variables(const ProblemDescDB& problem_db):
  referenceCount(1) // not used since this is the envelope, not the letter
{
#ifdef REFCOUNT_DEBUG
  Cout << "Variables::Variables(ProblemDescDB&) called to instantiate envelope."
       << endl;
#endif

  // Set the rep pointer to the appropriate derived variables class
  variablesRep = get_variables(problem_db);
  if ( !variablesRep ) // bad type or insufficient memory
    abort_handler(-1);
}


/** Initializes variablesRep to the appropriate derived type, as given
    by problem_db attributes.  The standard derived class constructors
    are invoked.  */
Variables* Variables::get_variables(const ProblemDescDB& problem_db)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Envelope instantiating letter in get_variables(ProblemDescDB&)." 
       << endl;
#endif

  pair<short,short> view = get_view(problem_db);

  // This get_variables version invokes the standard constructor.
  int active_view = view.first;
  switch (active_view) {
  case MIXED_ALL:
    return new AllVariables(problem_db, view); break;
  //case MERGED_ALL:
  // The AllMerged case has been deleted since this combination is unlikely
  // (discrete variable relaxation is only envisioned for optimization; perhaps
  // global surrogate-based branch and bound would require it).
  //  return new AllMergedVariables(problem_db, view); break;
  case MIXED_DISTINCT_DESIGN: case MIXED_DISTINCT_ALEATORY_UNCERTAIN:
  case MIXED_DISTINCT_EPISTEMIC_UNCERTAIN: case MIXED_DISTINCT_UNCERTAIN:  
  case MIXED_DISTINCT_STATE:
    return new DistinctVariables(problem_db, view); break;
  case MERGED_DISTINCT_DESIGN: case MERGED_DISTINCT_ALEATORY_UNCERTAIN:
  case MERGED_DISTINCT_EPISTEMIC_UNCERTAIN: case MERGED_DISTINCT_UNCERTAIN:
  case MERGED_DISTINCT_STATE:
    return new MergedVariables(problem_db, view); break;
  default:
    Cerr << "Variables active view " << active_view
	 << " not currently supported in derived Variables classes." << endl;
    return NULL; break;
  }
}


/* This is the alternate envelope constructor for minimal
   instantiations on the fly.  This constructor executes
   get_variables(view), which invokes the default derived/base
   constructors.  Prior to use, a reshape() is needed. */
/*
Variables::Variables(const pair<short,short>& view) :
  referenceCount(1) // not used since this is the envelope, not the letter
{
#ifdef REFCOUNT_DEBUG
  Cout << "Variables::Variables(pair<short,short>&) called to instantiate "
       << "envelope." << endl;
#endif

  // Set the rep pointer to the appropriate derived variables class
  variablesRep = get_variables(view);
  if ( !variablesRep ) // bad type or insufficient memory
    abort_handler(-1);
}
*/


/** This is the alternate envelope constructor for instantiations on
    the fly.  This constructor executes get_variables(view), which
    invokes the default derived/base constructors, followed by a
    reshape() based on vars_comps. */
Variables::
Variables(const pair<short,short>& view, const Sizet2DArray& vars_comps) :
  referenceCount(1) // not used since this is the envelope, not the letter
{
#ifdef REFCOUNT_DEBUG
  Cout << "Variables::Variables(pair<short,short>&, Sizet2DArray&) called to "
       << "instantiate envelope." << endl;
#endif

  // Set the rep pointer to the appropriate derived variables class
  variablesRep = get_variables(view);
  if (variablesRep)
    variablesRep->reshape(vars_comps);
  else // bad type or insufficient memory
    abort_handler(-1);
}


/** Initializes variablesRep to the appropriate derived type, as given
    by view.  The default derived class constructors are invoked. */
Variables* Variables::get_variables(const pair<short,short>& view) const
{
#ifdef REFCOUNT_DEBUG
  Cout << "Envelope instantiating letter in get_variables(pair<short,short>&)."
       << endl;
#endif

  // This get_variables version invokes the default constructor.  Virtual
  // copy_rep() function must then be called to copy derived class attributes.
  int active_view = view.first;
  Variables* vars;
  switch (active_view) {
  case MIXED_ALL:
    vars = new AllVariables(); break;
  //case MERGED_ALL:
  // The AllMerged case has been deleted since this combination is unlikely
  // (discrete variable relaxation is only envisioned for optimization; perhaps
  // global surrogate-based branch and bound would require it).
  //  vars = new AllMergedVariables(); break;
  case MIXED_DISTINCT_DESIGN: case MIXED_DISTINCT_ALEATORY_UNCERTAIN:
  case MIXED_DISTINCT_EPISTEMIC_UNCERTAIN: case MIXED_DISTINCT_UNCERTAIN:  
  case MIXED_DISTINCT_STATE:
    vars = new DistinctVariables(); break;
  case MERGED_DISTINCT_DESIGN: case MERGED_DISTINCT_ALEATORY_UNCERTAIN:
  case MERGED_DISTINCT_EPISTEMIC_UNCERTAIN: case MERGED_DISTINCT_UNCERTAIN:
  case MERGED_DISTINCT_STATE:
    vars = new MergedVariables(); break;
  default:
    Cerr << "Variables active view " << active_view
	 << " not currently supported in derived Variables classes." << endl;
    vars = NULL; break;
  }
  if (vars)
    vars->variablesView = view;
  return vars;
}


pair<short,short> Variables::get_view(const ProblemDescDB& problem_db) const
{
  // Default variables behavior uses the distinct arrays for continuous
  // and discrete vars/bnds (used by most iterators as well as mixed variable 
  // strategies with separate continuous and discrete iterators).  In some 
  // cases (e.g., branch & bound), a merged data approach is used in which 
  // continuous and discrete arrays are combined into a single continuous array
  // (integrality is relaxed; the converse of truncating or discretizing reals
  // could also be used in the future if needed).  And in other cases (e.g., 
  // parameter studies and dace), no distinction is drawn between design,
  // uncertain, and state variable types and all types are collected into 
  // allVars arrays.

  // Since this is used by the envelope, don't set variablesView
  pair<short,short> view; // view.first = active, view.second = inactive

  // Possibilities:  distinct types vs. all types combined
  //                 mixed cont./discrete vs. merged cont./discrete
  // If distinct, then active/inactive could be design, uncertain, or state.
  const String& strat_name  = problem_db.get_string("strategy.type");
  const String& method_name = problem_db.get_string("method.algorithm");
  if (strat_name == "branch_and_bound") {
    view.first  = MERGED_DISTINCT_DESIGN;            // active
    view.second = EMPTY;//MERGED_DISTINCT_UNCERTAIN; // inactive
  }
  else if ( method_name.ends("_parameter_study") || method_name == "dace" ||
	    method_name.begins("fsu_") || method_name.begins("psuade_") || 
	    ( method_name.begins("nond_") &&
	      problem_db.get_bool("method.nond.all_variables") ) ) {
    view.first  = MIXED_ALL;                  // active
    view.second = EMPTY;                      // inactive
  }
  else if (method_name.begins("nond_")) {
    if (method_name == "nond_sampling") { // MC, LHS, Increm MC, Increm LHS
      size_t num_auv = problem_db.get_sizet("variables.aleatory_uncertain"),
	     num_euv = problem_db.get_sizet("variables.epistemic_uncertain");
      //if (problem_db.get_bool("method.nond.aleatory_variables"))
      //  view.first  = MIXED_DISTINCT_ALEATORY_UNCERTAIN;  // active
      //else if (problem_db.get_bool("method.nond.epistemic_variables"))
      //  view.first  = MIXED_DISTINCT_EPISTEMIC_UNCERTAIN; // active
      //else { // default is whatever is specified
      if (num_auv && num_euv)
	view.first  = MIXED_DISTINCT_UNCERTAIN;           // active
      else if (num_euv)
	view.first  = MIXED_DISTINCT_EPISTEMIC_UNCERTAIN; // active
      else if (num_auv)
	view.first  = MIXED_DISTINCT_ALEATORY_UNCERTAIN;  // active
      else {
	Cerr << "Error: uncertain variables required for nond_sampling in "
	     << "Variables::get_view()." << endl;
	abort_handler(-1);
      }
      //}
      view.second = EMPTY;//MIXED_DISTINCT_DESIGN;        // inactive
    }
    else if (method_name == "nond_evidence" ||
             method_name == "nond_interval_est") {
      view.first  = MIXED_DISTINCT_EPISTEMIC_UNCERTAIN;   // active
      view.second = EMPTY;//MIXED_DISTINCT_DESIGN;        // inactive
    }
    else { // stoch exp and reliability methods
      view.first  = MIXED_DISTINCT_ALEATORY_UNCERTAIN;    // active
      view.second = EMPTY;//MIXED_DISTINCT_DESIGN;        // inactive
    }
  }
  else {
    view.first  = MIXED_DISTINCT_DESIGN;                  // active
    view.second = EMPTY;//MIXED_DISTINCT_UNCERTAIN;       // inactive
  }
  return view;
}


/** Copy constructor manages sharing of variablesRep and incrementing
    of referenceCount. */
Variables::Variables(const Variables& vars)
{
  // Increment new (no old to decrement)
  variablesRep = vars.variablesRep;
  if (variablesRep) // Check for an assignment of NULL
    variablesRep->referenceCount++;

#ifdef REFCOUNT_DEBUG
  Cout << "Variables::Variables(Variables&)" << endl;
  if (variablesRep)
    Cout << "variablesRep referenceCount = " << variablesRep->referenceCount
	 << endl;
#endif
}


/** Assignment operator decrements referenceCount for old variablesRep, assigns
    new variablesRep, and increments referenceCount for new variablesRep. */
Variables Variables::operator=(const Variables& vars)
{
  if (variablesRep != vars.variablesRep) { // normal case: old != new
    // Decrement old
    if (variablesRep) // Check for NULL
      if ( --variablesRep->referenceCount == 0 ) 
	delete variablesRep;
    // Assign new
    variablesRep = vars.variablesRep;
  }
  // Increment new (either case: old == new or old != new)
  if (variablesRep) // Check for NULL
    variablesRep->referenceCount++;

#ifdef REFCOUNT_DEBUG
  Cout << "Variables::operator=(Variables&)" << endl;
  if (variablesRep)
    Cout << "variablesRep referenceCount = " << variablesRep->referenceCount
	 << endl;
#endif

  return *this; // calls copy constructor since returned by value
}


/** Destructor decrements referenceCount and only deletes variablesRep
    when referenceCount reaches zero. */
Variables::~Variables()
{ 
  // Check for NULL pointer 
  if (variablesRep) {
    --variablesRep->referenceCount;
#ifdef REFCOUNT_DEBUG
    Cout << "variablesRep referenceCount decremented to " 
         << variablesRep->referenceCount << endl;
#endif
    if (variablesRep->referenceCount == 0) {
#ifdef REFCOUNT_DEBUG
      Cout << "deleting variablesRep" << endl;
#endif
      delete variablesRep;
    }
  }
}


void Variables::build_active_views()
{
  if (variablesRep)
    variablesRep->build_active_views(); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual build_active_views"
	 << "() function.\nNo default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::build_inactive_views()
{
  if (variablesRep)
    variablesRep->build_inactive_views(); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual build_inactive_views"
	 << "() function.\nNo default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::inactive_view(short view2)
{
  if (variablesRep)
    variablesRep->inactive_view(view2);
  else {
    // view assigment checks are performed here (rather than in the NestedModel
    // ctor or within the Model recursion) so that all levels of recursion are
    // adequately addressed.  If active view is {MERGED,MIXED}_ALL, outer level
    // active view is aggregated in inner loop all view and inactive view must
    // remain EMPTY.
    short view1 = variablesView.first;
    if (view2 != view1 && view1 != MERGED_ALL && view1 != MIXED_ALL) {
      variablesView.second = view2;
      check_view_compatibility();
      build_inactive_views();
    }
  }
}


void Variables::check_view_compatibility()
{
  short active_view = variablesView.first, inactive_view = variablesView.second;
#ifdef DEBUG
  Cout << "Variables view update: active_view = " << active_view
       << " inactive_view = " << inactive_view << endl;
#endif // DEBUG

  // check assigned subModel inactive view for compatibility with its
  // active view
  bool error_flag = false;
  if ( ( active_view   >= MERGED_DISTINCT_DESIGN &&
	 active_view   <= MERGED_DISTINCT_STATE  &&
	 inactive_view >= MIXED_DISTINCT_DESIGN  &&
	 inactive_view <= MIXED_DISTINCT_STATE ) ||
       ( active_view   >= MIXED_DISTINCT_DESIGN  &&
	 active_view   <= MIXED_DISTINCT_STATE   &&
	 inactive_view >= MERGED_DISTINCT_DESIGN &&
	 inactive_view <= MERGED_DISTINCT_STATE ) ) {
    // don't overlap MERGED and MIXED
    Cerr << "Error: subModel active and inactive views are inconsistent in "
	 << "MERGED/MIXED definition in NestedModel::update_submodel_view()."
	 << endl;
    error_flag = true;
  }
  if ( ( ( active_view   == MERGED_DISTINCT_UNCERTAIN ||
	   active_view   == MIXED_DISTINCT_UNCERTAIN ) &&
	 ( inactive_view == MERGED_DISTINCT_ALEATORY_UNCERTAIN  ||
	   inactive_view == MERGED_DISTINCT_EPISTEMIC_UNCERTAIN ||
	   inactive_view == MIXED_DISTINCT_ALEATORY_UNCERTAIN   ||
	   inactive_view == MIXED_DISTINCT_EPISTEMIC_UNCERTAIN ) ) ||
       ( ( active_view   == MERGED_DISTINCT_ALEATORY_UNCERTAIN  ||
	   active_view   == MERGED_DISTINCT_EPISTEMIC_UNCERTAIN ||
	   active_view   == MIXED_DISTINCT_ALEATORY_UNCERTAIN   ||
	   active_view   == MIXED_DISTINCT_EPISTEMIC_UNCERTAIN ) &&
	 ( inactive_view == MERGED_DISTINCT_UNCERTAIN ||
	   inactive_view == MIXED_DISTINCT_UNCERTAIN ) ) ) {
    // don't overlap aggregated and separated UNCERTAIN
    Cerr << "Error: subModel active and inactive views are inconsistent in "
	 << "UNCERTAIN variable sets in NestedModel::update_submodel_view()."
	 << endl;
    error_flag = true;
  }
  if (error_flag)
    abort_handler(-1);
}


void Variables::initialize_all_continuous_var_types_ids(bool relax)
{
  size_t i, id_offset, index_offset, acv_cntr = 0,
    num_cdv = variablesComponents[0][1], num_ddv = variablesComponents[0][2],
    num_auv = variablesComponents[1][0], num_euv = variablesComponents[2][0],
    num_csv = variablesComponents[3][1], num_dsv = variablesComponents[3][2],
    num_uv  = num_auv + num_euv,
    num_acv = num_cdv + num_uv + num_csv,
    num_nuv = variablesComponents[1][1],  num_lnuv = variablesComponents[1][2],
    num_uuv = variablesComponents[1][3],  num_luuv = variablesComponents[1][4],
    num_tuv = variablesComponents[1][5],  num_exuv = variablesComponents[1][6],
    num_buv = variablesComponents[1][7],  num_gauv = variablesComponents[1][8],
    num_guuv = variablesComponents[1][9], num_fuv  = variablesComponents[1][10],
    num_wuv = variablesComponents[1][11], num_huv  = variablesComponents[1][12],
    num_iuv = variablesComponents[2][1];

  if (relax)
    num_acv += num_ddv + num_dsv;

  allContinuousVarIds.resize(boost::extents[num_acv]);
  allContinuousVarTypes.resize(boost::extents[num_acv]);
  for (i=0; i<num_cdv; ++i, ++acv_cntr) {
    allContinuousVarTypes[acv_cntr] = "continuous_design";
    allContinuousVarIds[acv_cntr] = i + 1;
  }
  if (relax) {
    id_offset = num_cdv + 1;
    for (i=0; i<num_ddv; ++i, ++acv_cntr) {
      allContinuousVarTypes[acv_cntr] = "discrete_design";
      allContinuousVarIds[acv_cntr] = i + id_offset;
    }
  }

  for (i=0; i<num_nuv; ++i, ++acv_cntr)
    allContinuousVarTypes[acv_cntr] = "normal_uncertain";
  for (i=0; i<num_lnuv; ++i, ++acv_cntr)
    allContinuousVarTypes[acv_cntr] = "lognormal_uncertain";
  for (i=0; i<num_uuv; ++i, ++acv_cntr)
    allContinuousVarTypes[acv_cntr] = "uniform_uncertain";
  for (i=0; i<num_luuv; ++i, ++acv_cntr)
    allContinuousVarTypes[acv_cntr] = "loguniform_uncertain";
  for (i=0; i<num_tuv; ++i, ++acv_cntr)
    allContinuousVarTypes[acv_cntr] = "triangular_uncertain";
  for (i=0; i<num_exuv; ++i, ++acv_cntr)
    allContinuousVarTypes[acv_cntr] = "exponential_uncertain";
  for (i=0; i<num_buv; ++i, ++acv_cntr)
    allContinuousVarTypes[acv_cntr] = "beta_uncertain";
  for (i=0; i<num_gauv; ++i, ++acv_cntr)
    allContinuousVarTypes[acv_cntr] = "gamma_uncertain";
  for (i=0; i<num_guuv; ++i, ++acv_cntr)
    allContinuousVarTypes[acv_cntr] = "gumbel_uncertain";
  for (i=0; i<num_fuv; ++i, ++acv_cntr)
    allContinuousVarTypes[acv_cntr] = "frechet_uncertain";
  for (i=0; i<num_wuv; ++i, ++acv_cntr)
    allContinuousVarTypes[acv_cntr] = "weibull_uncertain";
  for (i=0; i<num_huv; ++i, ++acv_cntr)
    allContinuousVarTypes[acv_cntr] = "histogram_uncertain";
  for (i=0; i<num_iuv; ++i, ++acv_cntr)
    allContinuousVarTypes[acv_cntr] = "interval_uncertain";

  id_offset = num_cdv + num_ddv + 1;
  index_offset = (relax) ? num_cdv+num_ddv: num_cdv;
  for (i=0; i<num_uv; ++i)
    allContinuousVarIds[i + index_offset] = i + id_offset;
  id_offset += num_uv;
  for (i=0; i<num_csv; ++i, ++acv_cntr) {
    allContinuousVarTypes[acv_cntr] = "continuous_state";
    allContinuousVarIds[acv_cntr] = i + id_offset;
  }
  if (relax) {
    id_offset += num_csv;
    for (i=0; i<num_dsv; ++i, ++acv_cntr) {
      allContinuousVarTypes[acv_cntr] = "discrete_state";
      allContinuousVarIds[acv_cntr] = i + id_offset;
    }
  }
}


void Variables::initialize_all_discrete_var_types_ids()
{
  size_t i, num_ddv = variablesComponents[0][2],
    num_dsv = variablesComponents[3][2];
  allDiscreteVarTypes.resize(boost::extents[num_ddv + num_dsv]);
  for (i=0; i<num_ddv; ++i)
    allDiscreteVarTypes[i] = "discrete_design";
  for (i=0; i<num_dsv; ++i)
    allDiscreteVarTypes[i+num_ddv] = "discrete_state";
}


void Variables::read(istream& s)
{
  if (variablesRep)
    variablesRep->read(s); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual read function.\n"
         << "No default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::write(ostream& s) const
{
  if (variablesRep)
    variablesRep->write(s); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual write function.\n"
         << "No default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::write_aprepro(ostream& s) const
{
  if (variablesRep)
    variablesRep->write_aprepro(s); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual write_aprepro "
         << "function.\nNo default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::read_annotated(istream& s)
{
  pair<short,short> view;
  s >> view.first;
  if (s.eof()) // exception handling since EOF may not be captured properly
    throw String("Empty record in Variables::read_annotated(istream&)");
  s >> view.second;

  if (variablesRep) { // should not occur in current usage
    if (variablesRep->variablesView == view)
      variablesRep->read_annotated(s); // envelope fwd to existing letter
    else { // decrement old reference count and replace with new letter
      Cerr << "Warning: variables type mismatch in Variables::read(BiStream&)."
	   << endl;
      if (--variablesRep->referenceCount == 0) 
        delete variablesRep;
      variablesRep = get_variables(view);
      variablesRep->read_annotated(s);
      // types/ids not required
    }
  }
  else { // read from neutral file: variablesRep must be instantiated
    variablesRep = get_variables(view);
    variablesRep->read_annotated(s); // envelope fwd to new letter
    // types/ids not required
  }
}


void Variables::write_annotated(ostream& s) const
{
  if (variablesRep) {
    s << variablesRep->variablesView.first  << ' '
      << variablesRep->variablesView.second << ' ';
    variablesRep->write_annotated(s); // envelope fwd to letter
    s << '\n';
    // types/ids not required
  }
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual write_annotated "
         << "function.\nNo default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::write_tabular(ostream& s) const
{
  if (variablesRep)
    variablesRep->write_tabular(s); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual write_tabular "
         << "function.\nNo default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::read(BiStream& s)
{
  pair<short,short> view;
  s >> view.first;
  if (s.eof()) // exception handling since EOF not captured properly by BiStream
    throw String("Empty record in Variables::read(BiStream&)");
  s >> view.second;

  if (variablesRep) { // should not occur in current usage
    if (variablesRep->variablesView == view)
      variablesRep->read(s); // envelope fwd to existing letter
    else { // decrement old reference count and replace with new letter
      Cerr << "Warning: variables type mismatch in Variables::read(BiStream&)."
	   << endl;
      if (--variablesRep->referenceCount == 0) 
        delete variablesRep;
      variablesRep = get_variables(view);
      variablesRep->read(s);
      // types/ids not required
    }
  }
  else { // read from restart: variablesRep must be instantiated
    variablesRep = get_variables(view);
    variablesRep->read(s); // envelope fwd to new letter
    // types/ids not required
  }
}


void Variables::write(BoStream& s) const
{
  if (variablesRep) {
    s << variablesRep->variablesView.first <<variablesRep->variablesView.second;
    variablesRep->write(s); // envelope fwd to letter
    // types/ids not required
  }
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual write function.\n"
         << "No default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::read(MPIUnpackBuffer& s)
{
  bool letter;
  s >> letter;
  if (letter) {
    pair<short,short> view;
    s >> view.first >> view.second;
    if (variablesRep) { // should not occur in current usage
      if (variablesRep->variablesView == view)
	variablesRep->read(s); // envelope fwd to existing letter
      else { // decrement old reference count and replace with new letter
	Cerr << "Warning: variables type mismatch in "
	     << "Variables::read(MPIUnpackBuffer&)." << endl;
	if (--variablesRep->referenceCount == 0) 
	  delete variablesRep;
	variablesRep = get_variables(view);
	variablesRep->read(s);
	// types/ids not required
      }
    }
    else { // buffer read on slaves: variablesRep must be instantiated
      variablesRep = get_variables(view);
      variablesRep->read(s); // envelope fwd to new letter
      // types/ids not required
    }
  }
  else if (variablesRep) {
    if (--variablesRep->referenceCount == 0)
      delete variablesRep;
    variablesRep == NULL;
  }
}


void Variables::write(MPIPackBuffer& s) const
{
  s << !is_null();
  if (variablesRep) {
    s << variablesRep->variablesView.first
      << variablesRep->variablesView.second;
    // VarTypes/VarIds not required
    variablesRep->write(s); // envelope fwd to letter
    // types/ids not required
  }
}


/** Deep copies are used for history mechanisms such as bestVariables
    and data_pairs since these must catalogue copies (and should not
    change as the representation within currentVariables changes). */
Variables Variables::copy() const
{
  // the envelope class instantiates a new envelope and a new letter and copies
  // current attributes into the new objects.

#ifdef REFCOUNT_DEBUG
  Cout << "Variables::copy() called to generate a deep copy with no "
       << "representation sharing." << endl;
#endif

  Variables vars; // new envelope: referenceCount=1, variablesRep=NULL

  if (variablesRep) {
    // new letter: allocate a variablesRep and copy derived attributes
    vars.variablesRep = get_variables(variablesRep->variablesView);

    // Copy attributes used by all letters (so code not replicated in copy_rep)
    vars.variablesRep->variablesComponents = variablesRep->variablesComponents;
    vars.variablesRep->allContinuousVars   = variablesRep->allContinuousVars;
    vars.variablesRep->allDiscreteVars     = variablesRep->allDiscreteVars;

    vars.variablesRep->allContinuousLabels.resize(
      boost::extents[variablesRep->allContinuousLabels.size()]);
    vars.variablesRep->allContinuousLabels = variablesRep->allContinuousLabels;
    vars.variablesRep->allContinuousVarTypes.resize(
      boost::extents[variablesRep->allContinuousVarTypes.size()]);
    vars.variablesRep->allContinuousVarTypes
      = variablesRep->allContinuousVarTypes;
    vars.variablesRep->allContinuousVarIds.resize(
      boost::extents[variablesRep->allContinuousVarIds.size()]);
    vars.variablesRep->allContinuousVarIds = variablesRep->allContinuousVarIds;

    vars.variablesRep->allDiscreteLabels.resize(
      boost::extents[variablesRep->allDiscreteLabels.size()]);
    vars.variablesRep->allDiscreteLabels = variablesRep->allDiscreteLabels;
    vars.variablesRep->allDiscreteVarTypes.resize(
      boost::extents[variablesRep->allDiscreteVarTypes.size()]);
    vars.variablesRep->allDiscreteVarTypes = variablesRep->allDiscreteVarTypes;

    vars.copy_rep(variablesRep); // or vice-versa as convenient
  }

  return vars;
}


void Variables::copy_rep(const Variables* vars_rep)
{
  if (variablesRep)
    variablesRep->copy_rep(vars_rep);
  //else default is no-op
}


void Variables::reshape(const Sizet2DArray& vars_comps)
{
  if (variablesRep) { // envelope
    variablesRep->variablesComponents = vars_comps;
    variablesRep->reshape_rep(vars_comps);
  }
  else { // letter
    variablesComponents = vars_comps;
    reshape_rep(vars_comps);
  }
}


void Variables::reshape_rep(const Sizet2DArray& vars_comps)
{
  if (variablesRep)
    variablesRep->reshape_rep(vars_comps);
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual reshape_rep "
	 << "function.\n       No default defined at base class." << endl;
    abort_handler(-1);
  }
}


const UIntArray& Variables::merged_discrete_ids() const
{
  if (!variablesRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual merged_discrete_ids"
	 << "() function.\n       No default defined at base class." << endl;
    abort_handler(-1);
  }
 
  // envelope fwd to letter
  return variablesRep->merged_discrete_ids();
}


/// equality operator for Variables
bool operator==(const Variables& vars1, const Variables& vars2)
{
  // this operator is a friend of Variables

  Variables* v1_rep = vars1.variablesRep;
  Variables* v2_rep = vars2.variablesRep;

  // Check derived type since operator== below is only defined for identical 
  // derived types.  If it becomes needed in the future, could add translations
  // between types or use all_merged_variables (nonconst and expensive).
  if (v1_rep->variablesView != v2_rep->variablesView)
    return false;

  // Check content using tolerance-based equality
  // (sizes/labels/types/ids not required)
  if (v2_rep->allContinuousVars != v1_rep->allContinuousVars ||
      v2_rep->allDiscreteVars   != v1_rep->allDiscreteVars)
    return false;

  return true;
}


#ifdef HAVE_BOOST
/// binary_equal_to (since 'operator==' is not suitable for boost/hashed lookup)
bool binary_equal_to(const Variables& vars1, const Variables& vars2)
{
  // this function is a friend of Variables

  Variables* v1_rep = vars1.variablesRep;
  Variables* v2_rep = vars2.variablesRep;

  // Require identical view
  if (v1_rep->variablesView != v2_rep->variablesView)
    return false;

  // Require identical content in variable array lengths and values
  // (labels are ignored)
  size_t cv_size = v1_rep->allContinuousVars.length();
  size_t dv_size = v1_rep->allDiscreteVars.length();
  if (v2_rep->allContinuousVars.length() != cv_size ||
      v2_rep->allDiscreteVars.length()   != dv_size)
    return false;
  size_t i;
  for (i=0; i<cv_size; ++i)
    if (v2_rep->allContinuousVars[i] != v1_rep->allContinuousVars[i])
      return false;
  for (i=0; i<dv_size; ++i)
    if (v2_rep->allDiscreteVars[i]   != v1_rep->allDiscreteVars[i])
      return false;

  return true;
}


/// hash_value for Variables - required by the new BMI hash_set of PRPairs
std::size_t hash_value(const Variables& vars)
{
  // this function is a friend of Variables

  std::size_t seed = 0;
  // require identical views and variables data
  Variables* v_rep = vars.variablesRep;
  boost::hash_combine(seed, v_rep->variablesView);
  boost::hash_combine(seed, v_rep->allContinuousVars);
  boost::hash_combine(seed, v_rep->allDiscreteVars);
  return seed;
}
#endif // HAVE_BOOST

} // namespace Dakota
