/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       Dakota::Constraints
//- Description: Container class of variables and bounds, each with fundamental
//-              arrays (from specification) and derived arrays (for special 
//-              cases).
//- Owner:       Mike Eldred
//- Version: $Id: DakotaConstraints.H 5793 2009-03-31 03:15:00Z mseldre $

#ifndef DAKOTA_CONSTRAINTS_H
#define DAKOTA_CONSTRAINTS_H

#include "data_types.h"


namespace Dakota {

class ProblemDescDB;


/// Base class for the variable constraints class hierarchy.

/** The Constraints class is the base class for the class hierarchy
    managing bound, linear, and nonlinear constraints.  Using the
    variable lower and upper bounds arrays from the input
    specification, different derived classes define different views of
    this data.  The linear and nonlinear constraint data is consistent
    in all views and is managed at the base class level.  For memory
    efficiency and enhanced polymorphism, the variable constraints
    hierarchy employs the "letter/envelope idiom" (see Coplien
    "Advanced C++", p. 133), for which the base class (Constraints)
    serves as the envelope and one of the derived classes (selected in
    Constraints::get_constraints()) serves as the letter. */

class Constraints
{
public:

  //
  //- Heading: Constructors, destructor, assignment operator
  //

  /// default constructor
  Constraints();
  /// standard constructor
  Constraints(const ProblemDescDB& problem_db, const pair<short,short>& view,
	      const Sizet2DArray& vars_comps);
  // alternate constructor for minimal instantiations on the fly (reshape reqd)
  //Constraints(const pair<short,short>& view);
  /// alternate constructor for instantiations on the fly
  Constraints(const pair<short,short>& view, const Sizet2DArray& vars_comps);
  /// copy constructor
  Constraints(const Constraints& con);

  /// destructor
  virtual ~Constraints();

  /// assignment operator
  Constraints operator=(const Constraints& con);

  //
  //- Heading: Virtual functions
  //

  // INPUT/OUTPUT

  /// write a variable constraints object to an ostream
  virtual void write(ostream& s)    const;
  /// read a variable constraints object from an istream
  virtual void read(istream& s);

  //
  //- Heading: Member functions
  //

  // ACTIVE VARIABLES

  /// return the active continuous variable lower bounds
  const RealDenseVector& continuous_lower_bounds() const;
  /// set the active continuous variable lower bounds
  void continuous_lower_bounds(const RealDenseVector& c_l_bnds);
  /// set an active continuous variable lower bound
  void continuous_lower_bound(const Real& c_l_bnd, const size_t& i);
  /// return the active continuous variable upper bounds
  const RealDenseVector& continuous_upper_bounds() const;
  /// set the active continuous variable upper bounds
  void continuous_upper_bounds(const RealDenseVector& c_u_bnds);
  /// set an active continuous variable upper bound
  void continuous_upper_bound(const Real& c_u_bnd, const size_t& i);

  /// return the active discrete variable lower bounds
  const IntDenseVector& discrete_lower_bounds() const;
  /// set the active discrete variable lower bounds
  void discrete_lower_bounds(const IntDenseVector& d_l_bnds);
  /// set an active discrete variable lower bound
  void discrete_lower_bound(const int& d_l_bnd, const size_t& i);
  /// return the active discrete variable upper bounds
  const IntDenseVector& discrete_upper_bounds() const;
  /// set the active discrete variable upper bounds
  void discrete_upper_bounds(const IntDenseVector& d_u_bnds);
  /// set an active discrete variable upper bound
  void discrete_upper_bound(const int& d_u_bnd, const size_t& i);

  // INACTIVE VARIABLES

  /// return the inactive continuous lower bounds
  const RealDenseVector& inactive_continuous_lower_bounds() const;
  /// set the inactive continuous lower bounds
  void inactive_continuous_lower_bounds(const RealDenseVector& i_c_l_bnds);
  /// return the inactive continuous upper bounds
  const RealDenseVector& inactive_continuous_upper_bounds() const;
  /// set the inactive continuous upper bounds
  void inactive_continuous_upper_bounds(const RealDenseVector& i_c_u_bnds);

  /// return the inactive discrete lower bounds
  const IntDenseVector& inactive_discrete_lower_bounds() const;
  /// set the inactive discrete lower bounds
  void inactive_discrete_lower_bounds(const IntDenseVector& i_d_l_bnds);
  /// return the inactive discrete upper bounds
  const IntDenseVector& inactive_discrete_upper_bounds() const;
  /// set the inactive discrete upper bounds
  void inactive_discrete_upper_bounds(const IntDenseVector& i_d_u_bnds);

  // ALL VARIABLES

  /// returns a single array with all continuous lower bounds
  const RealDenseVector& all_continuous_lower_bounds() const;
  /// sets all continuous lower bounds using a single array
  void all_continuous_lower_bounds(const RealDenseVector& a_c_l_bnds);
  /// set a lower bound within the all continuous lower bounds array
  void all_continuous_lower_bound(const Real& a_c_l_bnd, const size_t& i);
  /// returns a single array with all continuous upper bounds
  const RealDenseVector& all_continuous_upper_bounds() const;
  /// sets all continuous upper bounds using a single array
  void all_continuous_upper_bounds(const RealDenseVector& a_c_u_bnds);
  /// set an upper bound within the all continuous upper bounds array
  void all_continuous_upper_bound(const Real& a_c_u_bnd, const size_t& i);

  /// returns a single array with all discrete lower bounds
  const IntDenseVector& all_discrete_lower_bounds() const;
  /// sets all discrete lower bounds using a single array
  void all_discrete_lower_bounds(const IntDenseVector& a_d_l_bnds);
  /// set a lower bound within the all discrete lower bounds array
  void all_discrete_lower_bound(const int& a_d_l_bnd, const size_t& i);
  /// returns a single array with all discrete upper bounds
  const IntDenseVector& all_discrete_upper_bounds() const;
  /// sets all discrete upper bounds using a single array
  void all_discrete_upper_bounds(const IntDenseVector& a_d_u_bnds);
  /// set an upper bound within the all discrete upper bounds array
  void all_discrete_upper_bound(const int& a_d_u_bnd, const size_t& i);

  // LINEAR CONSTRAINTS

  /// return the number of linear inequality constraints
  size_t num_linear_ineq_constraints() const;
  /// return the number of linear equality constraints
  size_t num_linear_eq_constraints() const;

  /// return the linear inequality constraint coefficients
  const RealDenseMatrix& linear_ineq_constraint_coeffs() const;
  /// set the linear inequality constraint coefficients
  void linear_ineq_constraint_coeffs(const RealDenseMatrix& lin_ineq_coeffs);
  /// return the linear inequality constraint lower bounds
  const RealDenseVector& linear_ineq_constraint_lower_bounds() const;
  /// set the linear inequality constraint lower bounds
  void linear_ineq_constraint_lower_bounds(
    const RealDenseVector& lin_ineq_l_bnds);
  /// return the linear inequality constraint upper bounds
  const RealDenseVector& linear_ineq_constraint_upper_bounds() const;
  /// set the linear inequality constraint upper bounds
  void linear_ineq_constraint_upper_bounds(
    const RealDenseVector& lin_ineq_u_bnds);
  /// return the linear equality constraint coefficients
  const RealDenseMatrix& linear_eq_constraint_coeffs() const;
  /// set the linear equality constraint coefficients
  void linear_eq_constraint_coeffs(const RealDenseMatrix& lin_eq_coeffs);
  /// return the linear equality constraint targets
  const RealDenseVector& linear_eq_constraint_targets() const;
  /// set the linear equality constraint targets
  void linear_eq_constraint_targets(const RealDenseVector& lin_eq_targets);

  // NONLINEAR CONSTRAINTS

  /// return the number of nonlinear inequality constraints
  size_t num_nonlinear_ineq_constraints() const;
  /// return the number of nonlinear equality constraints
  size_t num_nonlinear_eq_constraints() const;

  /// return the nonlinear inequality constraint lower bounds
  const RealDenseVector& nonlinear_ineq_constraint_lower_bounds() const;
  /// set the nonlinear inequality constraint lower bounds
  void nonlinear_ineq_constraint_lower_bounds(
    const RealDenseVector& nln_ineq_l_bnds);
  /// return the nonlinear inequality constraint upper bounds
  const RealDenseVector& nonlinear_ineq_constraint_upper_bounds() const;
  /// set the nonlinear inequality constraint upper bounds
  void nonlinear_ineq_constraint_upper_bounds(
    const RealDenseVector& nln_ineq_u_bnds);
  /// return the nonlinear equality constraint targets
  const RealDenseVector& nonlinear_eq_constraint_targets() const;
  /// set the nonlinear equality constraint targets
  void nonlinear_eq_constraint_targets(const RealDenseVector& nln_eq_targets);

  /// for use when a deep copy is needed (the representation is _not_ shared)
  Constraints copy() const;

  /// reshape the linear/nonlinear/bound constraint arrays arrays within the
  /// Constraints hierarchy
  void reshape(const size_t& num_nln_ineq_cons, const size_t& num_nln_eq_cons,
	       const size_t& num_lin_ineq_cons, const size_t& num_lin_eq_cons,
	       const Sizet2DArray& vars_comps);
  /// reshape the linear/nonlinear constraint arrays within the
  /// Constraints hierarchy
  void reshape(const size_t& num_nln_ineq_cons, const size_t& num_nln_eq_cons,
	       const size_t& num_lin_ineq_cons, const size_t& num_lin_eq_cons);
  /// reshape the lower/upper bound arrays within the Constraints hierarchy
  void reshape(const Sizet2DArray& vars_comps);

  /// sets the inactive view based on higher level (nested) context
  void inactive_view(short view2);

  /// function to check constraintsRep (does this envelope contain a letter)
  bool is_null() const;

protected:

  //
  //- Heading: Constructors
  //

  /// constructor initializes the base class part of letter classes
  /// (BaseConstructor overloading avoids infinite recursion in the
  /// derived class constructors - Coplien, p. 139)
  Constraints(BaseConstructor, const ProblemDescDB& problem_db,
	      const pair<short,short>& view, const Sizet2DArray& vars_comps);

  //
  //- Heading: Virtual functions
  //

  /// Used by copy() to copy the contents of a letter class
  virtual void copy_rep(const Constraints* con_rep);
  /// Used by reshape(Sizet2DArray&) to rehape the contents of a letter class
  virtual void reshape_rep();

  /// construct active views of all variables bounds arrays
  virtual void build_active_views();
  /// construct inactive views of all variables bounds arrays
  virtual void build_inactive_views();

  //
  //- Heading: Member functions
  //

  /// construct active/inactive views of all variables arrays
  void build_views();

  /// perform checks on user input, convert linear constraint
  /// coefficient input to matrices, and assign defaults
  void manage_linear_constraints(const ProblemDescDB& problem_db);

  //
  //- Heading: Data
  //

  /// the variables view pair containing active (first) and inactive
  /// (second) view enumerations.  This is a pointer to the instance
  /// from the corresponding Variables object.
  pair<short,short> variablesView;
  /// two-dimensional array containing variable type counts for
  /// design/uncertain/state (first index) by sub-type (second index).  This
  /// is a pointer to the instance from the corresponding Variables object.
  const Sizet2DArray* variablesComponents;

  /// a continuous lower bounds array combining continuous design,
  /// uncertain, and continuous state variable types (all view).
  RealDenseVector allContinuousLowerBnds;
  /// a continuous upper bounds array combining continuous design,
  /// uncertain, and continuous state variable types (all view).
  RealDenseVector allContinuousUpperBnds;
  /// a discrete lower bounds array combining discrete design and
  /// discrete state variable types (all view).
  IntDenseVector  allDiscreteLowerBnds;
  /// a discrete upper bounds array combining discrete design and
  /// discrete state variable types (all view).
  IntDenseVector  allDiscreteUpperBnds;

  /// number of nonlinear inequality constraints
  size_t numNonlinearIneqCons;
  /// number of nonlinear equality constraints
  size_t numNonlinearEqCons;
  /// nonlinear inequality constraint lower bounds
  RealDenseVector nonlinearIneqConLowerBnds;
  /// nonlinear inequality constraint upper bounds
  RealDenseVector nonlinearIneqConUpperBnds;
  /// nonlinear equality constraint targets
  RealDenseVector nonlinearEqConTargets;

  /// number of linear inequality constraints
  size_t numLinearIneqCons;
  /// number of linear equality constraints
  size_t numLinearEqCons;
  /// linear inequality constraint coefficients
  RealDenseMatrix linearIneqConCoeffs; 
  /// linear equality constraint coefficients
  RealDenseMatrix linearEqConCoeffs;
  /// linear inequality constraint lower bounds
  RealDenseVector linearIneqConLowerBnds;
  /// linear inequality constraint upper bounds
  RealDenseVector linearIneqConUpperBnds;
  /// linear equality constraint targets
  RealDenseVector linearEqConTargets;

  //
  //- Heading: Data views
  //

  /// the active continuous lower bounds array view
  RealDenseVector continuousLowerBnds;
  /// the active continuous upper bounds array view
  RealDenseVector continuousUpperBnds;
  /// the active discrete lower bounds array view
  IntDenseVector discreteLowerBnds;
  /// the active discrete upper bounds array view
  IntDenseVector discreteUpperBnds;

  /// the inactive continuous lower bounds array view
  RealDenseVector inactiveContinuousLowerBnds;
  /// the inactive continuous upper bounds array view
  RealDenseVector inactiveContinuousUpperBnds;
  /// the inactive discrete lower bounds array view
  IntDenseVector inactiveDiscreteLowerBnds;
  /// the inactive discrete upper bounds array view
  IntDenseVector inactiveDiscreteUpperBnds;

private:

  //
  //- Heading: Member functions
  //

  /// Used only by the constructor to initialize constraintsRep to the 
  /// appropriate derived type.
  Constraints* get_constraints(const ProblemDescDB& problem_db,
			       const pair<short,short>& view,
			       const Sizet2DArray& vars_comps);
  /// Used by copy() to initialize constraintsRep to the appropriate
  /// derived type.
  Constraints* get_constraints(const pair<short,short>& view) const;

  //
  //- Heading: Data
  //

  /// pointer to the letter (initialized only for the envelope)
  Constraints* constraintsRep;
  /// number of objects sharing constraintsRep
  int referenceCount;
};


inline const RealDenseVector& Constraints::continuous_lower_bounds() const
{
  return (constraintsRep) ?
    constraintsRep->continuousLowerBnds : continuousLowerBnds;
}


inline void Constraints::
continuous_lower_bounds(const RealDenseVector& c_l_bnds)
{
  if (constraintsRep)
    constraintsRep->continuousLowerBnds.assign(c_l_bnds);
  else
    continuousLowerBnds.assign(c_l_bnds);
}


inline void Constraints::
continuous_lower_bound(const Real& c_l_bnd, const size_t& i)
{
  if (constraintsRep)
    constraintsRep->continuousLowerBnds[i] = c_l_bnd;
  else
    continuousLowerBnds[i] = c_l_bnd;
}


inline const RealDenseVector& Constraints::continuous_upper_bounds() const
{
  return (constraintsRep) ?
    constraintsRep->continuousUpperBnds : continuousUpperBnds;
}


inline void Constraints::
continuous_upper_bounds(const RealDenseVector& c_u_bnds)
{
  if (constraintsRep)
    constraintsRep->continuousUpperBnds.assign(c_u_bnds);
  else
    continuousUpperBnds.assign(c_u_bnds);
}


inline void Constraints::
continuous_upper_bound(const Real& c_u_bnd, const size_t& i)
{
  if (constraintsRep)
    constraintsRep->continuousUpperBnds[i] = c_u_bnd;
  else
    continuousUpperBnds[i] = c_u_bnd;
}


inline const IntDenseVector& Constraints::discrete_lower_bounds() const
{
  return (constraintsRep) ?
    constraintsRep->discreteLowerBnds : discreteLowerBnds;
}


inline void Constraints::discrete_lower_bounds(const IntDenseVector& d_l_bnds)
{
  if (constraintsRep)
    constraintsRep->discreteLowerBnds.assign(d_l_bnds);
  else
    discreteLowerBnds.assign(d_l_bnds);
}


inline void Constraints::
discrete_lower_bound(const int& d_l_bnd, const size_t& i)
{
  if (constraintsRep)
    constraintsRep->discreteLowerBnds[i] = d_l_bnd;
  else
    discreteLowerBnds[i] = d_l_bnd;
}


inline const IntDenseVector& Constraints::discrete_upper_bounds() const
{
  return (constraintsRep) ?
    constraintsRep->discreteUpperBnds : discreteUpperBnds;
}


inline void Constraints::discrete_upper_bounds(const IntDenseVector& d_u_bnds)
{
  if (constraintsRep)
    constraintsRep->discreteUpperBnds.assign(d_u_bnds);
  else
    discreteUpperBnds.assign(d_u_bnds);
}


inline void Constraints::
discrete_upper_bound(const int& d_u_bnd, const size_t& i)
{
  if (constraintsRep)
    constraintsRep->discreteUpperBnds[i] = d_u_bnd;
  else
    discreteUpperBnds[i] = d_u_bnd;
}


inline const RealDenseVector& Constraints::
inactive_continuous_lower_bounds() const
{
  return (constraintsRep) ?
    constraintsRep->inactiveContinuousLowerBnds : inactiveContinuousLowerBnds;
}


inline void Constraints::
inactive_continuous_lower_bounds(const RealDenseVector& i_c_l_bnds)
{
  if (constraintsRep)
    constraintsRep->inactiveContinuousLowerBnds.assign(i_c_l_bnds);
  else
    inactiveContinuousLowerBnds.assign(i_c_l_bnds);
}


inline const RealDenseVector& Constraints::
inactive_continuous_upper_bounds() const
{
  return (constraintsRep) ?
    constraintsRep->inactiveContinuousUpperBnds : inactiveContinuousUpperBnds;
}


inline void Constraints::
inactive_continuous_upper_bounds(const RealDenseVector& i_c_u_bnds)
{
  if (constraintsRep)
    constraintsRep->inactiveContinuousUpperBnds.assign(i_c_u_bnds);
  else
    inactiveContinuousUpperBnds.assign(i_c_u_bnds);
}


inline const IntDenseVector& Constraints::inactive_discrete_lower_bounds() const
{
  return (constraintsRep) ?
    constraintsRep->inactiveDiscreteLowerBnds : inactiveDiscreteLowerBnds;
}


inline void Constraints::
inactive_discrete_lower_bounds(const IntDenseVector& i_d_l_bnds)
{
  if (constraintsRep)
    constraintsRep->inactiveDiscreteLowerBnds.assign(i_d_l_bnds);
  else
    inactiveDiscreteLowerBnds.assign(i_d_l_bnds);
}


inline const IntDenseVector& Constraints::inactive_discrete_upper_bounds() const
{
  return(constraintsRep) ?
    constraintsRep-> inactiveDiscreteUpperBnds : inactiveDiscreteUpperBnds;
}


inline void Constraints::
inactive_discrete_upper_bounds(const IntDenseVector& i_d_u_bnds)
{
  if (constraintsRep)
    constraintsRep->inactiveDiscreteUpperBnds.assign(i_d_u_bnds);
  else
    inactiveDiscreteUpperBnds.assign(i_d_u_bnds);
}


inline const RealDenseVector& Constraints::all_continuous_lower_bounds() const
{
  return (constraintsRep) ?
    constraintsRep->allContinuousLowerBnds : allContinuousLowerBnds;
}


inline void Constraints::
all_continuous_lower_bounds(const RealDenseVector& a_c_l_bnds)
{
  if (constraintsRep)
    constraintsRep->allContinuousLowerBnds.assign(a_c_l_bnds);
  else
    allContinuousLowerBnds.assign(a_c_l_bnds);
}


inline void Constraints::
all_continuous_lower_bound(const Real& a_c_l_bnd, const size_t& i)
{
  if (constraintsRep)
    constraintsRep->allContinuousLowerBnds[i] = a_c_l_bnd;
  else
    allContinuousLowerBnds[i] = a_c_l_bnd;
}


inline const RealDenseVector& Constraints::all_continuous_upper_bounds() const
{
  return (constraintsRep) ?
    constraintsRep->allContinuousUpperBnds : allContinuousUpperBnds;
}


inline void Constraints::
all_continuous_upper_bounds(const RealDenseVector& a_c_u_bnds)
{
  if (constraintsRep)
    constraintsRep->allContinuousUpperBnds.assign(a_c_u_bnds);
  else
    allContinuousUpperBnds.assign(a_c_u_bnds);
}


inline void Constraints::
all_continuous_upper_bound(const Real& a_c_u_bnd, const size_t& i)
{
  if (constraintsRep)
    constraintsRep->allContinuousUpperBnds[i] = a_c_u_bnd;
  else
    allContinuousUpperBnds[i] = a_c_u_bnd;
}


inline const IntDenseVector& Constraints::all_discrete_lower_bounds() const
{
  return (constraintsRep) ?
    constraintsRep->allDiscreteLowerBnds : allDiscreteLowerBnds;
}


inline void Constraints::
all_discrete_lower_bounds(const IntDenseVector& a_d_l_bnds)
{
  if (constraintsRep)
    constraintsRep->allDiscreteLowerBnds.assign(a_d_l_bnds);
  else
    allDiscreteLowerBnds.assign(a_d_l_bnds);
}


inline void Constraints::
all_discrete_lower_bound(const int& a_d_l_bnd, const size_t& i)
{
  if (constraintsRep)
    constraintsRep->allDiscreteLowerBnds[i] = a_d_l_bnd;
  else
    allDiscreteLowerBnds[i] = a_d_l_bnd;
}


inline const IntDenseVector& Constraints::all_discrete_upper_bounds() const
{
  return (constraintsRep) ?
    constraintsRep->allDiscreteUpperBnds : allDiscreteUpperBnds;
}


inline void Constraints::
all_discrete_upper_bounds(const IntDenseVector& a_d_u_bnds)
{
  if (constraintsRep)
    constraintsRep->allDiscreteUpperBnds.assign(a_d_u_bnds);
  else
    allDiscreteUpperBnds.assign(a_d_u_bnds);
}


inline void Constraints::
all_discrete_upper_bound(const int& a_d_u_bnd, const size_t& i)
{
  if (constraintsRep)
    constraintsRep->allDiscreteUpperBnds[i] = a_d_u_bnd;
  else
    allDiscreteUpperBnds[i] = a_d_u_bnd;
}


// nonvirtual functions can access letter attributes directly (only need to fwd
// member function call when the function could be redefined).
inline size_t Constraints::num_linear_ineq_constraints() const
{
  return (constraintsRep) ? 
    constraintsRep->numLinearIneqCons : numLinearIneqCons;
}


inline size_t Constraints::num_linear_eq_constraints() const
{ return (constraintsRep) ? constraintsRep->numLinearEqCons : numLinearEqCons; }


inline const RealDenseMatrix& Constraints::linear_ineq_constraint_coeffs() const
{
  return (constraintsRep) ?
    constraintsRep->linearIneqConCoeffs : linearIneqConCoeffs;
}


inline void Constraints::
linear_ineq_constraint_coeffs(const RealDenseMatrix& lin_ineq_coeffs)
{
  if (constraintsRep)
    constraintsRep->linearIneqConCoeffs = lin_ineq_coeffs;
  else
    linearIneqConCoeffs = lin_ineq_coeffs;
}


inline const RealDenseVector& Constraints::
linear_ineq_constraint_lower_bounds() const
{
  return (constraintsRep) ?
    constraintsRep->linearIneqConLowerBnds : linearIneqConLowerBnds;
}


inline void Constraints::
linear_ineq_constraint_lower_bounds(const RealDenseVector& lin_ineq_l_bnds)
{
  if (constraintsRep)
    constraintsRep->linearIneqConLowerBnds = lin_ineq_l_bnds;
  else
    linearIneqConLowerBnds = lin_ineq_l_bnds;
}


inline const RealDenseVector& Constraints::
linear_ineq_constraint_upper_bounds() const
{
  return (constraintsRep) ?
    constraintsRep->linearIneqConUpperBnds : linearIneqConUpperBnds;
}


inline void Constraints::
linear_ineq_constraint_upper_bounds(const RealDenseVector& lin_ineq_u_bnds)
{
  if (constraintsRep)
    constraintsRep->linearIneqConUpperBnds = lin_ineq_u_bnds;
  else
    linearIneqConUpperBnds = lin_ineq_u_bnds;
}


inline const RealDenseMatrix& Constraints::linear_eq_constraint_coeffs() const
{
  return (constraintsRep) ?
    constraintsRep->linearEqConCoeffs : linearEqConCoeffs;
}


inline void Constraints::
linear_eq_constraint_coeffs(const RealDenseMatrix& lin_eq_coeffs)
{
  if (constraintsRep)
    constraintsRep->linearEqConCoeffs = lin_eq_coeffs;
  else
    linearEqConCoeffs = lin_eq_coeffs;
}


inline const RealDenseVector& Constraints::linear_eq_constraint_targets() const
{
  return (constraintsRep) ?
    constraintsRep->linearEqConTargets : linearEqConTargets;
}


inline void Constraints::
linear_eq_constraint_targets(const RealDenseVector& lin_eq_targets)
{
  if (constraintsRep)
    constraintsRep->linearEqConTargets = lin_eq_targets;
  else
    linearEqConTargets = lin_eq_targets;
}


inline size_t Constraints::num_nonlinear_ineq_constraints() const
{
  return (constraintsRep) ?
    constraintsRep->numNonlinearIneqCons : numNonlinearIneqCons;
}


inline size_t Constraints::num_nonlinear_eq_constraints() const
{
  return (constraintsRep) ?
    constraintsRep->numNonlinearEqCons : numNonlinearEqCons;
}


inline const RealDenseVector& Constraints::
nonlinear_ineq_constraint_lower_bounds() const
{
  return (constraintsRep) ?
    constraintsRep->nonlinearIneqConLowerBnds : nonlinearIneqConLowerBnds;
}


inline void Constraints::
nonlinear_ineq_constraint_lower_bounds(const RealDenseVector& nln_ineq_l_bnds)
{
  if (constraintsRep)
    constraintsRep->nonlinearIneqConLowerBnds = nln_ineq_l_bnds;
  else
    nonlinearIneqConLowerBnds = nln_ineq_l_bnds;
}


inline const RealDenseVector& Constraints::
nonlinear_ineq_constraint_upper_bounds() const
{
  return (constraintsRep) ?
    constraintsRep->nonlinearIneqConUpperBnds : nonlinearIneqConUpperBnds;
}


inline void Constraints::
nonlinear_ineq_constraint_upper_bounds(const RealDenseVector& nln_ineq_u_bnds)
{
  if (constraintsRep)
    constraintsRep->nonlinearIneqConUpperBnds = nln_ineq_u_bnds;
  else
    nonlinearIneqConUpperBnds = nln_ineq_u_bnds;
}


inline const RealDenseVector& Constraints::
nonlinear_eq_constraint_targets() const
{
  return (constraintsRep) ?
    constraintsRep->nonlinearEqConTargets : nonlinearEqConTargets;
}


inline void Constraints::
nonlinear_eq_constraint_targets(const RealDenseVector& nln_eq_targets)
{
  if (constraintsRep)
    constraintsRep->nonlinearEqConTargets = nln_eq_targets;
  else
    nonlinearEqConTargets = nln_eq_targets;
}


inline bool Constraints::is_null() const
{ return (constraintsRep) ? false : true; }


inline void Constraints::build_views()
{
  // called only from letters
  if (variablesView.first)  //!= EMPTY)
    build_active_views();
  if (variablesView.second) //!= EMPTY)
    build_inactive_views();
}


// Having overloaded operators call read/write means that the operators need 
// not be a friend to private data because read/write functions are public.

/// istream extraction operator for Constraints
inline istream& operator>>(istream& s, Constraints& con)
{ con.read(s); return s; }


/// ostream insertion operator for Constraints
inline ostream& operator<<(ostream& s, const Constraints& con)
{ con.write(s); return s; }

} // namespace Dakota

#endif
