/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       Dakota::BaseVector<T>
//- Description: The Dakota::BaseVector class is the base class for 
//-		 Dakota::Vector and Dakota::Matrix classes. It is a
//-              simple vector class with little added functionality.
//- Owner:       Mario Alleva
//- Checked by:
//- Version: $Id

#ifndef DAKOTA_BASE_VECTOR_H
#define DAKOTA_BASE_VECTOR_H

#include "template_defs.h"


namespace Dakota {

/// Base class for the Dakota::Matrix and Dakota::Vector classes

/** The Dakota::BaseVector class is the base class for the
    Dakota::Matrix class. It is used to define a common vector
    interface for both the STL and RW vector classes. If the STL
    version is based on the valarray class then some basic vector
    operations such as + , * are available. */

template<class T>
class BaseVector : public DAKOTA_BASE_VECTOR<T>
{
public:

  //
  //- Heading: Constructors and destructor
  //

  /// Default constructor
  BaseVector();
  /// Constructor, creates vector of size
  BaseVector(size_t size);
  /// Constructor, creates vector of size with initial value of initial_val
  BaseVector(size_t size, const T& initial_val);

  /// Destructor
  ~BaseVector();

  /// Copy constructor
  BaseVector(const BaseVector<T>& a);

  //
  //- Heading: Assignment operators
  //

  /// Normal assignment operator.
  BaseVector<T>& operator=(const BaseVector<T>& a);

  /// Assigns all values of vector to ival
  BaseVector<T>& operator=(const T& ival);

  //
  //- Heading: Indexing operators
  //

#ifdef HAVE_PARTIAL_ORDERING
  // Quiets errors occurring for instances of array[int], e.g.:
  //   more than one operator [] matches these operands:
  //   built-in operator "pointer-to-object[integer]"
  //   function "Vector<T>::operator[](size_t) [with T=int]"

  /// alternate bounds-checked indexing operator for int indices
  T& operator[](int i);
  /// alternate bounds-checked const indexing operator for int indices
  const T& operator[](int i) const;
  // For completeness sake, could add:
  //T& operator() (int i);
  //const T& operator() (int i) const;
#endif // HAVE_PARTIAL_ORDERING

  // Many of the methods below add functionality to STL vectors to mimic old
  // functions from the RWTValVector class in order to avoid source changes in
  // the main DAKOTA code.  Note that the bounds-checked and nonbounds-checked
  // operators in BaseVector are different from STL vectors. 

  /// Returns the object at index i, (can use as lvalue)
  T& operator[](size_t i);
  /// Returns the object at index i, const (can't use as lvalue)
  const T& operator[](size_t i) const;

  /// Index operator, not bounds checked
  T& operator() (size_t i);
  /// Index operator const , not bounds checked
  const T& operator() (size_t i) const;

  //
  //- Heading: Member functions 
  //

  /// Returns size of vector
  size_t length() const;

  /// Resizes vector to size sz
  void reshape(size_t sz);

  /// Returns const pointer to standard C array.  Use with care.
  const T* data() const;

protected:

  //
  //- Heading: Member functions 
  //

  /// Returns pointer to standard C array.  Use with care.
  T* array() const;
};


template <class T>
inline BaseVector<T>::BaseVector(): DAKOTA_BASE_VECTOR<T>()
{ }


template <class T>
inline BaseVector<T>::BaseVector(size_t size): DAKOTA_BASE_VECTOR<T>(size)
{ }


/** Constructor which takes an initial size and an initial value,
    allocates an area of initial size and initializes it with input
    value.  Calls base class constructor */
template <class T>
inline BaseVector<T>::BaseVector(size_t size, const T& initial_val):
  DAKOTA_BASE_VECTOR<T>(size, initial_val)
{ }


template <class T>
inline BaseVector<T>::~BaseVector()
{ }


template <class T>
inline BaseVector<T>::BaseVector(const BaseVector<T>& a):
  DAKOTA_BASE_VECTOR<T>(a)
{ }


template <class T>
inline BaseVector<T>& BaseVector<T>::operator=(const BaseVector<T>& a)
{
  if (this != &a) // verify that these are different objects
    DAKOTA_BASE_VECTOR<T>::operator=(a); // base class assignment operator
  return *this;
}


template <class T>
inline BaseVector<T>& BaseVector<T>::operator=(const T& ival)
{
  // set the values in the array using the base class assign() function
  assign(DAKOTA_BASE_VECTOR<T>::size(), ival);
  return *this;
}


#ifdef HAVE_PARTIAL_ORDERING
template <class T> 
inline T& BaseVector<T>::operator[](int i)
{
#ifdef HAVE_VECTOR_AT
  return DAKOTA_BASE_VECTOR<T>::at(i);
#else
  return DAKOTA_BASE_VECTOR<T>::operator[](i);
#endif // HAVE_VECTOR_AT
}


template <class T> 
inline const T& BaseVector<T>::operator[](int i) const
{
#ifdef HAVE_VECTOR_AT
  return DAKOTA_BASE_VECTOR<T>::at(i);
#else
  return DAKOTA_BASE_VECTOR<T>::operator[](i);
#endif // HAVE_VECTOR_AT
}
#endif // HAVE_PARTIAL_ORDERING


/** Index operator, calls the STL method at() which is bounds checked.  Mimics
    the RW vector class.  Note: the at() method is not supported by the __GNUC__
    STL implementation or by builds omitting exceptions (e.g., SIERRA). */
template <class T> 
inline T& BaseVector<T>::operator[](size_t i)
{
#ifdef HAVE_VECTOR_AT
  return DAKOTA_BASE_VECTOR<T>::at(i);
#else
  return DAKOTA_BASE_VECTOR<T>::operator[](i);
#endif // HAVE_VECTOR_AT
}


/** Const versions of the index operator calls the STL method at() which is
    bounds checked.  Mimics the RW vector class.  Note: the at() method is not
    supported by the __GNUC__ STL implementation or by builds omitting
    exceptions (e.g., SIERRA). */
template <class T>
inline const T& BaseVector<T>::operator[](size_t i) const
{
#ifdef HAVE_VECTOR_AT
  return DAKOTA_BASE_VECTOR<T>::at(i);
#else
  return DAKOTA_BASE_VECTOR<T>::operator[](i);
#endif // HAVE_VECTOR_AT
}


/** Non bounds check index operator, calls the STL operator[] which is not
    bounds checked. Needed to mimic the RW vector class */
template <class T>
inline T& BaseVector<T>::operator()(size_t i)
{ return DAKOTA_BASE_VECTOR<T>::operator[](i); }


/** Const version of the non-bounds check index operator, calls the
    STL operator[] which is not bounds checked.  Needed to mimic the
    RW vector class */
template <class T>
inline const T& BaseVector<T>::operator()(size_t i) const
{ return DAKOTA_BASE_VECTOR<T>::operator[](i); }


/** Returns the length of the array by calling the STL size method.
    Needed to mimic the RW vector class */
template <class T>
inline size_t BaseVector<T>::length() const
{ return DAKOTA_BASE_VECTOR<T>::size(); }


/** Resizes the array to size sz by calling the STL resize method.
    Needed to mimic the RW vector class */
template <class T>
inline void BaseVector<T>::reshape(size_t sz)
{ DAKOTA_BASE_VECTOR<T>::resize(sz); }


/** Returns a const pointer to the data within the array.
    USE WITH CARE. Needed to mimic RW vector class. */
template <class T>
inline const T* BaseVector<T>::data() const
{
  return (DAKOTA_BASE_VECTOR<T>::empty()) ? NULL :
    (T*)&DAKOTA_BASE_VECTOR<T>::front();
}


/** Returns a non-const pointer to the data within the array.
    Non-const version of data() used by derived classes. */
template <class T>
inline T* BaseVector<T>::array() const
{
  return (DAKOTA_BASE_VECTOR<T>::empty()) ? NULL :
    (T*)&DAKOTA_BASE_VECTOR<T>::front();
}

} // namespace Dakota

#endif
