/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       Analyzer
//- Description: Implementation code for the Analyzer class
//- Owner:       Mike Eldred
//- Checked by:

#include "system_defs.h"
#include "data_io.h"
#include "DakotaModel.H"
#include "DakotaAnalyzer.H"
#include "ProblemDescDB.H"
#include "ParallelLibrary.H"
#ifdef DAKOTA_FSUDACE
#include "fsu.H"
#endif

static const char rcsId[]="@(#) $Id: DakotaAnalyzer.C 5824 2009-04-16 19:11:45Z briadam $";


namespace Dakota {

Analyzer::Analyzer(Model& model): Iterator(BaseConstructor(), model)
{ iteratedModel = model; }


Analyzer::Analyzer(NoDBBaseConstructor, Model& model):
  Iterator(NoDBBaseConstructor(), model)
{ iteratedModel = model; }


Analyzer::Analyzer(NoDBBaseConstructor): Iterator(NoDBBaseConstructor())
{ }


/** Convenience function for derived classes with sets of function
    evaluations to perform (e.g., NonDSampling, DDACEDesignCompExp,
    FSUDesignCompExp, ParamStudy). */
void Analyzer::
evaluate_parameter_sets(Model& model, bool log_resp_flag, bool log_best_flag)
{
  size_t i, num_evals = allVariables.length();
  if (probDescDB.parallel_library().command_line_dry_run()) {
    ofstream tabular_file("dakota_dry_run.dat");
    tabular_file << '%'; // matlab comment syntax
    write_data_tabular(tabular_file, model.all_continuous_variable_labels());
    write_data_tabular(tabular_file, model.all_discrete_variable_labels());
    tabular_file << '\n';
    for (i=0; i<num_evals; i++) {
      allVariables[i].write_tabular(tabular_file);
      tabular_file << '\n';
    }
    tabular_file.flush();
    tabular_file.close();
    // Can't just return as subsequent operations would be in error.
    // Consider throw here with catch in main().
    Cout << "\nDry run complete: Analyzer points written to file "
	 << "dakota_dry_run.dat; exiting.\n" << endl;
    abort_handler(0);
  }

  // This function does not need an iteratorRep fwd because it is a
  // protected fn only called by letter classes.

  // allVariables defines the set of function evaluations to be performed
  bool header_flag = (allHeaders.length() == num_evals);
  if (log_resp_flag && !asynchFlag)
    allResponses.reshape(num_evals);

  // Loop over parameter sets and compute responses.  Collect data
  // and track best evaluations based on flags.
  short model_view = model.current_variables().view().first;
  for (i=0; i<num_evals; i++) {
    // output the evaluation header (if present)
    if (header_flag)
      Cout << allHeaders[i];

    // set the variables
    // Note: allow allVariables to be used as a partial container,
    // by only setting variables that are present within the model.
    short vars_view = allVariables[i].view().first;
    if (vars_view == model_view) {
      //model.active_variables(allVariables[i]);
      size_t num_cv = allVariables[i].cv(), num_dv = allVariables[i].dv();
      if (num_cv)
	model.continuous_variables(allVariables[i].continuous_variables());
      if (num_dv)
	model.discrete_variables(allVariables[i].discrete_variables());
    }
    else if ( vars_view >= MERGED_DISTINCT_DESIGN &&
	      ( model_view == MERGED_ALL || model_view == MIXED_ALL ) ) {
      size_t num_acv = allVariables[i].acv(), num_adv = allVariables[i].adv();
      if (num_acv)
	model.continuous_variables(allVariables[i].all_continuous_variables());
      if (num_adv)
	model.discrete_variables(allVariables[i].all_discrete_variables());
    }
    else if ( ( vars_view == MERGED_ALL || vars_view == MIXED_ALL ) &&
	      model_view >= MERGED_DISTINCT_DESIGN ) {
      size_t num_cv = allVariables[i].cv(), num_dv = allVariables[i].dv();
      if (num_cv)
	model.all_continuous_variables(allVariables[i].continuous_variables());
      if (num_dv)
	model.all_discrete_variables(allVariables[i].discrete_variables());
    }
    else if ( ( vars_view  == MERGED_DISTINCT_UNCERTAIN &&
	        model_view == MERGED_DISTINCT_ALEATORY_UNCERTAIN ) || 
	      ( vars_view  == MIXED_DISTINCT_UNCERTAIN &&
		model_view == MIXED_DISTINCT_ALEATORY_UNCERTAIN ) ) {
      const RealDenseVector& uv = allVariables[i].continuous_variables();
      size_t num_auv = model.cv(), num_euv = uv.length() - num_auv;
      if (num_euv) {
	RealDenseVector auv(Teuchos::View, uv.values(), num_auv);
	model.continuous_variables(auv);
	//model.discrete_variables(dauv);
      }
      else
	model.continuous_variables(uv);
	//model.discrete_variables(duv);
    }
    else if ( ( vars_view  == MERGED_DISTINCT_UNCERTAIN &&
		model_view == MERGED_DISTINCT_EPISTEMIC_UNCERTAIN ) || 
	      ( vars_view  == MIXED_DISTINCT_UNCERTAIN &&
		model_view == MIXED_DISTINCT_EPISTEMIC_UNCERTAIN ) ) {
      const RealDenseVector& uv = allVariables[i].continuous_variables();
      size_t num_euv = model.cv(), num_auv = uv.length() - num_euv;
      if (num_auv) {
	RealDenseVector euv(Teuchos::View, const_cast<Real*>(&uv[num_auv]),
			    num_euv);
	model.continuous_variables(euv);
	//model.discrete_variables(deuv);
      }
      else
	model.continuous_variables(uv);
	//model.discrete_variables(duv);
    }
    else if ( ( model_view  == MERGED_DISTINCT_UNCERTAIN &&
		( vars_view == MERGED_DISTINCT_ALEATORY_UNCERTAIN ||
		  vars_view == MERGED_DISTINCT_EPISTEMIC_UNCERTAIN ) ) || 
	      ( model_view == MIXED_DISTINCT_UNCERTAIN &&
		( vars_view  == MIXED_DISTINCT_ALEATORY_UNCERTAIN ||
		  vars_view  == MIXED_DISTINCT_EPISTEMIC_UNCERTAIN ) ) ) {
      if (model.cv() != allVariables[i].cv()) {
	const RealDenseVector& acv = allVariables[i].all_continuous_variables();
	StringMultiArrayConstView acv_types
	  = allVariables[i].all_continuous_variable_types();
	size_t num_uv = model.cv(), num_dv
	  = std::count(acv_types.begin(), acv_types.end(), "continuous_design");
	if (model_view == MERGED_DISTINCT_UNCERTAIN)
	  num_dv +=
	    std::count(acv_types.begin(), acv_types.end(), "discrete_design");
	RealDenseVector uv(Teuchos::View, const_cast<Real*>(&acv[num_dv]),
			   num_uv);
	model.continuous_variables(uv);
	//model.discrete_variables(duv);
      }
      else
	model.continuous_variables(allVariables[i].continuous_variables());
	//model.discrete_variables(allVariables[i].discrete_variables());
    }
    else {
      Cerr << "Error: unsupported view mapping in Analyzer::"
	   << "evaluate_parameter_sets()" << endl;
      abort_handler(-1);
    }

    // compute the response
    if (asynchFlag)
      model.asynch_compute_response(activeSet);
    else {
      model.compute_response(activeSet);
      const Response& local_response = model.current_response();
      // log response data
      if (log_best_flag)
        update_best(allVariables[i], local_response, i);
      if (log_resp_flag)
        allResponses[i] = local_response.copy();
    }
  }

  // synchronize asynchronous evaluations
  if (asynchFlag) {
    const IntResponseMap& resp_map = model.synchronize();
    // log response data
    if (log_resp_flag)
      copy_data(resp_map, allResponses); // discards integer keys
    if (log_best_flag) {
      size_t num_resp = allVariables.length();
      IntRespMCIter r_cit = resp_map.begin();
      for (i=0; i<num_resp; ++i, ++r_cit)
        update_best(allVariables[i], r_cit->second, i);
    }
  }
}


/** Calculation of sensitivity indices obtained by variance based
    decomposition.  These indices are obtained by the Saltelli version
    of the Sobol VBD which uses (K+2)*N function evaluations, where K
    is the number of dimensions (uncertain vars) and N is the number
    of samples.  */
void Analyzer::var_based_decomp(const int ndim, const int num_samples)
{
  // run whatever sampling routine twice to generate two sets of 
  // input matrices, M1 and M2

  size_t i, j, k;
  RealDenseVector2DArray total_c_vars(ndim+2);
  for (i=0; i<ndim+2; i++)
    total_c_vars[i].reshape(num_samples);

  // get first sample block
  vary_pattern(true);
  get_parameter_sets(iteratedModel);
  for (j=0; j<num_samples; j++)
    copy_data(allVariables[j].continuous_variables(), total_c_vars[0][j]);

  // get second sample block
  get_parameter_sets(iteratedModel);
  for (j=0; j<num_samples; j++)
    copy_data(allVariables[j].continuous_variables(), total_c_vars[1][j]);

  for (i=2; i<ndim+2; i++)
    for (j=0; j<num_samples; j++)
      copy_data(allVariables[j].continuous_variables(), total_c_vars[i][j]);

  for (i=0; i<ndim; i++)
    for (j=0; j<num_samples; j++)    
      total_c_vars[i+2][j][i] = total_c_vars[0][j][i];

  // call evaluate parameter sets (ndim)*num_samples to get data
  RealVector2DArray total_fn_vals(numFunctions);
  for (k=0; k<numFunctions; k++)
    total_fn_vals[k].reshape(ndim+2);

  for (i=0; i<ndim+2; i++) {
    for (j=0; j<num_samples; j++)    
      allVariables[j].continuous_variables(total_c_vars[i][j]);
    // evaluate each of the parameter sets in allVariables
    evaluate_parameter_sets(iteratedModel, true, false);
    for (k=0; k<numFunctions; k++){
      total_fn_vals[k][i].reshape(num_samples);
      for (j=0; j<num_samples; j++)
	total_fn_vals[k][i][j] = allResponses[j].function_value(k);
    }
  }
  
  RealVectorArray S(numFunctions), T(numFunctions);
  for (k=0; k<numFunctions; k++){
    S[k].reshape(ndim);
    T[k].reshape(ndim);
  }

  // Loop over number of responses to obtain sensitivity indices for each
  for (k=0; k<numFunctions; k++){
 
#ifdef DEBUG
    Cout << "Total Samples\n"; 
    for (i=0; i<ndim+2; i++) 
      Cout << "Sample: " << i << '\n' << total_c_vars[i] << '\n';
    Cout << "Total Responses\n"; 
    for (i=0; i<ndim+2; i++) 
      Cout << "Response " << i << '\n' << total_fn_vals[i] << '\n';
#endif

    // calculate expected value of Y
    Real mean_hatY = 0., var_hatYS = 0., var_hatYT = 0.,
      mean_sq1  = 0., mean_sq2  = 0.;
    // mean estimate of Y (possibly average over matrices later)
    for (j=0; j<num_samples; j++)
      mean_hatY += total_fn_vals[k][0][j];
    mean_hatY /= (Real)num_samples;
    mean_sq1 = mean_hatY*mean_hatY;
    for (j=0; j<num_samples; j++)
      mean_sq2 += total_fn_vals[k][0][j]*total_fn_vals[k][1][j];
    mean_sq2 /= (Real)(num_samples);
    // variance estimate of Y for S indices
    for (j=0; j<num_samples; j++)
      var_hatYS += pow(total_fn_vals[k][0][j], 2.);
    var_hatYS = var_hatYS/(Real)(num_samples-1) - mean_sq2;
    // variance estimate of Y for T indices
    for (j=0; j<num_samples; j++)
      var_hatYT += pow(total_fn_vals[k][1][j], 2.);
    var_hatYT = var_hatYT/(Real)(num_samples-1) - mean_sq1;

#ifdef DEBUG
    Cout << "\nVariance of Yhat for S " << var_hatYS 
	 << "\nVariance of Yhat for T " << var_hatYT
	 << "\nMeanSq1 " << mean_sq1 << "\nMeanSq2 " << mean_sq2 << '\n';
#endif

  // calculate first order sensitivity indices and first order total indices

    for(i=0; i<ndim; i++) {
      Real sum_S = 0., sum_T = 0.;
      for (j=0; j<num_samples; j++) {
	sum_S += total_fn_vals[k][0][j]*total_fn_vals[k][i+2][j];
	sum_T += total_fn_vals[k][1][j]*total_fn_vals[k][i+2][j];
      }
      S[k][i] = (sum_S/(Real)(num_samples-1) - mean_sq2)/var_hatYS;     // ***
      T[k][i] = 1. - (sum_T/(Real)(num_samples-1) - mean_sq2)/var_hatYS;// TO DO: YT?
    }
  }
  print_vbd(Cout, S, T);
}


/** printing of variance based decomposition indices. */
void Analyzer::
print_vbd(ostream& s, const RealVectorArray& S, const RealVectorArray& T) const
{
  StringMultiArrayConstView cv_labels
    = iteratedModel.continuous_variable_labels();
  const StringArray& resp_labels = iteratedModel.response_labels();
  // output explanatory info
  s.setf(ios::scientific);
  s << "Variance Based Decomposition Sensitivity Indices\n"
    << "These indices measure the importance of the uncertain input\n"
    << "variables in determining the uncertainty (variance) of the output.\n"
    << "Si measures the main effect for variable i itself, while Ti\n"
    << "measures the total effect (including the interaction effects\n" 
    << "of variable i with other uncertain variables.)\n" << endl;
  s <<  setprecision(5);
  
  for (size_t k=0; k<numFunctions; k++){
    s << setw(12) << resp_labels[k] << ' ';
    s << '\n';	
    for(size_t i=0; i<numContinuousVars; i++) {
    s << setw(12) << cv_labels[i] << ":  Si = " << S[k][i] << "  Ti = " 
      << T[k][i]  << '\n';
    }
  }
}


/** Calculation of volumetric quality measures developed by FSU. */
void Analyzer::
volumetric_quality(int ndim, int num_samples, double* sample_points)
{
  qualityFlag = true;
  int num_trials = 100000;
  int seed_init = 1+rand();

#ifdef DAKOTA_FSUDACE
  chiMeas = chi_measure(ndim, num_samples, sample_points, num_trials,seed_init);
  dMeas   = d_measure(ndim, num_samples, sample_points, num_trials, seed_init);
  hMeas   = h_measure(ndim, num_samples, sample_points, num_trials, seed_init);
  tauMeas = tau_measure(ndim, num_samples, sample_points, num_trials,seed_init);
#endif
}


void Analyzer::
update_best(const Variables& vars, const Response& response,
	    const int eval_num)
{
  Cerr << "Error: Analyzer lacking redefinition of virtual update_best() "
       << "function.\n       This analyzer does not track best solutions."
       << endl;
  abort_handler(-1);
}


void Analyzer::vary_pattern(bool pattern_flag)
{
  Cerr << "Error: Analyzer lacking redefinition of virtual vary_pattern() "
       << "function.\n       This analyzer does not support pattern variance."
       << endl;
  abort_handler(-1);
}


void Analyzer::get_parameter_sets(const Model& model)
{
  Cerr << "Error: Analyzer lacking redefinition of virtual get_parameter_sets()"
       << " function.\n       This analyzer does not support parameter sets."
       << endl;
  abort_handler(-1);
}

} // namespace Dakota
