/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        AnalysisCode
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "system_defs.h"
#include "DakotaVariables.H"
#include "DakotaResponse.H"
#include "AnalysisCode.H"
#include "ProblemDescDB.H"
#include "ParallelLibrary.H"

static const char rcsId[]="@(#) $Id";


namespace Dakota {

AnalysisCode::AnalysisCode(const ProblemDescDB& problem_db):
  suppressOutputFlag(false), 
  // See base constructor in DakotaIterator.C for full discussion of output
  // verbosity.  Quiet is the default for AnalysisCode file operations.
  outputLevel(problem_db.get_short("method.output")),
  fileTagFlag(problem_db.get_bool("interface.application.file_tag")),
  fileSaveFlag(problem_db.get_bool("interface.application.file_save")),
  commandLineArgs(!problem_db.get_bool("interface.application.verbatim")),
  apreproFlag(problem_db.get_bool("interface.application.aprepro")),
  multipleParamsFiles(false),
  iFilterName(problem_db.get_string("interface.application.input_filter")),
  oFilterName(problem_db.get_string("interface.application.output_filter")),
  programNames(problem_db.get_dsa("interface.application.analysis_drivers")),
  numPrograms(programNames.length()),
  specifiedParamsFileName(
    problem_db.get_string("interface.application.parameters_file")),
  specifiedResultsFileName(
    problem_db.get_string("interface.application.results_file")),
  analysisComponents(
    problem_db.get_ds2a("interface.application.analysis_components")),
  parallelLib(problem_db.parallel_library())
{
  if (numPrograms > 1 && !analysisComponents.empty())
    multipleParamsFiles = true;
}


void AnalysisCode::define_filenames(const int id)
{
  // Define modified file names by handling Unix temp file and tagging options.
  // These names are used in writing the parameters file, in defining the 
  // Command Shell overloaded operator behavior in SysCallAnalysisCode.C, in 
  // reading the results file, and in file cleanup.

  // Different analysis servers _must_ share the same parameters and results
  // file root names.  If temporary file names are not used, then each evalComm
  // proc can define the same names without need for interproc communication.
  // However, if temporary file names are used, then evalCommRank 0 must define
  // the names and broadcast them to other evalComm procs.
  const ParallelConfiguration& pc = parallelLib.parallel_configuration();
  int eval_comm_rank   = parallelLib.ie_parallel_level_defined()
	? pc.ie_parallel_level().server_communicator_rank() : 0;
  int analysis_servers = parallelLib.ea_parallel_level_defined()
	? pc.ea_parallel_level().num_servers() : 1;
  bool bcast_flag = ( analysis_servers > 1 && ( specifiedParamsFileName.empty()
		   || specifiedResultsFileName.empty() ) ) ? true : false;

  if (eval_comm_rank == 0 || !bcast_flag) {
    char ctr_tag[16];
    if (fileTagFlag)
      sprintf(ctr_tag, ".%d", id);
    if (specifiedParamsFileName.empty()) { // no user spec -> use temp files
//#ifdef LINUX
      /*
      // mkstemp generates a unique filename using tmp_str _and_ creates the
      // file.  This prevents potential issues with race conditions, but has
      // the undesirable feature of requiring additional file deletions in the
      // case of secondary tagging for multiple analysis drivers.
      char tmp_str[20] = "/tmp/dakvars_XXXXXX";
      mkstemp(tmp_str); // replaces XXXXXX with unique id
      paramsFileName = tmp_str;
      */
//#else
      // tmpnam generates a unique filename that does not exist at the time of
      // invocation, but does not create the file.  It is more dangerous than
      // mkstemp, since multiple applications could potentially generate the
      // same temp filename prior to creating the file.
      paramsFileName = tmpnam(NULL);
//#endif
    }
    else {
      paramsFileName = specifiedParamsFileName;
      if (fileTagFlag)
	paramsFileName += ctr_tag;
    }
    if (specifiedResultsFileName.empty()) { // no user spec -> use temp files
//#ifdef LINUX
      /*
      // mkstemp generates a unique filename using tmp_str _and_ creates the
      // file.  This prevents potential issues with race conditions, but has
      // the undesirable feature of requiring additional file deletions in the
      // case of secondary tagging for multiple analysis drivers.
      char tmp_str[20] = "/tmp/dakresp_XXXXXX";
      mkstemp(tmp_str); // replaces XXXXXX with unique id
      resultsFileName = tmp_str;
      */
//#else
      // tmpnam generates a unique filename that does not exist at the time of
      // invocation, but does not create the file.  It is more dangerous than
      // mkstemp, since multiple applications could potentially generate the
      // same temp filename prior to creating the file.
      resultsFileName = tmpnam(NULL);
//#endif
    }
    else {
      resultsFileName = specifiedResultsFileName;
      if (fileTagFlag)
	resultsFileName += ctr_tag;
    }
  }

  if (bcast_flag) {
    // Note that evalComm and evalAnalysisIntraComm are equivalent in this case
    // since multiprocessor analyses are forbidden when using system calls/forks
    // (enforced by ApplicationInterface::init_communicators()).
    if (eval_comm_rank == 0) {
      // pack the buffer with root file names for the evaluation
      MPIPackBuffer send_buffer;
      send_buffer << paramsFileName << resultsFileName;
      // bcast buffer length so that other procs can allocate MPIUnpackBuffer
      //int buffer_len = send_buffer.len();
      //parallelLib.bcast_e(buffer_len);
      // broadcast actual buffer
      parallelLib.bcast_e(send_buffer);
    }
    else {
      // receive incoming buffer length and allocate space for MPIUnpackBuffer
      //int buffer_len;
      //parallelLib.bcast_e(buffer_len);
      // receive incoming buffer
      //MPIUnpackBuffer recv_buffer(buffer_len);
      MPIUnpackBuffer recv_buffer(256); // 256 should be plenty for 2 filenames
      parallelLib.bcast_e(recv_buffer);
      recv_buffer >> paramsFileName >> resultsFileName;
    }
  }
}


static void
cleanup_and_abort(const String &resname, bool keepfiles,
		  map<int, pair<String, String> > *fnames)
{
  Cerr << "\nError: cannot open results file " << resname << endl;
  if (!keepfiles) {
    map<int, pair<String, String> >::iterator It, Ite;
    It = fnames->begin();
    Ite = fnames->end();
    for(; It != Ite; ++It)
      remove((It->second).first);
  }
  abort_handler(-1);
}


void AnalysisCode::
write_parameters_files(const Variables& vars, const ActiveSet& set, 
		       const int id)
{
  pair<String, String> file_names(paramsFileName, resultsFileName);

  // If a new evaluation, insert the modified file names into map for use in
  // identifying the proper file names within read_results_files for the case
  // of asynch fn evaluations.  If a replacement eval (e.g., failure capturing 
  // with retry or continuation), then overwrite old file names with new ones.
  map<int, pair<String, String> >::iterator map_iter = fileNameMap.find(id);
  if (map_iter != fileNameMap.end()) {
    // remove old files.  This is only necessary for the tmp file case, since 
    // all other cases (including tagged) will overwrite the old files.  Since
    // we don't want to save tmp files, don't bother to check fileSaveFlag.
    remove((map_iter->second).first);  // parameters file name
    remove((map_iter->second).second); // results file name
    // replace file names in map, avoiding 2nd lookup
    map_iter->second = file_names;
  }
  else // insert new filenames.
    fileNameMap[id] = file_names;

  /* Uncomment for filename debugging
  Cout << "\nFile name entries at fn. eval. " << id << '\n';
  for (map_iter = fileNameMap.begin(); map_iter != fileNameMap.end();map_iter++)
    Cout << map_iter->first << " " << (map_iter->second).first << " " 
         << (map_iter->second).second << '\n';
  Cout << endl;
  */

  // Write paramsFileName without prog_num tag if there's an input filter or if
  // multiple sets of analysisComponents are not used.
  if (!multipleParamsFiles || !iFilterName.empty()) {
    StringArray all_an_comps;
    if (!analysisComponents.empty())
      copy_data(analysisComponents, all_an_comps);
    write_parameters_file(vars, set, all_an_comps, paramsFileName);
  }
  // If analysis components are used for multiple analysis drivers, then the 
  // parameters filename is tagged with program number, e.g. params.in.20.2
  // provides the parameters for the 2nd analysis for the 20th fn eval.
  if (multipleParamsFiles) { // append prog counter
    char prog_num[16];
    for (size_t i=0; i<numPrograms; i++) {
      sprintf(prog_num,".%d",i+1);
      String tag_params_fname = paramsFileName + prog_num;
      write_parameters_file(vars, set, analysisComponents[i], tag_params_fname);
    }
  }
}


void AnalysisCode::
write_parameters_file(const Variables& vars, const ActiveSet& set,
		      const StringArray& an_comps, const String& params_fname)
{
  // Write the parameters file
  ofstream parameter_stream(params_fname);
  if (!parameter_stream) {
    Cerr << "\nError: cannot create parameters file " << params_fname << endl;
    abort_handler(-1);
  }
  const ShortArray& asv = set.request_vector();
  const UIntArray&  dvv = set.derivative_vector();
  size_t asv_len = asv.length(), dvv_len = dvv.length(),
    ac_len = an_comps.length();
  StringArray asv_labels(asv_len), dvv_labels(dvv_len), ac_labels(ac_len);
  build_labels(asv_labels, "ASV_");
  build_labels(dvv_labels, "DVV_");
  build_labels(ac_labels,  "AC_");
  int prec = write_precision; // for restoration
  //write_precision = 16; // 17 total digits = full double precision
  write_precision = 15; // 16 total digits (17 would eliminate round off in last digit)
  int w = write_precision+7;
  if (apreproFlag) {
    String sp20("                    ");
    parameter_stream << sp20 << "{ DAKOTA_VARS     = " << setw(w) << vars.tv()
		     << " }\n";
    vars.write_aprepro(parameter_stream);
    parameter_stream << sp20 << "{ DAKOTA_FNS      = " << setw(w) << asv_len
		     << " }\n"; //<< setiosflags(ios::left);
    asv.write_aprepro(parameter_stream, asv_labels);
    parameter_stream << sp20 << "{ DAKOTA_DER_VARS = " << setw(w) << dvv_len
		     << " }\n";
    dvv.write_aprepro(parameter_stream, dvv_labels);
    parameter_stream << sp20 << "{ DAKOTA_AN_COMPS = " << setw(w) << ac_len
		     << " }\n";
    an_comps.write_aprepro(parameter_stream, ac_labels);
    //parameter_stream << resetiosflags(ios::adjustfield);
  }
  else {
    String sp21("                     ");
    parameter_stream << sp21 << setw(w) << vars.tv() << " variables\n" << vars
		     << sp21 << setw(w) << asv_len   << " functions\n";
		   //<< setiosflags(ios::left);
    asv.write(parameter_stream, asv_labels);
    parameter_stream << sp21 << setw(w) << dvv_len << " derivative_variables\n";
    dvv.write(parameter_stream, dvv_labels);
    parameter_stream << sp21 << setw(w) << ac_len  << " analysis_components\n";
    an_comps.write(parameter_stream, ac_labels);
    //parameter_stream << resetiosflags(ios::adjustfield);
  }
  write_precision = prec; // restore

  // Explicit flush and close added 3/96 to prevent Solaris problem of input 
  // filter reading file before the write was completed.
  parameter_stream.flush();
  parameter_stream.close();
}


void AnalysisCode::read_results_files(Response& response, const int id)
{
  // Retrieve parameters & results file names using fn. eval. id.  A map of
  // filenames is used because the names of tmp files must be available here 
  // and asynch_recv operations can perform output filtering out of order 
  // (which rules out making the file names into attributes of AnalysisCode or
  // rebuilding the file name from the root name plus the counter).
  map<int, pair<String, String> >::iterator map_iter = fileNameMap.find(id);
  const String& params_filename  = (map_iter->second).first;
  const String& results_filename = (map_iter->second).second;

  // If multiple analysis programs are used, then results_filename is tagged
  // with program number (to match ForkAnalysisCode::fork_application), e.g.
  // results.out.20.2 is the 2nd analysis results from the 20th fn eval.
  size_t i;
  char prog_num[16];
  StringArray results_filenames;
  if (numPrograms > 1) { // append program counter to results file
    results_filenames.reshape(numPrograms);
    for (i=0; i<numPrograms; i++) {
      sprintf(prog_num, ".%d", i+1);
      results_filenames[i] = results_filename + prog_num;
    }
  }

  // Read the results file(s).  If there's more than one program, then partial
  // responses must be overlaid to create the total response.  If an output
  // filter is used, then it has the responsibility to perform the overlay and
  // map results.out.[eval#].[1->numPrograms] to results.out.[eval#].  If no
  // output filter is used, then perform the overlay here.
  if (numPrograms > 1 && oFilterName.empty()) {
    response.reset();
    Response partial_response = response.copy();
    for (i=0; i<numPrograms; i++) {
      ifstream recovery_stream(results_filenames[i]);
      if (!recovery_stream)
	cleanup_and_abort(results_filenames[i], fileSaveFlag, &fileNameMap);
      recovery_stream >> partial_response;
      response.overlay(partial_response);
    }
  }
  else {
    ifstream recovery_stream(results_filename);
    if (!recovery_stream)
	cleanup_and_abort(results_filename, fileSaveFlag, &fileNameMap);
    recovery_stream >> response;
  }

  // Clean up files based on fileSaveFlag.
  if (!fileSaveFlag) {
    if (!suppressOutputFlag && outputLevel > NORMAL_OUTPUT) {
      Cout << "Removing " << params_filename;
      if (multipleParamsFiles) {
        if (!iFilterName.empty())
          Cout << " and " << params_filename;
        Cout << ".[1-" << numPrograms << ']';
      }
      Cout << " and " << results_filename;
      if (numPrograms > 1) {
        if (!oFilterName.empty())
          Cout << " and " << results_filename;
        Cout << ".[1-" << numPrograms << ']';
      }
      Cout << '\n';
    }
    if (!multipleParamsFiles || !iFilterName.empty())
      remove(params_filename);
    if (multipleParamsFiles) {
      for (i=0; i<numPrograms; i++) {
	sprintf(prog_num, ".%d", i+1);
	String tag_params_filename = params_filename + prog_num;
        remove(tag_params_filename);
      }
    }
    if (numPrograms == 1 || !oFilterName.empty())
      remove(results_filename);
    if (numPrograms > 1)
      for (i=0; i<numPrograms; i++)
        remove(results_filenames[i]);
  }

  // Prevent overwriting of files with reused names for which a file_save
  // request has been given.
  if (fileSaveFlag && !fileTagFlag && ( !specifiedParamsFileName.empty() ||
					!specifiedResultsFileName.empty() ) ) {
    // specifiedParamsFileName/specifiedResultsFileName are original user input
    if (!suppressOutputFlag && outputLevel > NORMAL_OUTPUT)
      Cout << "Files with nonunique names will be tagged for file_save:\n";
    char ctr_tag[16];
    sprintf(ctr_tag, ".%d", id);
    if (!specifiedParamsFileName.empty()) {
      String new_str = specifiedParamsFileName + ctr_tag;
      if (!multipleParamsFiles || !iFilterName.empty()) {
	if (!suppressOutputFlag && outputLevel > NORMAL_OUTPUT)
	  Cout << "Moving " << specifiedParamsFileName << " to " << new_str
	       << '\n';
	rename(specifiedParamsFileName, new_str);
      }
      if (multipleParamsFiles) { // append program counters to old/new strings
        for (i=0; i<numPrograms; i++) {
          sprintf(prog_num, ".%d", i+1);
          String old_tagged_str = specifiedParamsFileName + prog_num;
          String new_tagged_str = new_str + prog_num;
          if (!suppressOutputFlag && outputLevel > NORMAL_OUTPUT)
            Cout << "Moving " << old_tagged_str << " to " << new_tagged_str
                 << '\n';
          rename(old_tagged_str, new_tagged_str);
	}
      }
    }
    if (!specifiedResultsFileName.empty()) {
      String new_str = specifiedResultsFileName + ctr_tag;
      if (numPrograms == 1 || !oFilterName.empty()) {
        if (!suppressOutputFlag && outputLevel > NORMAL_OUTPUT)
          Cout << "Moving " << specifiedResultsFileName << " to " << new_str
	       << '\n';
        rename(specifiedResultsFileName, new_str);
      }
      if (numPrograms > 1) { // append program counters to old/new strings
        for (i=0; i<numPrograms; i++) {
          sprintf(prog_num, ".%d", i+1);
          String old_tagged_str = specifiedResultsFileName + prog_num;
          String new_tagged_str = new_str + prog_num;
          if (!suppressOutputFlag && outputLevel > NORMAL_OUTPUT)
            Cout << "Moving " << old_tagged_str << " to " << new_tagged_str
                 << '\n';
          rename(old_tagged_str, new_tagged_str);
	}
      }
    }
  }

  // Remove the evaluation which has been processed from the bookkeeping
  fileNameMap.erase(map_iter);
}

} // namespace Dakota
