// $Id: APPSPACK_Scheduler.hpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Scheduler.hpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Scheduler.hpp
  \brief Class definition for APPSPACK::Scheduler
*/
#ifndef APPSPACK_SCHEDULER_HPP
#define APPSPACK_SCHEDULER_HPP

#include "APPSPACK_Conveyor.hpp"
#include "APPSPACK_Citizen_Interface.hpp"
#include "APPSPACK_Evaluator_Interface.hpp"
#include "APPSPACK_Constraints_Linear.hpp"
#include "APPSPACK_Combiner_Generic.hpp"
#include "APPSPACK_Citizen_Generator.hpp"

namespace APPSPACK
{

/*! \brief This class schedules MPI jobs for both point evaluation and point generation 
  in parallel.  For for information \see \ref pageMediator */
class Scheduler
{
public:

  //! Scheduler.
  Scheduler(int argc, char* argv[]);

  //! Performs initialization procedures for masters and workers.
  void init();

  //! Perform any necessary clean up.
  void cleanup();

  //! Evaluate points while Mediator loops.
  void masterLoop(Citizen::Generator& generator);

  //! Evaluate points while Mediator loops.
  void workerLoop(Evaluator::Interface& evaluator);

  //! Generate points while Mediator loops.
  void citizenLoop(Citizen::Generator& generator);

  //! Returns true if master processors.
  bool isMaster() const;

  //! Returns true if dedicated citizen processor.
  bool isCitizen() const;

  //! Returns true of worker processor.
  bool isWorker() const;

  //! Returns processor rank.
  int getRank() const;

  //! Returns list of parameters.
  Parameter::List& getParameters();
private:

  //! Performs intialization procedures master.
  void initMaster();

  //! Performs intialization procedures for point gerating workers.
  void initCitizenWorker();

  //! Performs initializion procedures for point evaluation workers.
  void initEvalWorker();

  //! Determine number of evaluation and citizen workers and send.
  /*! Called only by master.  Master determines assignment vector,
    then sends this vector to the citizen and evaluation workers. */
  void sendAssignments();
  
  //! Recieves assignment vector from master.
  void recvAssignments();

  //! Determines name, type, and worker ranks of citizen i from params file.
  /*!
    \param i (input) denotes the ith citizen as in "Citizen i"
    \param type (output) type of citizen
    \param name (output) name is "Citizen i (type)"
    \param workerRank (output) denotes ranks of processors owned by Citizen i.
  */
  void getCitizenInfo(int i, string& type, string& name,
                      vector<int>& workerRank);

  //! Converts an integer into a string.
  string IntToString(int n) const;

  //! Throws an error if check is true, and displays msg to stderr.
  /*! If masterOnly is false, message only displayed by master. */
  void exitIf( bool check, string msg, bool masterOnly=false) const;

private:
  
  //! Rank of scheduler.
  const int rank;

  //! Number of processors.
  const int nprocs;

  //! Name of parameter's file.
  string paramsFile;

  //! Stores parameter for given task.
  Parameter::List params;

  //! Number of citizens.  Must be greater than 0.
  int nCitizens;

  //! Number of worrkers.  Must be greater than 0.
  int nWorkers;
  
  //! Assignment vector.  
  vector<string> assign;

};

}

#endif
