//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Direct_Wrappers.hpp
*/

#ifndef APPSPACK_DIRECT_WRAPPERS_HPP
#define APPSPACK_DIRECT_WRAPPERS_HPP

#include "APPSPACK_Common.hpp"
#include "APPSPACK_Vector.hpp"
#include "APPSPACK_Matrix.hpp"

#ifdef __cplusplus
extern "C" {
#endif

  typedef int (*Fpr)(int *n, double c[], double l[], double u[],
                      int point[], int* maxI, int* start, int* maxfunc, 
                      double fvec[],
		      int iidata[], int *iisize, double ddata[], int* idsize, 
		      char cdata[], int* icsize);
  
  /*C+-----------------------------------------------------------------------+
    C|    SUBROUTINE Direct                                                  |
    C| On entry                                                              |
    C|     fep--  Exchanges unevaluated points for evaluated points.         |
    C|       n -- The dimension of the problem.                              |
    C|     eps -- Exceeding value. If eps > 0, we use the same epsilon for   |
    C|            all iterations. If eps < 0, we use the update formula from |
    C|            Jones:                                                     |
    C|               eps = max(1.D-4*abs(fmin),epsfix),                      |
    C|            where epsfix = abs(eps), the absolute value of eps which is|
    C|            passed to the function.                                    |
    C|    maxf -- The maximum number of function evaluations.                |
    C|    maxT -- The maximum number of iterations.                          |
    C|            Direct stops when either the maximum number of iterations  |
    C|            is reached or more than maxf function-evalutions were made.|
    C|       l -- The lower bounds of the hyperbox.                          |
    C|       u -- The upper bounds of the hyperbox.                          |
    C|algmethod-- Choose the method, that is either use the original method  |
    C|            as described by Jones et.al. (0) or use our modification(1)|
    C| logfile -- File-Handle for the logfile. DIRECT expects this file to be|
    C|            opened and closed by the user outside of DIRECT. We moved  |
    C|            this to the outside so the user can add extra informations |
    C|            to this file before and after the call to DIRECT.          |
    C| fglobal -- Function value of the global optimum. If this value is not |
    C|            known (that is, we solve a real problem, not a testproblem)|
    C|            set this value to -1.D100 and fglper (see below) to 0.D0.  |
    C|  fglper -- Terminate the optimization when the percent error          |
    C|                100(f_min - fglobal)/max(1,abs(fglobal)) < fglper.     |
    C|  volper -- Terminate the optimization when the volume of the          |
    C|            hyperrectangle S with f(c(S)) = fmin is less then volper   |
    C|            percent of the volume of the original hyperrectangle.      |
    C|sigmaper -- Terminate the optimization when the measure of the         |
    C|            hyperrectangle S with f(c(S)) = fmin is less then sigmaper.|
    C|                                                                       |
    C| User data that is passed through without being changed:               |
    C|  iidata -- Integer Array of user data. This array of size iisize is   |
    C|            passed to the function to be optimized and can be used to  |
    C|            transfer data to this function. The contents are not       |
    C|            changed by DIRECT.                                         |
    C|  iisize -- Size of array iidata.                                      |
    C|   ddata -- DOUBLE PRECISION array of user data. See iidata.           |
    C|  idsize -- Size of array ddata.                                       |
    C|   cdata -- Character array. See iidata.                               |
    C|  icsize -- Size of array ddata.                                       |
    C|                                                                       |
    C| On return                                                             |
    C|                                                                       |
    C|       x -- The final point obtained in the optimization process.      |
    C|            X should be a good approximation to the global minimum     |
    C|            for the function within the hyper-box.                     |
    C|                                                                       |
    C|    fmin -- The value of the function at x.                            |
    C|  Ierror -- Error flag. If Ierror is lower 0, an error has occured. The|
    C|            values of Ierror mean                                      |
    C|            Fatal errors :                                             |
    C|             -1   u(i) <= l(i) for some i.                             |
    C|             -2   maxf is too large.                                   |
    C|             -3   Initialization in DIRpreprc failed.                  |
    C|             -4   Error in DIRSamplepoints, that is there was an error |
    C|                  in the creation of the sample points.                |
    C|             -5   Error in DIRSamplef, that is an error occured while  |
    C|                  the function was sampled.                            |
    C|             -6   Error in DIRDoubleInsert, that is an error occured   |
    C|                  DIRECT tried to add all hyperrectangles with the same|
    C|                  size and function value at the center. Either        |
    C|                  increase maxdiv or use our modification (Jones = 1). |
    C|            Termination values :                                       |
    C|              1   Number of function evaluations done is larger then   |
    C|                  maxf.                                                |
    C|              2   Number of iterations is equal to maxT.               |
    C|              3   The best function value found is within fglper of    |
    C|                  the (known) global optimum, that is                  |
    C|                   100(fmin - fglobal/max(1,|fglobal|))  < fglper.     |
    C|                  Note that this termination signal only occurs when   |
    C|                  the global optimal value is known, that is, a test   |
    C|                  function is optimized.                               |
    C|              4   The volume of the hyperrectangle with fmin at its    |
    C|                  center is less than volper percent of the volume of  |
    C|                  the original hyperrectangle.                         |
    C|              5   The measure of the hyperrectangle with fmin at its   |
    C|                  center is less than sigmaper.                        |
    C|                                                                       |
    C| SUBROUTINEs used :                                                    |
    C|                                                                       |
    C| DIRheader, DIRInitSpecific, DIRInitList, DIRpreprc, DIRInit, DIRChoose|
    C| DIRDoubleInsert, DIRGet_I, DIRSamplepoints, DIRSamplef, DIRDivide     |
    C| DIRInsertList, DIRreplaceInf, DIRWritehistbox, DIRsummary, Findareas  |
    C|                                                                       |
    C| Functions used :                                                      |
    C|                                                                       |
    C| DIRgetMaxdeep, DIRgetlevel                                            |
    C+-----------------------------------------------------------------------+
  */
  void direct_(Fpr fep, double x[], int* n, double* eps, int* maxf, 
               int* maxT, double* fmin, double l[], double u[],
               int* algmethod, int* Ierror, int* logfile, 
               double* fglobal, double* fglper, double* volper, double* sigmaper,
               int iidata[], int* iisize, double ddata[], int* idsize, 
               char cdata[], int* icsize, int cdata_len);

  void f77open_(int* iunit, char* name, int* state, int name_len);

  void f77close_(int* iunit);
  
#ifdef __cplusplus
}
#endif

void callDIRECT(const APPSPACK::Vector& xlow,
		const APPSPACK::Vector& xupp,
                string logfile,
		vector<int>& idata, vector<double>& ddata,
		vector<char>& cdata);

//! Returns false if direct should stop.
bool exchangePoints(const vector<APPSPACK::Vector>& T, vector<APPSPACK::Vector>& F);

#endif
