// $Id: APPSPACK_Citizen_Listener.hpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_Listener.hpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_Listener.hpp
  \brief Class definition for APPSPACK::Citizen_Listener
*/

#ifndef APPSPACK_CITIZEN_LISTENER_HPP
#define APPSPACK_CITIZEN_LISTENER_HPP

#include "APPSPACK_ConveyorList.hpp"
#include "APPSPACK_Citizen_Interface.hpp"
#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Constraints_Linear.hpp"
#include "APPSPACK_List.hpp"
#include <time.h>

namespace APPSPACK
{

//! Namespace for algorithms that generate points to be evaluated by Conveyor.  
namespace Citizen
{
  
//! Calls listens and record points.  
/* This class will also send a APPSPACK::Citizen::MustStop signal if
   "Percent Error" is defined in parameter list and "Known Global Minimum"
   is also defined and exists.
*/
class Listener : public Interface
{
public:
  //! Constructor.
  Listener(Parameter::List& params, const Constraints::Linear& constraints,
           Combiner::Generic& combiner, const string name);

  //! Destructor.
  ~Listener();

  //! Called once by mediator prior to processing points for any citizen.
  virtual void preprocess();
  
  //! Called once by mediator when point processing by all citizens complete.
  virtual void postprocess();
  
  //! Take in list of evaluated appspacks. Give list of new points.
  /*!
    \param R - Points that have been evaluated since last call.
    \param Wnew - Citizen may add points it wishes to be evaluated.
    \param ownermap - Map of Citizen names to point tags signifying which 
    points this Citizen currently owns in W, R, and that are currently being evaluated by
    the Conveyor.
    
    \note Citizen will have a chance to see all points generated when
    vote() is called.
  */
  virtual void exchange(const ConveyorList& R, ConveyorList& Wnew, 
                        const map<string, vector<int> >& ownermap);
  
  //! Vote on points to be evaluated.
  /*! 
    \param W (input) a list of points currently in the queue.  
    \param ownermap (input) Provides a list of which tags are owned by which citizen.
    So ownermap[\e name] returns a list of tags of points currently owned by
    Citizen \e name.
    \param tagVote (output) A map from tags in W to the number of votes this
    Citizen wishes to assign that tag.  Citizen's may vote on all points in W.  votes may
    be any nonnegative integer.  

    So, for example, if W contains a point with tag \f$ t_k\f$, then the user may set
    tagVote[\f$t_k\f$] = \f$ v_k\f$, where \f$v_k\f$ denotes the number of votes.
    The higher the number of votes a point receives, the higher its priority is
    in the resulting evaluation queue.

    \warning An error will be thrown if a Citizen attempt to vote on a point not in W.

    \note If a point in W receives 0 votes by all Citizens, the point is pruned.
  */
  virtual void vote(const ConveyorList& W, 
		    const map<string, vector<int> >& ownermap, 
		    vector<int>& tagOrder);
  
  //! Returns the Citizen's current state.
  virtual Citizen::State getState();
  
  //! Returns Citizens name.
  virtual const string& getName() const;

  //! Must be defined by Citizen if one or more processors requested.
  /*! Returns true if worker wishes to continue. */
  virtual bool worker();

private:

  //! Compute percent error 
  /*!
    Compute the relative error according to the formula in Gablonsky and Kelley, 2001.
   */
  double getPercentError(double fMin) const;

  //! Name of this Citizen.
  string name;

  //! Used to Combine value to determine best point so far.
  Combiner::Generic& combiner;

  //! Parameters file.
  Parameter::List& params;

  //! Output stream to store results.
  ofstream resultsStream;

  //! Output file name.
  string resultsFile;

  //! Best x point found so far.
  Vector xBest;

  //! Objective value of best x, f(x).
  Vector fBest;

  //! Stores evalCount when current best was found.
  int evalCountBest;

  //! Name of Citizen who found current best point.
  string nameEvalBest;
  
  //! Tag of current best point.
  int tagBest;

  //! Time of evaluation complete for current best point.
  double diffTimeBest;

  //! Evaluation count.
  int evalCount;

  //! Initial start time;
  time_t startTime;

  //! Stores global minimum value.
  double globalMin;

  //! Stores percent error desired.  
  double percentErrorTol;

  //! Determines if Listener will be sending a MustStop signal.
  bool canStop;

  //! Output precision.
  int precision;

};

}
}

//! Outputs stream state.
ostream& operator<<(ostream& stream, APPSPACK::Citizen::State state);

#endif
