// $Id: APPSPACK_Citizen_Interface.hpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_Interface.hpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_Interface.hpp
  \brief Class definition for APPSPACK::Citizen_Interface
*/

#ifndef APPSPACK_CITIZEN_INTERFACE_HPP
#define APPSPACK_CITIZEN_INTERFACE_HPP

#include "APPSPACK_ConveyorList.hpp"
#include <map>

namespace APPSPACK
{

//! Namespace for algorithms that generate points to be evaluated by Conveyor.  
namespace Citizen
{
//! State of Citizen
enum State {

  MustContinue,         //!< Citizen force everyone continue, exchange called for everyone.
  PassiveContinue,      //!< Citizen wishes to continue.
  StopRequest,          //!< Citizen is ready to stop, but still wishes exchange to be called.
  MustStop,             //!< Citizen demands everyone stop.
  Retired               //!< Citizen is ready to stop, and does not want exchange called.
};
  
 //! Abstract interface for Citizen point generators.
 /*! Citizens communicate with the Mediator exchanging unevaluated
points for evaluated ones.  All Citizens are competing for CPU time
and may vote on the priority of the points they wish evaluated.
Points that recieve a higher number of votes are placed higher in the
evaluation queue sent to the Conveyor.  Points that do not recieve any
votes from any Citizen are automatically deleted from the queue
without being evaluated.

For further details, see \ref pageMediator   
 */
class Interface
{
public:

  //! Called once by mediator prior to processing points for any citizen.
  virtual void preprocess() = 0;
  
  //! Called once by mediator when point processing by all citizens complete.
  virtual void postprocess() = 0;
  
  //! Take in list of evaluated appspacks. Give list of new points.
  /*!
    \param R - Points that have been evaluated since last call.
    \param Wnew - Citizen may add points it wishes to be evaluated.
    \param ownermap - Map of Citizen names to point tags signifying which 
    points this Citizen currently owns in W, R, and that are currently being evaluated by
    the Conveyor.
    
    \note Citizen will have a chance to see all points generated when
    vote() is called.
  */
  virtual void exchange(const ConveyorList& R, ConveyorList& Wnew, 
                        const map<string, vector<int> >& ownermap) = 0;
  
  //! Vote on points to be evaluated.
  /*! 
    \param W (input) a list of points currently in the queue.  
    \param ownermap (input) Provides a list of which tags are owned by which citizen.
    So ownermap[\e name] returns a list of tags of points currently owned by
    Citizen \e name.
    \param tagVote (output) A map from tags in W to the number of votes this
    Citizen wishes to assign that tag.  Citizen's may vote on all points in W.  votes may
    be any nonnegative integer.  

    So, for example, if W contains a point with tag \f$ t_k\f$, then the user may set
    tagVote[\f$t_k\f$] = \f$ v_k\f$, where \f$v_k\f$ denotes the number of votes.
    The higher the number of votes a point receives, the higher its priority is
    in the resulting evaluation queue.

    \warning An error will be thrown if a Citizen attempt to vote on a point not in W.

    \note If a point in W receives 0 votes by all Citizens, the point is pruned.
  */
  virtual void vote(const ConveyorList& W, 
		    const map<string, vector<int> >& ownermap, 
		    vector<int>& tagOrder) = 0;
  
  //! Returns the Citizen's current state.
  virtual Citizen::State getState() = 0;
  
  //! Returns Citizens name.
  virtual const string& getName() const = 0;

  //! Must be defined by Citizen if one or more processors requested.
  /*! Returns true if worker wishes to continue. */
  virtual bool worker();
private:
  
private:
  
};

}
}

//! Outputs stream state.
ostream& operator<<(ostream& stream, APPSPACK::Citizen::State state);

#endif
