//@HEADER
// ************************************************************************
// 
//          APPSPACK: Asynchronous Parallel Pattern Search
//                 Copyright (2003) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_DIRECTserial.hpp
  \brief Class definition for APPSPACK::Citizen_DIRECTserial
*/

#ifndef APPSPACK_CITIZEN_DIRECTSERIAL_HPP
#define APPSPACK_CITIZEN_DIRECTSERIAL_HPP

#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Citizen_Interface.hpp"
#include "APPSPACK_Constraints_Linear.hpp"

namespace APPSPACK
{

namespace Citizen
{

//! A class DIRECT serial class.
class DIRECTserial : public Interface
{
public:

  //! Constructor 
  DIRECTserial(Parameter::List& params_in,
	       Constraints::Linear& linear,
	       Combiner::Generic& combiner_in,	       
	       const string name = "");
  
  //! Destructor
  ~DIRECTserial();

  //! Prints initial state of citizen.
  virtual void preprocess();

  //! Prints final state of citizen.
  virtual void postprocess();
  
  //! Take in list of evaluated appspacks. Give list of new points.
  virtual void exchange(const ConveyorList& R, ConveyorList& Wnew,
                        const map<string, vector<int> >& ownermap);

  //! Vote on points to be evaluated.
  virtual void vote(const ConveyorList& W, 
		    const map<string, vector<int> >& ownermap,
		    vector<int>& tagOrder);

  //! Return Citizen's state.
  virtual Citizen::State getState();

  //! Returns Citizen's name.
  virtual const string& getName() const;

private:
  
  //! Creates command strings and file names.
  void createStrings();

  //! Writes W and R to an input file.
  void writeInputFile(const ConveyorList& R, const vector<int>& citizen_tags);
  
  //! Run executable on new points.
  void runProgram();
  
  //! Reads output file
  void readOutputFile(ConveyorList& Wnew);
  
  /*! \brief Parse a line of the form 
    x = [x1 x2 ... xn ] vote = [ int ] step = [ double ]
    where step is optional. \todo finish comments.
  */
  Point* processNewX(const string& line);

  //! Deletes input/output files when finished.
  void deleteFiles();  
  
  //!  Returns contents of name = [ string ].
  string getContent(const string& line, const string& name) const;
							    
  //! Convert a string to a int.
  int str2int(const string& item) const;

  //! Convert a string to a double.
  double str2dbl(const string& item) const;

  //! Convert a string to vector.
  void str2vec(const string& item, Vector& d) const;

private:
  
  //! Parameters
  Parameter::List& params;

  //! Combiner reference needed when generating points
  Combiner::Generic& combiner;

  //! Static counter used to generate unique tags
  static int staticCount;

  //! Constraints.
  Constraints::Linear& constraints;

  //! Name of citizen.
  string name;

  //! Prefix for input file
  string inputPrefix;

  //! File name for input file
  string inputFileName;

  //! Input file created.
  bool isInputFile;

  //! Prefix for output file
  string outputPrefix;

  //! File name for output file
  string outputFileName;

  //! Output file created.
  bool isOutputFile;

  //! Name of executable that generates points.
  string executableName;

  //! They system command used.
  string execString;
  
  //! File precision.
  int precision;

  //! Command tag.
  string fileTag;

  //! Stores tag votes from last call to exchange.
  vector<int> tagOrder;

  //! Default step size.
  const double defaultStep;

  //! Default number of votes given to a generated point.
  const int defaultVotes;
};

}
}
#endif
