// $Id: APPSPACK_Citizen_DIRECT.hpp,v 1.3 2008/05/03 16:22:31 jgriffi Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_DIRECT.hpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_DIRECT.hpp
  \brief Class definition for APPSPACK::Citizen_DIRECT
*/

#ifndef APPSPACK_CITIZEN_DIRECT_HPP
#define APPSPACK_CITIZEN_DIRECT_HPP

#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Citizen_Interface.hpp"
#include "APPSPACK_Constraints_Linear.hpp"
#include "APPSPACK_DIRECT_Wrappers.hpp"

namespace APPSPACK
{

namespace Citizen
{

//! A class that exchanges points using a local system call.
class DIRECT : public Interface
{
public:

  //! Constructor 
  DIRECT(Parameter::List& params_in,
      const Constraints::Linear& constraints_in,
      Combiner::Generic& combiner_in, int rank,
      const string name = "");
  
  //! Destructor
  ~DIRECT();

  //! Prints initial state of citizen.
  virtual void preprocess();

  //! Prints final state of citizen.
  virtual void postprocess();
  
  //! Take in list of evaluated appspacks. Give list of new points.
  virtual void exchange(const ConveyorList& R, ConveyorList& Wnew,
                        const map<string, vector<int> >& ownermap);

  //! Vote on points to be evaluated.
  virtual void vote(const ConveyorList& W, 
		    const map<string, vector<int> >& ownermap,
		    vector<int>& tagOrder);

  //! Return Citizen's state.
  virtual Citizen::State getState();

  //! Returns Citizen's name.
  virtual const string& getName() const;
  
  //! Call's tgp oracle.
  virtual bool worker();
private:

  //! Converts an int to a string.
  string IntToString(int n) const;

  //! Converts a string of form line = "1 2 3" to a vector x =<1 2 3>.
  void StringToVector(const string& line, Vector& x) const;

private:

  //! Name of citizen.
  const string name;

  //! Parameters
  Parameter::List& params;
  
  //! Combiner reference needed when generating points
  Combiner::Generic& combiner;

  //! Constraints
  const Constraints::Linear& constraints;

  //! Static counter used to generate unique tags
  static int staticCount;

  //! Static counter for evaluation to prevent DIRECT's from simultaneously running.
  static int staticEvalCount;

  //! Rank of this.
  const int workerRank;

  //! Records tags of points we currently need evaluated.
  vector<int> currentTags;

  //! Default step to use for GSS
  double defaultStep;

  //! List of evaluated points to be returned to DIRECT.
  vector<Vector> X;

  //! List of values corresponding to X.
  vector<Vector> F;

  //! List of tags corresponding to X.
  vector<int> Xtags;

  //! Signifies the worker is waiting for evaluations to complete.
  bool workerWaiting;

  //! Signals that DIRECT has finished.
  APPSPACK::Citizen::State state;

  //! Maximum function evaluations.
  int maxEvaluations;

  //! Maximum number of iterations.
  int maxIterations;

  //! Epsilon used by DIRECT
  double directEpsilon;

  //! Global percentage desired.  Set in mediator block.
  double globalPercentage;

  //! Desired or known global minimum.  Set in evaluator block.
  double valueOfGlobal;

  //! Specifies whether to use Jones method or Gablonsky's.
  bool useJones;

  //! Log file for DIRECT
  string logFile;
};

}
}
#endif
