// $Id: APPSPACK_Citizen_DIRECT.cpp,v 1.4 2008/05/03 16:22:31 jgriffi Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_DIRECT.cpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_DIRECT.cpp
  \brief Implements APPSPACK::Citizen_DIRECT

  \todo Josh Comment parameters.
*/

#include "APPSPACK_Citizen_DIRECT.hpp"
#include "APPSPACK_Combiner_Generic.hpp"
#include "APPSPACK_Executor_Oracle.hpp"
#include "APPSPACK_Print.hpp"
#include "APPSPACK_DIRECT_Wrappers.hpp"

int APPSPACK::Citizen::DIRECT::staticCount = 0;
int APPSPACK::Citizen::DIRECT::staticEvalCount = 0;

APPSPACK::Citizen::DIRECT::DIRECT(Parameter::List& params_in,
                            const Constraints::Linear& constraints_in,
                            Combiner::Generic& combiner_in, int rank_in,
                            const string name_in) :
  params(params_in.sublist(name_in)),
  name(name_in+"(DIRECT)"),
  combiner(combiner_in),
  workerRank(rank_in),
  constraints(constraints_in),
  defaultStep(params.getParameter("Default Step", 1.0)),
  workerWaiting(false),
  state(APPSPACK::Citizen::PassiveContinue),
  maxEvaluations(params.getParameter("Max Evaluations", 500)),
  maxIterations(params.getParameter("Max Iterations", 10)),
  directEpsilon(params.getParameter("Epsilon", 1e-3)),
  useJones(params.getParameter("Use Jones", true)),
  globalPercentage(params_in.sublist("Mediator").getParameter("Percent Error", 0.0)),
  valueOfGlobal(params_in.sublist("Evaluator").getParameter("Known Global Minimum", -1e20)),
  logFile(params.getParameter("Log File", "log.txt"))
{
}

APPSPACK::Citizen::DIRECT::~DIRECT()
{
}

void APPSPACK::Citizen::DIRECT::preprocess()
{ 
}

void APPSPACK::Citizen::DIRECT::postprocess()
{

}

void APPSPACK::Citizen::DIRECT::exchange(const APPSPACK::ConveyorList& R,
					 APPSPACK::ConveyorList& Wnew, 
                                         const map<string, vector<int> >& ownermap)
{
  if (state != APPSPACK::Citizen::PassiveContinue)
  {
    cerr << "Why are you still calling me?" << endl;
    return;
  }

  if (GCI::probe(APPSPACK::Executor::Oracle::Generate))
  {
    int msgtag, rank;
    string exit_status;

    APPSPACK::GCI::recv(APPSPACK::Executor::Oracle::Generate, workerRank);
    APPSPACK::GCI::bufinfo(msgtag, rank);
    APPSPACK::GCI::unpack(exit_status);

    if (exit_status == "DIRECT Finished")
    {
      cout << "DIRECT is retiring!!! " << endl;
      state = APPSPACK::Citizen::Retired;
      return;
    }
    
    vector<Vector> P;    
    APPSPACK::GCI::unpack(P);
    
    Vector f;
    string msg="(" + name + ")";
    for (int i=0; i<P.size(); i++)
    {
      Point* p = new Point(P[i], f, defaultStep, 0.0, combiner, msg);    
      Wnew.push(p);
      currentTags.push_back(p->getTag());
    }

    workerWaiting=true;
  }
    
  list<APPSPACK::Point*>::const_iterator point_itr;
  for (point_itr=R.begin(); point_itr!=R.end(); point_itr++)
  {
    int tag = (*point_itr)->getTag();
    vector<int>::iterator tag_itr;
    tag_itr = find(currentTags.begin(), currentTags.end(), tag);
    if (tag_itr != currentTags.end())
    {
      X.push_back((*point_itr)->getX());
      F.push_back((*point_itr)->getVecF());
      Xtags.push_back((*point_itr)->getTag());
      currentTags.erase(tag_itr);
    }
  }
  
  if (X.size() != F.size())
  {
    cerr << "Number of points must match number of function values";
    throw "APPSPACK Error";
  }

  if ((currentTags.size() == 0) && (workerWaiting == true))
  {
    workerWaiting=false;

    // Now tell DIRECT we are finished writing evaluations.
    string status="Evals Complete";
    GCI::initSend();
    GCI::pack(status);
    GCI::pack(F);
    GCI::pack(X);
    GCI::pack(Xtags);
    GCI::send(APPSPACK::Executor::Oracle::Generate, workerRank);  

    X.clear();
    F.clear();
    Xtags.clear();
  }
}

void APPSPACK::Citizen::DIRECT::vote(const ConveyorList& W, 
				     const map<string, vector<int> >& ownermap,
				     vector<int>& tagOrder)
{
  tagOrder.clear();

  // Get list of tags in the queue W.
  vector<int> wtags;
  W.getTagList(wtags);

  // Get list of tags that this owns.
  const vector<int>& direct_tags = ownermap.find(name)->second;

  // Give all points that we own a vote equal to GSS.
  for (int i=0; i<direct_tags.size(); i++)
  {
    int tag = direct_tags[i];
    if (find(wtags.begin(), wtags.end(), tag) != wtags.end())
      tagOrder.push_back(tag);
  }
}

APPSPACK::Citizen::State APPSPACK::Citizen::DIRECT::getState()
{
  return state;
}

const string& APPSPACK::Citizen::DIRECT::getName() const
{
  return name;
}

bool APPSPACK::Citizen::DIRECT::worker()
{  
  vector<int> iidata;
  vector<double> ddata;
  vector<char> cdata;

  // DIRECT will exit if iidata[0] != 0;
  iidata.push_back(1);
  iidata.push_back(workerRank);
  iidata.push_back(maxEvaluations);
  iidata.push_back(maxIterations);
  if (useJones)
    iidata.push_back(0);
  else
    iidata.push_back(1);

  ddata.push_back(directEpsilon);
  ddata.push_back(globalPercentage);
  ddata.push_back(valueOfGlobal);

  callDIRECT(constraints.getLower(), constraints.getUpper(), 
	     logFile, iidata, ddata, cdata);

  // Delete the file that DIRECT creates.
  unlink(logFile.c_str());

  return false;
}

string APPSPACK::Citizen::DIRECT::IntToString(int n) const
{
  ostringstream ss;
  ss << n;
  return ss.str();
}

void APPSPACK::Citizen::DIRECT::StringToVector(const string& item, 
					    APPSPACK::Vector& x) const
{
  x.resize(0);
  istringstream ss(item);
  
  double xi;
  while (ss >> xi)
    x.push_back(xi);
}

