//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*! 
  \file APPSPACK_Combiner_Smooth.cpp
  \brief Implementation of APPSPACK::Combiner::Smooth
*/

#include <math.h> // for fmax with Sun's stlport4
#include "APPSPACK_Common.hpp"
#include "APPSPACK_Combiner_Smooth.hpp"
#include "APPSPACK_Float.hpp"
#include <cassert>

APPSPACK::Combiner::Smooth::Smooth(const string& methodName_in) : 
  nlic(-1), nlec(-1), rho(dne()), methodName(methodName_in),
  debug(0)
{
}

APPSPACK::Combiner::Smooth::~Smooth()
{
}

double APPSPACK::Combiner::Smooth::operator()(const APPSPACK::Vector& fc) const
{
  if (fc.size() == 0)
    return dne();

  // Check constraints
  for (int i = 0; i < fc.size(); i ++)
    if (!exists(fc[i]))
      return dne();
      
  double phik;
  if (methodName == "Merit Max")
  {
    phik = getPenaltyMax(fc);
  }
  else if (methodName == "Merit Max Smooth")
  {
    phik = getPenaltyMaxSmooth(fc);
  }    
  else if (methodName == "Merit One")
  {
    phik = getPenaltyOne(fc);
  }
  else if (methodName == "Merit One Smooth")
  {
    phik = getPenaltyOneSmooth(fc);
  }
  else if (methodName == "Merit Two")
  {
    phik = getPenaltyTwo(fc);
  }
  else if (methodName == "Merit Two Smooth")
  {
    phik = getPenaltyTwoSmooth(fc);
  }
  else if (methodName == "Merit Two Squared")
  {
    phik = getPenaltyTwoSquared(fc);
  }
  else
  {
    cerr << "APPSPACK::Combiner::Smooth -- penatly method not set" << endl;
    throw "APPSPACK Error";
  }

  if ( debug > 4)
    cout << methodName << endl;

  // Return objective value
  return phik;
}

void  APPSPACK::Combiner::Smooth::setDebug(int debug_in)
{
  debug = debug_in;
}

void  APPSPACK::Combiner::Smooth::setConstraintSize(int nlec_in, int nlic_in)
{
  nlec = nlec_in;
  nlic = nlic_in;
}

void  APPSPACK::Combiner::Smooth::setRho(double rho_in)
{
  rho = rho_in;
}

double  APPSPACK::Combiner::Smooth::getRho() const
{
  return rho;
}

void  APPSPACK::Combiner::Smooth::setAlpha(double alpha_in)
{
  alpha = alpha_in;
}

double  APPSPACK::Combiner::Smooth::getAlpha() const
{
  return alpha;
}

double APPSPACK::Combiner::Smooth::infnorm(const Vector& fc) const
{
  assert( (fc.size() == (nlic+nlec+1)) && "fc has incorrect size!");

  // Equality constraints (recall fc[0] = f(x)):
  // fc[1], ... , fc[nlec] 
  // Inequality constraints. (note that we want fc[i] <= 0):
  // fc[nlec+1], ... , fc[fc.size() - 1]

  double norm=0;
  for (int i=1; i<fc.size(); i++)
  {
    if (i <= nlec)
      norm = max(fabs(fc[i]), norm);
    else
      norm = max(fc[i], norm);
  }
  
  return norm;
}

void APPSPACK::Combiner::Smooth::getInfVector(const Vector& fc,
                                              Vector& nu) const
{
  assert( (fc.size() == (nlic+nlec+1)) && "fc has incorrect size!");

  // Equality constraints (recall fc[0] = f(x)):
  // fc[1], ... , fc[nlec] 
  // Inequality constraints. (note that we want fc[i] <= 0):
  // fc[nlec+1], ... , fc[fc.size() - 1]

  nu.resize(0);
  for (int i=1; i<fc.size(); i++)
  {
    if (i <= nlec)
      nu.push_back(fabs(fc[i]));
    else
      nu.push_back(fmax(fc[i], 0.0));
  }
}

double APPSPACK::Combiner::Smooth::getErrorTol(const APPSPACK::Vector& fc) const
{
  double errtol = 0;
  if (methodName == "Merit Max Smooth")
  {
    errtol = 2*(fc.size() - 1)*alpha;
  }    
  else if (methodName == "Merit One Smooth")
  {
    errtol = 2*(fc.size() - 1)*alpha;
  }
  else if (methodName == "Merit Two Smooth")
  {
    errtol = 2*(fc.size() - 1)*alpha;
  }

  // Return objective value
  return errtol;
}

// PRIVATE
double APPSPACK::Combiner::Smooth::getPenaltyMax(const APPSPACK::Vector& fc) const
{
  double phik = 0;

  // Handle equality constraints.  
  for (int i = 1; i <= nlec; i++)
  {
    double ci = fabs(fc[i]);
    phik = max(phik, ci);
  }

  // Handle inequality constraints.
  for (int i = nlec+1; i < fc.size(); i++)
  {
    double ci = fmax(fc[i], 0.0);
    phik = max(phik, ci);
  }

  phik = fc[0] + rho*phik;

  return phik;
}

// PRIVATE
double APPSPACK::Combiner::Smooth::getPenaltyMaxSmooth(const APPSPACK::Vector& fc) const
{
  double shift = infnorm(fc)*rho/alpha;    
  double phik = exp(-shift);

  // Handle equality constraints.  
  for (int i = 1; i <= nlec; i++)
  {
    double ci = fc[i];
    phik += exp(ci*rho/alpha - shift);
    phik += exp(-ci*rho/alpha - shift);
  }
  
  // Handle inequality constraints.
  for (int i = nlec+1; i < fc.size(); i++)
  {
    double ci = fc[i];
    phik += exp(ci*rho/alpha - shift);
  }
  
  phik = fc[0] + alpha*(shift+log(phik));

  return phik;
}

// PRIVATE
double APPSPACK::Combiner::Smooth::getPenaltyOne(const APPSPACK::Vector& fc) const
{
  double phik = 0;

  // Handle equality constraints.  
  for (int i = 1; i <= nlec; i++)
  {
    double ci = fabs(fc[i]);
    phik += ci;
  }
  
  // Handle inequality constraints.
  for (int i = nlec+1; i < fc.size(); i++)
  {
    double ci = fmax(fc[i], 0.0);
    phik += ci;
  }

  phik = fc[0] + rho*phik;

  return phik;
}

// PRIVATE
double APPSPACK::Combiner::Smooth::getPenaltyOneSmooth(const APPSPACK::Vector& fc) const
{
  double phik = 0;

  // Handle equality constraints.  
  for (int i = 1; i <= nlec; i++)
  {
    double t = rho*fc[i]/alpha;
    double s = fmax(0.0,t);
    phik += alpha*(s + log(exp(-s) + exp(t - s)));  
    s = fmax(0.0,-t);
    phik += alpha*(s + log(exp(-s) + exp(-t - s)));
  }
  
  // Handle inequality constraints.
  for (int i = nlec+1; i < fc.size(); i++)
  {
    double t = rho*fc[i]/alpha;
    double s = fmax(0.0, t);
    phik += alpha*(s + log(exp(-s) + exp(t - s)));
  }

  phik = fc[0] + phik;
    
  return phik;
}

// PRIVATE
double APPSPACK::Combiner::Smooth::getPenaltyTwo(const APPSPACK::Vector& fc) const
{
  double phik = 0;
  
  // Handle equality constraints.  
  for (int i = 1; i <= nlec; i++)
  {
    double ci = fc[i];
    phik += ci*ci;
  }
  
  // Handle inequality constraints.
  for (int i = nlec+1; i < fc.size(); i++)
  {
    double ci = fmax(fc[i], 0.0);
    phik += ci*ci;
  }
  
  phik = fc[0] + rho*sqrt(phik);
  
  return phik;
}

// PRIVATE
double APPSPACK::Combiner::Smooth::getPenaltyTwoSmooth(const APPSPACK::Vector& fc) const
{
  double phik = 0;
  
  // Handle equality constraints.  
  for (int i = 1; i <= nlec; i++)
  {
    double ci = fc[i];
    phik += ci*ci;
  }
  
  // Handle inequality constraints.
  for (int i = nlec+1; i < fc.size(); i++)
  {
    double ci = fmax(fc[i], 0.0);
    phik += ci*ci;
  }
  
  phik = fc[0] + rho*sqrt(phik + (alpha*alpha)/(rho*rho));
  
  return phik;
}

// PRIVATE
double APPSPACK::Combiner::Smooth::getPenaltyTwoSquared(const APPSPACK::Vector& fc) const
{
  double phik = 0;
  
  // Handle equality constraints.  
  for (int i = 1; i <= nlec; i++)
  {
    double ci = fc[i];
    phik += ci*ci;
  }
  
  // Handle inequality constraints.
  for (int i = nlec+1; i < fc.size(); i++)
  {
    double ci = fmax(fc[i], 0.0);
    phik += ci*ci;
  }
  
  phik = fc[0] + rho*phik;
  
  return phik;
}

