// $Id: APPSPACK_ConveyorList.cpp,v 1.3 2008/05/02 00:57:22 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src-conveyor/APPSPACK_ConveyorList.cpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
x  \file APPSPACK_ConveyorList.cpp
  \brief Implementation of APPSPACK::ConveyorList
*/

#include <algorithm> // for find
#include "APPSPACK_ConveyorList.hpp"
#include "APPSPACK_Print.hpp"
#include "APPSPACK_Cache_Point.hpp"

APPSPACK::ConveyorList::ConveyorList()
{

}

APPSPACK::ConveyorList::ConveyorList(const APPSPACK::ConveyorList& source, 
				     const vector<int>& taglist)
{
  copyFrom(source, taglist);
}

APPSPACK::ConveyorList::~ConveyorList()
{
  prune();
}

bool APPSPACK::ConveyorList::isEmpty() const
{
  return tpl.empty();
}

bool APPSPACK::ConveyorList::isNotEmpty() const
{
  return !(tpl.empty());
}

list<APPSPACK::Point*>::const_iterator APPSPACK::ConveyorList::begin() const
{
  return tpl.begin();
}

list<APPSPACK::Point*>::const_iterator APPSPACK::ConveyorList::end() const
{
  return tpl.end();
}


list<APPSPACK::Point*>::const_reverse_iterator APPSPACK::ConveyorList::rbegin() const
{
  return tpl.rbegin();
}

list<APPSPACK::Point*>::const_reverse_iterator APPSPACK::ConveyorList::rend() const
{
  return tpl.rend();
}

int APPSPACK::ConveyorList::size() const
{
  return tpl.size();
}

void APPSPACK::ConveyorList::copyFrom(const APPSPACK::ConveyorList& source, 
				      const vector<int>& taglist)
{
  TPL::const_reverse_iterator it;
  for (it=source.tpl.rbegin(); it != source.tpl.rend(); it++)
  {
    // If the points tag is in tag list we create as normal.
    int tag = (*it)->getTag();
    if (find(taglist.begin(), taglist.end(), tag) != taglist.end())
      push(new Point(**it));
    else // Otherwise we give a parent tag of -1.
      push(new Point(**it, -1));    
  }
}


void APPSPACK::ConveyorList::getTagList(vector<int>& t) const
{
  t.clear();
  TPL::const_iterator it = tpl.begin();
  for (; it != tpl.end(); it ++)
  {
    int tag = (*it)->getTag();
    t.insert(t.begin(), tag);
  }
}

void APPSPACK::ConveyorList::prune(int n)
{
  if (n <= 0)
  {
    for (TPL::iterator it = tpl.begin(); it != tpl.end(); it ++)
      delete *it;
    
    tpl.clear();
  }
  else
  {
    int p = size() - n;
    Point* popped;
    for (int i = 0; i < p; i ++)
    {
      popped = pop();
      delete popped;
    }
  }
}

APPSPACK::Point* APPSPACK::ConveyorList::pop()
{
  if (tpl.empty())
    return NULL;

  Point* popped = tpl.back();
  tpl.pop_back();
  return popped;
}

APPSPACK::Point* APPSPACK::ConveyorList::pop(int tag)
{
  if (tpl.empty())
    return NULL;

  TPL::iterator it = tpl.begin();
  for (; it != tpl.end(); it ++)
    if (((*it)->getTag()) == tag)
      break;

  if (it == tpl.end())
    return NULL;

  Point* popped = *it;
  tpl.erase(it);

  return popped;
}

void APPSPACK::ConveyorList::push(Point* pushed)
{
  tpl.push_front(pushed);
}


void APPSPACK::ConveyorList::prependList(ConveyorList& source)
{
  tpl.insert(tpl.end(), source.tpl.begin(), source.tpl.end());
  source.tpl.clear();
}

void APPSPACK::ConveyorList::appendList(ConveyorList& source)
{
  tpl.insert(tpl.begin(), source.tpl.begin(), source.tpl.end());
  source.tpl.clear();
}


void APPSPACK::ConveyorList::print(const string label) const
{
  cout << label << ":\n";
  
  if (tpl.empty())
  {
    cout << "<empty>" << endl;
    return;
  }

  for (TPL::const_iterator it = tpl.begin(); it != tpl.end(); it ++)
    cout << *(*it) << endl;
}

bool APPSPACK::ConveyorList::contains(const APPSPACK::Point* w, int& tag) const
{  
  tag = -1;
  if (tpl.empty())
    return false;
  
  const APPSPACK::Vector& wx = w->getX();
  TPL::const_iterator tpi = tpl.begin();
  for (; tpi != tpl.end(); tpi ++)
  {
    if ( pointsEqual(w, (*tpi)) )
    {
      tag = (*tpi)->getTag();
      return true;
    }
  }
  
  return false;
}

bool APPSPACK::ConveyorList::pointsEqual(const APPSPACK::Point* w, 
					 const APPSPACK::Point* z) const
{
  const Vector& s = Cache::Point::scaling;
  double tol = Cache::Point::tolerance;

  APPSPACK::Vector wx=w->getX();
  APPSPACK::Vector zx=z->getX();
  
  if (wx.size() != zx.size())
  {
    cerr << "APPSPACK::ConveyorList::pointsEqual: different sized points" << endl;
    throw "APPSPACK Error";
  }
  
  for (int i=0; i<wx.size(); i++)
  {
    if (fabs(wx[i] - zx[i]) > s[i]*tol)
      return false;
  }

  return true;
}
