// $Id: APPSPACK_Conveyor.cpp,v 1.2 2008/05/02 00:57:22 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src-conveyor/APPSPACK_Conveyor.cpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Conveyor.cpp
  \brief Implements APPSPACK::Conveyor
*/

#include "APPSPACK_Conveyor.hpp"

APPSPACK::Conveyor::Conveyor(Parameter::List& params, 
			     const Vector& scaling_in,
			     Executor::Interface& executor_in) :
  executor(executor_in),
  cache(params, scaling_in),
  cacheRef(cache),
  doSync(params.getParameter("Synchronous", false)),
  minReturn(params.getParameter("Minimum Exchange Return", 1)),
  maxReturn(params.getParameter("Maximum Exchange Return", max(minReturn, 1000) ))
{
}

APPSPACK::Conveyor::Conveyor(Parameter::List& params, 
			     const Vector& scaling_in,
			     Executor::Interface& executor_in,
			     Cache::Manager& cache_in) :
  executor(executor_in),
  cacheRef(cache_in),
  doSync(params.getParameter("Synchronous", false)),
  minReturn(params.getParameter("Minimum Exchange Return", 1)),
  maxReturn(params.getParameter("Maximum Exchange Return", max(minReturn, 1000) ))
{
}

APPSPACK::Conveyor::~Conveyor()
{
  pendingList.prune();
  multimap<int, Point*>::iterator it;
  for (it=pendingPrime.begin(); it!=pendingPrime.end(); it++)
    delete (it->second);
}

const APPSPACK::Counter& APPSPACK::Conveyor::getCounter() const
{
  return counter;
}

int APPSPACK::Conveyor::getNumPending() const
{
  return pendingList.size();
}

void APPSPACK::Conveyor::exchange(APPSPACK::ConveyorList& queueList, 
				  APPSPACK::ConveyorList& evalList)
{
  Point* ptr;
  int tag;
  bool isf;
  Vector f;
  string msg;

  //cout << "Cache tolerance:" <<  APPSPACK::Cache::Point::tolerance << endl;

  while (((queueList.size() > 0) || (pendingList.size() > 0)) 
	 && ((doSync) || (evalList.size() < minReturn)))
  {
    // Try to submit as many items as possible to the executor
    while ((queueList.isNotEmpty()) && (executor.isWaiting()))
    {
      int pendingTag;
      ptr = queueList.pop();
      // Is point cached?
      if ( cacheRef.isCached( ptr->getX(), f) )
      {
	counter.incrementCached();
	ptr->setCachedFunctionValue(f, counter.getCountString());
	evalList.push(ptr);
      }
      // Is point currently being evaluated?
      else if (pendingList.contains(ptr, pendingTag))
      {
	pendingPrime.insert(make_pair(pendingTag, ptr));
      }
      // Point is neither in cache nor pending.  Submit new point.
      else
      {
 	executor.spawn( ptr->getX(), ptr->getTag() );
	pendingList.push(ptr);
      }
    }

    // Process items coming out of the executor
    int id;
    multimap<int, Point*>::iterator it;
    while (((id = executor.recv(tag,f,msg)) != 0) && 
	   ((doSync) || (evalList.size() < maxReturn)))
    {
      // Process received point.
      counter.incrementEvaluated(id, msg);
      ptr = pendingList.pop(tag);

      if (ptr == NULL)
      {
	cout << "Warning. Ignoring results from tag " << tag << endl;
      }
      else
      {
	ptr->setEvaluatedFunctionValue(f, counter.getCountString());
	cacheRef.insert(ptr->getX(),f);
	evalList.push(ptr);
	
	// Assign all points in pendingPrime, corresponding to tag
	// the value of the returned point *ptr.
	for (it = pendingPrime.lower_bound(tag);
	     it != pendingPrime.upper_bound(tag); it++)
	  {
	    counter.incrementPendingCached();
	    it->second->setCachedFunctionValue(f, counter.getCountString());
	    evalList.push(it->second);
	  }
	// evalList now owns these points, delete ownership from pendingPrime.
	pendingPrime.erase(pendingPrime.lower_bound(tag), pendingPrime.upper_bound(tag));
      }
    }
  }
}

void APPSPACK::Conveyor::print() const
{
  executor.print();
}
