// $Id: APPSPACK_Solver.hpp,v 1.2 2008/05/02 00:57:22 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src-appspack/APPSPACK_Solver.hpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Solver.hpp
  \brief Class definition for APPSPACK::Solver
*/

#ifndef APPSPACK_SOLVER_HPP
#define APPSPACK_SOLVER_HPP

#include "APPSPACK_Point.hpp"
#include "APPSPACK_Conveyor.hpp"
#include "APPSPACK_Directions.hpp"
#include "APPSPACK_List.hpp"
#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Print.hpp"
#include "APPSPACK_Constraints_Linear.hpp"
#include "APPSPACK_Combiner_Generic.hpp"
#include "APPSPACK_Iterator.hpp"

namespace APPSPACK
{

//! The solver itself.
/*!  <b>Parameters</b>

  These parameters are are stored in the Parameter::List that is
  passed to the Solver constructor. See \ref pageParameters for full details on
  these parameters.

  <ul>
  <li>"Function Tolerance"
  <li>"Maximum Evaluations"
  <li>"Bounds Tolerance"
  <li>"Snap To Boundary"
  <li>"Solution File"
  <li>"Use Random Order"
  </ul>

  The class Solver serves as a wrapper and a driver for the core class Iterator
  that can perform a single iteration of the generating set search algorithm
  implemented within APPSPACK.  The class Iterator uses reverse communication
  and hence Solver contains as members both #queueList and #exchangeList
  that store the points #gss_iterator request be evaluated and the points
  that have been evaluated respectively.  Evaluations of points within
  the queueList are performed through the member #conveyor whose purpose
  is to evaluate points either in serial or parallel depending upon the
  Executor chosen.
*/
class Solver
{
  
public:
  
  //! State of the solver
  enum State {
    Continue,			//!< Default state
    StepConverged,		//!< Step length converged
    FunctionConverged,		//!< Function tolerance converged
    EvaluationsExhausted        //!< Number of function evaluations exhausted
  };
  
  //! Constructor 
  Solver(const Parameter::List& params_in, 
	 Executor::Interface& executor_in, 
	 const Constraints::Linear& constraints_in,
	 Combiner::Generic& combiner_in,
	 Cache::Manager& cache_in);
  
  //! Constructor 
  Solver(const Parameter::List& params_in, 
	 Executor::Interface& executor_in, 
	 const Constraints::Linear& constraints_in);

  //! Destructor 
  ~Solver();


  // --------------------------------------------------------------------------------

  //@{ \name Accessors (for after solve() is called)
  
  //! Return the x-vector corresponding to the best point
  const vector<double>& getBestX() const;

  //! Return the finite function value (if any) associated with the best point
  double getBestF() const;

  //! Returns instance of class Value associated with current best point.
  const vector<double>& getBestVecF() const;
  
  //! Returns total number of points evaluated.
  int getNumEvaluations() const;

  //! Returns total number of cached evaluations.
  int getNumCached() const;

  //@}

  // --------------------------------------------------------------------------------

  //@{ \name Manipulators

  //! Find the minimum of the function that was set up in the constructor
  APPSPACK::Solver::State solve();
  //@}

private:

  //@{ \name Print Functions

  //! Prints initial information including parameter values.
  void printInitializationInformation() const;

  //! Prints final state information once solve completes.
  void printFinalInformation() const;

  //@}

  //! Prune the queue to the size specified by #maxQueue.
  void prune();
  
  //! Assign solver state based upon state of #gss_iterator.
  Solver::State getIteratorState();

private:
  //! The state of the solver.
  Solver::State state;

  //! Parameters
  Parameter::List params;
  
  //! Trial Point Evaluator
  Conveyor conveyor; 
  
  //! Performs a single APPSPACK iteration using reverse communication.
  Iterator gss_iterator;

  //! Trial points that are queued to be processed
  List queueList;

  //! Trial points that have been evaluated evaluated
  List exchangeList;

  //! How big the queue is allowed to be after a prune (set according to "Max Queue Size" parameter)
  int maxQueue;

  //! Enforce function evaluation budget
  bool isMaxEvaluations;

  //! Function evaluation budget
  int maxEvaluations;
};

}

//! Printing an APPSPACK::Solver::State
ostream& operator<<(ostream& stream, APPSPACK::Solver::State state);

#endif
