// $Id: APPSPACK_Iterator.cpp,v 1.3 2008/05/03 16:13:25 jgriffi Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src-appspack/APPSPACK_Iterator.cpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Iterator.cpp
  \brief Implementation of APPSPACK::Iterator
*/

#include "APPSPACK_Iterator.hpp"
#include "APPSPACK_Print.hpp"
#include "APPSPACK_Constraints_Linear.hpp"
#include "APPSPACK_Float.hpp"
#include "APPSPACK_LAPACK_Wrappers.hpp"
#include <algorithm>

APPSPACK::Iterator::Iterator(Parameter::List& params_in,
			     const Constraints::Linear& constraints_in,
			     Combiner::Generic& combiner_in) :
  constraints(constraints_in),
  params(params_in),
  print(params),		// modifies params and sets globals
  bestPointPtr(initializeBestPointPtr(params, constraints, combiner_in)), // modifies params
  directions(params, constraints), // modifies params
  exchangeList(),
  state(Continue),
  functionTolerance(dne()),
  useSnapTo(params.getParameter("Snap To Boundary",true))
{
  setup(constraints);
}

APPSPACK::Iterator::Iterator(Parameter::List& params_in,
			     const Constraints::Linear& constraints_in) :
  constraints(constraints_in),
  params(params_in),
  print(params),		// modifies params and sets globals
  bestPointPtr(initializeBestPointPtr(params, constraints, combiner)), // modifies params
  directions(params, constraints), // modifies params
  exchangeList(),
  state(Continue),
  functionTolerance(dne()),
  useSnapTo(params.getParameter("Snap To Boundary",true))
{
  setup(constraints);
}

void APPSPACK::Iterator::setup(const Constraints::Linear& constraints_in)
{
  useRandomOrder = params.getParameter("Use Random Order", false);

  // Stopping Criteria
  if (params.isParameterDouble("Function Tolerance"))
    functionTolerance = params.getDoubleParameter("Function Tolerance");

  boundsTolerance = params.getParameter("Bounds Tolerance", params.getDoubleParameter("Step Tolerance") / 2.0);
  if (boundsTolerance <= 0)
  {
    cout << "APPSPACK::Iterator::Iterator - Invalid non-positive value for \"Bounds Tolerance\"." << endl;
    throw "APPSPACK Error";
  }
  
  processNewBestPoint();

  // Push a copy of unevaluated initial points onto initialList.
  if (getBestVecF().size() == 0)
    initialList.push(new Point(*bestPointPtr,-1));
}

APPSPACK::Iterator::~Iterator()
{
  delete bestPointPtr;
}

const vector<double>& APPSPACK::Iterator::getBestX() const
{
  return bestPointPtr->getX().getStlVector();
}

double APPSPACK::Iterator::getBestF() const
{
  return bestPointPtr->getF(); 
}

const vector<double>& APPSPACK::Iterator::getBestVecF() const
{
  return bestPointPtr->getVecF().getStlVector(); 
}

int APPSPACK::Iterator::getBestTag() const
{
  return bestPointPtr->getTag(); 
}

APPSPACK::Point* APPSPACK::Iterator::initializeBestPointPtr(Parameter::List& params, 
							    const Constraints::Linear& constraints,
							    Combiner::Generic& combiner)
{
  // Set up the initial point
  Vector nominalX;
  if (!params.isParameter("Initial X"))
    constraints.getNominalX(nominalX);
  
  Vector initialX = params.getParameter("Initial X", nominalX);
  
  // Check that the initial point is the right size
  if (initialX.size() != constraints.getScaling().size())
  {
    cerr << "Error: Size mismatch\n";
    cerr << "The size of the initial X is not the same as the lower bounds.\n";
    throw "APPSPACK Error";
  }  
  
  bool useInitialF=true;
  // Try making feasible wrt to simple bounds.
  if (!constraints.isFeasible(initialX))
  {
    constraints.makeBoundsFeasible(initialX);
    useInitialF=false;
  }
  
#ifdef HAVE_LAPACK
  double epsilonSnap = params.getParameter("Bounds Tolerance", params.getDoubleParameter("Step Tolerance") / 2.0);
  if (epsilonSnap <= 0)
  {
    cout << "APPSPACK::Iterator::Iterator - Invalid non-positive value for \"Bounds Tolerance\"." << endl;
    throw "APPSPACK Error";
  }
  
  // Try snapping to
  if (!constraints.isFeasible(initialX))
  { 
    constraints.snapToBoundary(initialX,epsilonSnap);
    useInitialF=false;
  }
#endif

  // Check that x is feasible
  constraints.assertFeasible(initialX);

  // Initial step
  double initialStep = params.getParameter("Initial Step", 1.0);
  
  // Point
  double alpha = params.getParameter("Sufficient Decrease Factor", 0.01);
  
  // Initial F
  Vector initialF;
  if (useInitialF == true)
  {
    if (params.isParameterDouble("Initial F"))
    {
      initialF.assign(1,params.getDoubleParameter("Initial F"));
      params.setParameter("Initial F",initialF); // reset to be a vector!
    }
    else if (params.isParameterVector("Initial F"))
    {
      initialF = params.getVectorParameter("Initial F");
    }
  }
  
  return new Point(initialX, initialF, initialStep, alpha, combiner,
                   "(Initial Point)");
}

bool APPSPACK::Iterator::pointExchange(APPSPACK::List& newList)
{
  bool doPrune = false;
  exchangeList.insertList(newList);
  if ( !(exchangeList.isEmpty()) )
    doPrune = processEvaluatedTrialPoints();
 
  generateTrialPoints();

  newList.insertList(exchangeList);

  return doPrune;
}

APPSPACK::Iterator::State APPSPACK::Iterator::getState() const
{
  return state;
}

// PRIVATE
void APPSPACK::Iterator::processNewBestPoint(Point* newBestPointPtr)
{
  // Update the best point
  if (newBestPointPtr != NULL)
  {
    delete bestPointPtr;
    bestPointPtr = newBestPointPtr;
  }
  
  // Check for convergence based on the function value
  if ((exists(functionTolerance)) && (bestPointPtr->getF() < functionTolerance))
  {
    state = FunctionConverged;
    return;
  }

  // Update the (scaled) search directions
  directions.computeNewDirections(*bestPointPtr);
}

// PRIVATE
void APPSPACK::Iterator::generateTrialPoints()
{
  // Add initial points if provided.  Happens only on first iteration.
  if ( !(initialList.isEmpty()) )
    exchangeList.insertList(initialList);  
  
  // Local references
  const Vector& parentX = bestPointPtr->getX();
  
  Vector& x = tmpVector;
  int n = parentX.size();
  x.resize(n);

  vector<int> indices;
  directions.getDirectionIndices(indices);

  if (useRandomOrder == true)
  {
    // Randomize access order.    
    random_shuffle(indices.begin(), indices.end());
  }

  for (int i = 0; i < indices.size(); i ++)
  {
    int idx = indices[i];
    const Vector& direction = directions.getDirection(idx);
    const double step = directions.getStep(idx);
    double maxStep = constraints.maxStep(parentX, direction, step);
  
    // Only add nontrivial points to the exchange list.
    if (maxStep <= 0)
    {
      // Cannot move along this direction and remain feasible.
      directions.setStepConverged(idx);
    }
    else
    {
      for (int j = 0; j < n; j++)
	x[j] = parentX[j] + maxStep*direction[j];
      
#ifdef HAVE_LAPACK
      if (useSnapTo)
      {
	Vector xsnap = x;
	constraints.snapToBoundary(xsnap,boundsTolerance);
	if (constraints.isFeasible(xsnap))
	  x = xsnap;
	//! \todo Josh: put in maxStep search here.
      }
#endif
      
      constraints.assertFeasible(x);
      
      // Create a new trial point
      Point* newPointPtr = new Point(x, step, *bestPointPtr, idx);
      
      // Save off trial point information
      directions.setTrueStepAndTag(idx, maxStep, newPointPtr->getTag());
      
      // Push this trial point onto the new trial point list.
      // Ownership of the pointer transfers to the list.
      exchangeList.push(newPointPtr);
    }
  }
}

bool APPSPACK::Iterator::processEvaluatedTrialPoints()
{
  bool pruneConveyor=false;
  
  // Check for a new best point
  if (exchangeList.best() < *bestPointPtr)      
  {
    processNewBestPoint(exchangeList.popBest());
    exchangeList.prune();
    pruneConveyor = true;
  }
  else 
  {
    // Otherwise, just process the list
    Point* ptr;
    bool stepReduced = false;
    while (!exchangeList.isEmpty())
    {
      ptr = exchangeList.pop();
      if (ptr->getParentTag() == bestPointPtr->getTag())
      {
	directions.reduceStep(ptr->getIndex());
	stepReduced = true;
      }
      delete ptr;
    }
    // Check for step length convergence.  If not converged, append new directions to list.
    if (directions.isStepConverged())
      state = StepConverged;
    else if (stepReduced)
      directions.appendNewDirections();
  }

  return pruneConveyor;
}

void APPSPACK::Iterator::printInitializationInformation() const
{
  // Paramters
  cout << "*** Parameter List ***" << "\n";
  params.print();

  // Constraints
  cout << "\n" << "*** Constraints ***" << "\n";
  constraints.print();
}

void APPSPACK::Iterator::printBestPoint(const string label) const
{
  cout << "\n" << label << ": " << *bestPointPtr << endl;
}

void APPSPACK::Iterator::printDirections(const string label) const
{
  directions.print(label);
}

void APPSPACK::Iterator::writeSolutionFile(const string filename) const
{
  // Determine output precision.
  int precision = params.getParameter("Solution File Precision", 14);
    
  // Open file for append
  ofstream fout;
  fout.open(filename.c_str(), std::ios::out);
  if (!fout)
  {
    cerr << "APPSPACK::Iterator::writeSolutionFile() - Unable to open solution file" << endl;
    return;
  }

  // Write best value to outname using cache file output style.
  fout << "f=[ ";
  (bestPointPtr->getVecF()).leftshift(fout,precision);
  fout << " ]";
  
  fout << " x=[ ";
  (bestPointPtr->getX()).leftshift(fout,precision);
  fout << " ]" << endl;
}

ostream& operator<<(ostream& stream, APPSPACK::Iterator::State state) 
{
  switch (state)
  {
  case APPSPACK::Iterator::Continue:
    stream << "Continue";
    break;
  case APPSPACK::Iterator::StepConverged:
    stream << "Step Converged";
    break;
  case APPSPACK::Iterator::FunctionConverged:
    stream << "Function Converged";
    break;
  }

  return stream;
}
