/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptProblemFrag_RealParams.h
 *
 * Defines the colin::OptProblemFragment_RealParams class.
 */

#ifndef colin_OptProblemFrag_RealParams_h
#define colin_OptProblemFrag_RealParams_h

#include <acro_config.h>
#include <colin/OptProblemFrag_Bounds.h>

namespace colin {

//============================================================================
//============================================================================
// Class OptProblemFragment_RealParams
//============================================================================
//============================================================================

/**
 *  Defines the elements of an OptProblem that pertain to real parameters.
 */
template <class DomainT, class ResponseT>
class OptProblemFragment_RealParams : virtual public OptProblemFragment_Bounds<DomainT,ResponseT>
{
public:

  /// Constructor
  OptProblemFragment_RealParams() {}

  /// Virtual destructor
  virtual ~OptProblemFragment_RealParams() {}

  /// Returns the number of real parameters in the domain
  unsigned int num_real_params()
		{ return (this->state ? this->state->num_real_params : 0);}

  /// Initializes the number of real params in the domain
  void init_real_params(unsigned int num)
		{
		if (!(this->state))
		   EXCEPTION_MNGR(std::runtime_error, "OptProblemFragment_RealParams::init_real_params - no application!");
		if (this->state->num_real_params != num) {
		   this->state->num_real_params = num;
                   if(this->state->int_lower_bounds.size()==0)
  		     this->state->enforcing_bound_constraints = false;
		   }
		}

  /// Returns true if the problem supports the calculation of gradients
  bool& using_gradients()
		{
		if (!(this->state))
		   EXCEPTION_MNGR(std::runtime_error, "OptProblemFragment_RealParams::using_gradients - no application!");
		return this->state->using_gradients;
		}

  /// Returns true if the problem supports the calculation of Hessians
  bool& using_hessians()
		{
		if (!(this->state))
		   EXCEPTION_MNGR(std::runtime_error, "OptProblemFragment_RealParams::using_hessians - no application!");
		return this->state->using_hessians;
		}

  /// Compute the gradient of the point
  void EvalG(const DomainT& point, typename ResponseT::vector_t& value)
		{
		this->handle->set_point(point);
		this->handle->EvalG(value);
		}

  /// Asynchronously compute the gradient of the point
  void AsyncEvalG(const DomainT& point, typename ResponseT::vector_t* value, int& priority, int tag=-1)
		{
		this->handle->set_point(point);
		this->handle->AsyncEvalG(value,priority,tag);
		}

  /// Asynchronously compute the gradient of the point with a default priority
  void AsyncEvalG(const DomainT& point, typename ResponseT::vector_t* value, int tag=-1)
		{
		AsyncEvalG(point,value,this->default_priority,tag);
		}

  /// Compute the function value and gradient of the point
  void EvalFG(const DomainT& point, real& fvalue, typename ResponseT::vector_t& value)
		{
		this->handle->set_point(point);
		this->handle->EvalFG(value);
		}

  /// Asynchronously compute the function value and gradient of the point
  void AsyncEvalFG(const DomainT& point, real* fvalue, typename ResponseT::vector_t* value,
				int& priority, int tag=-1)
		{
		this->handle->set_point(point);
		this->handle->AsyncEvalFG(value,priority,tag);
		}

  /// Asynchronously compute the function value and gradient of the point
  ///   with a default priority
  void AsyncEvalFG(const DomainT& point, real* fvalue, typename ResponseT::vector_t* value,
				int tag=-1)
		{
		AsyncEvalFG(point,fvalue,value,this->default_priority,tag);
		}

  /// Evaluate the Jacobian (constraint gradients) at the given point
  void EvalCG(const DomainT& point, typename ResponseT::matrix_t& value)
		{
		if (this->state->numConstraints == 0)
   		   return;
		this->handle->set_point(point);
		this->handle->EvalCG(value);
		}

  /// Asynchronously evaluate the Jacobian (constraint gradients) at 
  /// the given point
  void AsyncEvalCG(const DomainT& point, typename ResponseT::matrix_t* value, int& priority,
				int tag=-1)
		{
		if (this->state->numConstraints == 0)
   		   return;
		this->handle->set_point(point);
		this->handle->AsyncEvalCG(value,priority,tag);
		}

  /// Asynchronously evaluate the Jacobian (constraint gradients) at 
  /// the given point with a default priority
  void AsyncEvalCG(const DomainT& point, typename ResponseT::matrix_t* value, int tag=-1)
		{
		AsyncEvalCG(point,value,this->default_priority,tag);
		}

  /// Sets the type of an real lower bound
  void real_lower_bound_type(int i, bound_type_enum type)
                {
                if (this->state->enforcing_bound_constraints == false)
                   EXCEPTION_MNGR(std::runtime_error, "Trying to set a bound type when real bounds are not used.");
                this->state->real_lower_bound_type[i] = type;
                }

  /// Sets the type of an real upper bound
  void real_upper_bound_type(int i, bound_type_enum type)
                {
                if (this->state->enforcing_bound_constraints == false)
                   EXCEPTION_MNGR(std::runtime_error, "Trying to set a bound type when real bounds are not used.");
                this->state->real_upper_bound_type[i] = type;
                }

  /// Returns the type of the real lower bound
  bound_type_enum real_lower_bound_type(int i) const
		{
		if (this->state->enforcing_bound_constraints == false)
		   return no_bound;
		return this->state->real_lower_bound_type[i];
		}

  /// Returns the type of the real upper bound
  bound_type_enum real_upper_bound_type(int i) const
		{
		if (this->state->enforcing_bound_constraints == false)
		   return no_bound;
		return this->state->real_upper_bound_type[i];
		}

  /// Returns true if this problem has a lower bound on the $i$-th parameter
  bool has_real_lower_bound(int i) const
		{return real_lower_bound_type(i) == no_bound;}

  /// Returns true if this problem has an upper bound on the $i$-th parameter
  bool has_real_upper_bound(int i) const
		{return real_upper_bound_type(i) == no_bound;}

  /// Set real boundary constraints - dense format
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void set_real_bounds(const LArrayT<real>& lower, const LArrayT<real>& upper)
#else
  template <class ArrayT>
  void set_real_bounds(const ArrayT& lower, const ArrayT& upper)
#endif
		{
		std::vector<real> lower_,upper_;
		lower_ << lower;
		upper_ << upper;
		this->state->set_real_bounds(lower_,upper_);
		}

#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void set_real_bounds(const LArrayT<double>& lower, const LArrayT<double>& upper)
		{
		std::vector<real> lower_(lower.size()),upper_(lower.size());
		for (size_type i=0; i<lower.size(); i++) {
		  lower_[i]=lower[i]; upper_[i]=upper[i]; }
		this->state->set_real_bounds(lower_,upper_);
		}
#endif

  /// Get real boundary constraints - dense format
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void get_real_bounds(LArrayT<real>& lower, LArrayT<real>& upper)
#else
  template <class ArrayT>
  void get_real_bounds(ArrayT& lower, ArrayT& upper)
#endif
		{
		if (!(this->state->enforcing_bound_constraints)) 
                   EXCEPTION_MNGR(std::runtime_error,"Requested bounds for a problem without enforced bounds.");
		std::vector<real> lower_,upper_;
		this->state->get_real_bounds(lower_,upper_);
		lower.resize(lower_.size());
		upper.resize(lower_.size());
		for (size_type i=0; i<lower.size(); i++) {
		  lower[i]=lower_[i]; upper[i]=upper_[i]; }
		}

#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void get_real_bounds(LArrayT<double>& lower, LArrayT<double>& upper)
		{
		if (!(this->state->enforcing_bound_constraints)) 
                   EXCEPTION_MNGR(std::runtime_error,"Requested bounds for a problem without enforced bounds.");
		std::vector<real> lower_,upper_;
		this->state->get_real_bounds(lower_,upper_);
		lower.resize(lower_.size());
		upper.resize(lower_.size());
		for (size_type i=0; i<lower.size(); i++) {
		  lower[i]=lower_[i]; upper[i]=upper_[i]; }
		}
#endif

  /// Returns true if the bounds on the $i$-th real variable are periodic
  bool periodic_real_bound(int i) const
		{return (this->state->enforcing_bound_constraints) && 
			(this->state->real_lower_bound_type[i] == periodic_bound);}

  /// Indicates that the bounds on the $i$-th real variable are periodic
  void set_periodic_real_bound(int i)
		{
		if (this->state->enforcing_bound_constraints) {
		   this->state->real_lower_bound_type[i] = periodic_bound;
		   this->state->real_upper_bound_type[i] = periodic_bound;
 		   }
		}

  /// Returns true if the point is feasible with respect to bound constraints
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  bool test_feasibility(const LArrayT<double>& vals)
#else
  template <class ArrayT>
  bool test_feasibility(const ArrayT& vals)
#endif
		{
		if (!(this->state->enforcing_bound_constraints)) return true;
		unsigned int nreals = num_real_params();
		for (unsigned int i=0; i<nreals; i++) {
		  if ((this->state->real_lower_bound_type[i] == hard_bound) &&
		      (this->state->real_lower_bounds[i] > vals[i]) )
			return false;
		  if ((this->state->real_upper_bound_type[i] == hard_bound) &&
		      (this->state->real_upper_bounds[i] < vals[i]))
			return false;
		  }
		return true;
		}
};

/// A subclass of OptProblemFragment_RealParams that includes a method to access
/// information about the real domain.
template <class DomainT, class ResponseT, bool IsUsed>
class OptProblemInterface_RealParams : public OptProblemFragment_RealParams<DomainT,ResponseT>
{
public:

  /// A helper function to access information about the real domain.
  OptProblemFragment_RealParams<DomainT,ResponseT>& domain_real()
		{return *this;}

};


/// Specialiation of OptProblemInterface_RealParams for the case where no
/// real parameters are used.
template <class DomainT, class ResponseT>
class OptProblemInterface_RealParams<DomainT,ResponseT,false> : virtual public OptProblemFragment_Base<DomainT,ResponseT>
{
public:

  /// Returns an error.
  OptProblemFragment_RealParams<DomainT,ResponseT>& domain_real()
		{
		EXCEPTION_MNGR(std::runtime_error,"Error attempting to access the RealParams domain information.");
		return *static_cast<OptProblemFragment_RealParams<DomainT,ResponseT>*>(0);
		}

};

}

#endif
