/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       NonDPolynomialChaos
//- Description: Implementation code for NonDPolynomialChaos class
//- Owner:       Mike Eldred, Sandia National Laboratories

#include "system_defs.h"
#include "NonDPolynomialChaos.H"
#include "DakotaModel.H"
#include "DakotaResponse.H"
#include "ProblemDescDB.H"
#include "DataFitSurrModel.H"
#include "OrthogPolyApproximation.H"


namespace Dakota {

NonDPolynomialChaos::NonDPolynomialChaos(Model& model): NonDExpansion(model),
  expansionImportFile(
    probDescDB.get_string("method.nond.expansion_import_file"))
{
  // This constructor is called for a standard letter-envelope iterator 
  // instantiation.

  // -------------------
  // Recast g(x) to G(u)
  // -------------------
  Model g_u_model;
  construct_g_u_model(g_u_model);

  // -------------------------
  // Construct u_space_sampler
  // -------------------------
  // LHS/Incremental LHS/Quadrature/SparseGrid samples in u-space
  // generated using active sampling view:
  Iterator u_space_sampler;
  // expansion_order defined for expansion_samples/collocation_pts
  UShortArray exp_order = probDescDB.get_dusa("method.nond.expansion_order");
  if (exp_order.length() == 1) {
    unsigned short order = exp_order[0];
    exp_order.reshape(numContinuousVars);
    exp_order = order;
  }
  if (expansionImportFile.empty()) {
    const UShortArray& quad_order_spec
      = probDescDB.get_dusa("method.nond.quadrature_order");
    const UShortArray& sparse_grid_level_spec
      = probDescDB.get_dusa("method.nond.sparse_grid_level");
    if (!quad_order_spec.empty()) {
      construct_quadrature(u_space_sampler, g_u_model, quad_order_spec);
      expansionCoeffsApproach = QUADRATURE;
    }
    else if (!sparse_grid_level_spec.empty()) {
      construct_sparse_grid(u_space_sampler, g_u_model, sparse_grid_level_spec);
      expansionCoeffsApproach = SPARSE_GRID;
    }
    else { // expansion_samples or collocation_points
      int orig_seed   = probDescDB.get_int("method.random_seed"),
	  exp_samples = probDescDB.get_int("method.nond.expansion_samples"),
	  colloc_pts  = probDescDB.get_int("method.nond.collocation_points");
      const Real& colloc_ratio
	= probDescDB.get_real("method.nond.collocation_ratio");
      if (exp_samples > 0) {
	numSamplesOnModel       = exp_samples;
	expansionCoeffsApproach = SAMPLING;
      }
      else if (colloc_pts > 0) {
	numSamplesOnModel       = colloc_pts;
	expansionCoeffsApproach = REGRESSION;
      }
      else if (colloc_ratio > 0.) {
	int num_exp_terms = (exp_order.empty()) ?
	  probDescDB.get_int("method.nond.expansion_terms") : 
	  BasisPolyApproximation::total_order_terms(exp_order);
	numSamplesOnModel = (int)floor(colloc_ratio*num_exp_terms + .5);
	expansionCoeffsApproach = REGRESSION;
      }
      construct_lhs(u_space_sampler, g_u_model, orig_seed);
    }

    // iteratedModel concurrency is defined by the number of samples
    // used in constructing the PC expansion
    if (numSamplesOnModel) // optional with default = 0
      maxConcurrency *= numSamplesOnModel;
  }

  // --------------------------------
  // Construct G-hat(u) = uSpaceModel
  // --------------------------------
  // G-hat(u) uses an orthogonal polynomial approximation over the
  // active/uncertain variables (using same view as iteratedModel/g_u_model:
  // not the typical All view for DACE).  No correction is employed.
  // *** Note: for PCBDO with polynomials over {u}+{d}, change view to All.
  String approx_type = "global_orthogonal_polynomial", sample_reuse, corr_type;
  if (expansionCoeffsApproach == REGRESSION)
    sample_reuse=probDescDB.get_string("method.nond.collocation_sample_reuse");
  short corr_order = -1;
  const Variables& g_u_vars = g_u_model.current_variables();
  uSpaceModel.assign_rep(new DataFitSurrModel(u_space_sampler, g_u_model,
    g_u_vars.view(), g_u_vars.variables_components(),
    g_u_model.current_response().active_set(), approx_type, exp_order,
    corr_type, corr_order, sample_reuse), false);
  // Define orthog_poly_basis for use by OrthogPolyApproximation in
  // instantiating the derived OrthogonalPolynomial instances.
  ShortArray orthog_poly_basis(numContinuousVars);
  const Pecos::ShortArray& u_types = natafTransform.u_types();
  bool extra_beta_params = false, extra_gamma_params = false;
  size_t i;
  for (i=0; i<numContinuousVars; i++) {
    switch (u_types[i]) {
    case NORMAL:
      orthog_poly_basis[i] = HERMITE;  break;
    case UNIFORM:
      orthog_poly_basis[i] = LEGENDRE; break;
    case EXPONENTIAL:
      orthog_poly_basis[i] = LAGUERRE; break;
    case BETA:
      orthog_poly_basis[i] = JACOBI;
      extra_beta_params    = true;     break;
    case GAMMA:
      orthog_poly_basis[i] = GENERALIZED_LAGUERRE;
      extra_gamma_params   = true;     break;
    }
  }
  // if all variables mode, initialize key to random variable subset
  bool all_vars = (numDesignVars || numEpistemicUncVars || numStateVars);
  BoolDeque random_vars_key;
  if (all_vars) {
    random_vars_key.resize(numContinuousVars);
    for (i=0; i<numContinuousVars; i++)
      random_vars_key[i]
	= (i >= numDesignVars && i < numDesignVars + numAleatoryUncVars);
  }
  // The expansion_order specification is passed through the DataFitSurrModel
  // ctor.  Any additional specification data needed by OrthogPolyApproximation
  // must be passed through by diving through the hierarchy.
  Array<Approximation>& orthog_poly_approxs = uSpaceModel.approximations();
  int expansion_terms = probDescDB.get_int("method.nond.expansion_terms");
  for (i=0; i<numFunctions; i++) {
    OrthogPolyApproximation* orthog_poly_approx_rep
      = (OrthogPolyApproximation*)orthog_poly_approxs[i].approx_rep();
    if (orthog_poly_approx_rep) { // may be NULL based on approxFnIndices
      orthog_poly_approx_rep->solution_approach(expansionCoeffsApproach);
      orthog_poly_approx_rep->basis_types(orthog_poly_basis);
      if (all_vars)
	orthog_poly_approx_rep->random_variables_key(random_vars_key);
      if (expansionCoeffsApproach == QUADRATURE ||
	  expansionCoeffsApproach == SPARSE_GRID)
	orthog_poly_approx_rep->integration_iterator(u_space_sampler);
      else if (expansion_terms)
	orthog_poly_approx_rep->expansion_terms(expansion_terms);
      if (extra_beta_params) {
	orthog_poly_approx_rep->beta_alphas(iteratedModel.beta_alphas());
	orthog_poly_approx_rep->beta_betas(iteratedModel.beta_betas());
      }
      if (extra_gamma_params)
	orthog_poly_approx_rep->gamma_alphas(iteratedModel.gamma_alphas());
    }
  }

  // -------------------------------------
  // Construct expansionSampler, if needed
  // -------------------------------------
  construct_expansion_sampler();

  // uSpaceModel concurrency is defined by the number of samples used
  // in evaluating the PC expansion
  uSpaceModel.init_communicators(numSamplesOnExpansion);
}


NonDPolynomialChaos::~NonDPolynomialChaos()
{
  uSpaceModel.free_communicators(numSamplesOnExpansion);
}


void NonDPolynomialChaos::quantify_uncertainty()
{
  initialize_expansion();

  if (expansionImportFile.empty())
    // ------------------------------
    // Calculate the PCE coefficients
    // ------------------------------
    compute_expansion();
  else {
    // ---------------------------
    // Import the PCE coefficients
    // ---------------------------
    ifstream import_stream(expansionImportFile);
    if (!import_stream) {
      Cerr << "\nError: cannot open polynomial chaos expansion import file "
	   << expansionImportFile << endl;
      abort_handler(-1);
    }
    if (subIteratorFlag || !finalStatistics.function_gradients().empty()) {
      Cerr << "\nError: PCE coefficient import not supported in advanced modes"
	   << endl;
      abort_handler(-1);
    }
    Array<Approximation>& orthog_poly_approxs = uSpaceModel.approximations();
    RealVectorArray chaos_coeffs_array(numFunctions);
    for (size_t i=0; i<numFunctions; i++) {
      OrthogPolyApproximation* orthog_poly_approx_rep
	= (OrthogPolyApproximation*)orthog_poly_approxs[i].approx_rep();
      orthog_poly_approx_rep->resolve_inputs();
      orthog_poly_approx_rep->allocate_arrays();
      chaos_coeffs_array[i].reshape(orthog_poly_approx_rep->expansion_terms());
    }
    import_stream >> chaos_coeffs_array;
    uSpaceModel.approximation_coefficients(chaos_coeffs_array);
  }

  compute_statistics();
  update_final_statistics();
  numUncertainQuant++;

#ifdef DEBUG
  if (subIteratorFlag)
    print_results(Cout);
#endif // DEBUG
}


void NonDPolynomialChaos::print_results(ostream& s)
{
  s << "-------------------------------------------------------------------\n";

  s.setf(ios::scientific);
  s << setprecision(write_precision);
  size_t i, j;
  char tag[10];
  const StringArray& fn_labels = iteratedModel.response_labels();
  for (i=0; i<numFunctions; i++) {
    s << "Polynomial Chaos coefficients for " << fn_labels[i] << ":\n";
    // header
    s << "  " << setw(17) << "coefficient";
    for (j=0; j<numDesignVars; j++)
      { sprintf(tag, "d%i", j+1); s << setw(5) << tag; }
    for (j=0; j<numUncertainVars; j++)
      { sprintf(tag, "u%i", j+1); s << setw(5) << tag; }
    for (j=0; j<numStateVars; j++)
      { sprintf(tag, "s%i", j+1); s << setw(5) << tag; }
    s << "\n  " << setw(17) << "-----------";
    for (j=0; j<numContinuousVars; j++)
      s << " ----";
    uSpaceModel.print_coefficients(s, i);
  }

  NonDExpansion::print_results(s);
}

} // namespace Dakota
