//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Combiner_Lagrange.hpp
  \brief Class description for APPSPACK::Combiner::Lagrange
*/

#ifndef APPSPACK_COMBINER_LAGRANGE
#define APPSPACK_COMBINER_LAGRANGE

#include "APPSPACK_Common.hpp"
#include "APPSPACK_Vector.hpp"
#include "APPSPACK_Combiner_Generic.hpp"

namespace APPSPACK
{

namespace Combiner
{

/*! \brief A class that impelements an Augmented Lagrangian (AL) approach
to handling nonlinear equality constraitns.

This class that combines a vector containing the objective value
\f$f\f$ and constraint values \f$c_i\f$, \f$i = 1,\ldots,m\f$, using
the Augmented Lagrangian merit function.  That is, given a vector
\f[
F = \left[\begin{array}{l}
f \\
c_1 \\
\vdots \\
c_m 
\end{array} \right]
\f]
this class returns the value
\f[
\Phi(F,\lambda,\mu) =  f + \sum_{i=1}^m (c_i\lambda_i + \frac{1}{2\mu} c_i^2).
\f]
Here \f$\lambda\f$ and \f$\mu\f$ are member variables and assigned values via
the methos setLambda() and setMu().
*/
class Lagrange : public Generic
{
public:
  //! Constructor. 
  Lagrange();

  //! Destructor.
  ~Lagrange();

  //! Returns the value of the Augmented Lagrangian
  virtual double operator()(const Vector& fc) const;

  //! Returns current value of mu.
  double getMu() const;
  //! Returns current value of lambda.
  const Vector& getLambda() const;

  //! Sets mu = mu_in.
  void setMu(double mu_in);
  //! Sets lambda = lambda_in
  void setLambda(const Vector& lambda_in);
  //! Set constraints
  void setBounds(const Vector& crhs_in, 
		 const Vector& hlow_in, 
		 const Vector& hupp_in);
private:
  //! Stores Lagrange multipliers estimates.
  Vector lambda;
  //! Stores penalty paramter mu.
  double mu;

  //! Stores equality constriants right-hand side.
  Vector crhs;
  //! Stores inequality constraints lower bound.
  Vector hlow;
  //! Stores inequality constraints upper bound.
  Vector hupp;
};


}
}
#endif
