/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       PStudyDACE
//- Description: Base class for ParamStudy, DDACEDesignCompExp, and
//-              FSUDesignCompExp
//- Owner:       Mike Eldred
//- Version: $Id: DakotaPStudyDACE.H 5780 2009-03-26 22:14:19Z mseldre $

#ifndef DAKOTA_PSTUDY_DACE_H
#define DAKOTA_PSTUDY_DACE_H

#include "DakotaAnalyzer.H"


namespace Dakota {

/// Base class for managing common aspects of parameter studies and
/// design of experiments methods.

/** The PStudyDACE base class manages common data and functions,
    such as those involving the best solutions located during the
    parameter set evaluations or the printing of final results. */

class PStudyDACE: public Analyzer
{
public:

protected:

  //
  //- Heading: Constructors and destructors
  //

  /// constructor
  PStudyDACE(Model& model);
  /// alternate constructor for instantiations "on the fly"
  PStudyDACE(NoDBBaseConstructor, Model& model);
  /// destructor
  ~PStudyDACE();
    
  //
  //- Heading: Virtual member function redefinitions
  //

  void  run();
  const Variables& variables_results() const;
  const Response&  response_results() const;
  void  response_results_active_set(const ActiveSet& set);
  void  print_results(ostream& s);

  //
  //- Heading: New virtual member functions
  //

  /// Redefines the run_iterator virtual function for the PStudy/DACE branch.
  virtual void extract_trends() = 0;

  //
  //- Heading: Convenience/internal member functions
  //

  /// compares current evaluation to best evaluation and updates best
  void update_best(const Variables& vars, const Response& response,
		   const int eval_num);

  //
  //- Heading: Data
  //

  // Data needed for update_best() so that param studies can be used in
  // strategies such as MultilevelOptStrategy

  Variables bestVariables; ///< best variables found during the study
  Response  bestResponse;  ///< best response found during the study

  /// best objective function found during the study
  Real bestObjFn;
  /// best constraint violations found during the study.  In the
  /// current approach, constraint violation reduction takes strict
  /// precedence over objective function reduction.
  Real bestConViol;

  size_t     numObjFns;        ///< number of objective functions
  size_t     numLSqTerms;      ///< number of least squares terms
};


inline void PStudyDACE::run()
{ extract_trends(); }


inline const Variables& PStudyDACE::variables_results() const
{ return bestVariables; }


inline const Response& PStudyDACE::response_results() const
{ return bestResponse; }


inline void PStudyDACE::response_results_active_set(const ActiveSet& set)
{ bestResponse.active_set(set); }

} // namespace Dakota

#endif
