/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implemenation of class SingleObjectiveStatistician.

    NOTES:

        See notes of SingleObjectiveStatistician.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Thu May 22 08:02:43 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the SingleObjectiveStatistician class.
 */




/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <../Utilities/include/Logging.hpp>
#include <../Utilities/include/DesignGroup.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <../Utilities/include/WeightedSumMap.hpp>
#include <../Utilities/include/DesignMultiSet.hpp>
#include <../Utilities/include/DesignGroupVector.hpp>
#include <../Utilities/include/SingleObjectiveStatistician.hpp>


/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;
using namespace JEGA;
using namespace JEGA::Logging;





/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Utilities {







/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/








/*
================================================================================
Accessors
================================================================================
*/








/*
================================================================================
Public Methods
================================================================================
*/
std::pair<double, std::vector<const Design*> >
SingleObjectiveStatistician::FindMinSumDesigns(
    const WeightedSumMap& wsm
    )
{
    EDDY_FUNC_DEBUGSCOPE

    double minSum = DBL_MAX;
    vector<const Design*> bests;

    for(WeightedSumMap::const_iterator it(wsm.begin()); it!=wsm.end(); ++it)
    {
        if((*it).second < minSum)
        {
            minSum = (*it).second;
            bests.assign(1, (*it).first);
        }
        else if(minSum == (*it).second) bests.push_back((*it).first);
    }


    return make_pair(minSum, bests);
}


double
SingleObjectiveStatistician::ComputeWeightedSum(
    const Design& des,
    const JEGA::DoubleVector& weights
    )
{
    EDDY_FUNC_DEBUGSCOPE

    JEGAIFLOG_CF_G(!des.IsEvaluated(), lquiet(),
        text_entry(lquiet(), "Computing weighted sum on non-evaluated Design. "
                   " Results may be nonsensical.")
        )

    JEGAIFLOG_CF_G(des.IsIllconditioned(), lquiet(),
        text_entry(lquiet(), "Computing weighted sum on illconditioned Design."
                   "  Result may be nonsensical.")
        )

    DesignTarget& target = des.GetDesignTarget();
    const ObjectiveFunctionInfoVector& ofInfos =
        target.GetObjectiveFunctionInfos();

    EDDY_ASSERT(weights.size() == target.GetNOF());

    double sum = 0.0;
    size_t of = 0;
    for(ObjectiveFunctionInfoVector::const_iterator it(ofInfos.begin());
        it!=ofInfos.end(); ++it)
            sum += weights.at(of++) * (*it)->WhichForMinimization(des);

    return sum;

} // SingleObjectiveStatistician::ComputeWeightedSum

WeightedSumMap
SingleObjectiveStatistician::ComputeWeightedSums(
    const DesignGroup& designs,
    const JEGA::DoubleVector& weights
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // create and return a weighted sum map of all Designs.
    WeightedSumMap ret(weights);

    designs.SynchronizeOFAndDVContainers();

    for(DesignDVSortSet::const_iterator it(designs.BeginDV());
        it!=designs.EndDV(); ++it)
            ret.AddWeightedSum(*it, ComputeWeightedSum(**it, weights));

    return ret;
}

WeightedSumMap
SingleObjectiveStatistician::ComputeWeightedSums(
    const DesignGroupVector& designs,
    const JEGA::DoubleVector& weights
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // create and return a weighted sum map of all Designs.
    WeightedSumMap ret(weights);

    for(DesignGroupVector::const_iterator it(designs.begin());
        it!=designs.end(); ++it)
            for(DesignDVSortSet::const_iterator jt((*it)->BeginDV());
                jt!=(*it)->EndDV(); ++jt)
                    ret.AddWeightedSum(*jt, ComputeWeightedSum(**jt, weights));

    return ret;
}

/*
================================================================================
Subclass Visible Methods
================================================================================
*/







/*
================================================================================
Subclass Overridable Methods
================================================================================
*/






/*
================================================================================
Private Methods
================================================================================
*/







/*
================================================================================
Structors
================================================================================
*/






/*
================================================================================
File Scope Utility Class Implementations
================================================================================
*/







/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Utilities
} // namespace JEGA
