/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 NonDSampling
//- Description: Implementation code for NonDSampling class
//- Owner:       Mike Eldred
//- Checked by:
//- Version:

#include "data_types.h"
#include "system_defs.h"
#include "DakotaModel.H"
#include "DakotaResponse.H"
#include "NonDSampling.H"
#include "ProblemDescDB.H"
#ifdef DAKOTA_DDACE
#include "Distribution.h"
#elif defined(DAKOTA_UTILIB)
#include <utilib/seconds.h>
#endif
#include "pecos_data_types.hpp"
#include "pecos_stat_util.hpp"

static const char rcsId[]="@(#) $Id: NonDSampling.C 5852 2009-05-01 22:56:36Z mseldre $";

using std::endl;
using std::setprecision;


namespace Dakota {

RealArray NonDSampling::rawData;


/** This constructor is called for a standard letter-envelope iterator
    instantiation.  In this case, set_db_list_nodes has been called and
    probDescDB can be queried for settings from the method specification. */
NonDSampling::NonDSampling(Model& model): NonD(model),
  originalSeed(probDescDB.get_int("method.random_seed")),
  samplesSpec(probDescDB.get_int("method.samples")), numSamples(samplesSpec),
  sampleType(probDescDB.get_string("method.sample_type")),
  statsFlag(true), allDataFlag(false), sampleRanksMode(IGNORE_RANKS),
  varyPattern(!probDescDB.get_bool("method.fixed_seed")), numLHSRuns(0)
{
  if (numEpistemicUncVars && totalLevelRequests) {
    Cerr << "\nError: nond_sampling does not support level requests for "
	 << "analyses containing epistemic uncertainties." << endl;
    abort_handler(-1);
  }

  // Since the sampleType is shared with other iterators for other purposes, its
  // default in DataMethod.C is the NULL string.  Explicitly enforce the LHS
  // default here.
  if (sampleType.empty())
    sampleType = "lhs";

  // initialize finalStatistics using the default statistics set
  initialize_final_statistics();

  // update concurrency
  if (numSamples) // samples is now optional (default = 0)
    maxConcurrency *= numSamples;
}


/** This alternate constructor is used for generation and evaluation
    of on-the-fly sample sets. */
NonDSampling::
NonDSampling(NoDBBaseConstructor, Model& model, int samples, int seed):
  NonD(NoDBBaseConstructor(), model), originalSeed(seed), samplesSpec(samples),
  numSamples(samples), sampleType("lhs"), statsFlag(false), allDataFlag(true),
  sampleRanksMode(IGNORE_RANKS), varyPattern(false), numLHSRuns(0)
{
  subIteratorFlag = true; // suppress some output
  // not used but included for completeness
  if (numSamples) // samples is now optional (default = 0)
    maxConcurrency *= numSamples;
}


/** This alternate constructor is used by ConcurrentStrategy for
    generation of uniform, uncorrelated sample sets. */
NonDSampling::
NonDSampling(NoDBBaseConstructor, int samples, int seed,
	     const RealDenseVector& lower_bnds,
	     const RealDenseVector& upper_bnds):
  NonD(NoDBBaseConstructor(), lower_bnds, upper_bnds), originalSeed(seed),
  samplesSpec(samples), numSamples(samples), sampleType("lhs"),
  statsFlag(false), allDataFlag(true), sampleRanksMode(IGNORE_RANKS),
  varyPattern(false), numLHSRuns(0)

{
  subIteratorFlag = true; // suppress some output

  // not used but included for completeness
  if (numSamples) // samples is now optional (default = 0)
    maxConcurrency *= numSamples;
}


NonDSampling::~NonDSampling()
{ }


/** This version of get_parameter_sets() extracts data from the
    user-defined model in any of the four sampling modes. */
void NonDSampling::get_parameter_sets(const Model& model)
{
  initialize_lhs(true);
  RealDenseMatrix samples_array;

  short model_view = model.current_variables().view().first;
#ifdef DAKOTA_PECOS
  if (samplingVarsMode == UNCERTAIN_UNIFORM || 
      samplingVarsMode == ACTIVE_UNIFORM    ||
      samplingVarsMode == ALL_UNIFORM) {
    if ( samplingVarsMode == ACTIVE_UNIFORM || 
	 ( samplingVarsMode == ALL_UNIFORM && 
	   ( model_view == MERGED_ALL || model_view == MIXED_ALL ) ) ||
	 ( samplingVarsMode == UNCERTAIN_UNIFORM && 
	   ( model_view == MERGED_DISTINCT_UNCERTAIN           ||
	     model_view == MERGED_DISTINCT_ALEATORY_UNCERTAIN  ||
	     model_view == MERGED_DISTINCT_EPISTEMIC_UNCERTAIN ||
	     model_view == MIXED_DISTINCT_UNCERTAIN            ||
	     model_view == MIXED_DISTINCT_ALEATORY_UNCERTAIN   ||
	     model_view == MIXED_DISTINCT_EPISTEMIC_UNCERTAIN ) ) ) {
      // sample uniformly from ACTIVE lower/upper bounds (regardless of model
      // view), from UNCERTAIN lower/upper bounds (with model in DISTINCT view),
      // or from ALL lower/upper bounds (with model in ALL view).
      // TO DO: verify loss of sampleRanks control is OK
      lhsDriver.generate_uniform_samples(model.continuous_lower_bounds(),
					 model.continuous_upper_bounds(),
					 numSamples, samples_array);
    }
    else if (samplingVarsMode == ALL_UNIFORM) {
      // sample uniformly from ALL lower/upper bounds with model using
      // a DISTINCT view
      // TO DO: verify loss of sampleRanks control is OK
      lhsDriver.generate_uniform_samples(model.all_continuous_lower_bounds(),
					 model.all_continuous_upper_bounds(),
					 numSamples, samples_array);
    }
    else if (samplingVarsMode == UNCERTAIN_UNIFORM) {
      // sample uniformly from UNCERTAIN lower/upper bounds with model
      // using a view other than
      // {MERGED,MIXED}_DISTINCT_{,_ALEATORY,_EPISTEMIC}UNCERTAIN
      RealDenseVector uncertain_l_bnds, uncertain_u_bnds;
      if (model_view == MERGED_ALL || model_view == MIXED_ALL) {
	const RealDenseVector& c_l_bnds = model.continuous_lower_bounds();
	const RealDenseVector& c_u_bnds = model.continuous_upper_bounds();
	uncertain_l_bnds = RealDenseVector(Teuchos::View,
	  const_cast<Real*>(&c_l_bnds[numDesignVars]), numUncertainVars);
	uncertain_u_bnds = RealDenseVector(Teuchos::View,
	  const_cast<Real*>(&c_u_bnds[numDesignVars]), numUncertainVars);
      }
      else { // MERGED/MIXED_DISTINCT_DESIGN or MERGED/MIXED_DISTINCT_STATE
	// This case should not occur. If it does, numDesignVars may not be set.
	const RealDenseVector& all_l_bnds = model.all_continuous_lower_bounds();
	const RealDenseVector& all_u_bnds = model.all_continuous_upper_bounds();
	uncertain_l_bnds = RealDenseVector(Teuchos::View,
	  const_cast<Real*>(&all_l_bnds[numDesignVars]), numUncertainVars);
	uncertain_u_bnds = RealDenseVector(Teuchos::View,
	  const_cast<Real*>(&all_u_bnds[numDesignVars]), numUncertainVars);
      }
      // TO DO: verify loss of sampleRanks control is OK
      lhsDriver.generate_uniform_samples(uncertain_l_bnds, uncertain_u_bnds,
					 numSamples, samples_array);
    }
  }
  else { // UNCERTAIN || ACTIVE || ALL

    // extract design and state bounds
    RealDenseVector design_l_bnds, design_u_bnds, state_l_bnds, state_u_bnds;
    if ( samplingVarsMode == ACTIVE || ( samplingVarsMode == ALL && 
	 ( model_view == MERGED_ALL || model_view == MIXED_ALL ) ) ) {
      const RealDenseVector& c_l_bnds = model.continuous_lower_bounds();
      const RealDenseVector& c_u_bnds = model.continuous_upper_bounds();
      design_l_bnds = RealDenseVector(Teuchos::View,
	const_cast<Real*>(&c_l_bnds[0]), numDesignVars);
      design_u_bnds = RealDenseVector(Teuchos::View,
	const_cast<Real*>(&c_u_bnds[0]), numDesignVars);
      state_l_bnds = RealDenseVector(Teuchos::View,
	const_cast<Real*>(&c_l_bnds[numDesignVars + numUncertainVars]),
	numStateVars);
      state_u_bnds  = RealDenseVector(Teuchos::View,
	const_cast<Real*>(&c_u_bnds[numDesignVars + numUncertainVars]),
	numStateVars);
    }
    else if (samplingVarsMode == ALL) {
      const RealDenseVector& all_l_bnds = model.all_continuous_lower_bounds();
      const RealDenseVector& all_u_bnds = model.all_continuous_upper_bounds();
      design_l_bnds = RealDenseVector(Teuchos::View,
	const_cast<Real*>(&all_l_bnds[0]), numDesignVars);
      design_u_bnds = RealDenseVector(Teuchos::View,
	const_cast<Real*>(&all_u_bnds[0]), numDesignVars);
      state_l_bnds = RealDenseVector(Teuchos::View,
	const_cast<Real*>(&all_l_bnds[numDesignVars + numUncertainVars]),
	numStateVars);
      state_u_bnds  = RealDenseVector(Teuchos::View,
	const_cast<Real*>(&all_u_bnds[numDesignVars + numUncertainVars]),
	numStateVars);
    }

    // Call LHS to generate the specified samples within the specified
    // distributions.  Moved from constructors to allow publication of changes
    // after construction (e.g., sampling_reset() or change in UQ samples as OUU
    // progresses).  Note that this requires LHS to be more stable than before
    // (i.e., needs to be reentrant so that the same set of samples is generated
    // if nothing changes between quantify_uncertainty calls -->> important for
    // finite difference accuracy in OUU).
    lhsDriver.generate_samples(design_l_bnds, design_u_bnds, state_l_bnds,
      state_u_bnds, model.normal_means(), model.normal_std_deviations(),
      model.normal_lower_bounds(), model.normal_upper_bounds(),
      model.lognormal_means(), model.lognormal_std_deviations(),
      model.lognormal_error_factors(), model.lognormal_lower_bounds(),
      model.lognormal_upper_bounds(), model.uniform_lower_bounds(),
      model.uniform_upper_bounds(), model.loguniform_lower_bounds(),
      model.loguniform_upper_bounds(), model.triangular_modes(),
      model.triangular_lower_bounds(), model.triangular_upper_bounds(),
      model.exponential_betas(), model.beta_alphas(), model.beta_betas(),
      model.beta_lower_bounds(), model.beta_upper_bounds(),
      model.gamma_alphas(), model.gamma_betas(), model.weibull_alphas(),
      model.weibull_betas(), model.gumbel_alphas(), model.gumbel_betas(),
      model.frechet_alphas(), model.frechet_betas(),
      model.histogram_bin_pairs(), model.histogram_point_pairs(), 
      model.interval_probabilities(), model.interval_bounds(),
      model.uncertain_correlations(), numSamples, samples_array, sampleRanks);
  }
#endif // DAKOTA_PECOS

  finalize_lhs(samples_array); // samples_array -> allVariables
}


/** This version of get_parameter_sets() does not extract data from the
    user-defined model, but instead relies on the incoming bounded region
    definition.  It only support a UNIFORM sampling mode, where the
    distinction of ACTIVE_UNIFORM vs. ALL_UNIFORM is handled elsewhere. */
void NonDSampling::
get_parameter_sets(const RealDenseVector& lower_bnds,
		   const RealDenseVector& upper_bnds)
{
  initialize_lhs(true);
  RealDenseMatrix samples_array;
#ifdef DAKOTA_PECOS
  lhsDriver.generate_uniform_samples(lower_bnds, upper_bnds, numSamples,
				     samples_array);
#endif // DAKOTA_PECOS
  finalize_lhs(samples_array); // samples_array -> allVariables
}


void NonDSampling::initialize_lhs(bool write_message)
{
  // keep track of number of LHS executions for this object
  numLHSRuns++;

#ifdef DAKOTA_PECOS
  // Set seed value for input to LHS's random number generator.  Emulate DDACE
  // behavior in which a user-specified seed gives you repeatable behavior but
  // no specification gives you random behavior.  A system clock is used to
  // randomize in the no user specification case.  For cases where
  // get_parameter_sets() may be called multiple times for the same sampling
  // iterator (e.g., SBO), support a deterministic sequence of seed values.
  // This renders the study repeatable but the sampling pattern varies from
  // one run to the next.
  if (numLHSRuns == 1) { // set initial seed
    if (originalSeed) // user seed specification: repeatable behavior
      lhsDriver.seed(originalSeed);
    else { // no user specification: nonrepeatable behavior
      // Generate initial seed from a system clock.  NOTE: the system clock
      // should not used for multiple LHS calls since (1) clock granularity can
      // be too coarse (can repeat on subsequent runs for inexpensive test fns)
      // and (2) seed progression can be highly structured, which could induce
      // correlation between sample sets.  Instead, the clock-generated case
      // varies the seed below using the same approach as the user-specified
      // case.  This has the additional benefit that a random run can be
      // recreated by specifying the clock-generated seed in the input file.
      int random_seed = 1;
#ifdef DAKOTA_DDACE
      random_seed += DistributionBase::timeSeed(); // microsecs, time of day
#elif defined(DAKOTA_UTILIB)
      random_seed += (int)CurrentTime();           // secs, time of day
#else
      random_seed += (int)clock();                 // clock ticks, exec time
#endif
      lhsDriver.seed(random_seed);
    }
  }
  else if (varyPattern) // define sequence of seed values for numLHSRuns > 1
    lhsDriver.advance_seed_sequence();

  // Needed a way to turn this off when LHS sampling is being used in
  // NonDAdaptImpSampling because it gets written a _LOT_
  if (write_message) {
    Cout << "\nNonD " << sampleType << " Samples = " << numSamples;
    if (varyPattern && numLHSRuns > 1)
      Cout << " Seed (deterministic sequence) = ";
    else if (originalSeed)
      Cout << " Seed (user-specified) = ";
    else
      Cout << " Seed (system-generated) = ";
    Cout << lhsDriver.seed() << '\n';
  }

  lhsDriver.initialize(sampleType, sampleRanksMode, !subIteratorFlag);
#endif // DAKOTA_PECOS
}


void NonDSampling::finalize_lhs(RealDenseMatrix& samples_array)
{
  if (samples_array.empty()) {
    Cerr << "Error: empty samples_array in NonDSampling::finalize_lhs()."
	 << endl;
    abort_handler(-1);
  }

  pair<short,short> vars_view;
  RealDenseVector   ic_vars;
  switch (samplingVarsMode) { // TO DO: add additional sampling modes?
  case UNCERTAIN: case UNCERTAIN_UNIFORM:
    vars_view.first  = MIXED_DISTINCT_UNCERTAIN; // TO DO: aleatory/epist flags?
    vars_view.second = MIXED_DISTINCT_DESIGN;
    if (!iteratedModel.is_null()) {
      const Variables& model_vars = iteratedModel.current_variables();
      short model_view = model_vars.view().first;
      if ( model_view == MERGED_ALL || model_view == MIXED_ALL )
	ic_vars = RealDenseVector(Teuchos::View,
	  model_vars.continuous_variables().values(), numDesignVars);
    }
    break;
  case ACTIVE: case ACTIVE_UNIFORM:
    if (iteratedModel.is_null()) {
      vars_view.first  = MIXED_DISTINCT_UNCERTAIN;
      vars_view.second = MIXED_DISTINCT_DESIGN;
    }
    else
      vars_view = iteratedModel.current_variables().view();
    break;
  case ALL: case ALL_UNIFORM:
    vars_view.first  = MIXED_ALL;
    vars_view.second = EMPTY;
    break;
  }

  size_t i, j, num_av = samples_array.numRows();
  Sizet2DArray null_comps;
  if (iteratedModel.is_null()) {
    null_comps.reshape(4);
    null_comps[0].reshape(3);  null_comps[0] = 0; // design
    null_comps[1].reshape(13); null_comps[1] = 0; // aleatory uncertain
    null_comps[1][0] = null_comps[1][3] = num_av; // --> uniform uncertain
    null_comps[2].reshape(2);  null_comps[2] = 0; // epistemic uncertain
    null_comps[3].reshape(3);  null_comps[3] = 0; // state
  }
  const Sizet2DArray& vars_comps = (iteratedModel.is_null()) ? null_comps :
    iteratedModel.current_variables().variables_components();

  // avoid unnecessary copying as much as possible (may be large sample sets)
  RealDenseVector c_vars(num_av);
  if (allVariables.length() != numSamples)
    allVariables.reshape(numSamples);
  for (i=0; i<numSamples; i++) {
    Variables& all_vars_i = allVariables[i];
    Real*   samples_col_i = samples_array[i];
    if (all_vars_i.is_null() || vars_view != all_vars_i.view()) {
      all_vars_i = Variables(vars_view, vars_comps); // minimal instantiation
      // continuous vars not yet sized: set by vector
      for (j=0; j<num_av; j++)
	c_vars[j] = samples_col_i[j]; // [i][j] or (j,i)
      all_vars_i.continuous_variables(c_vars);
    }
    else // continuous vars already sized: avoid vector copy
      for (j=0; j<num_av; j++)
	all_vars_i.continuous_variable(samples_col_i[j], j); // [i][j] or (j,i)
    if (!ic_vars.empty())
      all_vars_i.inactive_continuous_variables(ic_vars);
  }
}


void NonDSampling::
compute_statistics(const VariablesArray& vars_samples,
		   const ResponseArray&  resp_samples)
{
  if (numEpistemicUncVars) // Epistemic/mixed
    compute_intervals(resp_samples); // compute min/max response intervals
  else { // Aleatory
    // compute means and std deviations with confidence intervals
    compute_moments(resp_samples);
    // compute CDF/CCDF mappings of z to p/beta and p/beta to z
    if (totalLevelRequests)
      compute_distribution_mappings(resp_samples);
  }
  compute_correlations(vars_samples, resp_samples);
  update_final_statistics();
}


void NonDSampling::compute_intervals(const ResponseArray& samples)
{
  // For the samples array, calculate min/max response intervals

  size_t i, j, num_obs = samples.length(), num_samp;
  const StringArray& resp_labels = iteratedModel.response_labels();

  if (minValues.empty()) minValues.reshape(numFunctions);
  if (maxValues.empty()) maxValues.reshape(numFunctions);
  for (i=0; i<numFunctions; i++) {
    num_samp = 0;
    Real min = DBL_MAX, max = -DBL_MAX;
    for (j=0; j<num_obs; j++) {
      const Real& sample = samples[j].function_value(i);
      if (Isfinite(sample)) { // neither NaN nor +/-Inf
	if (sample < min) min = sample;
	if (sample > max) max = sample;
	num_samp++;
      }
    }
    minValues[i] = min;
    maxValues[i] = max;
    if (num_samp != num_obs)
      Cerr << "Warning: sampling statistics for " << resp_labels[i] << " omit "
	   << num_obs-num_samp << " failed evaluations out of " << num_obs
	   << " samples.\n";
  }
}


void NonDSampling::compute_moments(const ResponseArray& samples)
{
  // For the samples array, calculate means and standard deviations
  // with confidence intervals

  size_t i, j, num_obs = samples.length(), num_samp;
  Real sum, var;
  const StringArray& resp_labels = iteratedModel.response_labels();

  if (meanStats.empty())           meanStats.reshape(numFunctions);
  if (stdDevStats.empty())         stdDevStats.reshape(numFunctions);
  if (mean95CIDeltas.empty())      mean95CIDeltas.reshape(numFunctions);
#if defined(HAVE_BOOST)
  if (stdDev95CILowerBnds.empty()) stdDev95CILowerBnds.reshape(numFunctions);
  if (stdDev95CIUpperBnds.empty()) stdDev95CIUpperBnds.reshape(numFunctions);
#endif // HAVE_BOOST
  for (i=0; i<numFunctions; i++) {

    num_samp  = 0;
    sum = var = 0.;
    // means
    for (j=0; j<num_obs; j++) {
      const Real& sample = samples[j].function_value(i);
      if (Isfinite(sample)) { // neither NaN nor +/-Inf
	sum += sample;
	num_samp++;
      }
    }

    if (num_samp != num_obs)
      Cerr << "Warning: sampling statistics for " << resp_labels[i] << " omit "
	   << num_obs-num_samp << " failed evaluations out of " << num_obs
	   << " samples.\n";
    if (!num_samp) {
      Cerr << "Error: Number of samples for " << resp_labels[i]
	   << " must be nonzero for moment calculation in NonDSampling::"
	   << "compute_statistics()." << endl;
      abort_handler(-1);
    }

    meanStats[i] = sum/((Real)num_samp);
    // standard deviations
    for (j=0; j<num_obs; j++) {
      const Real& sample = samples[j].function_value(i);
      if (Isfinite(sample)) // neither NaN nor +/-Inf
	var += pow(sample - meanStats[i], 2);
    }
    stdDevStats[i] = (num_samp > 1) ? sqrt(var/(Real)(num_samp-1)) : 0.;

    if (num_samp > 1) {
      // 95% confidence intervals (2-sided interval, not 1-sided limit)
      Real dof = num_samp - 1, s = stdDevStats[i];
#ifdef HAVE_BOOST
      // mean: the better formula does not assume known variance but requires
      // a function for the Student's t-distr. with (num_samp-1) degrees of
      // freedom (Haldar & Mahadevan, p. 127).
      Pecos::students_t_dist t_dist(dof);
      mean95CIDeltas[i]
	= s*(bmth::quantile(t_dist,0.975))/sqrt((Real)num_samp);
      // std dev: chi-square distribution with (num_samp-1) degrees of freedom
      // (Haldar & Mahadevan, p. 132).
      Pecos::chi_squared_dist chisq(dof);
      stdDev95CILowerBnds[i] = s*sqrt(dof/bmth::quantile(chisq,0.975));
      stdDev95CIUpperBnds[i] = s*sqrt(dof/bmth::quantile(chisq,0.025));
/* #elif HAVE_GSL
      // mean: the better formula does not assume known variance but requires
      // a function for the Student's t-distr. with (num_samp-1) degrees of
      // freedom (Haldar & Mahadevan, p. 127).
      mean95CIDeltas[i]
	= s*gsl_cdf_tdist_Pinv(0.975,dof)/sqrt((Real)num_samp);
      // std dev: chi-square distribution with (num_samp-1) degrees of freedom
      // (Haldar & Mahadevan, p. 132).
      stdDev95CILowerBnds[i] = s*sqrt(dof/gsl_cdf_chisq_Pinv(0.975, dof));
      stdDev95CIUpperBnds[i] = s*sqrt(dof/gsl_cdf_chisq_Pinv(0.025, dof));
*/
#else
      // mean: k_(alpha/2) = Phi^(-1)(0.975) = 1.96 (Haldar & Mahadevan,
      // p. 123).  This simple formula assumes a known variance, which
      // requires a sample of sufficient size (i.e., greater than 10).
      mean95CIDeltas[i] = 1.96*stdDevStats[i]/sqrt((Real)num_samp);
#endif // HAVE_BOOST
    }
    else {
      mean95CIDeltas = 0.;
      stdDev95CILowerBnds = stdDev95CIUpperBnds = stdDevStats;
    }
  }
}


void NonDSampling::compute_distribution_mappings(const ResponseArray& samples)
{
  // For the samples array, calculate the following statistics:
  // > CDF/CCDF mappings of response levels to probability/reliability levels
  // > CDF/CCDF mappings of probability/reliability levels to response levels

  // Size the output arrays here instead of in the ctor in order to support
  // alternate sampling ctors.  Sizing is done in a minimal way for sampling
  // methods since there is no distinction between requested and computed levels
  // for the same measure (the request is always achieved) and since probability
  // (computed by binning) and reliability (computed using mu,sigma,target) are
  // not directly related.  [In NonDReliability, a requested level may differ
  // from that computed/achieved and probability and reliability are carried
  // along in parallel.]
  if (computedRespLevels.empty() || computedProbLevels.empty() ||
      computedRelLevels.empty()  || computedGenRelLevels.empty()) {
    computedRespLevels.reshape(numFunctions);
    computedProbLevels.reshape(numFunctions);
    computedRelLevels.reshape(numFunctions);
    computedGenRelLevels.reshape(numFunctions);
    for (size_t i=0; i<numFunctions; i++) {
      size_t num_levels = requestedRespLevels[i].length();
      switch (respLevelTarget) {
      case PROBABILITIES:
	computedProbLevels[i].reshape(num_levels);   break;
      case RELIABILITIES:
	computedRelLevels[i].reshape(num_levels);    break;
      case GEN_RELIABILITIES:
	computedGenRelLevels[i].reshape(num_levels); break;
      }
      num_levels = requestedProbLevels[i].length() +
	requestedRelLevels[i].length() + requestedGenRelLevels[i].length();
      computedRespLevels[i].reshape(num_levels);
    }
  }

  size_t i, j, k, num_obs = samples.length(), num_samp;
  const StringArray& resp_labels = iteratedModel.response_labels();
  RealArray sorted_samples; // STL-based array for sorting

  for (i=0; i<numFunctions; i++) {

    num_samp = 0;
    for (j=0; j<num_obs; j++)
      if (Isfinite(samples[j].function_value(i))) // neither NaN nor +/-Inf
	num_samp++;

    // CDF/CCDF mappings: z -> p/beta/beta* and p/beta/beta* -> z
    size_t rl_len = requestedRespLevels[i].length(),
           pl_len = requestedProbLevels[i].length(),
           bl_len = requestedRelLevels[i].length(),
           gl_len = requestedGenRelLevels[i].length();
    for (j=0; j<rl_len; j++) {
      const Real& z = requestedRespLevels[i][j];
      switch (respLevelTarget) {
      case PROBABILITIES: case GEN_RELIABILITIES: {
	// z -> p/beta* (based on binning)
	size_t less_eq_z = 0;
	for (k=0; k<num_obs; k++) {
	  const Real& sample = samples[k].function_value(i);
	  if (Isfinite(sample)) // neither NaN nor +/-Inf
	    less_eq_z += (sample <= z) ? 1 : 0;
	}
	Real cdf_prob = (Real)less_eq_z/(Real)num_samp;
	Real computed_prob = (cdfFlag) ? cdf_prob : 1. - cdf_prob;
	if (respLevelTarget == PROBABILITIES)
	  computedProbLevels[i][j] = computed_prob;
	else
	  computedGenRelLevels[i][j] = -Pecos::Phi_inverse(computed_prob);
	break;
      }
      case RELIABILITIES: // z -> beta (based on moment projection)
	if (stdDevStats[i] > 1.e-25) {
	  Real ratio = (meanStats[i] - z)/stdDevStats[i];
	  computedRelLevels[i][j] = (cdfFlag) ? ratio : -ratio;
	}
	else
	  computedRelLevels[i][j] = ( (cdfFlag && meanStats[i] <= z) ||
	    (!cdfFlag && meanStats[i] > z) ) ? -1.e50 : 1.e50;
	break;
      }
    }
    if (pl_len || gl_len) { // sort samples array for p/beta* -> z mappings
      sorted_samples.reshape(num_samp);
      size_t cntr = 0;
      for (j=0; j<num_obs; j++) {
	const Real& sample = samples[j].function_value(i);
	if (Isfinite(sample))
	  sorted_samples[cntr++] = sample;
      }
      sort(sorted_samples.begin(), sorted_samples.end()); // ascending order
    }
    for (j=0; j<pl_len+gl_len; j++) { // p/beta* -> z
      Real p = (j<pl_len) ? requestedProbLevels[i][j] :
	Pecos::Phi(-requestedGenRelLevels[i][j-pl_len]);
      // since each sample has 1/N probability, a probability level can be
      // directly converted to an index within a sorted array (index =~ p * N)
      Real cdf_p_x_obs = (cdfFlag) ? p*(Real)num_samp : (1.-p)*(Real)num_samp;
      // convert to an int and round down using floor().  Apply a small
      // numerical adjustment so that probabilities on the boundaries
      // (common with round probabilities and factor of 10 samples)
      // are consistently rounded down (consistent with CDF p(g<=z)).
      Real order = (cdf_p_x_obs > .9)
	         ? pow(10., ceil(log10(cdf_p_x_obs))) : 0.;
      int index = (int)floor(cdf_p_x_obs - order*DBL_EPSILON);
      // clip at array ends due to possible roundoff effects
      if (index < 0)         index = 0;
      if (index >= num_samp) index = num_samp - 1;
      if (j<pl_len)
	computedRespLevels[i][j] = sorted_samples[index];
      else
	computedRespLevels[i][j+bl_len] = sorted_samples[index];
    }
    for (j=0; j<bl_len; j++) { // beta -> z
      const Real& beta = requestedRelLevels[i][j];
      computedRespLevels[i][j+pl_len] = (cdfFlag) ?
	meanStats[i] - beta * stdDevStats[i] :
	meanStats[i] + beta * stdDevStats[i];
    }
  }
}


bool NonDSampling::rank_sort(const int& x, const int& y)
{ return rawData[x]<rawData[y]; }


void NonDSampling::
compute_correlations(const VariablesArray& vars_samples,
		     const ResponseArray&  resp_samples)
{
  // this method calculates four correlation matrices:
  // simple correlation coefficients, partial correlation coefficients
  // simple rank correlation coefficients, and partial rank correlation coeff.

  size_t num_obs = resp_samples.length();
  if (!num_obs) {
    Cerr << "Error: Number of samples must be nonzero in NonDSampling::"
         << "compute_correlations()." << endl;
    abort_handler(-1);
  }
  if (vars_samples.length() != num_obs) {
    Cerr << "Error: Mismatch in array lengths in NonDSampling::"
         << "compute_correlations()." << endl;
    abort_handler(-1);
  }
  bool compute_all = (numEpistemicUncVars) ? false : true;
  size_t i, j, k, rank_count, num_cv = vars_samples[0].cv(),
    num_fns = resp_samples[0].num_functions();

  //simple correlation coefficients
  int num_corr = num_cv + num_fns;
  size_t num_true_samples = 0, sample_count = 0;
  IntVector valid_sample(num_obs);

  for (j=0; j<num_obs; j++){
    for (k=0; k<num_fns; k++) {
      //any Nan or +/- Inf response will result in observation being dropped
      const Real& this_sample = resp_samples[j].function_value(k);
      valid_sample(j) = Isfinite(this_sample); 
      //Cout << this_sample << " valid " << valid_sample(j) << '\n';
    }
    if (valid_sample(j))
      num_true_samples++ ;
  }
  
  RealDenseMatrix total_data(num_corr,num_true_samples);
  for (i=0; i<num_corr; i++) {
    sample_count = 0;
    for (j=0; j<num_obs; j++) {
      if (valid_sample(j)) {
        total_data(i,sample_count) = (i < num_cv) ?
	  vars_samples[j].continuous_variable(i) :
	  resp_samples[j].function_value(i-num_cv);
        sample_count++;
      }
    }
  }
 
  //calculate simple rank correlation coeff
  if (compute_all)
    simple_corr(total_data, false, num_corr);
  else
    simple_corr(total_data, false, num_cv);

  //calculate partial correlation coeff
  partial_corr(total_data, false, num_cv);

  //simple rank correlations
  IntArray rank_col(num_true_samples);
  IntArray final_rank(num_true_samples);
  num_corr = num_cv + num_fns;
  rawData.reshape(num_true_samples);
  for (i=0; i<num_corr; i++) {
    rank_count = 0;
    for (j=0; j<num_obs; j++) {
      if (valid_sample(j)){
        rank_col[rank_count] = rank_count;
        rawData[rank_count] = (i < num_cv) ?
	  vars_samples[j].continuous_variable(i) :
	  resp_samples[j].function_value(i-num_cv);
	rank_count++;
      }
    }
    sort(rank_col.begin(),rank_col.end(),rank_sort);
    for (j=0; j<num_true_samples; j++)
      final_rank[rank_col[j]] = j;
    for (j=0; j<num_true_samples; j++)
      total_data(i,j) = 1.0*final_rank[j];
  }
  //calculate simple rank correlation coeff
  if (compute_all) 
    simple_corr(total_data, true, num_corr);
  else
    simple_corr(total_data, true, num_cv);

  //calculate partial rank correlation coeff
  partial_corr(total_data, true, num_cv);
}


void NonDSampling::
simple_corr(RealDenseMatrix& total_data, bool rank_on, const int& num_in)
{
  //this method calculates simple correlation coefficients from a matrix of data
  //num_corr is number of columns of total data
  //rank_on indicates if rank correlations should be calculated
  //num_in indicate if only pairs of correlations 
  //should be calculated between pairs of columns (num_in vs. num_corr-num_in) 
  //if num_in =  num_corr, correlations are calculated between all columns

  size_t i,j,k;
  int num_corr = total_data.numRows(), num_obs = total_data.numCols(),
    num_out = num_corr - num_in;

  if (num_corr == num_in) {
    if (rank_on)
      simpleRankCorr.shape(num_corr,num_corr);
    else
      simpleCorr.shape(num_corr,num_corr);
  }
  else {
    if (rank_on)
      simpleRankCorr.shape(num_in,num_out);
    else
      simpleCorr.shape(num_in,num_out);
  }

  RealDenseVector mean_column(num_corr, false),
    sumsquare_column(num_corr, false);

  //get means of each column
  for (i=0; i<num_corr; i++) {
    Real sum = 0.0;
    for (j=0; j<num_obs; j++)
      sum += total_data(i,j);
    mean_column(i) = sum/((Real)num_obs);
  }

  //subtract means from each value in each column
  for (i=0; i<num_corr; i++)
    for (j=0; j<num_obs; j++)
      total_data(i,j) -= mean_column(i);

  //calculate sum of squares for each column
  for (i=0; i<num_corr; i++) {
    Real sumsq = 0.0;
    for (j=0; j<num_obs; j++)
      sumsq += total_data(i,j)*total_data(i,j);
    sumsquare_column(i) = sqrt(sumsq);
  }

  //normalize the column values with the sumsquare term
  for (i=0; i<num_corr; i++)
    for (j=0; j<num_obs; j++)
      total_data(i,j) = (sumsquare_column(i)>1E-30)
                      ? total_data(i,j)/sumsquare_column(i) : 0.;

  //calculate simple correlation coeff, put in matrix
  if (num_corr == num_in) {
    for (i=0; i<num_corr; i++) {
      if (rank_on)
	simpleRankCorr(i,i) = 1.0;
      else
	simpleCorr(i,i) = 1.0;
    }
    for (i=0; i<num_corr; i++) {
      for (k=0; k<i; k++) {
	if (rank_on) {
	  simpleRankCorr(k,i) = 0.0;
	  for (j=0; j<num_obs; j++)
	    simpleRankCorr(i,k) += total_data(i,j)*total_data(k,j);
	}
	else {
	  simpleCorr(k,i) = 0.0;
	  for (j=0; j<num_obs; j++)
	    simpleCorr(i,k) += total_data(i,j)*total_data(k,j);
	}
      }
    }
  }
  else{//input/output case
    for (i=0; i<num_in; i++) {
      for (k=0; k<num_out; k++) {
	if (rank_on) {
	  simpleRankCorr(i,k) = 0.0;
	  for (j=0; j<num_obs; j++)
	    simpleRankCorr(i,k) += total_data(i,j)*total_data(k+num_in,j);
	}
	else {
	  simpleCorr(i,k) = 0.0;
	  for (j=0; j<num_obs; j++)
	    simpleCorr(i,k) += total_data(i,j)*total_data(k+num_in,j);
	}
      }
    }
  }
} 


void NonDSampling::
partial_corr(RealDenseMatrix& total_data, bool rank_on, const int& num_in)
{
  //this method calculates partial correlation coefficients from a
  //matrix of data.  Note that it is assumed that total data is
  //normalized (each value is the (original value-column value)/sumsqr col.)

  size_t i,j,k,m;
  int num_obs = total_data.numCols(), num_out = total_data.numRows() - num_in;
  RealDenseMatrix    total_pdata, partial_corr, partial_transpose, temp2_corr;
  RealSymDenseMatrix temp_corr;
  RealSpdDenseSolver corr_solve1;
  RealDenseSolver    corr_solve2;
  total_pdata.shape(num_in+1,num_obs);
  temp_corr.shape(num_in+1);
  partial_corr.shape(num_in+1,num_in+1);
  if (rank_on)
    partialRankCorr.shape(num_in, num_out);
  else
    partialCorr.shape(num_in, num_out);
  temp2_corr.shape(num_in+1,num_in+1);
  partial_transpose.shape(num_in+1,num_in+1);

  //start off by populating partial_corr
  //NOTE:  need to calculate partial correlations one response at a time
  for (i=0; i<num_in; i++)
    for (k=0; k<num_obs; k++)
      total_pdata(i,k) = total_data(i,k);

  for (m=0; m<num_out; m++) {
    //first finish populating data by adding one response at a time
    for (j=0; j<num_obs; j++)
      total_pdata(num_in,j) = total_data(num_in + m,j);
    for (i=0; i<num_in+1; i++) {
      for (k=0; k<=i; k++){
	temp_corr(i,k) = 0.0;
	if (k==i)
	  temp_corr(k,i) = 1.0;
	else {
	  for (j=0; j<num_obs; j++)
	    temp_corr(i,k) += total_pdata(i,j)*total_pdata(k,j);
	  temp_corr(k,i) = temp_corr(i,k);
	}
      }
    }
    corr_solve1.setMatrix( Teuchos::rcp(&temp_corr, false) );
    corr_solve1.factor();
    for (i=0; i<num_in+1; i++)
      for (k=0; k<=i; k++)
	temp2_corr(i,k) = (k <= i) ? temp_corr(i,k): 0.0;
    corr_solve2.setMatrix( Teuchos::rcp(&temp2_corr, false) );
    corr_solve2.invert();
    for (i=0; i<num_in+1; i++)
      for (k=0; k<num_in+1; k++)
	partial_transpose(k,i) = temp2_corr(i,k);
    partial_corr.multiply(Teuchos::NO_TRANS,Teuchos::NO_TRANS,1.0,
                          partial_transpose,temp2_corr,0.0);
    for (i=0; i<num_in; i++) {
      if (rank_on)
	partialRankCorr(i,m) = -partial_corr(i, num_in) /
	  sqrt(partial_corr(num_in, num_in) *
	       partial_corr(i, i));
      else
	partialCorr(i,m) = -partial_corr(i, num_in) /
	  sqrt(partial_corr(num_in, num_in) *
	       partial_corr(i, i));
    }
  }
}


void NonDSampling::update_final_statistics()
{
  //if (finalStatistics.is_null())
  //  initialize_final_statistics();

  size_t i, j, cntr = 0;
  if (numEpistemicUncVars) {
    for (i=0; i<numFunctions; i++) {
      finalStatistics.function_value(minValues[i], cntr++);
      finalStatistics.function_value(maxValues[i], cntr++);
    }
  }
  else {
    for (i=0; i<numFunctions; i++) {
      // final stats from compute_moments()
      if (!meanStats.empty() && !stdDevStats.empty()) {
	finalStatistics.function_value(meanStats[i],   cntr++);
	finalStatistics.function_value(stdDevStats[i], cntr++);
      }
      else
	cntr += 2;
      // final stats from compute_distribution_mappings()
      size_t rl_len = requestedRespLevels[i].length();
      for (j=0; j<rl_len; j++)
	switch (respLevelTarget) {
	case PROBABILITIES:
	  finalStatistics.function_value(computedProbLevels[i][j], cntr++);
	  break;
	case RELIABILITIES:
	  finalStatistics.function_value(computedRelLevels[i][j], cntr++);
	  break;
	case GEN_RELIABILITIES:
	  finalStatistics.function_value(computedGenRelLevels[i][j], cntr++);
	  break;
	}
      size_t pl_bl_gl_len = requestedProbLevels[i].length() +
	requestedRelLevels[i].length() + requestedGenRelLevels[i].length();
      for (j=0; j<pl_bl_gl_len; j++)
	finalStatistics.function_value(computedRespLevels[i][j], cntr++);
    }
  }
}


void NonDSampling::print_statistics(std::ostream& s) const
{
  if (numEpistemicUncVars) // output only min & max values in the epistemic case
    print_intervals(s);
  else {
    print_moments(s);
    if (totalLevelRequests)
      print_distribution_mappings(s);
  }
  print_correlations(s);
}


void NonDSampling::print_intervals(std::ostream& s) const
{
  const StringArray& resp_labels = iteratedModel.response_labels();

  s.setf(std::ios::scientific);
  s << setprecision(write_precision)
    << "\nMin and Max values for each response function:\n";
  for (size_t i=0; i<numFunctions; i++)
    s << resp_labels[i] << ":  Min = " << minValues[i]
      << "  Max = " << maxValues[i] << '\n';
}


void NonDSampling::print_moments(std::ostream& s) const
{
  const StringArray& resp_labels = iteratedModel.response_labels();

  s.setf(std::ios::scientific);
  s << setprecision(write_precision)
    << "\nMoments for each response function:\n";
  size_t i;
  for (i=0; i<numFunctions; i++) {
    s << resp_labels[i] << ":  Mean = " << meanStats[i]
      << "  Std. Dev. = " << stdDevStats[i] << "  Coeff. of Variation = ";
    if (fabs(meanStats[i]) > 1.e-25)
      s << stdDevStats[i]/meanStats[i] << '\n';
    else
      s << "Undefined\n";
  }

  if (numSamples > 1) {
    // output 95% confidence intervals as (,) interval
    s << "\n95% confidence intervals for each response function:\n";
    for (i=0; i<numFunctions; i++) {
      s << resp_labels[i] << ":  Mean = ( " << meanStats[i] - mean95CIDeltas[i]
	<< ", " << meanStats[i] + mean95CIDeltas[i] << " )";
#if defined(HAVE_BOOST)
      s << ", Std Dev = ( " << stdDev95CILowerBnds[i] << ", "
	<< stdDev95CIUpperBnds[i] << " )";
#endif // HAVE_BOOST
      s << '\n';
    }
  }
}


void NonDSampling::print_distribution_mappings(std::ostream& s) const
{
  const StringArray& resp_labels = iteratedModel.response_labels();

  // output CDF/CCDF probabilities resulting from binning or CDF/CCDF
  // reliabilities resulting from number of std devs separating mean & target
  s.setf(std::ios::scientific);
  s << setprecision(write_precision)
    << "\nProbabilities for each response function:\n";
  size_t i, j;
  for (i=0; i<numFunctions; i++) {
    if (!requestedRespLevels[i].empty() || !requestedProbLevels[i].empty() ||
	!requestedRelLevels[i].empty()  || !requestedGenRelLevels[i].empty()) {
      if (cdfFlag)
	s << "Cumulative Distribution Function (CDF) for ";
      else
	s << "Complementary Cumulative Distribution Function (CCDF) for ";
      s << resp_labels[i] << ":\n     Response Level  Probability Level  "
	<< "Reliability Index  General Rel Index\n     --------------  "
	<< "-----------------  -----------------  -----------------\n";
      size_t num_resp_levels = requestedRespLevels[i].length();
      for (j=0; j<num_resp_levels; j++) {
	s << "  " << setw(17) << requestedRespLevels[i][j] << "  ";
	switch (respLevelTarget) {
	case PROBABILITIES:
	  s << setw(17) << computedProbLevels[i][j]   << '\n'; break;
	case RELIABILITIES:
	  s << setw(36) << computedRelLevels[i][j]    << '\n'; break;
	case GEN_RELIABILITIES:
	  s << setw(55) << computedGenRelLevels[i][j] << '\n'; break;
	}
      }
      size_t num_prob_levels = requestedProbLevels[i].length();
      for (j=0; j<num_prob_levels; j++)
	s << "  " << setw(17) << computedRespLevels[i][j] << "  "
	  << setw(17) << requestedProbLevels[i][j] << '\n';
      size_t num_rel_levels = requestedRelLevels[i].length();
      for (j=0; j<num_rel_levels; j++)
	s << "  " << setw(17) << computedRespLevels[i][j+num_prob_levels]
	  << "  " << setw(36) << requestedRelLevels[i][j] << '\n';
      size_t num_gen_rel_levels = requestedGenRelLevels[i].length();
      for (j=0; j<num_gen_rel_levels; j++)
	s << "  " << setw(17)
	  << computedRespLevels[i][j+num_prob_levels+num_rel_levels] << "  "
	  << setw(55) << requestedGenRelLevels[i][j] << '\n';
    }
  }
}


void NonDSampling::print_correlations(std::ostream& s) const
{
  const StringArray& resp_labels = iteratedModel.response_labels();
  StringMultiArrayConstView cv_labels
    = iteratedModel.continuous_variable_labels();

  s.setf(std::ios::scientific);
  s << setprecision(5);

  // output correlation matrices
  size_t i, j; 
  bool print_all = (numEpistemicUncVars) ? false : true;
  int num_in_out = numContinuousVars + numFunctions;
  if (print_all && simpleCorr.numRows() == num_in_out &&
      simpleCorr.numCols() == num_in_out) {
    s << "\nSimple Correlation Matrix among all inputs and outputs:\n"
      << "             ";
    for (i=0; i<numContinuousVars; i++)
      s << setw(12) << cv_labels[i] << ' ';
    for (i=0; i<numFunctions; i++)
      s << setw(12) << resp_labels[i] << ' ';
    s << '\n';
    for (i=0; i<num_in_out; i++) {
      if (i < numContinuousVars)
	s << setw(12) << cv_labels[i] << ' ';
      else
	s << setw(12) << resp_labels[i-numContinuousVars] << ' ';
      for (j=0; j<=i; j++)
	s << setw(12) << simpleCorr(i,j) << ' ';
      s << '\n';
    }
  }
  else if (!print_all && simpleCorr.numRows() == numContinuousVars &&
	   simpleCorr.numCols() == numFunctions) {
    s << "\nSimple Correlation Matrix between input and output:\n"
      << "             ";
    for (j=0; j<numFunctions; j++)
      s << setw(12) << resp_labels[j] << ' ';
    s << '\n';
    for (i=0; i<numContinuousVars; i++) {
      s << setw(12) << cv_labels[i] << ' ';
      for (j=0; j<numFunctions; j++)
	s << setw(12) << simpleCorr(i,j) << ' ';
      s << '\n';
    }
  }

  if (partialCorr.numRows() == numContinuousVars &&
      partialCorr.numCols() == numFunctions) {
    s << "\nPartial Correlation Matrix between input and output:\n"
      << "             ";
    for (j=0; j<numFunctions; j++)
      s << setw(12) << resp_labels[j]<<' ';
    s << '\n';
    for (i=0; i<numContinuousVars; i++) {
      s << setw(12) << cv_labels[i]<<' ';
      for (j=0; j<numFunctions; j++)
	s << setw(12) << partialCorr(i,j) << ' ';
      s << '\n';
    }
  }

  if (print_all && simpleRankCorr.numRows() == num_in_out &&
      simpleRankCorr.numCols() == num_in_out) {
    s << "\nSimple Rank Correlation Matrix among all inputs and outputs:\n"
      << "             ";
    for (i=0; i<numContinuousVars; i++)
      s << setw(12) << cv_labels[i] << ' ';
    for (i=0; i<numFunctions; i++)
      s << setw(12) << resp_labels[i] << ' ';
    s << '\n';
    for (i=0; i<num_in_out; i++) {
      if (i<numContinuousVars)
	s << setw(12) << cv_labels[i] << ' ';
      else
	s << setw(12) << resp_labels[i-numContinuousVars] << ' ';
      for (j=0; j<=i; j++)
	s << setw(12) << simpleRankCorr(i,j) << ' ';
      s << '\n';
    }
  }
  else if (!print_all && simpleRankCorr.numRows() == numContinuousVars &&
	   simpleRankCorr.numCols() == numFunctions) {
    s << "\nSimple Rank Correlation Matrix between input and output:\n"
      << "             ";
    for (j=0; j<numFunctions; j++)
      s << setw(12) << resp_labels[j] << ' ';
    s << '\n';
    for (i=0; i<numContinuousVars; i++) {
      s << setw(12) << cv_labels[i] << ' ';
      for (j=0; j<numFunctions; j++)
	s << setw(12) << simpleRankCorr(i,j) << ' ';
      s << '\n';
    }
  }

  if (partialRankCorr.numRows() == numContinuousVars &&
      partialRankCorr.numCols() == numFunctions) {
    s << "\nPartial Rank Correlation Matrix between input and output:\n"
      << "             ";
    for (j=0; j<numFunctions; j++)
      s << setw(12) << resp_labels[j] << ' ';
    s << '\n';
    for (i=0; i<numContinuousVars; i++) {
      s << setw(12) << cv_labels[i] << ' ';
      for (j=0; j<numFunctions; j++)
	s << setw(12) << partialRankCorr(i,j) << ' ';
      s << '\n';
    }
  }
  s << setprecision(write_precision) << endl; // return to prev. precision
}

} // namespace Dakota

