// $Id: APPSPACK_Citizen_Listener.cpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_Listener.cpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_Listener.cpp
  \brief Implements APPSPACK::Citizen::Listener
*/

#include "APPSPACK_Citizen_Listener.hpp"
#include "APPSPACK_Float.hpp"
#include <time.h>

APPSPACK::Citizen::Listener::Listener(APPSPACK::Parameter::List& params_in,
                                      const APPSPACK::Constraints::Linear& constraints_in,
                                      APPSPACK::Combiner::Generic& combiner_in,
                                      const string name_in) :
  name(name_in + "(Listener)"),
  combiner(combiner_in),
  params(params_in.sublist(name_in)),
  resultsFile(params.getParameter("Results File", "listener.txt")),
  percentErrorTol(params_in.sublist("Mediator").getParameter("Percent Error", -1.0)),
  globalMin(params_in.sublist("Evaluator").getParameter("Known Global Minimum", dne())),
  precision(params.getParameter("Precision", 8)),
  evalCount(0),
  evalCountBest(0),
  canStop(false),
  tagBest(-1),
  nameEvalBest("undefined")  
{
  time(&startTime);
}

APPSPACK::Citizen::Listener::~Listener()
{
}

void APPSPACK::Citizen::Listener::preprocess()
{
  // Open output file.
  resultsStream.open(resultsFile.c_str());

  // Decide whether or not Listener can force a stop.
  if (exists(globalMin) && (percentErrorTol != -1))
    canStop = true;
}

void APPSPACK::Citizen::Listener::postprocess()
{
  // Close output file
  resultsStream.close();

  // Calculate runtime
  time_t endTime;
  time(&endTime);
  double runTime = difftime(endTime,startTime);

  
  cout.precision(precision);
  cout << "*** Listener Best Point Info ***" << endl;
  cout << "LISTENER: Best point found by: " << nameEvalBest << endl;
  cout << "LISTENER: Best point found at evaluation: " << evalCountBest << endl;
  cout << "LISTENER: fBest = ";
  fBest.leftshift(cout, precision);
  cout << endl;
  cout << "LISTENER: xBest = ";
  xBest.leftshift(cout, precision);
  cout << endl;
  cout << "LISTENER: tagBest = " << tagBest << endl;
  cout << "LISTENER: Best Point Found at Time = " << diffTimeBest << endl;
  if (exists(globalMin) && (fBest.size() > 0))
  {
    double pe = getPercentError(fBest[0]);
    cout << "LISTENER: Percent Error = " << pe << endl;
    cout << "LISTENER: Best Point Found? " << ((pe <= percentErrorTol)?'T':'F') << endl;
  }
  cout << "LISTENER: Total Run Time: " << runTime << endl;
  cout << "LISTENER: Total Eval Count: " << evalCount << endl;
}

void APPSPACK::Citizen::Listener::exchange(const APPSPACK::ConveyorList& R,
                                           APPSPACK::ConveyorList& Wnew,
                                           const map<string, vector<int> >& ownermap)
{
  // Record time immediately.
  time_t endTime;
  time(&endTime);
  double difTime = difftime(endTime,startTime);

  // First invert ownermap.
  map<int,string> tag2name;
  map<string, vector<int> >::const_iterator iter;
  for (iter = ownermap.begin(); iter != ownermap.end(); ++iter) 
  {
    const vector<int>& tlist = iter->second;
    for (int i = 0; i < tlist.size(); i++)
      tag2name[tlist[i]] = iter->first;    
  }

  // Now loop through evaluated points.
  vector<int> rtags;
  R.getTagList(rtags);
  list<APPSPACK::Point*>::const_iterator itr;
  for (itr = R.begin(); itr != R.end(); itr++)
  {
    string nameEval = "";
    Vector f = (*itr)->getVecF();
    Vector x = (*itr)->getX();
    Point::State state = (*itr)->getState();
    int tag = (*itr)->getTag();
    
    // Points coming from mediator should only be evaluated or cached.
    if ((state == Point::Infeasible) || (state == Point::Unevaluated))
    {
      cerr << "Error: Unexpected state.." << endl;
      cerr << "This point's state is " << state << endl;
      cerr << "Owned by " << tag2name[tag] << endl;
      throw "APPSPACK Error";
    }

    if (tag2name.find(tag) == tag2name.end())
    {
      cerr << "Error: failed to find point tag" << endl;
      throw "APPSPACK Error";
    }

    // If point is not Cached eval, process.
    if ( ((*itr)->getState() == Point::EvaluatedInsufficientDecrease) ||
         ((*itr)->getState() == Point::Evaluated) )
    {
      evalCount++;
      nameEval = tag2name[tag];

      // The first evaluated point will always be best.
      if ((fBest.size() == 0) || (combiner(f) < combiner(fBest)))
      {
        nameEvalBest = nameEval;
	nameEval += " newbest";
	xBest = x;
	fBest = f;
        evalCountBest = evalCount;
        tagBest = tag;
        diffTimeBest = difTime;
      }
      
      resultsStream.precision(precision);
      resultsStream << evalCount << ", " << combiner(f) << ", " << nameEval  
                    << ", tag=" << tag << ", f=[ ";
      resultsStream.precision(precision);
      for (int i = 0; i < f.size(); i++)
        resultsStream << f[i] << " ";
      resultsStream << "], x=[";
      for (int i = 0; i < x.size(); i++)
        resultsStream << x[i] <<  " ";
      resultsStream << "] time=[" << difTime << "]";
      if (exists(globalMin) && (f.size() > 0))
      {
        resultsStream << " pe=[" << getPercentError(f[0]) << "]";
      }
      resultsStream << endl;
    }
  }

  resultsStream << "======================================" << endl;
}

void APPSPACK::Citizen::Listener::vote(const APPSPACK::ConveyorList& W,
                                       const map<string, vector<int> >& ownermap,
                                       vector<int>& tagOrder)
{
}

bool APPSPACK::Citizen::Listener::worker()
{
  // Should never be called.
  cerr << "APPSPACK::Citizen::Listener::worker() should not be called" << endl;
  throw "APPSPACK Error";
}

APPSPACK::Citizen::State APPSPACK::Citizen::Listener::getState()
{
  if ((canStop == true) && (fBest.size() > 0))
  {
    double pe = getPercentError(fBest[0]);
    if (pe <= percentErrorTol)
    {
      cout << "Relative error: " << pe << ", Global Minimum: " << globalMin << endl;
      return APPSPACK::Citizen::MustStop;
    }
  }

  // Listener wishes to stop when everyone is ready.
  return APPSPACK::Citizen::StopRequest;
}

const string& APPSPACK::Citizen::Listener::getName() const
{
  return name;
}

double APPSPACK::Citizen::Listener::getPercentError(double fMin) const
{
  // Calculate the difference between the current minimum and the
  // global minimum.
  double pe = fMin - globalMin;

  // If we do *better* than the global min, then the percent error is zero.
  if (pe < 0)
    return 0.0;

  // Normalize the difference, except for cases where the globalmin is
  // small.
  if (fabs(globalMin) > 1.0e-4)
    pe = pe / fabs(globalMin);

  // Make it a percent.
  return 100 * pe;
}
