/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Inline methods of class GeneticAlgorithmSelector.

    NOTES:

        See notes of GeneticAlgorithmSelector.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Wed May 21 15:26:36 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the inline methods of the GeneticAlgorithmSelector class.
 */


/*
================================================================================
Includes
================================================================================
*/
#include <FitnessRecord.hpp>
#include <../Utilities/include/DesignGroup.hpp>
#include <../Utilities/include/DesignMultiSet.hpp>
#include <../Utilities/include/DesignGroupVector.hpp>







/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JU = JEGA::Utilities;

namespace JEGA {
    namespace Algorithms {





inline
bool
GeneticAlgorithmSelector::FitnessPred::operator ()(
    const JU::Design* des1,
    const JU::Design* des2
    ) const
{
    return this->_ftns.GetFitness(*des1) >
           this->_ftns.GetFitness(*des2);
}


inline
GeneticAlgorithmSelector::FitnessPred::FitnessPred(
    const FitnessRecord& ftns
    ) :
        _ftns(ftns)
{}


/*
================================================================================
Inline Mutators
================================================================================
*/








/*
================================================================================
Inline Accessors
================================================================================
*/








/*
================================================================================
Inline Public Methods
================================================================================
*/








/*
================================================================================
Inline Subclass Visible Methods
================================================================================
*/


template <typename CompT>
JU::DesignOFSortSet
GeneticAlgorithmSelector::SelectNBest(
    JU::DesignGroupVector& from,
    std::size_t n,
    CompT comp
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // if n is zero, we don't need to do anything at all.
    if(n == 0) return JU::DesignOFSortSet();

    // to make this easy, we will create a multimap of designs sorted using
    // the comparator and select the top n putting them in the return set.
    typedef std::multiset<JU::Design*, CompT> TopMap;

    TopMap theMap(comp);

    for(JU::DesignGroupVector::const_iterator git(from.begin());
        git!=from.end(); ++git)
    {
        const JU::DesignOFSortSet::const_iterator de((*git)->EndOF());
        for(JU::DesignOFSortSet::const_iterator dit((*git)->BeginOF());
            dit!=de; ++dit) theMap.insert(*dit);
    }

    // now go through and take the required number of Designs subject to the
    // size of the map.
    JU::DesignOFSortSet ret;

    std::size_t numSels = 0;
    for(typename TopMap::const_iterator it(theMap.begin());
        it!=theMap.end(); ++it)
    {
        ret.insert(*it);
        if(++numSels >= n) return ret;
    }

    return ret;
}







/*
================================================================================
Inline Private Methods
================================================================================
*/








/*
================================================================================
Inline Structors
================================================================================
*/








/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Algorithms
} // namespace JEGA
