// $Id: APPSPACK_Citizen_Random.hpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_Random.hpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_Random.hpp
  \brief Class definition for APPSPACK::Citizen_Random
*/

#ifndef APPSPACK_CITIZEN_RANDOM_HPP
#define APPSPACK_CITIZEN_RANDOM_HPP

#include "APPSPACK_Citizen_Interface.hpp"
#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Constraints_Linear.hpp"
#include "APPSPACK_Iterator.hpp"
#include "APPSPACK_List.hpp"

namespace APPSPACK
{

namespace Citizen
{

//! Generates a random feasible point every time a new best point is found.
/*! At every iteration that 1) Interface::exchange() is called, \e and
  2) a new best point is found, the method will generate a random
  (feasible) points using a positive linear combination of feasible
  search directions from the queue list and the evaluated points list
  respectively.  
  
  At every iteration, the best point found is stored in #bestPointPtr.
  Whenever a new best point is encountered on a call to Citizen::Random::exchange(),
  a new randomly generated point will be added to the queue.  This class votes
  seeking to place its point at the end of the queue by giving each random
  point that this generated a single vote. 

  The random point is generated as follows.  
  <ol>
  
  <li> Search directions are computed for the new best point using the
  \f$ {\cal B}(x_{\rm best}, \epsilon_{\rm best})\f$, where \f$\epsilon_{\rm best}\f$
  denotes step-size for the current best point stored by
  #bestPointPtr.
  
  <li> A random search direction is calculated from a positive linear combination
  of these search direction:
 
  \f[
  d_{\rm rand} = \sum_{i} w_i d_i, \text{ where } 
  w_i = {\rm min}(\alpha_i, \epsilon_{\rm best}) r_i 
  \f]
  
  where  
  - \f$\alpha_i\f$ equals the maximum feasible step all direction \f$d_i\f$
  - \f$r_i\f$ denotes a random number between 0 and 1.
  
  <li> The new random point added to the queue is then defined
  as 
  \f[
  x_{\rm rand} = x_{\rm best} + \alpha d_{\rm rand},
  \f]
  where \f$\alpha\f$ is used to ensure a feasible point was generated. 
  </ol>   
*/
class Random : public Interface
{
public:
  //! Constructor 
  Random(Parameter::List& params_in,
	 const Constraints::Linear& constraints_in,
	 Combiner::Generic& combiner_in,
	 const string name = "");
  
  //! Destructor
  ~Random();

  //! Derived. 
  virtual void preprocess();

  //! Derived
  virtual void postprocess();
  
  //! Generates a random feasible point whenever a new best point is found.
  /*! \see APPSPACK::Citizen::Random */
  virtual void exchange(const ConveyorList& R, ConveyorList& Wnew,
                        const map<string, vector<int> >& ownermap);

  //! Each point in W owned by this receives a single vote.
    virtual void vote(const ConveyorList& W, 
                      const map<string, vector<int> >& ownermap,
                      vector<int>& tagOrder);

  //! Always returns true.  Thus Random stops when solver stops.
  virtual Citizen::State getState();

  //! Returns Citizen's name.
  virtual const string& getName() const;

private:

  //! Returns pointer to new random Point transferring ownership to the invoker
  /*! Returns a feasible point constructed from a random
    weighted linear combinations of search directions.
    \see APPSPACK::Citize::Random
  */
  Point* getRandomPoint();

private:
  //! Constraints
  const Constraints::Linear& constraints;
  
  //! Parameters
  Parameter::List& params;

  //! The search directions 
  Directions directions;
  
  //! Pointer to the best trial point thus far
  Point* bestPointPtr;
 
  //! Trial points that have been evaluated evaluated
  List exchangeList;

  //! Name of citizen.
  string name;
};

}
}
#endif
