//*************************************************************************
// * Copyright (c) 2004   The Regents of the University of California.
// * Produced at the Lawrence Livermore National Laboratory.
// * Written by Charles Tong.
// * All rights reserved.
// *
// *  This program is free software: you can redistribute it and/or modify
// *  it under the terms of the GNU General Public License as published by
// *  the Free Software Foundation, either version 2 of the License, or
// *  (at your option) any later version.
// *
// *  This program is distributed in the hope that it will be useful,
// *  but WITHOUT ANY WARRANTY; without even the implied warranty of
// *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// *  GNU General Public License for more details.
// *  You should have received a copy of the GNU General Public License
// *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
// *
//*************************************************************************
// * functions for the Morris one-at-a-time class 
//*************************************************************************

#include <stdlib.h>
#include <math.h>
#include "MOATSampling.h"

// BMA: not sure why this wasn't included
#include <stdio.h>
// BMA: includes random number generator and glue to DAKOTA
#include "DakotaPsuade.H"
// BMA: do not need -- define away
#define PSUADE_randInit() 

//*************************************************************************
//* Constructor
//*------------------------------------------------------------------------

MOATSampling::MOATSampling(): DakotaPsuade()
{
   P_            = 4;
   nSamples_     = 0;
   nInputs_      = 0;
   sampleMatrix_ = NULL;
}

// BMA: alternate constructor with seed for use with DAKOTA
MOATSampling::MOATSampling(int seed, int partitions): DakotaPsuade(seed)
{
   P_            = partitions;
   nSamples_     = 0;
   nInputs_      = 0;
   sampleMatrix_ = NULL;
}

//********************************************************************
//* destructor 
//*-------------------------------------------------------------------

MOATSampling::~MOATSampling()
{
   if (sampleMatrix_ != NULL)
   {
      for (int ii = 0; ii < nSamples_; ii++)
         if (sampleMatrix_[ii] != NULL)
            delete [] sampleMatrix_[ii];
      delete [] sampleMatrix_;
      sampleMatrix_ = NULL;
   }
}

//*************************************************************************
//* initialize the sampling data
//*------------------------------------------------------------------------

int MOATSampling::initialize(sData &sdata)
{
   int    ii, ii2, rr, ss, nReps;
   double **BS, *ranges, ddata, delta;

   // ----------------------------------------------------------------
   // clean up 
   // ----------------------------------------------------------------

   nSamples_ = sdata.nSamples_;
   nInputs_  = sdata.nInputs_;
   if (nSamples_/(nInputs_+1) * (nInputs_+1) != nSamples_) 
   {
      printf("MOATSampling: nSamples should be multiples of nInputs+1.\n");
      printf("              nSamples reset to be 10*(nInputs+1).\n");
      nSamples_ = 10 * (nInputs_ + 1);
   }

   // ----------------------------------------------------------------
   // allocate matrices
   // ----------------------------------------------------------------

   BS = new double*[nSamples_];
   for (ii = 0; ii < nSamples_; ii++) BS[ii] = new double[nInputs_];

   // ----------------------------------------------------------------
   // preparation for generating the samples
   // ----------------------------------------------------------------

   sampleMatrix_ = new double*[nSamples_];
   for (ii = 0; ii < nSamples_; ii++)
      sampleMatrix_[ii] = new double[nInputs_];

   nReps  = nSamples_ / (nInputs_ + 1);
   ranges = new double[nInputs_];
   for (ii = 0;  ii < nInputs_;  ii++) 
      ranges[ii] = sdata.iUpperB_[ii] - sdata.iLowerB_[ii];

   // ----------------------------------------------------------------
   // generate a MOAT sample
   // ----------------------------------------------------------------

   for (rr = 0; rr < nReps; rr++) generate(&BS[rr*(nInputs_+1)]);

   // ----------------------------------------------------------------
   // convert the sample to application ranges
   // ----------------------------------------------------------------

   for (ss = 0; ss < nSamples_; ss+=(nInputs_+1))
   {
      for (ii = 0; ii <= nInputs_; ii++)
      {
         for (ii2 = 0; ii2 < nInputs_; ii2++)
         {
            ddata = BS[ss+ii][ii2];
            ddata = ddata * ranges[ii2] + sdata.iLowerB_[ii2];
            sampleMatrix_[ss+ii][ii2] = ddata;
         }
      }
   }

   // ----------------------------------------------------------------
   // clean up
   // ----------------------------------------------------------------

   delete [] ranges;
   for (ii = 0;  ii < nSamples_; ii++) delete [] BS[ii];
   delete [] BS;
   return 0;
}

//*************************************************************************
//* generate the BS matrix
//*------------------------------------------------------------------------

int MOATSampling::generate(double **BS)
{
   int    *permute, ss, ii, ii2, idata, imax;
   double **B, *D, *X, **B2, delta, ddata;

   // ----------------------------------------------------------------
   // initialize
   // ----------------------------------------------------------------

   PSUADE_randInit(); // initialize random number generator
   delta = P_ / ((double) (2*P_) - 2.0);

   // ----------------------------------------------------------------
   // build the B matrix, and allocate for the other
   // ----------------------------------------------------------------

   B = new double*[nInputs_+1];
   for (ii = 0; ii <= nInputs_; ii++)
   {
      B[ii] = new double[nInputs_];
      for (ii2 = 0; ii2 < ii; ii2++) B[ii][ii2] = 1.0;
      for (ii2 = ii; ii2 < nInputs_; ii2++) B[ii][ii2] = 0.0;
   }
   D = new double[nInputs_];
   X = new double[nInputs_];
   permute = new int[nInputs_];
   B2   = new double*[nInputs_+1];
   for (ii = 0; ii <= nInputs_; ii++) B2[ii] = new double[nInputs_];

   // ----------------------------------------------------------------
   // initialize the D matrix
   // ----------------------------------------------------------------

   for (ii = 0; ii < nInputs_; ii++)
   {
      D[ii] = PSUADE_drand(); // generate a random double in [0,1]
      if (D[ii] > 0.5) D[ii] = 1.0;
      else             D[ii] = -1.0;
   }

   // -----------------------------------------------------------------
   // initialize the X vector
   // ----------------------------------------------------------------

   imax = (P_ - 1) / 2;
   for (ii = 0; ii < nInputs_; ii++)
   {
       X[ii] = PSUADE_drand(); // generate a random double in [0,1]
       idata = (int) (X[ii] * (imax + 1));
       if (idata > imax) idata--;
       X[ii] = (double) idata / (double) (P_ - 1);
   }
   
   // ----------------------------------------------------------------
   // initialize the permutation matrix
   // ----------------------------------------------------------------

   // put [0:nInputs-1] in permute in random order
   generateRandomIvector(nInputs_, permute);
   
   // ----------------------------------------------------------------
   // form the BS matrix
   // ----------------------------------------------------------------

   for (ii = 0; ii <= nInputs_; ii++)
      for (ii2 = 0; ii2 < nInputs_; ii2++)
         B2[ii][ii2] = X[ii2]+delta/2*((B[ii][ii2]*2-1.0)*D[ii2]+1.0);
   for (ii = 0; ii <= nInputs_; ii++)
      for (ii2 = 0; ii2 < nInputs_; ii2++)
         BS[ii][ii2] = B2[ii][permute[ii2]];

   // ----------------------------------------------------------------
   // clean up
   // ----------------------------------------------------------------

   for (ii = 0;  ii <= nInputs_; ii++)
   {
      delete [] B[ii];
      delete [] B2[ii];
   }
   delete [] B;
   delete [] B2;
   delete [] D;
   delete [] X;
   delete [] permute;
   return 0;
}

