/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        Response
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "system_defs.h"
#include "DakotaBinStream.H"
#include "CtelRegExp.H"
#include "DakotaResponse.H"
#include "DakotaVariables.H"
#include "ProblemDescDB.H"
#include "data_io.h"

static const char rcsId[]="@(#) $Id: DakotaResponse.C 5802 2009-04-03 20:58:06Z mseldre $";


namespace Dakota {

/** Need a populated problem description database to build a meaningful
    Response object, so set the responseRep=NULL in default constructor
    for efficiency.  This then requires a check on NULL in the copy
    constructor, assignment operator, and destructor. */
Response::Response(): responseRep(NULL)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Response::Response(), responseRep = NULL" << endl;
#endif
}


Response::Response(const Variables& vars, const ProblemDescDB& problem_db):
  responseRep(new ResponseRep(vars, problem_db))
{
#ifdef REFCOUNT_DEBUG
  Cout << "Response::Response(vars, problem_db), responseRep referenceCount = "
       << responseRep->referenceCount << endl;
#endif
}


Response::Response(const ActiveSet& set): responseRep(new ResponseRep(set))
{
#ifdef REFCOUNT_DEBUG
  Cout << "Response::Response(set), responseRep referenceCount = "
       << responseRep->referenceCount << endl;
#endif
}


Response::Response(const Response& response)
{
  // Increment new (no old to decrement)
  responseRep = response.responseRep;
  if (responseRep) // Check for an assignment of NULL
    responseRep->referenceCount++;

#ifdef REFCOUNT_DEBUG
  Cout << "Response::Response(Response&)" << endl;
  if (responseRep)
    Cout << "responseRep referenceCount = " << responseRep->referenceCount
	 << endl;
#endif
}


Response Response::operator=(const Response& response)
{
  if (responseRep != response.responseRep) { // normal case: old != new
    // Decrement old
    if (responseRep) // Check for NULL
      if ( --responseRep->referenceCount == 0 ) 
	delete responseRep;
    // Assign and increment new
    responseRep = response.responseRep;
    if (responseRep) // Check for NULL
      responseRep->referenceCount++;
  }
  // else if assigning same rep, then do nothing since referenceCount
  // should already be correct

#ifdef REFCOUNT_DEBUG
  Cout << "Response::operator=(Response&)" << endl;
  if (responseRep)
    Cout << "responseRep referenceCount = " << responseRep->referenceCount
	 << endl;
#endif

  return *this;
}


Response::~Response()
{
  if (responseRep) { // Check for NULL
    --responseRep->referenceCount; // decrement
#ifdef REFCOUNT_DEBUG
    Cout << "responseRep referenceCount decremented to " 
         << responseRep->referenceCount << endl;
#endif
    if (responseRep->referenceCount == 0) {
#ifdef REFCOUNT_DEBUG
      Cout << "deleting responseRep" << endl;
#endif
      delete responseRep;
    }
  }
}


Response Response::copy() const
{
  Response response; // new handle: responseRep=NULL

  if (responseRep) {
    // allocate a responseRep body and copy data attributes
    response.responseRep = new ResponseRep();
    response.responseRep->functionValues    = responseRep->functionValues;
    response.responseRep->functionGradients = responseRep->functionGradients;
    response.responseRep->functionHessians  = responseRep->functionHessians;
    response.responseRep->responseActiveSet = responseRep->responseActiveSet;
    response.responseRep->functionLabels    = responseRep->functionLabels;
    response.responseRep->idResponses       = responseRep->idResponses;
  }

  return response;
}


/** The standard constructor used by Dakota::ModelRep. */
ResponseRep::
ResponseRep(const Variables& vars, const ProblemDescDB& problem_db):
  referenceCount(1), functionLabels(problem_db.get_dsa("responses.labels")),
  idResponses(problem_db.get_string("responses.id"))
{
  size_t num_fns,
    num_resp_fns = problem_db.get_sizet("responses.num_response_functions");
  if (num_resp_fns)
    num_fns = num_resp_fns;
  else {
    num_fns
      = problem_db.get_sizet("responses.num_nonlinear_inequality_constraints")
      + problem_db.get_sizet("responses.num_nonlinear_equality_constraints");
    size_t num_obj_fns
      = problem_db.get_sizet("responses.num_objective_functions"), num_lsq_terms
      = problem_db.get_sizet("responses.num_least_squares_terms");
    if (num_obj_fns)
      num_fns += num_obj_fns;
    else if (num_lsq_terms)
      num_fns += num_lsq_terms;
  }
  if (num_fns == 0) {
    Cerr << "Error: total number of response functions must be nonzero."<< endl;
    abort_handler(-1);
  }

  // the derivative arrays must accomodate either active or inactive variables,
  // but the default is active variables.  Derivative arrays are resized if a
  // DVV of different length is set within responseActiveSet.
  size_t num_params = vars.cv();

  // Resize & initialize response data
  // Conserve memory by checking DB info prior to sizing grad/hessian arrays
  bool grad_flag = (problem_db.get_string("responses.gradient_type") == "none")
                 ? false : true;
  bool hess_flag = (problem_db.get_string("responses.hessian_type")  == "none")
                 ? false : true;
  functionValues.resize(num_fns);
  functionValues = 0.;
  short asv_value = 1;
  if (grad_flag) {
    asv_value += 2;
    functionGradients.reshape_2d(num_fns, num_params);
    functionGradients = 0.;
  }
  if (hess_flag) {
    asv_value += 4;
    functionHessians.resize(num_fns);
    for (size_t i=0; i<num_fns; i++) {
      functionHessians[i].reshape(num_params);
      functionHessians[i] = 0.;
    }
  }

  // set up the response ActiveSet.  This object is copied by Iterator for its
  // initial activeSet and used by Model for MPI buffer size estimation.
  //responseActiveSet.reshape(num_fns, num_params);
  ShortArray asv(num_fns, asv_value);
  responseActiveSet.request_vector(asv);
  responseActiveSet.derivative_vector(vars.continuous_variable_ids());
}


/** Used for building a response object of the correct size on the
    fly (e.g., by slave analysis servers performing execute() on a
    local_response).  functionLabels is not needed for this purpose
    since it's not passed in the MPI send/recv buffers. However,
    NPSOLOptimizer's user-defined functions option uses this
    constructor to build bestResponse and bestResponse needs
    functionLabels for I/O, so construction of functionLabels has
    been added. */
ResponseRep::ResponseRep(const ActiveSet& set): referenceCount(1),
  functionValues(set.request_vector().length()), responseActiveSet(set)
{
  // Set flags according to asv content.
  const ShortArray& asv = set.request_vector();
  size_t i, num_fns = asv.length(),
    num_params = set.derivative_vector().length();
  bool grad_flag = false, hess_flag = false;
  for (i=0; i<num_fns; i++) {
    if (asv[i] & 2)
      grad_flag = true;
    if (asv[i] & 4)
      hess_flag = true;
  }

  // Reshape functionGradients and functionHessians according to content of asv
  if (grad_flag) {
    functionGradients.reshape_2d(num_fns, num_params);
    functionGradients = 0.;
  }
  if (hess_flag) {
    functionHessians.resize(num_fns);
    for (i=0; i<num_fns; i++) {
      functionHessians[i].reshape(num_params);
      functionHessians[i] = 0.;
    }
  }

  // Build a default functionLabels array (currently only used for
  // bestResponse by NPSOLOptimizer's user-defined functions option).
  functionLabels.resize(num_fns);
  build_labels(functionLabels, "f");
}


// I/O Notes:

// Bitwise AND used in ASV checks in read/write fns: the '&' operator does a bit
// by bit AND operation.  For example, 7 & 4 does a bitwise AND on 111 and 100
// and returns 100 or 4.  It is not necessary to verify that, for example,
// (asv[i] & 4) == 4 since (asv[i] & 4) will be nonzero/true only if the 4 bit
// is present in asv[i] (the "== value" can be omitted so long as only the
// presence of a single bit is of interest).

// Matrix sizing: since fn gradients and hessians are of size 0 if their type is
// "none", all grad loops are run from 0 to num rows and all Hessian loops are
// run from 0 to num hessians, rather than from 0 to num_fns.  For the number
// of derivative variables, the length of responseActiveSet.derivative_vector()
// is used rather than the array sizes since the matrices are originally sized
// according to the maximal case and all entries may not be used.

// When creating functionGradients/functionHessians on the fly in read() input
// functions, grad_flag/hess_flag are used rather than inferring sizes from
// responseActiveSet since it is safer to match the sizing of the response
// object that was written (originally sized by the Response constructor) than
// to base sizing on the particular constent of an asv (which can vary from
// eval. to eval.).  For example, assigning empty gradient/Hessian matrices to
// Model::currentResponse in a restart operation would cause problems.


/** ASCII version of read needs capabilities for capturing data omissions or
    formatting errors (resulting from user error or asynch race condition) and
    analysis failures (resulting from nonconvergence, instability, etc.). */
void ResponseRep::read(istream& s)
{
  // Failure capturing:
  // The read operator detects analysis failure through reading a failure 
  // string returned by the user's output_filter/analysis_driver.  This string
  // must occur at the beginning of the file, even if some portion of the 
  // results were successfully computed prior to failure.  Other mechanisms for
  // denoting analysis failure that were considered included (1) use of an 
  // empty file (disadvantage: ambiguity with asynch race condition) and (2) 
  // use of a success/fail string (disadvantage: would require addition of 
  // "success" field to non-failure output).  The current approach avoids these
  // problems by doing a read of 4 characters.  If these characters are not 
  // "fail" or "FAIL" then it resets the stream pointer to the beginning.  This
  // works for ifstreams, but may be problematic for other istreams.  Once 
  // failure is detected, an exception of type int is thrown (to distinguish 
  // from String exceptions) that will be caught after try{ execute() }.
  // NOTE: s.peek() triggering on 'f' or 'F' would be another possibility.
  // NOTE: reading the first token using "s >> fail_string;" should work for a
  //       file having less than four characters.
  size_t i;
  int fail_code = 1;
  char fail_chars[4];
  String fail_string("fail");
  // Old version failed for fewer than 4 characters in results file:
  //s >> fail_chars[0] >> fail_chars[1] >> fail_chars[2] >> fail_chars[3];
  for (i=0; i<4; i++) {
    s >> fail_chars[i];
    //Cout << "fail_char[" << i << "] = " << fail_chars[i] << endl;
    if (tolower(fail_chars[i]) != fail_string[i]) { // FAIL, Fail, fail, etc.
      fail_code = 0; // No failure communicated from results file 
      s.seekg(0); // Reset stream to beginning
      break; // exit for loop
    }
  }
  if (fail_code) {
    //Cerr << "Failure captured with string = " << fail_chars << endl;
    throw fail_code;
  }

  // Destroy old values and set to zero (so that else assignments are not needed
  // below). The arrays have been properly sized by the ResponseRep constructor.
  reset();

  // Get fn. values as governed by ASV requests
  String token;
  CtelRegexp reg_exp("[\\+-]?[0-9]*\\.?[0-9]+\\.?[0-9]*[eEdD]?[\\+-]?[0-9]*|[Nn][Aa][Nn]|[\\+-]?[Ii][Nn][Ff]");
  const ShortArray& asv = responseActiveSet.request_vector();
  size_t num_fns = asv.length();
  for (i=0; i<num_fns; i++) {
    if (asv[i] & 1) { // & 1 masks off 2nd and 3rd bit
      if (s) { // get value
	s >> token;
	// On RHEL 4, strtod preserves Inf/NaN, but atof doesn't
	//Cout << "Debug read: token = " << token << '\n';
	//Cout << "Debug read: atof of token = " << atof(token) << '\n';
	//Cout << "Debug read: strtod of token = " << strtod(token, NULL)<<'\n';
	functionValues[i] = atof(token); // handles NaN and +/-Inf
      }
      else {
        char err[80];
        sprintf(err, "At EOF: insufficient data for functionValue %d", i+1);
        throw String(err);
      }
      if (s) { // get optional tag
	//s.ignore(256, '\n'); // simple soln., but requires consistent '\n'
        int pos = s.tellg(); // save stream pos prior to token extraction
        s >> token; // get next field (may be a tag or a number)
        // Check to see if token matches the pattern (see CtelRegExp class docs)
	// of a numerical value (including +/-Inf and NaN) or of the beginning
	// of a gradient/Hessian block.  If it does, then rewind the stream.
        if ( !token.empty() &&
	     ( token[(size_t)0] == '[' || token == reg_exp.match(token) ) )
          s.seekg(pos); // token is not a tag, rewind
        // else field was properly extracted as a tag
      }
    }
  }

  // Get function gradients as governed by ASV requests
  // For brackets, chars are used rather than token strings to allow optional
  // white space between brackets and values.
  char l_bracket, r_bracket; // eat white space and grab 1 character
  char l_brackets[2], r_brackets[2]; // eat white space and grab 2 characters
  // l_bracket = '[' = 1 character; l_brackets = "[[" =2 characters
  size_t nr = functionGradients.num_rows();
  for (i=0; i<nr; i++) { // prevent loop if functionGradients not sized
    if (asv[i] & 2) { // & 2 masks off 1st and 3rd bit
      if (s)
        s >> l_bracket;
      else {
        char err[80];
        sprintf(err, "At EOF: insufficient data for functionGradient %d", i+1);
        throw String(err);
      }
      functionGradients.read_row_vector(s, i);
      if (s)
        s >> r_bracket;
      else {
        char err[80];
        sprintf(err, "At EOF: insufficient data for functionGradient %d", i+1);
        throw String(err);
      }
      if (l_bracket != '[' || r_bracket != ']') {
        char err[80];
        sprintf(err, "Response format error with functionGradient %d", i+1);
        throw String(err);
      }
    }
  }

  // Get function Hessians as governed by ASV requests
  size_t nh = functionHessians.length();
  for (i=0; i<nh; i++) { // prevent loop if functionHessians not sized
    if (asv[i] & 4) { // & 4 masks off 1st and 2nd bit
      if (s)
        s >> l_brackets[0] >> l_brackets[1];
      else {
        char err[80];
        sprintf(err, "At EOF: insufficient data for functionHessian %d", i+1);
        throw String(err);
      }
      // wjbVERIFY: functionHessians[i].read(s);
      Dakota::read_data(s, functionHessians[i]);
      if (s)
        s >> r_brackets[0] >> r_brackets[1];
      else {
        char err[80];
        sprintf(err, "At EOF: insufficient data for functionHessian %d", i+1);
        throw String(err);
      }
      if ((l_brackets[0] != '[' || l_brackets[1] != '[')  ||
	  (r_brackets[0] != ']' || r_brackets[1] != ']')) {
        char err[80];
        sprintf(err, "Response format error with functionHessian %d", i+1);
        throw String(err);
      }
    }
  }
}


/** ASCII version of write. */
void ResponseRep::write(ostream& s) const
{
  const ShortArray& asv = responseActiveSet.request_vector();
  const UIntArray&  dvv = responseActiveSet.derivative_vector();
  size_t i, num_fns = asv.length();
  bool deriv_flag = false;
  for (i=0; i<num_fns; i++)
    if (asv[i] & 6)
      { deriv_flag = true; break; }

  // Write ASV/DVV information
  s << "Active set vector = { ";
  asv.write_annotated(s, false);
  if (deriv_flag) { // dvv != vars.continuous_variable_ids() ?,
                    // outputLevel > NORMAL_OUTPUT ?
    s << "} Deriv vars vector = { ";
    dvv.write_annotated(s, false);
    s << "}\n";
  }
  else
    s << "}\n";

  // Make sure a valid set of functionLabels exists. This has been a problem
  // since there is no way to build these labels in the default Response
  // constructor (used by lists & vectors of Response objects).
  if (functionLabels.length() != num_fns) {
    Cerr << "Error with functionLabels in ResponseRep::write." << endl;
    abort_handler(-1);
  }

  // Write the function values if present
  for (i=0; i<num_fns; i++)
    if (asv[i] & 1) // & 1 masks off 2nd and 3rd bit
      s << "                     " << setw(write_precision+7) 
        << functionValues[i] << ' ' << functionLabels[i] << '\n';

  // Write the function gradients if present
  size_t nr = functionGradients.num_rows();
  for (i=0; i<nr; i++) {
    if (asv[i] & 2) { // & 2 masks off 1st and 3rd bit
      functionGradients.write_row_vector(s, i, true, true, false);
      s << functionLabels[i] << " gradient\n";
    }
  }

  // Write the function Hessians if present
  size_t nh = functionHessians.length();
  for (i=0; i<nh; i++) {
    if (asv[i] & 4) { // & 4 masks off 1st and 2nd bit
      Dakota::write_data(s, functionHessians[i], true, true, false);
      s << functionLabels[i] << " Hessian\n";
    }
  }
  s << endl;
}


/** read_annotated() is used for neutral file translation of restart files.
    Since objects are built solely from this data, annotations are used.
    This version closely mirrors the BiStream version. */
void ResponseRep::read_annotated(istream& s)
{
  // Read sizing data
  size_t i, num_fns, num_params;
  bool grad_flag, hess_flag;
  s >> num_fns >> num_params >> grad_flag >> hess_flag;

  // Reshape and read responseActiveSet and functionLabels.
  responseActiveSet.reshape(num_fns, num_params);
  functionLabels.resize(num_fns);
  s >> responseActiveSet >> functionLabels;

  // reshape response arrays and reset all data to zero
  reshape(num_fns, num_params, grad_flag, hess_flag);
  reset();

  // Get fn. values as governed by ASV requests
  const ShortArray& asv = responseActiveSet.request_vector();
  for (i=0; i<num_fns; i++)
    if (asv[i] & 1) // & 1 masks off 2nd and 3rd bit
      s >> functionValues[i];

  // Get function gradients as governed by ASV requests
  for (i=0; i<num_fns; i++)
    if (asv[i] & 2) // & 2 masks off 1st and 3rd bit
      functionGradients.read_row_vector(s, i);

  // Get function Hessians as governed by ASV requests
  for (i=0; i<num_fns; i++)
    if (asv[i] & 4) // & 4 masks off 1st and 2nd bit
      Dakota::read_data(s, functionHessians[i]);
}


/** write_annotated() is used for neutral file translation of restart files.
    Since objects need to be build solely from this data, annotations are used.
    This version closely mirrors the BoStream version, with the exception of
    the use of white space between fields. */
void ResponseRep::write_annotated(ostream& s) const
{
  const ShortArray& asv = responseActiveSet.request_vector();
  size_t i, num_fns = asv.length(),
    num_params = responseActiveSet.derivative_vector().length();
  bool grad_flag = (functionGradients.empty()) ? false : true;
  bool hess_flag = (functionHessians.empty())  ? false : true;

  // Write ResponseRep sizing data
  s << num_fns   << ' ' << num_params << ' '
    << grad_flag << ' ' << hess_flag  << ' ';

  // Write responseActiveSet and functionLabels.  Don't separately annotate
  // arrays with sizing data since ResponseRep handles this all at once.
  responseActiveSet.write_annotated(s);
  functionLabels.write_annotated(s, false);

  // Write the function values if present
  for (i=0; i<num_fns; i++)
    if (asv[i] & 1) // & 1 masks off 2nd and 3rd bit
      s << functionValues[i] << ' ';

  // Write the function gradients if present
  for (i=0; i<num_fns; i++)
    if (asv[i] & 2) // & 2 masks off 1st and 3rd bit
      functionGradients.write_row_vector(s, i, false, false, false);

  // Write the function Hessians if present
  for (i=0; i<num_fns; i++)
    if (asv[i] & 4) // & 4 masks off 1st and 2nd bit
      Dakota::write_data(s, functionHessians[i], false, false, false);
}


/** read_tabular is used to read functionValues in tabular format.  It
    is currently only used by ApproximationInterfaces in reading samples
    from a file.  There is insufficient data in a tabular file to build
    complete response objects; rather, the response object must be
    constructed a priori and then its functionValues can be set. */
void ResponseRep::read_tabular(std::istream& s)
{
  // Read the function values, regardless of ASV.  Since this is a table
  // format, there must be a field read even when data is inactive.

  // read_tabular_data(istream) continuously monitors for EOF
  // WJB - ToDo - consult with MSE, but for now, impl fnVal read_tabular here!? (see the write_tabular below) read_data_tabular(s, functionValues);
  //      ALSO, why is the asv NOT used for read (it is for the write step!?)
  //const ShortArray& asv = responseActiveSet.request_vector();
  size_t i, num_fns = functionValues.length();
  for (i=0; i<num_fns; ++i) {
    if (s)
      s >> functionValues[i];
    else {
      char err_msg[80];
      sprintf(err_msg,
              "At EOF: insufficient tabular data for SerialDenseVector[%d]", i);      throw String(err_msg);
    }
  }
}


/** write_tabular is used for output of functionValues in a tabular
    format for convenience in post-processing/plotting of DAKOTA results. */
void ResponseRep::write_tabular(std::ostream& s) const
{
  // Set stream format
  s << std::setprecision(10) << std::resetiosflags(std::ios::floatfield);

  // Print a field for each of the function values, even if inactive (since
  // this is a table and the header associations must be preserved).  Dropouts
  // (inactive data) could be printed as 0's (the inactive value), "N/A", -9999,
  // a blank field, etc.  Using either "N/A" or a blank field gives the correct
  // meaning visually, but both cause problems with data import.
  size_t i, num_fns = functionValues.length();
  const ShortArray& asv = responseActiveSet.request_vector();
  for (i=0; i<num_fns; i++) {
    if (asv[i] & 1)
      s << setw(14) << functionValues[i] << ' ';
    else
      s << "               "; // blank field for inactive data
  }
  s << std::endl; // table row completed
}


/** Binary version differs from ASCII version in 2 primary ways:
    (1) it lacks formatting.
    (2) the Response has not been sized a priori.  In reading data from
        the binary restart file, a ParamResponsePair was constructed with
        its default constructor which called the Response default 
        constructor.  Therefore, we must first read sizing data and resize 
        all of the arrays. */
void ResponseRep::read(BiStream& s)
{
  size_t i, num_fns, num_params;
  bool grad_flag, hess_flag;

  // Read sizing data, responseActiveSet, and functionLabels
  s >> num_fns >> num_params >> grad_flag >> hess_flag
    >> responseActiveSet >> functionLabels; // reshape not needed

  // reshape response arrays and reset all data to zero
  reshape(num_fns, num_params, grad_flag, hess_flag);
  reset();

  // Get fn. values as governed by ASV requests
  const ShortArray& asv = responseActiveSet.request_vector();
  for (i=0; i<num_fns; i++)
    if (asv[i] & 1) // & 1 masks off 2nd and 3rd bit
      s >> functionValues[i];

  // Get function gradients as governed by ASV requests
  for (i=0; i<num_fns; i++)
    if (asv[i] & 2) // & 2 masks off 1st and 3rd bit
      functionGradients.read_row_vector(s, i);

  // Get function Hessians as governed by ASV requests
  for (i=0; i<num_fns; i++)
    if (asv[i] & 4) // & 4 masks off 1st and 2nd bit
      Dakota::read_data(s, functionHessians[i]);
}


/** Binary version differs from ASCII version in 2 primary ways:
    (1) It lacks formatting.
    (2) In reading data from the binary restart file, ParamResponsePairs are
        constructed with their default constructor which calls the Response
        default constructor.  Therefore, we must first write sizing data so
        that ResponseRep::read(BoStream& s) can resize the arrays. */
void ResponseRep::write(BoStream& s) const
{
  const ShortArray& asv = responseActiveSet.request_vector();
  size_t i, num_fns = asv.length(),
    num_params = responseActiveSet.derivative_vector().length();
  bool grad_flag = (functionGradients.empty()) ? false : true;
  bool hess_flag = (functionHessians.empty())  ? false : true;

  // Write sizing data, responseActiveSet, and functionLabels
  s << num_fns << num_params << grad_flag << hess_flag
    << responseActiveSet << functionLabels;

  // Write the function values if present
  for (i=0; i<num_fns; i++)
    if (asv[i] & 1) // & 1 masks off 2nd and 3rd bit
      s << functionValues[i];

  // Write the function gradients if present
  for (i=0; i<num_fns; i++)
    if (asv[i] & 2) // & 2 masks off 1st and 3rd bit
      functionGradients.write_row_vector(s, i);

  // Write the function Hessians if present
  for (i=0; i<num_fns; i++)
    if (asv[i] & 4) // & 4 masks off 1st and 2nd bit
      Dakota::write_data(s, functionHessians[i]);
}


/** UnpackBuffer version differs from BiStream version in the omission
    of functionLabels.  Master processor retains labels and interface ids
    and communicates asv and response data only with slaves. */
void ResponseRep::read(MPIUnpackBuffer& s)
{
  size_t i, num_fns, num_params;
  bool grad_flag, hess_flag;

  // Read sizing data and responseActiveSet (reshape not needed)
  s >> num_fns >> num_params >> grad_flag >> hess_flag
    >> responseActiveSet; // >> functionLabels;

  // reshape response arrays and reset all data to zero
  reshape(num_fns, num_params, grad_flag, hess_flag);
  reset();

  // Get fn. values as governed by ASV requests
  const ShortArray& asv = responseActiveSet.request_vector();
  for (i=0; i<num_fns; i++)
    if (asv[i] & 1) // & 1 masks off 2nd and 3rd bit
      s >> functionValues[i];

  // Get function gradients as governed by ASV requests
  for (i=0; i<num_fns; i++)
    if (asv[i] & 2) // & 2 masks off 1st and 3rd bit
      functionGradients.read_row_vector(s, i);

  // Get function Hessians as governed by ASV requests
  for (i=0; i<num_fns; i++)
    if (asv[i] & 4) // & 4 masks off 1st and 2nd bit
      Dakota::read_data(s, functionHessians[i]);
}


/** MPIPackBuffer version differs from BoStream version only in the
    omission of functionLabels.  The master processor retains labels
    and ids and communicates asv and response data only with slaves. */
void ResponseRep::write(MPIPackBuffer& s) const
{
  const ShortArray& asv = responseActiveSet.request_vector();
  size_t i, num_fns = asv.length(),
    num_params = responseActiveSet.derivative_vector().length();
  bool grad_flag = (functionGradients.empty()) ? false : true;
  bool hess_flag = (functionHessians.empty())  ? false : true;

  // Write sizing data and responseActiveSet
  s << num_fns << num_params << grad_flag << hess_flag
    << responseActiveSet; // << functionLabels;

  // Write the function values if present
  for (i=0; i<num_fns; i++)
    if (asv[i] & 1) // & 1 masks off 2nd and 3rd bit
      s << functionValues[i];

  // Write the function gradients if present
  for (i=0; i<num_fns; i++)
    if (asv[i] & 2) // & 2 masks off 1st and 3rd bit
      functionGradients.write_row_vector(s, i);

  // Write the function Hessians if present
  for (i=0; i<num_fns; i++)
    if (asv[i] & 4) // & 4 masks off 1st and 2nd bit
      Dakota::write_data(s, functionHessians[i]);
}


int ResponseRep::data_size()
{
  // return the number of doubles active in response for sizing double* 
  // response_data arrays passed into read_data and write_data
  int size = 0;
  const ShortArray& asv = responseActiveSet.request_vector();
  size_t i, num_fns = functionValues.length(),
    grad_size = responseActiveSet.derivative_vector().length(),
    hess_size = grad_size*grad_size; // symmetry is not exploited
  for (i=0; i<num_fns; i++) {
    if (asv[i] & 1)
      size++;
    if (asv[i] & 2)
      size += grad_size;
    if (asv[i] & 4)
      size += hess_size;
  }
  return size;
}


void ResponseRep::read_data(double* response_data)
{
  // read from an incoming double* array
  const ShortArray& asv = responseActiveSet.request_vector();
  size_t i, j, k, cntr = 0, num_fns = functionValues.length(),
    num_params = responseActiveSet.derivative_vector().length();
  for (i=0; i<num_fns; i++)
    if (asv[i] & 1)
      functionValues[i] = response_data[cntr++];
  num_fns = functionGradients.num_rows();
  for (i=0; i<num_fns; i++)
    if (asv[i] & 2)
      for (j=0; j<num_params; j++)
        functionGradients[i][j] = response_data[cntr++];
  num_fns = functionHessians.size();
  for (i=0; i<num_fns; i++)
    if (asv[i] & 4)
      for (j=0; j<num_params; j++)
        for (k=0; k<num_params; k++) // symmetry is not exploited
          functionHessians[i](j,k) = response_data[cntr++];
}


void ResponseRep::write_data(double* response_data)
{
  // write to an incoming double* array
  const ShortArray& asv = responseActiveSet.request_vector();
  size_t i, j, k, cntr = 0, num_fns = functionValues.length(),
    num_params = responseActiveSet.derivative_vector().length();
  for (i=0; i<num_fns; i++)
    if (asv[i] & 1)
      response_data[cntr++] = functionValues[i];
  num_fns = functionGradients.num_rows();
  for (i=0; i<num_fns; i++)
    if (asv[i] & 2)
      for (j=0; j<num_params; j++)
        response_data[cntr++] = functionGradients[i][j];
  num_fns = functionHessians.size();
  for (i=0; i<num_fns; i++)
    if (asv[i] & 4)
      for (j=0; j<num_params; j++)
        for (k=0; k<num_params; k++) // symmetry is not exploited
          response_data[cntr++] = functionHessians[i](j,k);
}


void ResponseRep::overlay(const Response& response)
{
  // add incoming response to functionValues/Gradients/Hessians
  const ShortArray& asv = responseActiveSet.request_vector();
  size_t i, j, k, num_fns = functionValues.length(),
    num_params = responseActiveSet.derivative_vector().length();
  const RealDenseVector& partial_fn_vals  = response.function_values();
  const RealMatrix& partial_fn_grads = response.function_gradients();
  const RealSymDenseMatrixArray& partial_fn_hessians = response.function_hessians();
  for (i=0; i<num_fns; i++)
    if (asv[i] & 1)
      functionValues[i] += partial_fn_vals[i];
  num_fns = functionGradients.num_rows();
  for (i=0; i<num_fns; i++)
    if (asv[i] & 2)
      for (j=0; j<num_params; j++)
        functionGradients[i][j] += partial_fn_grads[i][j];
  num_fns = functionHessians.size();
  for (i=0; i<num_fns; i++)
    if (asv[i] & 4)
      for (j=0; j<num_params; j++)
        for (k=0; k<num_params; k++)
          functionHessians[i](j,k) += partial_fn_hessians[i](j,k);
}


/** Copy function values/gradients/Hessians data _only_.  Prevents unwanted
    overwriting of responseActiveSet, functionLabels, etc.  Also, care is
    taken to account for differences in derivative variable matrix sizing. */
void ResponseRep::
copy_results(const RealDenseVector& source_fn_vals,
	     const RealMatrix& source_fn_grads,
	     const RealSymDenseMatrixArray& source_fn_hessians,
	     const ActiveSet& source_set)
{
  // current response data
  const ShortArray& asv = responseActiveSet.request_vector();
  size_t i, j, k, num_fns = asv.length(), 
    num_params = responseActiveSet.derivative_vector().length();
  bool grad_flag = false, hess_flag = false;
  for (i=0; i<num_fns; i++) {
    if (asv[i] & 2)
      grad_flag = true;
    if (asv[i] & 4)
      hess_flag = true;
  }

  // verify that incoming data is of sufficient size
  if (source_set.request_vector().length() < num_fns) {
    Cerr << "Error: insufficient number of response functions to copy "
	 << "response results in ResponseRep::copy_results()." << endl;
    abort_handler(-1);
  }
  if ( (grad_flag || hess_flag) && 
       source_set.derivative_vector().length() < num_params) {
    Cerr << "Error: insufficient number of derivative variables to copy "
	 << "response results in ResponseRep::copy_results()." << endl;
    abort_handler(-1);
  }

  // for functionValues, using operator= does not allow vector size to differ.
  // for functionGradients/functionHessians, using operator= does not allow
  // num_fns or derivative variable matrix sizing to differ.  This includes the
  // extreme case in which the incoming derivative arrays may have zero size
  // (e.g., this can happen in a multilevel strategy where a captured duplicate
  // came from the evals of a previous nongradient-based algorithm).
  //functionValues    = source_fn_vals;
  //functionGradients = source_fn_grads;
  //functionHessians  = source_fn_hessians;

  for (i=0; i<num_fns; i++)
    if (asv[i] & 1) // assign each entry since size may differ
      functionValues[i] = source_fn_vals[i];

  // copy source_fn_grads and source_fn_hessians.  For now, a mapping from
  // responseActiveSet.derivative_vector() to source_set.derivative_vector()
  // is _not_ used, although this may be needed in the future.
  if (grad_flag) {
    if (source_fn_grads.num_rows() < num_fns) {
      Cerr << "Error: insufficient incoming gradient size to copy response "
	   << "results required in ResponseRep::copy_results()." << endl;
      abort_handler(-1);
    }
    for (i=0; i<num_fns; i++)
      if (asv[i] & 2) // assign each entry since size may differ
	for (j=0; j<num_params; j++)
	  functionGradients[i][j] = source_fn_grads[i][j];
  }
  if (hess_flag) {
    if (source_fn_hessians.length() < num_fns) {
      Cerr << "Error: insufficient incoming Hessian size to copy response "
	   << "results required in ResponseRep::copy_results()." << endl;
      abort_handler(-1);
    }
    for (i=0; i<num_fns; i++)
      if (asv[i] & 4) // assign each entry since size may differ
	for (j=0; j<num_params; j++)
	  for (k=0; k<num_params; k++)
	    functionHessians[i](j,k) = source_fn_hessians[i](j,k);
  }

  // Remove any data retrieved from response but not reqd by responseActiveSet
  // (id_vars_set_compare will detect duplication when the current asv is a
  // subset of the database asv and more can be retrieved than requested).
  if (responseActiveSet != source_set) // only reset if needed
    reset_inactive();
}


/** Copy function values/gradients/Hessians data _only_.  Prevents unwanted
    overwriting of responseActiveSet, functionLabels, etc.  Also, care is
    taken to account for differences in derivative variable matrix sizing. */
void ResponseRep::
copy_results_partial(size_t start_index_target, size_t num_items,
		     const RealDenseVector& source_fn_vals,
		     const RealMatrix& source_fn_grads,
		     const RealSymDenseMatrixArray& source_fn_hessians,
		     const ActiveSet& source_set, size_t start_index_source)
{
  // current response data
  const ShortArray& asv = responseActiveSet.request_vector();
  size_t i, j, k, num_fns = asv.length(), 
    num_params = responseActiveSet.derivative_vector().length();
  bool grad_flag = false, hess_flag = false;
  for (i=0; i<num_fns; i++) {
    if (asv[i] & 2)
      grad_flag = true;
    if (asv[i] & 4)
      hess_flag = true;
  }

  // verify that incoming data is of sufficient size
  if (start_index_target + num_items > num_fns ||
      start_index_source + num_items > source_set.request_vector().length() ) {
    Cerr << "Error: insufficient number of response functions to copy partial "
	 << "response results in ResponseRep::copy_results_partial()." << endl;
    abort_handler(-1);
  }
  if ( (grad_flag || hess_flag) && 
       source_set.derivative_vector().length() < num_params) {
    Cerr << "Error: insufficient number of derivative variables to copy "
	 << "partial response derivative results in ResponseRep::"
	 << "copy_results_partial()." << endl;
    abort_handler(-1);
  }

  for (i=0; i<num_items; i++)
    if (asv[start_index_target+i] & 1)
      functionValues[start_index_target+i]
	= source_fn_vals[start_index_source+i];

  // copy source_fn_grads and source_fn_hessians.  For now, a mapping from
  // responseActiveSet.derivative_vector() to source_set.derivative_vector()
  // is _not_ used, although this may be needed in the future.
  if (grad_flag) {
    if (source_fn_grads.num_rows() < start_index_source + num_items) {
      Cerr << "Error: insufficient incoming gradient size to copy partial "
	   << "response gradient results required in ResponseRep::"
	   << "copy_results_partial()." << endl;
      abort_handler(-1);
    }
    for (i=0; i<num_items; i++)
      if (asv[start_index_target+i] & 2)
	for (j=0; j<num_params; j++)
	  functionGradients[start_index_target+i][j]
	    = source_fn_grads[start_index_source+i][j];
  }
  if (hess_flag) {
    if (source_fn_hessians.length() < start_index_source + num_items) {
      Cerr << "Error: insufficient incoming Hessian size to copy partial "
	   << "response Hessian results required in ResponseRep::"
	   << "copy_results_partial()." << endl;
      abort_handler(-1);
    }
    for (i=0; i<num_items; i++)
      if (asv[start_index_target+i] & 4)
	for (j=0; j<num_params; j++)
	  for (k=0; k<num_params; k++)
	    functionHessians[start_index_target+i](j,k)
	      = source_fn_hessians[start_index_source+i](j,k);
  }

  // Remove any data retrieved from response but not reqd by responseActiveSet
  // (id_vars_set_compare will detect duplication when the current asv is a
  // subset of the database asv and more can be retrieved than requested).
  if (responseActiveSet != source_set) // only reset if needed
    reset_inactive();// active data not included in partial update are not reset
}


/** Reshape functionValues, functionGradients, and functionHessians
    according to num_fns, num_params, grad_flag, and hess_flag. */
void ResponseRep::reshape(const size_t& num_fns, const size_t& num_params,
			  bool grad_flag, bool hess_flag)
{
  if (responseActiveSet.request_vector().length()    != num_fns || 
      responseActiveSet.derivative_vector().length() != num_params)
    responseActiveSet.reshape(num_fns, num_params);
  if (functionLabels.size() != num_fns) {
    functionLabels.resize(num_fns);
    build_labels(functionLabels, "f");
  }
  if (functionValues.length() != num_fns)
    functionValues.resize(num_fns);
  if (grad_flag) {
    if (functionGradients.num_rows()    != num_fns || 
        functionGradients.num_columns() != num_params)
      functionGradients.reshape_2d(num_fns, num_params);
  }
  else if (!functionGradients.empty())
    functionGradients.clear();
  if (hess_flag) {
    if (functionHessians.size() != num_fns)
      functionHessians.resize(num_fns);
    for (size_t i=0; i<num_fns; i++) {
      if (functionHessians[i].numRows() != num_params)
        functionHessians[i].reshape(num_params);
    }
  }
  else if (!functionHessians.empty())
    functionHessians.clear();
}


/** Reset all numerical response data (not labels, ids, or active set)
    to zero. */
void ResponseRep::reset()
{
  functionValues = 0.;
  functionGradients = 0.;
  size_t nh = functionHessians.size();
  for (size_t i=0; i<nh; i++)
    functionHessians[i] = 0.;
}


/** Used to clear out any inactive data left over from previous evaluations. */
void ResponseRep::reset_inactive()
{
  const ShortArray& asv = responseActiveSet.request_vector();
  size_t i, asv_len = asv.length(), nr = functionGradients.num_rows(),
    nh = functionHessians.size();
  for (i=0; i<asv_len; i++)
    if ( !(asv[i] & 1) ) // value bit is omitted
      functionValues[i] = 0.;
  for (i=0; i<nr; i++)
    if ( !(asv[i] & 2) ) // gradient bit is omitted
      functionGradients[i] = 0.; // the i_th row vector (all columns)
  for (i=0; i<nh; i++)
    if ( !(asv[i] & 4) ) // Hessian bit is omitted
      functionHessians[i] = 0.;  // the i_th Hessian (all rows & columns)
}


/// equality operator for ResponseRep
bool operator==(const ResponseRep& rep1, const ResponseRep& rep2)
{
  return (rep1.functionValues    == rep2.functionValues    &&
	  rep1.functionGradients == rep2.functionGradients &&
	  rep1.functionHessians  == rep2.functionHessians  &&
	  rep1.responseActiveSet == rep2.responseActiveSet &&
	  rep1.functionLabels    == rep2.functionLabels);//&&
        //rep1.idResponses       == rep2.idResponses);
}


void ResponseRep::active_set_request_vector(const ShortArray& asrv)
{
  // a change in ASV length is not currently allowed
  if (responseActiveSet.request_vector().length() != asrv.length()) {
    Cerr << "Error: total number of response functions may not be changed in "
	 << "ResponseRep::active_set_request_vector(ShortArray&)." << endl;
    abort_handler(-1);
  }
  // assign the new request vector
  responseActiveSet.request_vector(asrv);
}


void ResponseRep::active_set_derivative_vector(const UIntArray& asdv)
{
  // resize functionGradients/functionHessians if needed to accomodate
  // a change in DVV size
  size_t new_deriv_vars = asdv.length();
  if (responseActiveSet.derivative_vector().length() != new_deriv_vars) {
    size_t num_fns = responseActiveSet.request_vector().length();
    //reshape(num_fns, new_deriv_vars, !functionGradients.empty(),
    //	      !functionHessians.empty());
    if (!functionGradients.empty())
      functionGradients.reshape_2d(num_fns, new_deriv_vars);
    if (!functionHessians.empty())
      for (size_t i=0; i<num_fns; i++)
	functionHessians[i].reshape(new_deriv_vars);
  }
  // assign the new derivative vector
  responseActiveSet.derivative_vector(asdv);
}

} // namespace Dakota
