%{
/*********************************************************************
Copyright 2008 Sandia Corporation.  Under the terms of Contract
DE-AC04-94AL85000 with Sandia Corporation, the U.S. Government
retains certain rights in this software.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

* Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

* Neither the name of Sandia Corporation nor the names of its
contributors may be used to endorse or promote products derived from
this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
***********************************************************************/

#include <ctype.h>
#include <math.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "nidr.h"	/* for KeyWordKind */

#ifndef DMBLK_GULP
#define DMBLK_GULP 1024
#endif
#ifndef DMLBK_SGULP
#define DMBLK_SGULP 16000
#endif
#ifndef MBLK_GULP
#define MBLK_GULP 8191
#endif
#ifndef KWSTACKLEN
#define KWSTACKLEN 100	/*should be overkill*/
#endif
#ifndef DEFTAGLEN
#define DEFTAGLEN 2048	/* should be overkill*/
#endif

 typedef union
YYSTYPE { char *s; } YYSTYPE;

 enum LastSeen {
	Saw_kwname	= 1,
	Saw_type	= 2,
	Saw_func	= 3,
	Saw_dflt	= 4,
	Saw_dfltname	= 5,
	Saw_lt		= 6,
	Saw_le		= 7,
	Saw_ge		= 8,
	Saw_gt		= 9,
	Saw_desc	= 10,
	Saw_group	= 11,
	Saw_tag		= 12,
	Saw_len		= 13
	};

 static const char *tokname[] = {
	"<bug>",
	"kwname",
	"type",
	"{funcspec}",
	":=",
	"#define name for :=",
	"<",
	"<=",
	">=",
	">",
	"DESC",
	"GROUP",
	"TAG",
	"LEN"
	};

 enum ParState {
	/* parstate values... */
	PPar_emtpy	= 0,
	PPar_starting	= 1,
	PPar_started	= 2,
	PPar_leftpar	= 3,
	Dupchecking	= 4,
	Dupchecked	= 5,
	Printing	= 6,
	KwStacked	= 7,
	Sorted		= 8,
	Leftput		= 9,
	Parchecked	= 10
	};

 /* values for lastseen also include single characters ( [ { } ] ) ; */

 /* enums to make constants visible to debuggers */
 enum {
	DMblk_gulp	= DMBLK_GULP,
	Dmblk_sgulp	= DMBLK_SGULP,
	Mblk_gulp	= MBLK_GULP,
	KW_stacklen	= KWSTACKLEN,
	Br_stacklen	= 2*KW_stacklen,
	DHTlen		= 997,
	DHTfactor	= 43,
	HTlen		= 127,
	KWPlen		= 32,
	KWOUTlen0	= 128 };

 typedef struct Defhash Defhash;
 typedef struct DefhashHead DefhashHead;
 typedef struct DefhashTab DefhashTab;
 typedef struct HashHead HashHead;
 typedef struct KWctx KWctx;
 typedef struct Mblk Mblk;
 typedef struct Taghash Taghash;
 typedef struct StrList StrList;

 struct
DefhashHead {
	DefhashHead *hnext, *thread;
	char *name;
	};

 struct
Defhash {
	DefhashHead hh;
	char *value;
	int qkeep;
	};

 struct
Taghash {
	DefhashHead hh;
	KWctx *kw;
	int uses;
	};

 struct
DefhashTab {
	DefhashHead *Tab[DHTlen];
	DefhashHead *lastdh;
	size_t nalloc;	/* size of new entry allocations in units of sizeof(void*) */
	int nent;
	};

 struct
Mblk {
	Mblk *next;
	void *stuff[Mblk_gulp];
	};

 struct
HashHead { HashHead *hnext, *hprev; };

 struct
StrList {
	HashHead h;
	StrList *next;
	char *val;
	int Lineno;
	int primary;
	};

 struct
KWctx {
	KWctx *next;
	StrList *names;	/* list of name and aliases */
	char *name1;	/* first name (for tagout) */
	KWctx *kw;	/* list of contained keywords */
	KWctx *master;	/* non-null ==> this is an alias; use master's kwkno */
	KWctx *pmaster;	/* used in computing paoff */
	KWctx *pparent;	/* paren parent, for ((...)|(...) ...) */
	KWctx *rparent;	/* "real" parent, for stuff inside ((...)) */
	StrList *funcs;	/* pre- and post order functions and (void*) args */
	char *Br_top;	/* Br_top when this keyword began */
	char *lb;	/* lower bound (if kind & KWKind_Lb is nonzero) */
	char *ub;	/* upper bound (if kind & KWKind_Ub is nonzero) */
	char *init;	/* initial value (if kind & KWKind_init is nonzero) and */
			/* kind & KWKind_Mask is 1 or 2 (KWKind_Int or KWKind_Real) */
	char *cinit;	/* initial value for STRING-valued keyword,	*/
			/* ==> (kind & (KWKind_Mask | KWKind_init))	*/
			/*	== (KWKind_Str | KWKind_init)		*/
	char *defname;	/* name of preprocessor token to be #defined */
	char *desc;
	char *group;
	char *tag;
	KWctx *len;	/* keyword giving array length */
	size_t dtnext;	/* for adjusting deftag when this keyword goes out of scope */
	size_t name1len;
	int alt;	/* alternative group number of this keyword */
	int kind;	/* kind of values for this keyword */
	int kno;	/* number of this keyword, for generating kw_n names */
	int kwkno;	/* number of this keyword's kw array (if any) */
	int kwknop;	/* if this is a pparent and has a pparent, kwkno for this and parents */
	int nalt;	/* number of alternative groups in contained keywords */
	int nfcn;	/* number of function and arg entries in funcs */
	int nkw;	/* number of contained keywords */
	int nreq;	/* number of required elements in contained keywords */
	int req;	/* required group for this keyword (0 if optional) */
	int parstate;	/* state in handling ((...)) */
	int altoffset;	/* for expand */
	int reqoffset;	/* for expand */
	int level;
	int objno;	/* for kwcomp2 */
	int agroup;	/* for GuiKeyWord */ /* deprecated */
	int paoff;	/* offset of primary keyword: replacement (?) for agroup */
	};

 static int lastagroup, lastobjno, nalias;

 Defhash *lastdef;
 DefhashTab DHTab, TagTab;
 FILE *mtagout, *tagin, *tagout;
 HashHead HTab[HTlen];
 KWctx KWbase, *lastkw, *lastkw1;
 KWctx *KWStack[KW_stacklen], **KWtop = KWStack;
 Mblk FirstMblk, *CurMblk = &FirstMblk;
 YYSTYPE yyval;
 char Br_stack[Br_stacklen], *Br_top = Br_stack;
 char deftag[DEFTAGLEN], *dtend = deftag + DEFTAGLEN - 2, *dtnext = deftag;
 char *infname, *lasttag, *progname, *specfile, *tagfname;
 int Lineno = 1, dfltgroup = 1, startwarn = 1, wantfuncs = 1;
 int brace, btype, dupnames, expand, guikeywds, ignoretags;
 int kwmult, lastkno, lastseen, missing, needcomma, nsquawk, or_mode, saweof;
 size_t mbavail = Mblk_gulp;
 void **mbnext = FirstMblk.stuff;
 static void Eat_comment(void);
 static void Keywd(char *);
 static void Saw_quote(void);

 /* Possible values for *Br_top:
 //	 0  ==> empty (for Br_stack[0] only)
 //	'(' ==> open paren seen, awaiting ')'
 //	'[' ==> open square bracket, awaiting ']'
 */

 static void
botch(const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	fprintf(stderr, "\n%s:\n\tbotch on line %d", progname, Lineno);
	if (infname)
		fprintf(stderr, " of \"%s\"", infname);
	fprintf(stderr, ":\n\t");
	vfprintf(stderr, fmt, ap);
	fputs(".\n", stderr);
	va_end(ap);
	exit(1);
	}

 static void
squawk(const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	fprintf(stderr, "\n%s: line %d", progname, Lineno);
	if (infname)
		fprintf(stderr, " of \"%s\"", infname);
	fprintf(stderr, ":\n\t");
	vfprintf(stderr, fmt, ap);
	fputs(".\n", stderr);
	va_end(ap);
	++nsquawk;
	}

 static void*
Malloc(size_t L, const char *where)
{
	void *rv = malloc(L);
	if (!rv)
		botch("malloc(%lu) failure in %s()", (unsigned long)L);
	return rv;
	}

 void
unexpected(const char *what)
{ botch("unexpected \"%s\"", what); }

 static void *
Alloc(size_t len)
{
	Mblk *mb;
	size_t L;
	void *rv;

	L = (len + sizeof(void*) - 1)/sizeof(void*);
	if (L >= Mblk_gulp)
		botch("Alloc(%lu) failure", (unsigned long)len);
	if (L > mbavail) {
		if (!(mb = CurMblk->next)) {
			mb = (Mblk*)Malloc(sizeof(Mblk), "Alloc");
			CurMblk->next = mb;
			}
		CurMblk = mb;
		mbavail = Mblk_gulp;
		mbnext = mb->stuff;
		mb->next = 0;
		memset(mbnext, 0, sizeof(mb->stuff));
		}
	rv = mbnext;
	mbnext += L;
	mbavail -= L;
	return rv;
	}

#if 0
 static void
Mblk_reset(void)
{
	CurMblk = FirstMblk;
	mbavail = Mblk_gulp;
	mbnext = FirstMblk.stuff;
	memset(mbnext, 0, sizeof(FirstMblk.stuff));
	}
#endif

 StrList *
new_StrList(const char *s, StrList *nxt)
{
	size_t L = strlen(s) + 1;
	StrList *rv = (StrList*)Alloc(sizeof(StrList) + L);
	strcpy(rv->val = (char*)(rv + 1), s);
	rv->next = nxt;
	rv->Lineno = Lineno;
	return rv;
	}

 static void
Alias(char *s)
{
	KWctx *kw;
	if (lastseen != Saw_kwname)
		botch("ALIAS out of place");
	kw = lastkw;
	for(s += 6; *s <= ' '; s++);
	kw->names = new_StrList(s, kw->names);
	++nalias;
	lastseen = Saw_kwname;
	}

 static void
reqchk(const char *a, const char *s)
{
	if (a)
		while(*a)
			if (*a++ == lastseen)
				return;
	botch("\"%s\" out of place", s);
	}

 static void
reqchk1(const char *a, const char *s)
{
	int c;

	if (*Br_top != *s) {
		if ((c = *Br_top)) {
			switch(c) {
			 case '[': c = ']'; break;
			 case '(': c = ')'; break;
			 default:  c = '?';
			 }
			botch("Missing '%c'", c);
			}
		else
			a = 0;
		}
	reqchk(a, s+1);
	}

 static void
Bar(void)
{
	static char allowed[] = { /*(*/ ')', Saw_kwname, Saw_type, 0 };
	reqchk(allowed, "|");
	lastseen = '|';
	lastkw = lastkw1 = 0;
	or_mode = 1;
	}

 static void
Lpar(void)
{
	KWctx *kw;
	int olastseen = lastseen;
	static char allowed[] = { '(', '[', '|', ']', ')', Saw_kwname, Saw_type, 0 };

	reqchk(allowed, "("/*)*/);
	lastseen = '('; /*)*/
	if (KWtop > KWStack) {
		kw = KWtop[0];
		if (or_mode && kw->parstate == PPar_leftpar && !kw->kw) {
			dtnext = deftag + kw->dtnext;
			--KWtop;
			Keywd(0);
			kw->pparent = KWtop[0];
			}
		else if (olastseen == '(' || olastseen == '[' /*])*/)
			Keywd(0);
		lastseen = '(' /*)*/;
		}
	if (++Br_top - Br_stack >= Br_stacklen)
		botch("Br_stack overflow");
	*Br_top = '('; /*)*/
	lastkw = lastkw1 = 0;
	}

 static void
Lsqb(void)
{
	static char allowed[] = { /*([*/ ']', ')', Saw_kwname, Saw_type, 0 };
	reqchk(allowed, "["/*]*/);
	if (++Br_top - Br_stack >= Br_stacklen)
		botch("Br_stack overflow");
	lastseen = *Br_top = '['; /*]*/
	lastkw = lastkw1 = 0;
	}

 static void
Rpar(void)
{
	static char allowed[] =  {/*[(*/ ')', ']', Saw_kwname, Saw_type, 0 };
	reqchk1(allowed, "()");
	if (KWtop <= KWStack)
		botch("bug: KW stack error");
	--Br_top;
	if (!KWtop[0]->kw && KWtop - KWStack > 2
	 && KWtop[-1]->parstate == PPar_starting)
		KWtop[-1]->Br_top = Br_top;
	--KWtop;
	lastseen = /*(*/ ')';
	lastkw = lastkw1 = 0;
	}

 static void
Rsqb(void)
{
	static char allowed[] = {/*(*/')', '[', ']', Saw_kwname, Saw_type, 0 };
	reqchk1(allowed, "[]");
	if (KWtop <= KWStack)
		botch("bug: KW stack error");
	--Br_top;
	--KWtop;
	lastseen = /*[*/ ']';
	lastkw = lastkw1 = 0;
	}

 static void
Lbrace(void)
{
	if (brace++ || (lastseen != Saw_kwname && lastseen != Saw_type))
		unexpected("{");
	lastseen = '{'; /*}}*/
	}

 static void
Rbrace(void)
{
	if (--brace || (lastseen != Saw_func && lastseen != ';'))
		/*{{*/unexpected("}");
	lastseen = Saw_type;
	}

 static void
Type(int n, char *s)
{
	if (lastseen != Saw_kwname || !lastkw)
		botch("type %s out of place", s);
	lastkw->kind |= n;
	lastkw = 0;
	lastseen = Saw_type;
	}

 static char *
def_name(char *s0)
{
	char *s;
	size_t L, L1;
	static char *slast, *snext;

	s = s0;
	while(*s++);
	L = s - s0;
	if (snext + L > slast) {
		snext = (char*)Malloc(L1 = Dmblk_sgulp + L, "def_name");
		slast = snext + L1;
		}
	strcpy(s = snext, s0);
	snext += L;
	return s;
	}

static void **DHHlast, **DHHnext;	/* Could be private to DHHlookup, but here */
					/* to make them visible to debuggers. */
 static DefhashHead *
DHHlookup(char *s0, DefhashTab *Htab, int add)
{
	DefhashHead *h, **hp;
	char *s;
	size_t L;
	unsigned int c, x;

	for(x = 0, s = s0; (c = *(unsigned char*)s); ++s)
		x += DHTfactor*x + c;
	x %= DHTlen;
	for(hp = &Htab->Tab[x]; (h = *hp); hp = &h->hnext) {
		if (!strcmp(s0, h->name))
			return h;
		}
	if (!add)
		return h;
	Htab->nent++;
	if (DHHnext + Htab->nalloc >= DHHlast) {
		DHHnext = (void**)Malloc(L = DMblk_gulp*sizeof(void*), "DHHlookup");
		DHHlast = DHHnext + DMblk_gulp;
		memset(DHHnext, 0, L);
		}
	*hp = h = (DefhashHead*)DHHnext;
	DHHnext += Htab->nalloc;
	h->name = def_name(s0);
	h->thread = Htab->lastdh;
	return Htab->lastdh = h;
	}

 static void
bothplaces(KWctx *kw, const char *name)
{ squawk("For %s, %s appears both in specfile and tagfile", kw->names->val, name); }

 static void
tagimport(KWctx *kw, KWctx *tw)
{
	typedef struct KWimport KWimport;
	struct KWimport { const char *name; size_t off; };
	static KWimport Imp[] = {
		{ "lower bound",	offsetof(KWctx,lb) },
		{ "upper bound",	offsetof(KWctx,ub) },
		{ "default (:= value)",	offsetof(KWctx,init) },
		{ "default (:= value)",	offsetof(KWctx,cinit) },
		{ "defname",		offsetof(KWctx,defname) },
		{ "DESC",		offsetof(KWctx,desc) },
		{ "GROUP",		offsetof(KWctx,group) },
		{ "LEN",		offsetof(KWctx,len) },
		{0}};
	KWimport *ki;
#define field(k,n) *(char**)((char*)k+n)
	for(ki = Imp; ki->name; ++ki) {
		if (field(tw,ki->off)) {
			if (field(kw,ki->off))
				bothplaces(kw, ki->name);
			else
				field(kw,ki->off) = field(tw,ki->off);
			}
		}
#undef field
	}

 static void
Keywd(char *s)
{
	KWctx *kw, *kw0, *kw1, *kw2, *kwm, **kwp;
	StrList *nam;
	Taghash *th;
	char *s1;
	int alt, i, req;
	size_t L, L1;
	static char allowed[] = { '(',')','[',']','|', Saw_kwname, Saw_type, 0 };

	if (tagin)
		botch("Inappropriate text \"%s\" in tagfile \"%s\"", s, tagfname);
	kw0 = kw1 = *KWtop;
	kwm = 0;
	if (!kw1->names && s) {
		switch(kw1->parstate) {
		  case PPar_emtpy:
			do {
				kw1->parstate = PPar_starting;
				kw1 = kw1->pparent;
				}
				while(kw1 && kw1->parstate == PPar_emtpy);
			kw1 = kw0->rparent;
			break;
		  case PPar_starting:
			if (or_mode)
				kw1 = kw0->rparent;
			else
				kw1->parstate = PPar_started;
			break;
		  }
		}
	if (or_mode && !kw1->kw && KWtop - KWStack > 1) {
		kw2 = KWtop[-1];
		if (!kw2->names && kw2->parstate == PPar_starting)
			kw2 = kw2->rparent;
		if (s)
			kw1->master = kwm = kw1;
		kw1 = kw2;
		}
	alt = req = 0;
	if (s) {
		kwp = KWStack + kw1->level + 1;
		if (kwp > KWtop)
			kwp = KWtop;
		while(kwp > KWStack) {
			kw2 = *kwp--;
			if (kw2->name1) {
				dtnext = deftag + kw2->dtnext;
				L = kw2->name1len;
				strcpy(dtnext-L, kw2->name1);
				break;
				}
			}
		alt = ++kw1->nalt;
		if (*Br_top != '['/*]*/ || Br_top == kw1->Br_top) {
			req = ++kw1->nreq;
			if (*Br_top == '(' /*)*/) /* allow [(...)|...] */
				for(s1 = Br_top; --s1 > kw1->Br_top; ) {
					if (*s1 != '(' /*)*/) {
						if (*s1 == '[' /*]*/) {
							--kw1->nreq;
							req = 0;
							}
						break;
						}
					}
			}
		if (or_mode) {
			alt = --kw1->nalt;
			if (req)
				req = --kw1->nreq;
			}
		if (lastseen)
			reqchk(allowed, s);
		or_mode = 0;
		}
	lastkw = lastkw1 = kw = (KWctx*) Alloc(sizeof(KWctx));
	if (!kw0->names)
		kw->pparent = kw0;
	if (s) {
		kw->names = nam = new_StrList(s,0);
		kw->name1 = nam->val;
		kw->name1len = L = strlen(s);
		for(kwp = KWtop; kwp > KWStack; --kwp) {
			kw2 = *kwp;
			if (kw2->names) {
				if (kwp < KWtop) {
					L1 = kw2->dtnext + L + 1;
					while(++kwp <= KWtop) {
						kw2 = *kwp;
						if (!kw2->name1) {
							kw2->name1 = kw->name1;
							kw2->name1len = L;
							kw2->dtnext = L1;
							}
						}
					}
				break;
				}
			}
		nam->primary = KWKind_primary;
		kw->next = kw1->kw;
		kw1->kw = kw;
		if (lastseen == '(' /*)*/)
			kw->parstate = PPar_leftpar;
		kw->agroup = lastagroup++;
		}
	else {
		kw->objno = ++lastobjno;
		if (!kw1->kw && (kw->rparent = kw1->rparent))
			kw1 = kw->rparent;
		else
			kw->rparent = kw1;
		alt = kw1->alt;
		req = kw1->req;
		}
	kw->alt = alt;
	kw->req = req;
	kw->Br_top = Br_top;
	kw->master = kwm;
	kw->level = KWtop - KWStack;
	if (lastseen == '(' || lastseen == '[' /*])*/ || KWtop == KWStack) {
		if (++KWtop - KWStack >= KW_stacklen)
			botch("KW stack overflow");
		*KWtop = kw;
		}
	lasttag = 0;
	if (s) {
		if (dtnext > deftag)
			*dtnext++ = '/';
		if (dtnext + L >= dtend)
			botch("deftag overflow");
		memcpy(dtnext, s, L);
		*(dtnext += L) = 0;
		if (tagout)
			fprintf(tagout, "TAG \"%s\"\n", deftag);
		if (tagfname) {
			if ((th = (Taghash*)DHHlookup(deftag, &TagTab, ignoretags))) {
				++th->uses;
				lasttag = th->hh.name;
				if (!ignoretags) {
					lastkw1->tag = lasttag;
					tagimport(lastkw1, th->kw);
					}
				}
			else if (mtagout)
				fprintf(mtagout, "TAG \"%s\" #missing\n", deftag);
			}
		}
	kw->dtnext = dtnext - deftag;
	if (lastseen == 0)
		kw->kind = kwmult;
	lastseen = Saw_kwname;
	}

 static void
kw_finish(int nextkind)
{
	if (*Br_top)
		botch("unmatched '%c'", *Br_top);
	if (KWtop - KWStack > 1)
		botch("bug: KW stack botch in kw_finish");
	KWtop = KWStack;
	dtnext = deftag;
	lastseen = 0;
	kwmult = nextkind;
	}

 static void
Fnc(char *s)
{
	KWctx *kw;

	if (!(kw = lastkw1))
		botch("misplaced {...}");
	if (++kw->nfcn > 4)
		botch("too many functions in {...}");
	kw->funcs = new_StrList(s, kw->funcs);
	lastseen = Saw_func;
	}

 static const char*
tname_adj(const char *tname, int what, int kbits)
{
	Uint k = lastkw1->kind;

	switch(what) {
	 case Saw_lt:
		if (k & KWKind_caneqUb)
			tname = "<=";
		break;
	 case Saw_le:
		if (k & KWKind_strictUb)
			tname = "<";
		break;
	 case Saw_ge:
		if (k & KWKind_strictLb)
			tname = ">";
		break;
	 case Saw_gt:
		if (k & KWKind_caneqLb)
			tname = ">=";
	 }
	return tname;
	}

 static void
Saw_tok(int what, int kbits)
{
	const char *tname;

	if ((lastseen == Saw_kwname || lastseen == Saw_type) && lastkw1) {
		if (lastkw1->kind & kbits) {
			tname = tokname[what];
			if (kbits & (KWKind_Lb | KWKind_Ub))
				tname = tname_adj(tname, what, kbits);
			botch("%s already seen for keyword %s",
				tname, lastkw1->names->val);
			}
		lastseen = what;
		}
	else
		unexpected(tokname[what]);
	}

 static void
Saw_stok(int what, size_t woff)
{
	if ((lastseen == Saw_kwname || lastseen == Saw_type) && lastkw1) {
		if (*(void**)((char*)lastkw1 + woff))
			botch("%s already seen for keyword %s",
				tokname[what], lastkw1->names->val);
		lastseen = what;
		}
	else
		unexpected(tokname[what]);
	}

 static KWctx *
lenfind(char *s)
{
	KWctx *kw, *kw0, **pkw;
	StrList *sl;

	kw0 = lastkw1;
	for(pkw = KWtop; pkw > KWStack; --pkw) {
		kw = *pkw;
		for(kw = kw->kw; kw; kw = kw->next) {
			for(sl = kw->names; sl; sl = sl->next)
				if (!strcmp(sl->val, s))
					return kw;
			}
		}
	return 0;
	}

 static void
alreadyseen(const char *what)
{
	squawk("%s already given for %s", what, lastkw1->names->val);
	}

 static int
Saw_name(char *s)
{
	KWctx *kw;
	Taghash *th;
	char *s1;
	static const char notfound[] =
		"keyword \"%s\" not found in this or an enclosing context";

	if (lastkw1) {
		switch(lastseen) {
		  case Saw_dflt:
			lastseen = Saw_dfltname;
			lastdef = (Defhash*)DHHlookup(s, &DHTab, 1);
			if (!lastkw1->defname)
				lastkw1->defname = def_name(s);
			else if (lastdef->value
				 && ((s1 = lastkw1->init) || (s1 = lastkw1->cinit))
				 && strcmp(s1, lastdef->value))
					squawk("Inconsistent definitions for %s: %s and %s\n",
						lastdef->hh.name, lastdef->value, s1);
			return 0;
		  case Saw_len:
			if (lastkw1->len) {
				alreadyseen("LEN");
				goto ret1;
				}
			kw = lenfind(s);
			if (!kw)
				squawk(notfound,s);
			else {
				if ((kw->kind & KWKind_Mask) != KWKind_Int)
					squawk("%s is not of type INTEGER", kw->names->val);
				lastkw1->len = kw;
				}
			goto ret1;
		  case Saw_group:
			if (lastkw1->group)
				alreadyseen("GROUP");
			else
				strcpy(lastkw1->group = (char*)Alloc(strlen(s)+1), s);
			goto ret1;
		  case Saw_tag:
			if (tagin)
				goto new_tag;
			if (lasttag) {
				alreadyseen("TAG");
				goto ret1;
				}
			if (!tagfname && !ignoretags) {
				squawk("TAG \"%s\" not found: no tag file given", s);
				goto ret1;
				}
			th = (Taghash*)DHHlookup(s, &TagTab, ignoretags);
			if (!th)
				squawk("TAG \"%s\" not found in tag file \"%s\"", s, tagfname);
			else {
				++th->uses;
				lasttag = th->hh.name;
				if (!ignoretags) {
					lastkw1->tag = lasttag;
					tagimport(lastkw1, th->kw);
					}
				}
 ret1:
			lastseen = Saw_type;
			return 1;
		  }
		}
	else if (tagin) {
 new_tag:
		th = (Taghash*)DHHlookup(s, &TagTab, 1);
		if (th->kw)
			squawk("Tag \"%s\" already seen in tagfile %s\n", s, tagfname);
		else
			th->kw = lastkw1 = (KWctx*) Alloc(sizeof(KWctx));
		goto ret1;
		}
	botch("unexpected name \"%s\"", s);
	return 0; /* not reached */
	}

 static void
boundalready(const char *what)
{
	squawk("%s bound already given for %s", what, lastkw1->names->val);
	}

 static void
set_lastdef(char *s, int qkeep)
{
	if (!lastdef->value) {
		lastdef->value = def_name(s);
		lastdef->qkeep = qkeep;
		}
	else if (strcmp(s, lastdef->value))
		squawk("Two := values for %s: %s and %s", lastdef->hh.name,
			lastdef->value, s);
	}

 static void
Saw_num(char *s)
{
	Defhash *ld;
	double x, xa;
	char *se, **sp;
	int k;
	static double b[3];

	static char allowed[] = { Saw_dflt, Saw_dfltname, Saw_lt, Saw_le, Saw_ge, Saw_gt, 0 };

	reqchk(allowed, "number");
	x = strtod(s,&se);
	if (*se)
		botch("invalid number: \"%s\"", s);
	ld = 0;
	switch(lastseen) {
		case Saw_dflt:
		case Saw_dfltname:
			sp = &lastkw1->init;
			lastkw1->kind |= KWKind_init;
			ld = lastdef;
			if ((lastkw1->kind & KWKind_Mask) == KWKind_Int)
				x = x >= 0. ? floor(x) : ceil(x);
			b[2] = x;
			if (lastkw1->lb && x < b[0])
				squawk("default value inconsistent with lower bound");
			if (lastkw1->ub && x > b[1])
				squawk("default value inconsistent with upper bound");
			break;
		case Saw_gt:
		case Saw_ge:
			if (lastkw1->lb) {
				boundalready("lower");
				goto ret;
				}
			if ((lastkw1->kind & KWKind_Mask) == KWKind_Int) {
				xa = ceil(x);
				if (xa > x) {
					if (btype == KWKind_strictLb)
						btype = KWKind_caneqLb;
					x = xa;
					}
				}
			sp = &lastkw1->lb;
			lastkw1->kind |= btype;
			b[0] = x;
			if (lastkw1->ub && x >= b[1])
				squawk("inconsistent bounds");
			if (lastkw1->init && x > b[2])
				squawk("lower bound inconsistent with default value");
			break;
		case Saw_le:
		case Saw_lt:
			if (lastkw1->ub) {
				boundalready("upper");
				goto ret;
				}
			if ((lastkw1->kind & KWKind_Mask) == KWKind_Int) {
				xa = floor(x);
				if (xa < x) {
					if (btype == KWKind_strictUb)
						btype = KWKind_caneqUb;
					x = xa;
					}
				}
			sp = &lastkw1->ub;
			lastkw1->kind |= btype;
			b[1] = x;
			if (lastkw1->lb && x <= b[0])
				squawk("inconsistent bounds");
			if (lastkw1->init && x < b[2])
				squawk("upper bound inconsistent with default value");
			break;
		default:
			unexpected(s);
		}
	k = lastkw1->kind & KWKind_Mask;
	if (k == KWKind_Void || k == KWKind_Str)
		squawk("%s cannot accept a %s value", lastkw1->names->val, tokname[lastseen]);
	else {
		strcpy(*sp = (char*)Alloc(strlen(s)+1), s);
		if (ld)
			set_lastdef(s, 0);
		}
 ret:
	lastseen = Saw_type;
	}

%}

ALias	"ALIAS"[ \t]+[a-z][a-z_0-9]*
D	[0-9]
Kword	[a-z][a-z_0-9]*
Func	[a-zA-Z_0-9()][,a-zA-Z_0-9<>():.+*/\-\[\] \t]*
Num	[\-+]?({D}+("."{D}*)?|"."{D}+)([eE]([\-+]?){D}+)?
PPname	[a-zA-Z][a-zA-Z0-9_]*
ws	[ \t]+

%s FNC VAL

%%

\n		{ ++Lineno; }
{ws}		{}
<INITIAL>{ALias}	{ Alias(yytext); }
<INITIAL>"INTEGER"	{ Type(KWKind_Int, yytext); }
<INITIAL>"INTEGERLIST"	{ Type(KWKind_Int | KWKind_List, yytext); }
<INITIAL>"KEYWORD"	{ kw_finish(0); }
<INITIAL>"KEYWORD1"	{ kw_finish(KWKind_1); }
<INITIAL>"KEYWORD01"	{ kw_finish(KWKind_01); }
<INITIAL>"KEYWORD12"	{ kw_finish(KWKind_12); }
<INITIAL>"REAL"		{ Type(KWKind_Real, yytext); }
<INITIAL>"REALLIST"	{ Type(KWKind_Real | KWKind_List, yytext); }
<INITIAL>"STRING"	{ Type(KWKind_Str, yytext); }
<INITIAL>"STRINGLIST"	{ Type(KWKind_Str | KWKind_List, yytext); }
<INITIAL>{Kword}	{ Keywd(yytext); }
<INITIAL>";"		{ kw_finish(0); }
<INITIAL>"DESC"		{ Saw_stok(Saw_desc, offsetof(KWctx,desc));	BEGIN VAL; }
<INITIAL>"TAG"		{ Saw_stok(Saw_tag, offsetof(KWctx,tag));	BEGIN VAL; }
<INITIAL>"GROUP"	{ Saw_stok(Saw_group, offsetof(KWctx,group));	BEGIN VAL; }
<INITIAL>"LEN1"		{ Saw_stok(Saw_len, offsetof(KWctx,len));
			  lastkw1->kind |= KWKind_Len1OK;		BEGIN VAL; }
<INITIAL>"LEN"		{ Saw_stok(Saw_len, offsetof(KWctx,len));	BEGIN VAL; }
<INITIAL>":="		{ Saw_tok(Saw_dflt, KWKind_init); lastdef = 0;	BEGIN VAL; }
<INITIAL>">="		{ Saw_tok(Saw_ge, KWKind_Lb); btype = KWKind_caneqLb;	BEGIN VAL; }
<INITIAL>">"		{ Saw_tok(Saw_gt, KWKind_Lb); btype = KWKind_strictLb;	BEGIN VAL; }
<INITIAL>"<="		{ Saw_tok(Saw_le, KWKind_Ub); btype = KWKind_caneqUb;	BEGIN VAL; }
<INITIAL>"<"		{ Saw_tok(Saw_lt, KWKind_Ub); btype = KWKind_strictUb;	BEGIN VAL; }
<VAL>{PPname}		{ if (Saw_name(yytext))	BEGIN INITIAL; }
<VAL>{Num}		{ Saw_num(yytext);	BEGIN INITIAL; }
<VAL>\"			{ Saw_quote();		BEGIN INITIAL; }

"{"		{ Lbrace(); BEGIN FNC; }
"}"		{ Rbrace(); BEGIN INITIAL; }
"|"		{ Bar(); }
"("		{ Lpar(); }
")"		{ Rpar(); }
"["		{ Lsqb(); }
"]"		{ Rsqb(); }
<FNC>{Func}	{ Fnc(yytext); }
<FNC>";"	{}
"#".*\n		{++Lineno;}
"/*"		{ Eat_comment(); }
.	{ botch("\"%s\" unexpected", yytext); }

%%

 static void
cantopen(char *what)
{ fprintf(stderr, "%s: cannot open \"%s\"\n", progname, what); }

 int
yywrap(void)
{
	if (tagin) {
		if (tagin != stdin)
			fclose(tagin);
		tagin = 0;
		Lineno = 1;
		saweof = 1;
		if (*specfile == '-' && !specfile[1]) {
			yyin = stdin;
			infname = 0;
			return 0;
			}
		if ((yyin = fopen(infname = specfile, "r")))
			return 0;
		cantopen(specfile);
		}
	return 1;
	}

 static void
Eat_comment(void)
{
	int c;

	for(;;) {
		c = input();
 c_check:
		if (c == EOF)
			botch("EOF within /* comment */");
		if (c == '*') {
			c = input();
			if (c == '/')
				return;
			goto c_check;
			}
		if (c == '\n')
			++Lineno;
		}
	}

 static void
Saw_quote(void)
{
	char buf[1024];
	char *b, *b0, *be;
	int c, c0;
	size_t L, L0, L1;
	static char escname[32];
	static char esetup[] = "\aa\bb\ff\nn\rr\tt\vv";
	static char hexdig[16] = "0123456789abcdef";

	if (!escname['\a'])
		for(b = esetup; *b; b += 2)
			escname[*b] = b[1];	/* assuming ASCII */
	b0 = b = buf;
	L = sizeof(buf);
	be = b + L - 3;	/* allow room for /xhh */
	c0 = 0;
	saweof = 0;
	for(;;) {
		c = input();
		if (c == EOF)
			botch("EOF within quoted string");
		if (b >= be) {
			L1 = b - b0;
			L0 = L;
			L <<= 1;
			be = (char*)Malloc(L, "Saw_quote");
			memcpy(be, b0, L1);
			if (b0 != buf)
				free(b0);
			b0 = be;
			b = b0 + L1;
			be = b0 + L - 3;
			}
		if (c < ' ') {
			*b++ = '\\';
			if (escname[c]) {
				if (c == '\n')
					++Lineno;
				c = escname[c];
				}
			else {
				*b++ = 'x';
				*b++ = '0' + (c >> 4);
				c = hexdig[c & 0xf];
				}
			}
		*b++ = c;
		if (c0 == '\\')
			c0 = 0;
		else {
			c0 = c;
			if (c == '"') {
				while((c = input()) <= ' ') {
					if (c == EOF) {
						c = ';';
						break;
						}
					if (c == '\n')
						++Lineno;
					}
				if (c == '"' && !saweof) {
					--b;
					continue;
					}
				unput(c);
				break;
				}
			}
		}
	b[-1] = 0;	/* replacing " */
	if (lastseen == Saw_tag) {
		if (saweof) /* kludge around poor flex design */ {
			tagin = stdin;
			Saw_name(b0);
			tagin = 0;
			}
		else
			Saw_name(b0);
		}
	else {
		L = b - b0;
		be = (char*)Alloc(L);
		memcpy(be, b0, L);
		switch(lastseen) {
		  case Saw_dflt:
		  case Saw_dfltname:
			if ((lastkw1->kind & KWKind_Mask) != KWKind_Str)
				botch("%s cannot have a string-valued := value",
					lastkw1->names->val);
			lastkw1->cinit = be;
			if (lastdef)
				set_lastdef(b0, 1);
			break;
		  case Saw_desc:
			lastkw1->desc = be;
			break;
		  case Saw_group:
			lastkw1->group = be;
			break;
		  default:
			unexpected("string");
		  }
		}
	if (b0 != buf)
		free(b0);
	lastseen = Saw_type;
	}

 static int
usage(int rc)
{
	fprintf(rc ? stderr : stdout,
	"Usage: %s [options] specfile [tagfile [defs_out [keywds_out]]]\n\
	where the last two are output files, \".\" means \"omit\", and \"-\"\n\
	means write to stdout.  Missing trailing arguments are not read\n\
	or written, except that if all outputs are missing, \". -\" is assumed.\n\
	Options may modify this behavior.\noptions:\n\
	-D		{ just write definitions to stdout }\n\
	-d		{ suppress check for ambiguity from duplicate keywords }\n\
	-e		{ expand ((...)) for debugging }\n\
	-f		{ omit functions, i.e., {...} with -p; without -p,\n\
			  supply NULL function pointers }\n\
	-G		{ for keywords with a TAG but not GROUP, do not assume\n\
			  that implictly GROUP == TAG }\n\
	-g		{ write GuiKeyWord rather than KeyWord header;\n\
			  with -p, include GUI details: TAG, DESC, GROUP, LEN }\n\
	-h headername	{ Specify header file(s), in addition to nidr.h;\n\
			  several -h options can be given;\n\
			  default = -h- ==> no headers }\n\
	-m mtagfile	{ write to mtagfile tags not found in tagfile (and quit) }\n\
	-n namespname	{ assume namespace namespname;\n\
			  default = -n Dakota\n\
			  -n- ==> no namespace }\n\
	-p[w]		{ pretty-print rather than produce KeyWord header;\n\
			  add w spaces (default 2) for each nested context. }\n\
	-s[w]		{ like -p[w], but sort the keywords }\n\
	-T outtagfile	{ write dummy tagfile to outtagfile (and quit) }\n\
	-t		{ ignore TAG fields if tagfile is not given }\n\
	-w		{ cancel warn mode: do not report (on stderr)\n\
			  the number of implicitly named start routines;\n\
			  still flag their names  with /*!!*/ }\n",
		progname);
	return rc;
	}

 static StrList *
reverse(StrList **x)
{
	StrList *s, *t, *u;

	t = 0;
	for(s = *x; s; s = u) {
		u = s->next;
		s->next = t;
		t = s;
		}
	return *x = t;
	}

 static int
kwcomp(const void *a, const void *b)
{
	return strcmp((*(const KWctx**)a)->names->val, (*(const KWctx**)b)->names->val);
	}

 static KWctx *
kwsort(KWctx *kw0)
{
	/* Sort keywords after expanding kw list with aliases; compute nkw, */
	/* including aliases. */

	KWctx *kw, *kw1, *kw2, *kwm, **x, **x0, **xe, *x00[512];
	StrList *dname, *minname, *nam, **pmin, **pnam;
	char *descsave, *groupsave;
	int agadj, nk;
	size_t L, nkw, nkwa, onkwa;

	reverse(&kw0->funcs);
	nkw = nkwa = 0;
	for(kw = kw0->kw; kw; kw = kw->next) {
		++nkw;
		pmin = &kw->names;
		minname = *pmin;
		pnam = &minname->next;
		onkwa = nkwa;
		while((nam = *pnam)) {
			dname = nam;
			++nkwa;
			if (strcmp(nam->val, minname->val) < 0) {
				pmin = pnam;
				minname = nam;
				}
			pnam = &nam->next;
			}
		agadj = 2*kw->agroup;
		kw->agroup = agadj + 1;
		if (nkwa > onkwa) {
			*pmin = minname->next;
			minname->next = kw->names;
			nam = kw->names = minname;
			nk = kw->kno;	/* ALias() made room for nk use below */
			descsave = kw->desc;
			groupsave = kw->group;
			if (dname != minname) {
				kw->desc = kw->group = 0;
				kw->agroup = agadj;
				}
			kwm = kw;
			kw = kw2 = 0;
			if (nam->primary)
				kw2 = kwm;
			while((nam = nam->next)) {
				kw1 = (KWctx*)Alloc(sizeof(KWctx));
				if (!kw)
					kw = kw1;	/* so kw->next is right */
				kw1->next = kwm->next;
				kwm->next = kw1;
				kw1->names = nam;
				kw1->master = kwm;
				kw1->kind = kwm->kind;
				kw1->kno = ++nk;
				kw1->agroup = agadj;
				if (nam == dname) {
					kw1->desc = descsave;
					kw1->group = groupsave;
					kw1->agroup = agadj + 1;
					}
				if (nam->primary)
					kw2 = kw1;
				}
			for(kw1 = kwm;; kw1 = kw1->next) {
				if (kw1 != kw2)
					kw1->pmaster = kw2;
				if (kw1 == kw)
					break;
				}
			}
		}
	kw0->nkw = (int)(nkw += nkwa);
	if (nkw <= 1)
		return kw0->kw;
	x0 = x00;
	if (nkw > sizeof(x00)/sizeof(x00[0]))
		x0 = (KWctx**)Malloc(L = nkw*sizeof(KWctx*), "kwsort");
	for(x = x0, kw = kw0->kw; kw; kw = kw->next)
		*x++ = kw;
	qsort(x0, nkw, sizeof(KWctx*), kwcomp);
	for(xe = x - 1, x = x0; x < xe; ++x) {
		if (!strcmp(x[0]->names->val, x[1]->names->val)) {
			fprintf(stderr, "%s: duplicate appearance of \"%s\"\n",
				progname, x[0]->names->val);
			++dupnames;
			}
		x[0]->next = x[1];
		}
	x[0]->next = 0;
	for(x = x0, ++xe; x < xe; ++x)
		(*x)->paoff = x - x0;
	for(x = x0; x < xe; ++x)
		if ((kw1 = (kw = *x)->pmaster))
			kw->paoff = kw1->paoff - kw->paoff;
	for(x = x0; x < xe; ++x)
		if (!(kw = *x)->pmaster)
			kw->paoff = 0;
	kw0->kw = kw = x0[0];
	if (x0 != x00)
		free(x0);
	return kw;
	}

 static int
alphanum(char *s)
{
	for(; *s; ++s)
		if (!isalnum(*s))
			return 0;
	return 1;
	}

 typedef struct Zbuf Zbuf;
 struct
Zbuf {
	char buf[80];	/* overkill */
	char *b;
	};

 static void
iput(int i, Zbuf *z)
{
	char *s;

	if (i == 0) {
		s = z->b;
		*s++ = ',';
		*s++ = '0';
		z->b = s;
		}
	else {
		if (z->b > z->buf) {
			*z->b = 0;
			printf(z->b = z->buf);
			}
		printf(",%d", i);
		}
	}

 static void
nput(char *s, Zbuf *z)
{
	char *fmt, *s1;

	if (!s || (strtod(s,&s1) == 0. && !*s1)) {
		s = z->b;
		*s++ = ',';
		*s++ = '0';
		*s++ = '.';
		z->b = s;
		}
	else {
		fmt = ",%s.";
		for(s1 = s; *s1; ++s1)
			switch(*s1) {
				case '.':
				case 'e':
				case 'E':
					fmt = ",%s";
					goto have_fmt;
				}
 have_fmt:
		if (z->b > z->buf) {
			*z->b = 0;
			printf(z->b = z->buf);
			}
		printf(fmt, s);
		}
	}

 static void
sput(char *s, Zbuf *z)
{
	if (!s) {
		s = z->b;
		*s++ = ',';
		*s++ = '0';
		z->b = s;
		}
	else {
		if (z->b > z->buf) {
			*z->b = 0;
			printf(z->b = z->buf);
			}
		printf(",\"%s\"", s);
		}
	}

 static void
kwout(KWctx *kw, KWctx **kwtop)
{
	KWctx *kw0, *kw1, *kw2, *kwm, **kws, **x, *x0[KWOUTlen0];
	StrList *S;
	Zbuf z;
	char *fmt, *s;
	int aoff, k, i, m, m1, n, roff, sentinnel;

	if (expand && !kw->names)
		return;
	if (needcomma) {
		printf(",\n");
		needcomma = 0;
		}
	kw1 = kw->kw;
	n = 0;
	if (kw->names) {
		kw0 = kw1;
		if (kw->parstate != KwStacked)
			for(kw2 = kw->pparent; kw2; kw2 = kw2->pparent) {
				kw2->next = kw1;
				kw1 = kw2;
				++n;
				}
		if (!kw0) {
			if (!--n) {
				kw->nkw = kw1->nkw;
				if (kw1->kwkno) {
					kw->kwkno = kw1->kwkno;
					return;
					}
				kw->kwkno = lastkno + 1;
				kw = kw1;
				kw1 = kw->kw;
				}
			else {
				kw2 = kw->pparent;
				if (kw2->kwknop) {
					kw->kwkno = kw2->kwknop;
					return;
					}
				kw2->kwknop = lastkno + 1;
				kw1 = kw2;
				while((kw2 = kw2->pparent)) {
					kw2->next = kw1;
					kw1 = kw2;
					}
				}
			}
		}
	for(m = 0, kw0 = kw1; kw0; kw0 = kw0->next)
		++m;
	if (expand)
		for(kw0 = kw1; kw0 && !kw0->names; kw0 = kw0->next)
			for(kw2 = kw0->kw; kw2; kw2 = kw2->next)
				++m;
	x = x0;
	if (m > KWOUTlen0)
		x = (KWctx**)Malloc(m*sizeof(KWctx*), "kwout");
	m = 0;
	kw0 = kw1;
	if (expand) {
		aoff = kw->nalt;
		roff = kw->nreq;
		for(; kw0 && !kw0->names; kw0 = kw0->next) {
			for(kw2 = kw0->kw; kw2; kw2 = kw2->next) {
				kw2->altoffset = aoff;
				kw2->reqoffset = roff;
				x[m++] = kw2;
				}
			aoff += kw0->nalt;
			roff += kw0->nreq;
			}
		kw->nalt = aoff;
		kw->nreq = roff;
		}
	for(; kw0; kw0 = kw0->next)
		x[m++] = kw0;
	sentinnel = 0;
	if (expand) {
		kw->nkw = m;
		qsort(x, m, sizeof(KWctx*), kwcomp);
		for(i = 1; i < m; ++i)
			if (!strcmp(x[i-1]->names->val, x[i]->names->val)) {
				fprintf(stderr, "%s: duplicate appearance of \"%s\"\n",
					progname, x[i]->names->val);
				++dupnames;
				}
		}
	else if (kw->nkw == 0)
		sentinnel = 1;
	m1 = m + sentinnel;
	printf("\tkw_%d[%d] = {\n", kw->kwkno = ++lastkno, m1);
	needcomma = 1;
	for(i = 0; i < m;) {
		kw1 = x[i++];
		if (!(kwm = kw1->master))
			kwm = kw1;
		fputs("\t\t{", stdout);
		z.b = z.buf;
		if (kw1->names) {
			k = kw1->kind | kw1->names->primary;
			printf(k < 16
				? "\"%s\",%d"
				: "\"%s\",0x%x", kw1->names->val, k);
			iput(kwm->nkw, &z);
			iput(kwm->alt + kwm->altoffset, &z);
			iput(kwm->req + kwm->reqoffset, &z);
			}
		else {
			printf("0,0");
			iput(kwm->nkw, &z);
			iput(0, &z);
			iput(0, &z);
			}
		if (guikeywds)
			iput(kw1->agroup, &z);
		if ((n = kwm->kwkno)) {
			if (z.b > z.buf) {
				*z.b = 0;
				printf(z.b = z.buf);
				}
			printf(",kw_%d", n);
			}
		else
			iput(0, &z);
		nput(kw1->lb, &z);
		nput(kw1->ub, &z);
		if (guikeywds) {
			nput(kw1->init, &z);
			sput(kw1->cinit, &z);
			sput(kw1->desc, &z);
			if (!(s = kw1->group) && dfltgroup)
				s = kw1->tag;
			sput(s, &z);
			if (kwm->len)
				sput(kwm->len->names->val, &z);
			goto nofuncs;
			}
		iput(kw1->paoff, &z);
		if ((S = kw1->funcs)) {
			if (z.b > z.buf) {
				*z.b = 0;
				printf(z.b = z.buf);
				}
			if (!kw1->kw) {
				reverse(&kw1->funcs);
				S = kw1->funcs;
				}
			goto funcput;
			}
		if ((S = kwm->funcs)) {
 funcput:
			if (S->next || strcmp(S->val,"0")) {
				if (z.b > z.buf) {
					*z.b = 0;
					printf(z.b = z.buf);
					}
				if (!wantfuncs)
					printf(",dummystart");
				else for(n = 0; S; S = S->next, ++n) {
					s = S->val;
					if (!(n & 1) || !strcmp(s,"0"))
						fmt = ",%s";
					else if (alphanum(s))
						fmt = ",(void*)%s";
					else
						fmt = ",(void*)(%s)";
					printf(fmt, s);
					}
				}
			}
		else if (kwm->names) {
			++missing;
			printf(",/*!!*/");
			for(kws = KWStack; ++kws < kwtop; ) {
				if (!(kw2 = *kws)->names)
					kw2 = kw2->rparent;
				printf("%s_", kw2->names->val);
				}
			printf("%s_start", kwm->names->val);
			}
 nofuncs:
		printf("}%s\n", i < m1 ? "," : "");
		}
	if (sentinnel)
		fputs("\t\t{\"\"}\n", stdout);
	printf("\t\t}");
	if (x != x0)
		free(x);
	}

 static void
hashclear(KWctx *kw)
{
	StrList *sl;

	kw->parstate = Dupchecked;
	for(kw = kw->kw; kw; kw = kw->next)
		for(sl = kw->names; sl; sl = sl->next) {
			sl->h.hnext->hprev = sl->h.hprev;
			sl->h.hprev->hnext = sl->h.hnext;
			sl->h.hprev = sl->h.hnext = 0;
			}
	}

 static void
dupfound(HashHead *h0, StrList *sl, StrList *sl1)
{
	StrList *sl2, *sl3;

	++dupnames;
	for(sl2 = sl1; sl2->h.hnext != h0; sl2 = sl3) {
		sl3 = (StrList*)sl2->h.hnext;
		if (strcmp(sl->val, sl3->val))
			break;
		}
	fprintf(stderr, "Warning: \"%s\" on line %d also appears on line%s %d",
		sl->val, sl->Lineno, "s" + (sl2 == sl1), sl1->Lineno);
	while(sl1 != sl2) {
		sl1 = (StrList*)sl1->h.hnext;
		fprintf(stderr, ", %d", sl1->Lineno);
		}
	fputs(".\n", stderr);
	}

 static KWctx **
hashadd(KWctx *kw, KWctx **kwtop)
{
	HashHead *h, *h0;
	KWctx *kw1;
	StrList *sl, *sl1;
	char *s;
	int n;
	unsigned int c, x;

	kw->parstate = Dupchecking;
	for(kw1 = kw->kw; kw1; kw1 = kw1->next)
		for(sl = kw1->names; sl; sl = sl->next) {
			x = 0;
			s = sl->val;
			while((c = *(unsigned char*)s++))
				x = 43*x + c - ' ';
			h0 = &HTab[x % HTlen];
			for(h = h0->hnext; h != h0; h = h->hnext) {
				sl1 = (StrList*)h;
				n = strcmp(sl->val, sl1->val);
				if (n <= 0) {
					if (n == 0)
						dupfound(h0,sl,sl1);
					break;
					}
				}
			sl->h.hnext = h;
			h->hprev = (sl->h.hprev = h->hprev)->hnext = (HashHead *)sl;
			}
	if ((kw1 = kw->kw))
		*++kwtop = kw1;
	return kwtop;
	}

 static void
dup_check(void)
{
	HashHead *h, *he;
	KWctx *kw, *kw1, *kw2, **kwtop, **kwbot, *plist;
	int lev;

	for(h = HTab, he = h + HTlen; h < he; ++h)
		h->hnext = h->hprev = h;
	plist = 0;
	kwtop = kwbot = KWStack;
	*kwtop = &KWbase;
	for(;;) {
		kw = *kwtop;
		lev = kw->level;
		while(plist && plist->level >= lev) {
			hashclear(plist);
			plist = plist->next;
			}
		if (kw->parstate == Dupchecking) {
			hashclear(kw);
			if ((*kwtop = kw->next))
				continue;
			if (kwtop <= kwbot)
				break;
			--kwtop;
			continue;
			}
		if ((kw1 = kw->pparent) && kw1->parstate != Dupchecking) {
			for(kw2 = kw1; plist && kw2; plist = plist->next) {
				while(kw2->level > plist->level && (kw2 = kw2->pparent));
				if (kw2 != plist)
					break;
				hashclear(plist);
				}
			kw2 = 0;
			do {
				if (kw1->kw) {
					kw1->next = kw2;
					kw2 = kw1;
					}
				}
				while((kw1 = kw1->pparent) && kw1->parstate != Dupchecking);
			while(kw2) {
				kwtop = hashadd(kw2, kwtop);
				kw1 = kw2;
				kw2 = kw2->next;
				kw1->next = plist;
				plist = kw1;
				}
			}
		kwtop = hashadd(kw, kwtop);
		}
	}

 static void
prkwname(KWctx *kw, const char *suf, int needrev)
{
	int k;
	StrList *S;
	static const char *kname[] = { "BUG", "INTEGER", "REAL", "STRING" };

	if (needrev)
		reverse(&kw->names);
	S = kw->names;
	printf("%s", S->val);
	while((S = S->next))
		printf(" ALIAS %s", S->val);
	if ((k = kw->kind & KWKind_Mask)) {
		printf(" %s", kname[k]);
		if (kw->kind & KWKind_List)
			printf("LIST");
		}
	if (kw->lb)
		printf(" %s %s", kw->kind & KWKind_strictLb ? ">" : ">=", kw->lb);
	if (kw->ub)
		printf(" %s %s", kw->kind & KWKind_strictUb ? "<" : "<=", kw->ub);
	if (kw->defname && wantfuncs) {
		if (kw->init)
			printf(" := %s %s", kw->defname, kw->init);
		else if (kw->cinit)
			printf(" := %s \"%s\"", kw->defname, kw->cinit);
		}
	else {
		if (kw->init)
			printf(" := %s", kw->init);
		else if (kw->cinit)
			printf(" := \"%s\"", kw->cinit);
		}
	if (!wantfuncs)
		goto sufchk;
	if (kw->tag)
		printf(ignoretags ? " TAG \"%s\"" : " /*TAG \"%s\"*/", kw->tag);
	if (kw->funcs) {
		fputs(" {", stdout);
		reverse(&kw->funcs);
		for(S = kw->funcs;;) {
			fputs(S->val, stdout);
			if (!(S = S->next))
				break;
			putchar(',');
			}
		putchar('}');
		}
	if (kw->len)
		printf(" LEN %s", kw->len->names->val);
	if (kw->group)
		printf(" GROUP \"%s\"", kw->group);
	if (kw->desc)
		printf(" DESC \"%s\"", kw->desc);
 sufchk:
	if (suf)
		fputs(suf, stdout);
	if (needrev == 2)
		reverse(&kw->names);
	}

 static KWctx *
kwrev(KWctx **pkw)
{
	KWctx *kw, *kw0, *kw1;
	kw1 = *pkw;
	for(kw0 = 0; (kw = kw1); kw0 = kw) {
		kw1 = kw->next;
		kw->next = kw0;
		}
	return *pkw = kw0;
	}

 static void
paradj(int needrev)
{
	KWctx *kw, *kw1, **kwp, **kwtop, **kwbot;

	kwtop = kwbot = KWStack;
	*kwtop = &KWbase;
	if (needrev)
		kwrev(kwtop);
	for(;; --kwtop) {
		for(kw = *kwtop; kw; kw = kw->next) {
 top:
			if (needrev && (kw1 = kw->kw) && kw1->next)
				kwrev(&kw->kw);
			if ((kw1 = kw->pparent) && kw1->parstate != Parchecked) {
				kwp = &kw->pparent;
				for(;;) {
					if (kw1->kw) {
						*kwtop++ = kw1;
						kw1->parstate = Parchecked;
						*kwp = kw1;
						kwp = &kw1->pparent;
						}
					if (!(kw1 = *kwp = kw1->pparent)
					 || kw1->parstate == Parchecked)
						break;
					}
				}
			if (kw->kw) {
				if ((*kwtop = kw->next))
					++kwtop;
				kw = kw->kw;
				goto top;
				}
			}
		if (kwtop <= kwbot)
			break;
		}
	}

 static KWctx *
masteradj(KWctx *kw)
{
	/* to help print (nm1 | nm2 foo goo), make nm2 "master" */
	KWctx *kw1, *kw2, *kw3, *kwm, *rv;

	for(kw1 = kw, kw3 = 0; kw1; kw1 = kw1->next) {
		if ((kw2 = kw1->master))
			kw3 = kw2->master = kw2;
		}
	if (!kw3)
		return kw;
	for(rv = kw; kw; kw = kw->next) {
		kwm = kw->master;
		if ((kw1 = kw->next) && kw1->master == kwm && kwm) {
			kw3 = kw->kw ? kw : 0;
			do {
				kw2 = kw1;
				if (kw1->kw) {
					if (kw3)
						botch("Two masters in masteradj!");
					kw3 = kw1;
					}
				}
				while((kw1 = kw1->next) && kw1->master == kwm);
			if (kw3 || kw2->pparent) {
				if (kw2 != kw3) {
					kw2->pparent = kw3->pparent;
					kw2->kw = kw3->kw;
					kw3->kw = 0;
					}
				for(;;kw = kw->next) {
					kw->master = kw2;
					if (kw == kw2)
						break;
					}
				}
			else
				for(;;kw = kw->next) {
					kw->master = 0;
					if (kw == kw2)
						break;
					}
			kw = kw2;
			}
		}
	return rv;
	}

 static void
xmasteradj(KWctx **x, int n)
{
	/* variant of masteradj for pretty_ex1 */
	KWctx *kw, *kw1, *kw2, *kw3, *kwm;
	int i, j;

	for(i = 0, kw3 = 0; i < n; ++i) {
		kw1 = x[i];
		if ((kw2 = kw1->master))
			kw3 = kw2->master = kw2;
		}
	if (!kw3)
		return;
	for(i = 0; i < n; i = j) {
		j = i + 1;
		kw = x[i];
		kwm = kw->master;
		if (j < n && kwm && (kw1 = x[j])->master == kwm) {
			kw3 = kw->kw ? kw : 0;
			do {
				kw2 = kw1;
				if (kw1->kw) {
					if (kw3)
						botch("Two masters in xmasteradj!");
					kw3 = kw1;
					}
				}
				while(++j < n && (kw1 = x[j])->master == kwm);
			if (kw3 || kw2->pparent) {
				if (kw2 != kw3) {
					kw2->pparent = kw3->pparent;
					kw2->kw = kw3->kw;
					kw3->kw = 0;
					}
				for(;;kw = x[++i]) {
					kw->master = kw2;
					if (kw == kw2)
						break;
					}
				}
			else
				for(;; kw = x[++i]) {
					kw->master = 0;
					if (kw == kw2)
						break;
					}
			}
		}
	}

 static void
do_indent(int indent)
{
	for(indent += 8; indent >= 8; indent -= 8)
		putchar('\t');
	for(; indent > 0; --indent)
		putchar(' ');
	}

 static void
do_indent1(int indent)	/* write indent-1 spaces */
{
	while(--indent > 0)
		putchar(' ');
	}

 static const char *KEYKIND[4] = { "KEYWORD ", "KEYWORD01 ", "KEYWORD1 ", "KEYWORD12 " };

 static void
pretty_print(int iinc, int needrev)
{
	KWctx *kw, *kw1, *kw2, *kw3, **kwtop, **kwbot, *plist;
	const char *s;
	int alt, i, indent, lev, needindent, needspace;

	kwtop = kwbot = KWStack;
	indent = 0;
	plist = 0;
	needindent = 1;
	for(kw1 = KWbase.kw; kw1;) {
		fputs(KEYKIND[(kw1->kind >> KWtopshift) & 3], stdout);
		prkwname(kw1, "\n", needrev);
		*kwtop = masteradj(kw1->kw);
		alt = 0;
		for(;;--kwtop) {
	 top:
			for(kw = *kwtop; kw; kw = kw->next) {
				lev = kw->level;
				if (needindent) {
					do_indent(indent);
					needindent = 0;
					}
				if (plist
				 && (plist->level > lev
				  || (plist->level == lev && plist != kw->pparent))) {
 finishpar:
					*kwtop = kw;
					*++kwtop = plist;
					*++kwtop = masteradj(plist->kw);
					plist->alt = alt;
					alt = 0;
					plist = plist->next;
					goto top;
					}
				if (kw->parstate == Printing) {
					s = !kw->names || /*[(*/kw->req ? ")\n" : "]\n";
					if (kw->kw && !kw->req
					 && kw->next && kw->next->alt == kw->alt) {
						s = /*(*/ ")\n";
						indent -= iinc;
						}
					fputs(s, stdout);
					needindent = 1;
					alt = kw->alt;
					indent -= iinc;
					if (!kw->names) {
						alt = kw->alt;
						goto bot;
						}
					continue;
					}
				if (plist)
					for (kw2 = kw->pparent;
							kw2 && kw2->parstate != Printing;
							kw2 = kw2->pparent) {
						if (plist->level >= kw2->level)
							goto finishpar;
						}
				needindent = 1;
				needspace = 0;
				if (alt == kw->alt) {
					putchar('|');
					needspace = 1;
					}
				if ((kw2 = kw->pparent) && kw2->parstate != Printing) {
					kw3 = 0;
					do {
						if (kw2->parstate == Printing)
							break;
						if (kw2->kw) {
							kw2->next = kw3;
							kw3 = kw2;
							kw2->parstate = Printing;
							if (needspace) {
								putchar('\n');
								needspace = 0;
								do_indent(indent);
								}
							indent += iinc;
							putchar('('); /*)*/
							do_indent1(iinc);
							}
						}
						while((kw2 = kw2->pparent));
					while(kw3) {
						kw2 = kw3->next;
						kw3->next = plist;
						plist = kw3;
						kw3 = kw2;
						}
					}
				if ((kw2 = kw->master) && kw == kw2)
					kw2 = 0;
				if (!kw->kw && (!kw2 || kw2->parstate == Leftput)) {
					if (needspace)
						putchar(' ');
					if (!kw->req) {
						if (alt != kw->alt) {
							fputs("[ ", stdout);
							alt = kw->alt;
							}
						s = /*[*/ " ]\n";
						if ((kw2 = kw->next) && kw2->alt == alt)
							s = "\n";
						prkwname(kw, s, needrev);
						continue;
						}
					alt = kw->alt;
					prkwname(kw, "\n", needrev);
					continue;
					}
				if (kw->parstate == Leftput) {
					if (needspace)
						goto putspace;
					}
				else {
					if (needspace) {
						putchar('\n');
						do_indent(indent);
						}
					putchar(kw->req ? '(' : '['); /*])*/
					if (kw->kw && !kw->req
					 && kw->next && kw->next->alt == kw->alt) {
						do_indent1(iinc);
						putchar('('/*)*/);
						indent += iinc;
						}
					indent += iinc;
 putspace:
					do_indent1(iinc);
					}
				kw->parstate = Printing;
				prkwname(kw, "\n", needrev);
				if (kw2) {
					alt = kw->alt;
					kw2->parstate = Leftput;
					continue;
					}
				alt = 0;
				*kwtop = kw;
				*++kwtop = masteradj(kw->kw);
				goto top;
				}
 bot:
			if (kwtop == kwbot) {
				if (plist) {
					*kwtop++ = plist;
					*kwtop++ = masteradj(plist->kw);
					plist->alt = alt;
					alt = 0;
					plist = plist->next;
					}
				else {
					if (indent) {
						if (needindent)
							do_indent(indent);
						fputs(/*(*/")\n", stdout);
						needindent = 1;
						indent -= iinc;
						}
					break;
					}
				}
			}
		if (!(kw1 = kw1->next))
			break;
		putchar('\n');
		}
	}

 static void
Output(StrList *headers, StrList *nsname, int hwant, int nwant)
{
	KWctx *kw, *kw1, *kw2, **kwtop, **kwbot;
	StrList *h;

	if (hwant && headers) {
		printf("#include \"nidr.h\"\n");
		for(h = reverse(&headers); h; h = h->next)
			printf(*h->val == '<' /*>*/
				? "#include %s\n"
				: "#include \"%s\"\n", h->val);
		}
	if (nwant) {
		if (!nsname)
			nsname = new_StrList("Dakota", nsname);
		putchar('\n');
		for(h = reverse(&nsname); h; h = h->next)
			printf("namespace %s {\n", h->val); /*}*/
		}
	printf("\n/** %d distinct keywords (plus %d aliases) **/\n\nstatic %sKeyWord\n",
		lastagroup, nalias, guikeywds ? "Gui" : "");
	kwtop = kwbot = KWStack;
	*kwtop = &KWbase;
	for(;;--kwtop) {
 top:
		for(kw = *kwtop; kw; kw = kw->next) {
			if ((kw->master && kw->master != kw)
			|| ((!(kw1 = kw->pparent) || kw->parstate == KwStacked) && !kw->kw)
			|| kw->kwkno)
				continue;
			if (kw1 && kw1->parstate != KwStacked) {
				if ((kw1 = kw->pparent) && kw1->parstate != KwStacked) {
					kw1->parstate = KwStacked;
					*kwtop = kw;
					*++kwtop = kw1;
					kw2 = *++kwtop = kwsort(kw1);
					do kw2->parstate = KwStacked;
						while((kw2 = kw2->next));
					while((kw1 = kw1->pparent) && kw1->parstate != KwStacked) {
						kw1->parstate = KwStacked;
						*++kwtop = kw1;
						kw2 = *++kwtop = kwsort(kw1);
						do kw2->parstate = KwStacked;
							while((kw2 = kw2->next));
						}
					goto top;
					}
				if (!kw1 && !kw->kw)
					continue;
				}
			if (!kw->nkw && kw->kw) {
				*kwtop = kw;
				*++kwtop = kwsort(kw);
				goto top;
				}
			kwout(kw, kwtop);
			}
		if (kwtop == kwbot)
			break;
		}
	printf(";\n\n");
	if (nwant) {
		for(h = reverse(&nsname); h; h = h->next)
			printf("} // namespace %s\n", h->val);
		putchar('\n');
		}
	printf("KeyWord Dakota_Keyword_Top = {\"KeywordTop\",0,%d,%d,%d",
		KWbase.nkw, KWbase.alt, KWbase.req);
	putchar(',');
	if (nwant)
		for(h = reverse(&nsname); h; h = h->next)
			printf("%s::", h->val);
	printf("kw_%d};\n", KWbase.kwkno);
	if (startwarn && missing && !guikeywds)
		fprintf(stderr, "%s: %d start routines implicitly named.\n",
			progname, missing);
	}

 typedef int (*KWcomp)(const void*, const void*);

 static const char **zalt;

 static int
kwcomp1(const void *a, const void *b)
{
	const KWctx *ka, *kb;

	ka = *(const KWctx**)a;
	kb = *(const KWctx**)b;
	if (ka->req) {
		if (!kb->req)
			return -1;
		}
	else if (kb->req)
		return 1;
	if (ka->alt == kb->alt) {
		if (ka->master != kb->master) {
			if (!ka->master)
				return -1;
			if (!kb->master)
				return 1;
			ka = ka->master;
			kb = kb->master;
			}
		return strcmp(ka->names->val, kb->names->val);
		}
	return strcmp(zalt[ka->alt], zalt[kb->alt]);
	}

 static int
kwcomp2(const void *a, const void *b)
{
	const KWctx *ka, *kb, *pa, *pb;

	ka = *(const KWctx**)a;
	kb = *(const KWctx**)b;
	if (ka->req) {
		if (!kb->req)
			return -1;
		}
	else if (kb->req)
		return 1;
	if (ka->alt == kb->alt) {
		pa = ka->master ? ka->master->pparent : ka->pparent;
		pb = kb->master ? kb->master->pparent : kb->pparent;
		if (pa != pb)
			return ka->objno - kb->objno;
		if (ka->master != kb->master) {
			if (!ka->master)
				return -1;
			if (!kb->master)
				return 1;
			ka = ka->master;
			kb = kb->master;
			}
		return strcmp(ka->names->val, kb->names->val);
		}
	return strcmp(zalt[ka->alt], zalt[kb->alt]);
	}

 static void
do_sort(KWctx *kw, KWcomp KWcmp)
{
	KWctx *kw1, *kwm, **x, *x0[KWPlen];
	const char *z0[KWPlen];
	int i, n, na;

	for(n = 0, kw1 = kw->kw; kw1; kw1 = kw1->next) {
		++n;
		reverse(&kw1->names);
		}
	if (n <= 1)
		return;
	x = n > KWPlen ? (KWctx**)Malloc(n*sizeof(KWctx*), "do_sort") : x0;
	na = kw->nalt + 1;
	zalt = na <= KWPlen ? z0 : (const char**)Malloc(na*sizeof(const char*), "do_sort");
	memset(zalt, 0, na*sizeof(const char*));
	for(n = 0, kw1 = kw->kw; kw1; kw1 = kw1->next) {
		x[n++] = kw1;
		if ((kwm = kw1->master))
			kwm->master = kwm;
		if (!zalt[i = kw1->alt]
		 || strcmp(kw1->names->val, zalt[i]) < 0)
			zalt[i] = kw1->names->val;
		}
	qsort(x, n--, sizeof(KWctx*), KWcmp);
	for(i = 0; i < n; ++i)
		x[i]->next = x[i+1];
	x[n]->next = 0;
	kw->kw = x[0];
	if (zalt != z0)
		free(zalt);
	if (x != x0)
		free(x);
	}

 static void
ksort(KWcomp KWcmp)
{
	KWctx *kw, *kw1, **kwtop, **kwtop0, **kwbot;

	kwtop = kwbot = KWStack;

	for(*kwtop = &KWbase;; --kwtop) {
 top:
		kw = *kwtop;
 top1:
		while(kw) {
			if ((kw1 = kw->pparent) && kw1->parstate != Sorted) {
				*kwtop = kw;
				kwtop0 = kwtop;
				do {
					kw1->parstate = Sorted;
					*++kwtop = kw1;
					}
					while((kw1 = kw1->pparent) && kw1->parstate != Sorted);
				if (kwtop > kwtop0)
					goto top;
				}
			if (kw->kw) {
				do_sort(kw, KWcmp);
				if ((*kwtop = kw->next))
					++kwtop;
				kw = *kwtop = kw->kw;
				goto top1;
				}
			kw = kw->next;
			}
		if (kwtop <= kwbot)
			break;
		}
	}

 static void
pretty_ex1(KWctx *kw, KWctx *prevkw, KWctx *nextkw, int indent, int iinc, int needrev)
{
	KWctx *kw0, *kw1, *kw2, *kwm, *kwp, *kwp0, **x, *y, *y1;
	const char *suf, *z0[KWPlen];
	int ep, i, i0, ia, j, n, na, np;

	x = 0;
	kw1 = kw->kw;
	if (needrev == 1)
		needrev = 2;
	if ((kwp0 = kwp = kw->pparent) && (!kw->master || kw->master == kw)) {
		for(n = 0;;) {
			if (kwp->parstate != Printing) {
				kwp->parstate = Printing;
				for(kw1 = kwp->kw; kw1; kw1 = kw1->next)
					kw1->pparent = 0;	/* avoid infinite loops */
				}
			for(kw1 = kwp->kw; kw1; kw1 = kw1->next)
				++n;
			if (!(kw1 = kwp->pparent))
				break;
			kwp = kwp->pparent = kw1;
			}
		np = n;
		for(kw1 = kw->kw; kw1; kw1 = kw1->next)
			++n;
		y = y1 = (KWctx*)Malloc(np*sizeof(KWctx) + n*sizeof(KWctx*), "pretty_ex1");
		x = (KWctx**)(y + np);
		for(i = 0, kw1 = kw->kw; kw1; kw1 = kw1->next)
			x[i++] = kw1;
		i0 = i;
		for(kwp = kwp0; kwp; kwp = kwp->pparent) {
			j = i;
			na = i + kwp->nalt;
			for(kw1 = kwp->kw; kw1; kw1 = kw1->next, ++y1) {
				memcpy(y1, kw1, sizeof(KWctx));
				x[i++] = y1;
				y1->alt += j;
				}
			}
		if (!needrev) {
			for(j = i0; j < n; ++j) {
				kw1 = x[j];
				kw1->nreq = -(kw1->nreq + 1);
				}
			zalt = ++na <= KWPlen ? z0
				: (const char**)Malloc(na*sizeof(const char*), "pretty_ex1");
			memset(zalt, 0, na*sizeof(const char*));
			for(j = 0; j < n; ++j) {
				kw1 = x[j];
				if ((kwm = kw1->master))
					kwm->master = kwm;
				if (!zalt[i = kw1->alt]
				 || strcmp(kw1->names->val, zalt[i]) < 0)
					zalt[i] = kw1->names->val;
				}
			qsort(x, n, sizeof(KWctx*), kwcomp1);
			if (zalt != z0)
				free(zalt);
			for(j = 0; j < n; ++j) {
				kw1 = x[j];
				if (kw1->nreq < 0) {
					kw1->nreq = -(kw1->nreq + 1);
					kw1->pparent = 0;
					}
				}
			}
		kw1 = x[0];
		xmasteradj(x, n);
		}
	if (kw1) {
		if (!x)
			kw1 = masteradj(kw->kw);
		ep = kw->req;
		if (prevkw && kw->alt == prevkw->alt) {
			ia = kw->master && kw->master == prevkw->master ? iinc : 0;
			do_indent(indent + ia);
			if (!prevkw->master || prevkw->master != kw->master) {
				fputs("|\n", stdout);
				do_indent(indent + ia);
				putchar('(');
				}
			else
				putchar('|');
			}
		else {
			do_indent(indent);
			if (kw->req)
				putchar('('/*)*/);
			else {
				putchar('[');
				if (nextkw && !nextkw->req && kw->alt == nextkw->alt) {
					ep = 1;
					do_indent1(iinc);
					putchar('(');
					indent += iinc;
					}
				}
			}
		do_indent1(iinc);
		indent += iinc;
		prkwname(kw, "\n", needrev);
		i = 0;
		kw0 = 0;
		while(kw1) {
			if (x)
				kw2 = ++i >= n ? 0 : x[i];
			else
				kw2 = kw1->next;
			pretty_ex1(kw1, kw0, kw2, indent, iinc, needrev);
			kw0 = kw1;
			kw1 = kw2;
			}
		if (x)
			free(y);
		do_indent(indent);
		fputs(ep ? ")\n" : "]\n", stdout);
		}
	else {
		if (prevkw && kw->alt == prevkw->alt
		 && kw->master && kw->master == prevkw->master)
			indent += iinc;
		do_indent(indent);
		suf = "\n";
		if (prevkw && kw->alt == prevkw->alt) {
			if (kw->master && kw->master != prevkw->master) {
				fputs("|\n", stdout);
				do_indent(indent);
				fputs("( ", stdout);
				}
			else {
				fputs("| ", stdout);
				if (!kw->req)
					goto rbcheck;
				}
			}
		else if (kw->req) {
			if (prevkw && prevkw->master != kw->master)
				fputs("( ", stdout);
			}
		else {
			fputs("[ ", stdout);
 rbcheck:
			if (!nextkw || nextkw->alt != kw->alt)
				suf = " ]\n";
			}
		prkwname(kw, suf, needrev);
		}
	}

 static void
pretty_expand(int iinc, int needrev)
{
	KWctx *kw, *kw0, *kw1, *kw2;

	if ((kw = KWbase.kw)) for(;;) {
		masteradj(kw->kw);
		fputs(KEYKIND[(kw->kind >> KWtopshift) & 3], stdout);
		prkwname(kw, "\n", needrev);
		kw0 = 0;
		for(kw1 = kw->kw; kw1; kw1 = kw2) {
			kw2 = kw1->next;
			pretty_ex1(kw1, kw0, kw2, 0, iinc, needrev);
			kw0 = kw1;
			}
		if (!(kw = kw->next))
			break;
		putchar('\n');
		}
	}

 static int
Str_option(char *s, char ***pav, int *want, StrList **ps)
{
	if (*++s || (s = *++*pav)) {
		if (*s == '-' && ! s[1])
			*want = 0;
		else
			*ps = new_StrList(s, *ps);
		return 0;
		}
	return 1;
	}

 static int
version(void)
{
	printf("nidrgen version 20090403\n");
	return 0;
	}

 static int
tagcomp(const void *a, const void *b)
{
	return strcmp(*(const char**)a, *(const char**)b);
	}

 static int
unused_tags()
{
	Taghash *th;
	char **x;
	size_t i, nbad;

	nbad = 0;
	for(th = (Taghash*)TagTab.lastdh; th; th = (Taghash*)th->hh.thread)
		if (!th->uses)
			++nbad;
	if (!nbad)
		return 0;
	x = (char**)Malloc(nbad*sizeof(char*), "unsed_tags");
	for(i = 0, th = (Taghash*)TagTab.lastdh; th; th = (Taghash*)th->hh.thread)
		if (!th->uses)
			x[i++] = th->hh.name;
	fprintf(stderr, "%lu unused tag definition%s in %s:\n",
		(unsigned long)nbad, "s" + (nbad == 1), tagfname);
	if (nbad > 1)
		qsort(x, nbad, sizeof(char*), tagcomp);
	for(i = 0; i < nbad; ++i)
		fprintf(stderr, "\t\"%s\"\n", x[i]);
	free(x);
	return 1;
	}

 static int
defcomp(const void *a, const void *b)
{
	return strcmp((*(const Defhash**)a)->hh.name, (*(const Defhash**)b)->hh.name);
	}

 static void
write_defs(FILE *f)
{
	Defhash *d, **x, **x1;
	DefhashHead *dh;
	size_t i, n;

	n = DHTab.nent;
	x = x1 = (Defhash**)Malloc(n*sizeof(Defhash*), "write_defs");
	for(dh = DHTab.lastdh, i = 0; dh; dh = dh->thread) {
		if (++i > n)
			botch("lastdef count botch");
		*x1++ = (Defhash*)dh;
		}
	qsort(x, n, sizeof(Defhash*), defcomp);
	for(i = 0; i < n; ++i) {
		d = x[i];
		fprintf(f, d->qkeep ? "#define %s \"%s\"\n"
			: "#define %s %s\n",
			d->hh.name, d->value);
		}
	}

 int
main(int argc, char **argv)
{
	FILE *f;
	StrList *headers, *mtagname, *nsname, *outtagname;
	char *defsout, *ofname, *s, *se;
	int defwrite, dupcheck, hwant, needrev, nwant, pprint, sort, x;

	KWStack[0] = &KWbase;
	DHTab.nalloc  = (sizeof(Defhash) + sizeof(void*) - 1)/sizeof(void*);
	TagTab.nalloc = (sizeof(Taghash) + sizeof(void*) - 1)/sizeof(void*);
	KWbase.names = new_StrList("$top",0);
	progname = argv[0];
	if (argc <= 1)
 usage1:	return usage(1);
	headers = mtagname = nsname = outtagname = 0;
	defsout = ofname = 0;
	dupcheck = hwant = needrev = nwant = 1;
	defwrite = pprint = sort = 0;
 nextopt:
	while((s = *++argv) && *s == '-' && s[1]) {
		while(*++s)
		switch(*s) {
		 case 'D':
			defwrite = 1;
			break;
		 case 'T':
			if (Str_option(s, &argv, &hwant, &outtagname))
				goto usage1;
			goto nextopt;
		 case 'd':
			dupcheck = 0;
			break;
		 case 'e':
			expand = 1;
			break;
		 case 'f':
			wantfuncs = 0;
			ignoretags = 1;
			break;
		 case 'g':
			guikeywds = 1;
			break;
		 case 'G':
			dfltgroup = 0;
			break;
		 case 'h':
			if (Str_option(s, &argv, &hwant, &headers))
				goto usage1;
			goto nextopt;
		 case 'm':
			if (Str_option(s, &argv, &hwant, &mtagname))
				goto usage1;
			goto nextopt;
		 case 'n':
			if (Str_option(s, &argv, &nwant, &nsname))
				goto usage1;
			goto nextopt;
		 case 't':
			ignoretags = 1;
			break;
		 case 's':
			sort = 1;
			needrev = 0;
			for(;;++s) {
				switch(s[1]) {
				  case 'e': expand = 1;
					    continue;
				  case 'f': wantfuncs = 0;
				  case 'p': continue;
				  }
				break;
				}
			/* no break */
		 case 'p':
			pprint = 2;
			for(;;++s) {
				switch(s[1]) {
				  case 'e':
					expand = 1;
					continue;
				  case 'f':
					wantfuncs = 0;
					continue;
				  case 's':
					sort = 1;
					needrev = 0;
					continue;
				  case 't':
					ignoretags = 1;
					continue;
				  }
				break;
				}
			if (*++s) {
				pprint = (int)strtol(s,&se,10);
				if (pprint < 1 || se)
					goto usage1;
				}
			goto nextopt;
		 case 'v':
			return version();
		 case 'w':
			startwarn = 0;
			break;
		 case '?':
			return usage(s[1] != 0);
		 case '-':
			if (!strcmp(++s,"help"))
				return usage(0);
			if (!strcmp(s,"version"))
				return version();
			if (!*s) {
				s = *++argv;
				goto argsdone;
				}
			/* no break */
		  default:
			goto usage1;
		  }
		}
 argsdone:
	if (!s || (argv[1] && argv[2] && argv[3] && argv[4]))
		return usage(1);
	if ((tagfname = argv[1])) {
		specfile = s;
		s = tagfname;
		lastseen = Saw_kwname;
		lastkw1 = &KWbase;
		if ((defsout = argv[2]))
			ofname = argv[3];
		}
	else if (mtagout) {
		fprintf(stderr, "%s: -m specified but no tagfile given.\n", progname);
		return 1;
		}
	if (*s == '-' && !s[1])
		yyin = stdin;
	else if (!(yyin = fopen(infname = s, "r"))) {
		cantopen(s);
		return 1;
		}
	if (specfile)
		tagin = yyin;
	if ((mtagname && (!(mtagout = fopen(s = mtagname->val, "w"))))
	 || (outtagname && !(tagout = fopen(s = outtagname->val, "w")))) {
		cantopen(s);
		return 1;
		}
	if ((x = yylex())) {
		fprintf(stderr, "\n%s: Surprise return %d from yylex()\n", progname, x);
		return x;
		}
	if (mtagout)
		fclose(mtagout);
	if (tagout)
		fclose(tagout);
	if (tagout || mtagout)
		goto done;
	if (Br_top > Br_stack) {
		fputs("Missing ", stderr);
		do putc(/*[*/ *Br_top-- == '(' ? ')' : ']', stderr);
			while(Br_top >Br_stack);
		fputs(" at end of file.\n", stderr);
		return 1;
		}
	if (nsquawk)
		return 1;
	if (tagfname && unused_tags())
		return 1;
	if (KWbase.kw) {
		paradj(needrev);
		if (dupcheck)
			dup_check();
		}
	if (ofname && !freopen(ofname, "w", stdout)) {
		fprintf(stderr, "%s: Cannot open output file \"%s\"\n", progname, ofname);
		return 1;
		}
	if (KWbase.kw) {
		if (pprint) {
			if (expand) {
				if (sort)
					ksort(kwcomp1);
				pretty_expand(pprint, needrev);
				}
			else {
				if (sort)
					ksort(kwcomp2);
				pretty_print(pprint, needrev);
				}
			}
		else if (!defwrite)
			Output(headers, nsname, hwant, nwant);
		else if (DHTab.lastdh)
			write_defs(stdout);
		}
	if (!defwrite && defsout) {
		f = fopen(defsout,"w");
		if (!f) {
			fprintf(stderr, "%s:  cannot open defs file \"%s\"\n",
				progname, defsout);
			return 1;
			}
		write_defs(f);
		fclose(f);
		}
 done:
	return dupnames > 0;
	}
