/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        GolubWelschOrthogPolynomial
//- Description:  Class for Golub-Welsch Orthogonal Polynomials
//-               
//- Owner:        Mike Eldred, Sandia National Laboratories

#ifndef GOLUB_WELSCH_ORTHOG_POLYNOMIAL_H
#define GOLUB_WELSCH_ORTHOG_POLYNOMIAL_H

#include "OrthogonalPolynomial.H"
#include "DataMethod.H"


namespace Dakota {

typedef Real ( *GWFPType ) ( const Real& x, const Real& p1, const Real& p2); //,... );


/// Derived orthogonal polynomial class for Golub-Welsch polynomials

/** The GolubWelschOrthogPolynomial class evaluates a univariate
    Golub-Welsch polynomial of a particular order.  It uses Gauss
    points and weights that are numerically generated by the
    Golub-Welsch algorithm (Mathematics of Computation, Vol. 23,
    No. 106, 1969).  It enables (mixed) multidimensional orthogonal
    polynomial basis functions within OrthogPolyApproximation. */

class GolubWelschOrthogPolynomial: public OrthogonalPolynomial
{
public:

  //
  //- Heading: Constructor and destructor
  //

  GolubWelschOrthogPolynomial();  ///< default constructor
  ~GolubWelschOrthogPolynomial(); ///< destructor

  //
  //- Heading: Public functions
  //

  void bounded_normal_distribution(const Real& mean,  const Real& std_dev,
				   const Real& l_bnd, const Real& u_bnd);

  void lognormal_distribution(const Real& mean, const Real& std_dev);

  void bounded_lognormal_distribution(const Real& mean,  const Real& std_dev,
				      const Real& l_bnd, const Real& u_bnd);

  void loguniform_distribution(const Real& l_bnd, const Real& u_bnd);

  void triangular_distribution(const Real& mode, const Real& l_bnd,
			       const Real& u_bnd);

  void gumbel_distribution(const Real& alpha, const Real& beta);

  void frechet_distribution(const Real& alpha, const Real& beta);

  void weibull_distribution(const Real& alpha, const Real& beta);

protected:

  //
  //- Heading: Virtual function redefinitions
  //

  /// retrieve the value of the Golub-Welsch polynomial of given order
  /// for a given parameter x
  const Real& get_value(const Real& x, unsigned short order);
  /// retrieve the Golub-Welsch polynomial gradient for a given parameter x 
  const Real& get_gradient(const Real& x, unsigned short order);

  /// return the inner product <GW_n,GW_n> = ||GW_n||^2
  const Real& norm_squared(unsigned short order);

  /// return the Gauss quadrature points corresponding to polynomial order
  const RealVector& gauss_points(unsigned short order);
  /// return the Gauss quadrature weights corresponding to polynomial order
  const RealVector& gauss_weights(unsigned short order);

private:

  //
  //- Heading: Convenience functions
  //

  /// solve a symmetric tridiagonal eigenvalue problem for the Gauss
  /// points and weights for an orthogonal polynomial of order n
  void solve_eigenproblem(unsigned short n);
  /// compute inner product of specified polynomial orders
  Real inner_product(const RealDenseVector& poly_coeffs1,
		     const RealDenseVector& poly_coeffs2);
  /// 
  Real laguerre_semibounded_integral(const RealDenseVector& poly_coeffs1,
				     const RealDenseVector& poly_coeffs2,
				     GWFPType weight_fn);
  /// 
  Real legendre_bounded_integral(const RealDenseVector& poly_coeffs1,
				 const RealDenseVector& poly_coeffs2,
				 GWFPType weight_fn);
  /// 
  Real hermite_unbounded_integral(const RealDenseVector& poly_coeffs1,
				  const RealDenseVector& poly_coeffs2,
				  GWFPType weight_fn);
  /// retrieve the value of the Golub-Welsch polynomial of given
  /// coefficients for a given parameter x
  const Real& get_value(const Real& x, const RealDenseVector& poly_coeffs);

  //
  //- Heading: Data
  //

  /// LOGNORMAL, LOGUNIFORM, TRIANGULAR, GUMBEL, FRECHET, or WEIBULL
  short distributionType;
  Real distParam1; ///<  first distribution parameter
  Real distParam2; ///< second distribution parameter
  Real distParam3; ///<  third distribution parameter
  Real distParam4; ///< fourth distribution parameter

  /// coefficients of the orthogonal polynomials, from order 0 to n
  RealDenseVectorArray polyCoeffs;

  /// norm-squared of all polynomials up to order n, as defined by the
  /// inner product <Poly_n, Poly_n> = ||Poly_n||^2
  RealDenseVector orthogPolyNormsSq;
};


inline GolubWelschOrthogPolynomial::GolubWelschOrthogPolynomial()
{ ptFactor = wtFactor = 1.; }


inline GolubWelschOrthogPolynomial::~GolubWelschOrthogPolynomial()
{ }


inline void GolubWelschOrthogPolynomial::
bounded_normal_distribution(const Real& mean,  const Real& std_dev,
			    const Real& l_bnd, const Real& u_bnd)
{
  distributionType = LOGNORMAL;
  distParam1 = mean;  distParam2 = std_dev;
  distParam3 = l_bnd; distParam4 = u_bnd;
  reset_gauss();
}


inline void GolubWelschOrthogPolynomial::
lognormal_distribution(const Real& mean, const Real& std_dev)
{
  distributionType = LOGNORMAL;
  distParam1 = mean; distParam2 = std_dev;
  reset_gauss();
}


inline void GolubWelschOrthogPolynomial::
bounded_lognormal_distribution(const Real& mean,  const Real& std_dev,
			       const Real& l_bnd, const Real& u_bnd)
{
  distributionType = LOGNORMAL;
  distParam1 = mean;  distParam2 = std_dev;
  distParam3 = l_bnd; distParam4 = u_bnd;
  reset_gauss();
}


inline void GolubWelschOrthogPolynomial::
loguniform_distribution(const Real& l_bnd, const Real& u_bnd)
{
  distributionType = LOGUNIFORM;
  distParam1 = l_bnd; distParam2 = u_bnd;
  reset_gauss();
}


inline void GolubWelschOrthogPolynomial::
triangular_distribution(const Real& mode, const Real& l_bnd, const Real& u_bnd)
{
  distributionType = TRIANGULAR;
  distParam1 = mode; distParam2 = l_bnd; distParam3 = u_bnd;
  reset_gauss();
}


inline void GolubWelschOrthogPolynomial::
gumbel_distribution(const Real& alpha, const Real& beta)
{
  distributionType = GUMBEL;
  distParam1 = alpha; distParam2 = beta;
  reset_gauss();
}


inline void GolubWelschOrthogPolynomial::
frechet_distribution(const Real& alpha, const Real& beta)
{
  distributionType = FRECHET;
  distParam1 = alpha; distParam2 = beta;
  reset_gauss();
}


inline void GolubWelschOrthogPolynomial::
weibull_distribution(const Real& alpha, const Real& beta)
{
  distributionType = WEIBULL;
  distParam1 = alpha; distParam2 = beta;
  reset_gauss();
}

} // namespace Dakota

#endif
