/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        DistinctConstraints
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "DistinctConstraints.H"
#include "ProblemDescDB.H"
#include "data_io.h"
#include "data_util.h"

static const char rcsId[]="@(#) $Id";


namespace Dakota {

/** In this class, the distinct approach (design, uncertain, and state
    types are distinct) is used.  Most iterators/strategies use this
    approach, which is the default in Constraints::get_constraints(). */
DistinctConstraints::
DistinctConstraints(const ProblemDescDB& problem_db,
		    const pair<short,short>& view,
		    const Sizet2DArray& vars_comps):
  Constraints(BaseConstructor(), problem_db, view, vars_comps)
{
  aggregate_data(
    problem_db.get_rdv("variables.continuous_design.lower_bounds"),
    problem_db.get_rdv("variables.uncertain.lower_bounds"),
    problem_db.get_rdv("variables.continuous_state.lower_bounds"),
    allContinuousLowerBnds);
  aggregate_data(
    problem_db.get_rdv("variables.continuous_design.upper_bounds"),
    problem_db.get_rdv("variables.uncertain.upper_bounds"),
    problem_db.get_rdv("variables.continuous_state.upper_bounds"),
    allContinuousUpperBnds);

  aggregate_data(
    problem_db.get_idv("variables.discrete_design.lower_bounds"),
    problem_db.get_idv("variables.discrete_state.lower_bounds"),
    allDiscreteLowerBnds);
  aggregate_data(
    problem_db.get_idv("variables.discrete_design.upper_bounds"),
    problem_db.get_idv("variables.discrete_state.upper_bounds"),
    allDiscreteUpperBnds);

  // construct active/inactive views of all arrays
  build_views();

  // Manage linear constraints.  Verify specification sanity and set defaults
  // if needed.
  if (variablesView.first == MIXED_DISTINCT_ALEATORY_UNCERTAIN  ||
      variablesView.first == MIXED_DISTINCT_EPISTEMIC_UNCERTAIN ||
      variablesView.first == MIXED_DISTINCT_UNCERTAIN) {
    if ( problem_db.get_rdv("method.linear_inequality_constraints").length()
      || problem_db.get_rdv("method.linear_equality_constraints").length() )
      Cerr << "Warning: linear constraints not supported with nondeterministic "
	   << "iterators\n         Input will be ignored.";
    numLinearIneqCons = numLinearEqCons = 0;
  }
  else
    manage_linear_constraints(problem_db);

#ifdef REFCOUNT_DEBUG
  Cout << "Letter instantiated: variablesView active = " << variablesView.first
       << " inactive = " << variablesView.second << endl;
#endif
}


void DistinctConstraints::copy_rep(const Constraints* con_rep)
{ build_views(); }


void DistinctConstraints::reshape_rep()
{
  size_t num_acv = (*variablesComponents)[0][1] + (*variablesComponents)[1][0] +
    (*variablesComponents)[2][0] + (*variablesComponents)[3][1],
    num_adv = (*variablesComponents)[0][2] + (*variablesComponents)[3][2];

  allContinuousLowerBnds.resize(num_acv);
  allContinuousUpperBnds.resize(num_acv);
  allDiscreteLowerBnds.resize(num_adv);
  allDiscreteUpperBnds.resize(num_adv);

  build_views();
}


void DistinctConstraints::build_active_views()
{
  // Initialize continuousVarTypes/discreteVarTypes/continuousVarIds.
  // Don't bleed over any logic about supported view combinations; rather,
  // keep this class general and encapsulated.
  size_t num_cdv = (*variablesComponents)[0][1],
    num_ddv = (*variablesComponents)[0][2],
    num_auv = (*variablesComponents)[1][0],
    num_euv = (*variablesComponents)[2][0],
    num_csv = (*variablesComponents)[3][1],
    num_dsv = (*variablesComponents)[3][2];

  // Initialize active views
  size_t cv_start = 0, dv_start = 0, num_cv = 0, num_dv = 0;
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    num_cv = num_cdv;                       num_dv = num_ddv;           break;
  case MIXED_DISTINCT_ALEATORY_UNCERTAIN:
    cv_start = num_cdv;                     num_cv = num_auv;           break;
  case MIXED_DISTINCT_EPISTEMIC_UNCERTAIN:
    cv_start = num_cdv+num_auv;             num_cv = num_euv;           break;
  case MIXED_DISTINCT_UNCERTAIN:
    cv_start = num_cdv;                     num_cv = num_auv + num_euv; break;
  case MIXED_DISTINCT_STATE:
    cv_start = num_cdv + num_auv + num_euv; num_cv = num_csv;
    dv_start = num_ddv;                     num_dv = num_dsv;           break;
  }
  if (num_cv) {
    continuousLowerBnds = RealDenseVector(Teuchos::View,
      &allContinuousLowerBnds[cv_start], num_cv);
    continuousUpperBnds = RealDenseVector(Teuchos::View,
      &allContinuousUpperBnds[cv_start], num_cv);
  }
  if (num_dv) {
    discreteLowerBnds = IntDenseVector(Teuchos::View,
      &allDiscreteLowerBnds[dv_start], num_dv);
    discreteUpperBnds = IntDenseVector(Teuchos::View,
      &allDiscreteUpperBnds[dv_start], num_dv);
  }
}


void DistinctConstraints::build_inactive_views()
{
  // Initialize continuousVarTypes/discreteVarTypes/continuousVarIds.
  // Don't bleed over any logic about supported view combinations; rather,
  // keep this class general and encapsulated.
  size_t num_cdv = (*variablesComponents)[0][1],
    num_ddv = (*variablesComponents)[0][2],
    num_auv = (*variablesComponents)[1][0],
    num_euv = (*variablesComponents)[2][0],
    num_csv = (*variablesComponents)[3][1],
    num_dsv = (*variablesComponents)[3][2];

  // Initialize inactive views
  size_t icv_start = 0, idv_start = 0, num_icv = 0, num_idv = 0;
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    num_icv = num_cdv;                       num_idv = num_ddv;           break;
  case MIXED_DISTINCT_ALEATORY_UNCERTAIN:
    icv_start = num_cdv;                     num_icv = num_auv;           break;
  case MIXED_DISTINCT_EPISTEMIC_UNCERTAIN:
    icv_start = num_cdv+num_auv;             num_icv = num_euv;           break;
  case MIXED_DISTINCT_UNCERTAIN:
    icv_start = num_cdv;                     num_icv = num_auv + num_euv; break;
  case MIXED_DISTINCT_STATE:
    icv_start = num_cdv + num_auv + num_euv; num_icv = num_csv;
    idv_start = num_ddv;                     num_idv = num_dsv;           break;
  }
  if (num_icv) {
    inactiveContinuousLowerBnds = RealDenseVector(Teuchos::View,
      &allContinuousLowerBnds[icv_start], num_icv);
    inactiveContinuousUpperBnds = RealDenseVector(Teuchos::View,
      &allContinuousUpperBnds[icv_start], num_icv);
  }
  if (num_idv) {
    inactiveDiscreteLowerBnds = IntDenseVector(Teuchos::View,
      &allDiscreteLowerBnds[idv_start], num_idv);
    inactiveDiscreteUpperBnds = IntDenseVector(Teuchos::View,
      &allDiscreteUpperBnds[idv_start], num_idv);
  }
}


void DistinctConstraints::read(istream& s)
{
  size_t num_cdv = (*variablesComponents)[0][1],
    num_ddv      = (*variablesComponents)[0][2],
    num_uv_csv   = allContinuousLowerBnds.length() - num_cdv,
    num_dsv      = allDiscreteLowerBnds.length()   - num_ddv;
  read_data_partial(s, 0, num_cdv, allContinuousLowerBnds);
  read_data_partial(s, 0, num_ddv, allDiscreteLowerBnds);
  read_data_partial(s, num_cdv, num_uv_csv, allContinuousLowerBnds);
  read_data_partial(s, num_ddv, num_dsv, allDiscreteLowerBnds);

  read_data_partial(s, 0, num_cdv, allContinuousUpperBnds);
  read_data_partial(s, 0, num_ddv, allDiscreteUpperBnds);
  read_data_partial(s, num_cdv, num_uv_csv, allContinuousUpperBnds);
  read_data_partial(s, num_ddv, num_dsv, allDiscreteUpperBnds);
}


void DistinctConstraints::write(ostream& s) const
{
  size_t num_cdv = (*variablesComponents)[0][1],
    num_ddv      = (*variablesComponents)[0][2],
    num_uv_csv   = allContinuousLowerBnds.length() - num_cdv,
    num_dsv      = allDiscreteLowerBnds.length()   - num_ddv;
  write_data_partial(s, 0, num_cdv, allContinuousLowerBnds);
  write_data_partial(s, 0, num_ddv, allDiscreteLowerBnds);
  write_data_partial(s, num_cdv, num_uv_csv, allContinuousLowerBnds);
  write_data_partial(s, num_ddv, num_dsv, allDiscreteLowerBnds);

  write_data_partial(s, 0, num_cdv, allContinuousUpperBnds);
  write_data_partial(s, 0, num_ddv, allDiscreteUpperBnds);
  write_data_partial(s, num_cdv, num_uv_csv, allContinuousUpperBnds);
  write_data_partial(s, num_ddv, num_dsv, allDiscreteUpperBnds);
}

} // namespace Dakota
