//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Combiner_Smooth.hpp
  \brief Class description for APPSPACK::Combiner::Smooth
*/

#ifndef APPSPACK_COMBINER_SMOOTH
#define APPSPACK_COMBINER_SMOOTH

#include "APPSPACK_Common.hpp"
#include "APPSPACK_Vector.hpp"
#include "APPSPACK_Combiner_Generic.hpp"

namespace APPSPACK
{
namespace Combiner
{

/*! \brief This class class can be used to define merit functions based upon
the \f$\|c^+(x)\|_1\f$, \f$\|c^+(x)\|_2\f$, and \f$\|c^+(x)\|_\infty\f$
norms of the constraint violations where 
\f[
c_i^+(x) = \begin{array}{ll}
|c_i(x)| & {\rm if} 1 \le i \le n_e \\
\max(c_i(x),0) & i > n_e
\end{array}
\f]
Where \f$n_e\f$ denotes the number of nonlinear equality constraints.  It is assumed that
\f$c(x)\f$ is ordered so that nonlinear equality constraint preceed 
the nonlinear inequality constraints.
*/
class Smooth : public Generic
{
public:
  //! Constructor. 
  Smooth(const string& methodName);
  
  //! Destructor.
  ~Smooth();

  //! Derived.
  virtual double operator()(const Vector& fc) const;

  //! Set the debug level for printing.
  void setDebug(int debug);

  //! Set sizes of constraints.
  void setConstraintSize(int nlec, int nlic);

  //! Set scale parameter for penalty term.
  void setRho(double rho);

  //! Get scale parameter for penalty term.
  double getRho() const;

  //! Return infnorm of constraint violation.
  double infnorm(const Vector& fc) const;

  /*! \brief Return vector \f$\nu\f$ of constraint violations. 

    <ul>
    <li> \f$ \nu_i = | c_i(x)| \f$ if \f$c_i(x)\f$ is an nonlinear equality constraint. 
    <li> \f$ \nu_i = \max(0, c_i(x)) \f$ if \f$c_i(x)\f$ is an nonlinear inequality constraint. 
    </ul> */
  void getInfVector(const Vector& fc, Vector& nu) const;

  //! Set scale parameter for penalty term.
  void setAlpha(double alpha);
  
  //! Get scale parameter for penalty term.
  double getAlpha() const;

  //! Get greatest violations allowed for current alpha.
  double getErrorTol(const Vector& fc) const;
private:

  /*! \brief  Return the max violation norm.

  This class defines merit functions based upon the \f$\ell_\infty\f$ penalty function.
    The \f$\ell_\infty\f$-based merit function defined by this method is given by
    \f[
    M(x,\rho_k) = f(x) + \rho_k \|c^+(x)\|_\infty.
    \f]
  */
  double getPenaltyMax(const Vector& fc) const;

  /*! \brief Returns a smoothed variant of the max violation norm.
    
  This class defines merit functions based upon the \f$\ell_\infty\f$ penalty function.
    The \f$\ell_\infty\f$-based merit function defined by this method is given by
    \f[
    M(x,\rho_k,\alpha_k) =  f(x) + \alpha_k\ln\left(1 + \sum_{i\le n_e} 2\cosh(\rho c_i(x)/\alpha_k)
      + \sum_{i>n_e} e^{\rho c_i(x)/\alpha_k} \right).
    \f]
    Here \f$n_e\f$ denotes the number of nonlinear equality constraints.
  */
  double getPenaltyMaxSmooth(const Vector& fc) const;


  /*! \brief This class defines merit functions based upon the \f$\ell_1\f$ penalty function.
    The \f$\ell_1\f$-based merit function defined by this method is given by
    \f[
    M(x,\rho_k) = f(x) + \rho_k \|c^+(x)\|_1.
    \f]
  */
  double getPenaltyOne(const Vector& fc) const;  

  /*! \brief Returns a smoothed variant of \f$\ell_1\f$ norm.
    
  Th1is class defines merit functions based upon the \f$\ell_1\f$ penalty function.
    The \f$\ell_\infty\f$-based merit function defined by this method is given by
    \f[
    M(x,\rho_k,\alpha_k) = f(x) + \sum_{i\le n_e} \alpha_k\ln(2+2\cosh(\rho_k c_i(x)/\alpha_k))
    +
    \sum_{i > n_e} \alpha_k\ln(1+\exp(\rho_k c_i(x)/\alpha_k)).  
    \f]
    Here \f$n_e\f$ denotes the number of nonlinear equality constraints.
  */
  double getPenaltyOneSmooth(const Vector& fc) const;  

  /*!\brief This class defines merit functions based upon the \f$\ell_2\f$ penalty function.
    The \f$\ell_2^2\f$-based merit function defined by this method is given by
    \f[
    M(x,\rho_k) = f(x) + \rho_k \|c^+(x)\|_2.
    \f]
  */
  double getPenaltyTwo(const Vector& fc) const;  
  
  /*!\brief This class defines merit functions based upon the \f$\ell_2\f$ penalty function.
    The \f$\ell_2\f$-based merit function defined by this method is given by
    \f[
    M(x,\rho_k,\alpha_k) = f(x) +  \rho_k \sqrt{\|c^+(x)\|_2^2+ (\alpha_k/\rho_k)^2}.
    \f]
  */
  double getPenaltyTwoSmooth(const Vector& fc) const;  

  /*!\brief This class defines merit functions based upon the \f$\ell_2^2\f$ penalty function.
    The \f$\ell_2^2\f$-based merit function defined by this method is given by
    \f[
    M(x,\rho_k) = f(x) + \rho_k \|c^+(x)\|_2^2.
    \f]
  */
  double getPenaltyTwoSquared(const Vector& fc) const; 

private:

  //! Debug level.
  int debug;

  //! Number of nonlinear equality constraints.
  int nlec;
  
  //! Number of nonlinear inequality constraints.
  int nlic;

  //! Scale parameter for penalty term.
  double rho; 
  
  //! Smoothing parameter for penalty term.
  double alpha; 

  //! Specifies whether to use scaled or piecewise.
  const string methodName;
};


}
}
#endif
