/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file ForkCallAnalysisCode.h
 *
 */

#ifndef colin_ForkCallAnalysisCode_h
#define colin_ForkCallAnalysisCode_h

#include <acro_config.h>
#if 0
#include <colin/ColinAnalysisCode.h>


namespace colin {

/** \class ForkCallAnalysisCode
  *
  * Derived class in the AnalysisCode class hierarchy which spawns
  * simulations using forks.
  *
  * ForkAnalysisCode creates a copy of the parent process using
  * fork()/vfork() and then replaces the copy with a simulation
  * process using execvp().  The parent process can then use waitpid()
  * to wait on completion of the simulation process.
  */
class ForkAnalysisCode: public AnalysisCode
{
  public:

    //
    //- Heading: Constructors and destructor
    //

    ForkAnalysisCode(const ProblemDescDB& problem_db); ///< constructor
    ~ForkAnalysisCode();                               ///< destructor

    //
    //- Heading: Methods
    //

    /// spawn a child process using fork()/vfork()/execvp() and wait
    /// for completion using waitpid() if block_flag is true
    pid_t fork_program(const short block_flag);

    /// check the exit status of a forked process and abort if an
    /// error code was returned
    void check_status(const int status);

    /// set argList[index] to arg
    void argument_list(const int index, const DakotaString& arg);

    /// append an additional tag to argList[index] (beyond that
    /// already present in the modified file names) for managing
    /// concurrent analyses within a function evaluation
    void tag_argument_list(const int index, const int tag);

  private:

    //
    //- Heading: Data
    //

    /// an array of strings for use with execvp(const char *, char * const *)
    /// (an argList entry can be passed as the first argument, and the entire
    /// argList can be cast as the second argument)
    const char* argList[4];
};


inline void 
ForkAnalysisCode::argument_list(const int index, const DakotaString& arg)
{
  if (index >=0 && index <= 2)
    argList[index] = arg;
  else {
    Cerr << "Error: index out of range in ForkAnalysisCode::argument_list "
	 << "assignment." << endl;
    abort_handler(-1);
  }
}


inline void 
ForkAnalysisCode::tag_argument_list(const int index, const int tag)
{
  char tag_str[16];
  sprintf(tag_str, ".%d", tag);
  if (index == 1) {
    DakotaString tempFilename(modifiedParamsFileName);
    tempFilename += tag_str;
    argList[1] = tempFilename;
  }
  else if (index == 2) {
    DakotaString tempFilename(modifiedResFileName);
    tempFilename += tag_str;
    argList[2] = tempFilename;
  }
  else { // tagging only used with input/output filenames
    Cerr << "Error: index invalid in ForkAnalysisCode::tag_argument_list."
         << endl;
    abort_handler(-1);
  }
}

}

#endif

#endif
