/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implementation of class MOGAConverger.

    NOTES:

        See notes of MOGAConverger.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Mon Jul 07 09:24:18 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the MOGAConverger class.
 */



/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <cfloat>
#include <FitnessRecord.hpp>
#include <GeneticAlgorithm.hpp>
#include <../Utilities/include/Logging.hpp>
#include <utilities/include/numeric_limits.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <../Utilities/include/DesignStatistician.hpp>
#include <../MOGA/include/Convergers/MOGAConverger.hpp>
#include <../Utilities/include/MultiObjectiveStatistician.hpp>



/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;
using namespace JEGA::Utilities;
using namespace JEGA::Logging;
using namespace eddy::utilities;







/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Algorithms {








/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/








/*
================================================================================
Accessors
================================================================================
*/








/*
================================================================================
Public Methods
================================================================================
*/


const string&
MOGAConverger::Name(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    static const string* ret = new string("metric_tracker");
    return *ret;
}

const string&
MOGAConverger::Description(
    )
{
    EDDY_FUNC_DEBUGSCOPE

    static const string ret(
        "This converger computes various metrics for the "
        "population and determines if sufficient improvement "
        "is being made.  If not, this converger returns true."
        );
    return ret;
}

GeneticAlgorithmOperator*
MOGAConverger::Create(
    GeneticAlgorithm& algorithm
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return new MOGAConverger(algorithm);
}


/*
================================================================================
Subclass Visible Methods
================================================================================
*/

double
MOGAConverger::GetMaxRangeChange(
    const eddy::utilities::DoubleExtremes& newExtremes
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(newExtremes.size() == _prevExtremes.size());

    JEGAIFLOG_CF_II_F(newExtremes.size() != _prevExtremes.size(),
        GetLogger(), this,
        text_entry(lfatal(), GetName() + ": Dimensional disagreement between "
                   "old Pareto extremes and new Pareto extremes.")
        );

    // compute the percentage change in range along each dimension as the
    // difference in the ranges divided by the old range.  Return the maximum
    // of them.
    double maxChng = eddy::utilities::numeric_limits<double>::smallest();
    DoubleExtremes::size_type of = 0;
    DoubleExtremes::size_type size = newExtremes.size();

    for(DoubleExtremes::size_type i=0; i<size; ++i)
    {
        double overallRange = _prevExtremes.get_range(i);
        double currChng = overallRange == 0.0 ?
            newExtremes.get_range(i) : Math::Abs(
                (newExtremes.get_range(i) - overallRange) / overallRange
                );

        if(currChng > maxChng) { maxChng = currChng; of = i; }
    }

    // do a little outputting if appropriate
    JEGAIFLOG_CF_II(maxChng != 0.0, GetLogger(), lverbose(), this,
        ostream_entry(
            lverbose(), GetName() + ": Max \"best fitness\" expansion of "
            ) << Math::Round(maxChng*100.0, GetNumDP())
              << "% found for objective \""
              << GetDesignTarget().GetObjectiveFunctionInfos().
                 at(of)->GetLabel()
              << "\"."
        )

    JEGAIFLOG_CF_II(maxChng == 0.0, GetLogger(), lverbose(), this,
        text_entry(lverbose(), GetName() + ": No \"best fitness\" expansion "
            "occured this generation.")
        )
    return maxChng;
}

double
MOGAConverger::GetDensityChange(
    const JEGA::Utilities::DesignOFSortSet& newSet,
    const eddy::utilities::DoubleExtremes& newExtremes
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    // This is computed as the percentage change in density where density
    // is npts/volume.

    // start with our volumes.
    double oldVol = ComputeVolume(_prevExtremes);
    double newVol = ComputeVolume(newExtremes);

    // now do our densities.
    double oldDen = _prevSet.size() / oldVol;
    double newDen = newSet.size() / newVol;

    // now compute and return our percent change.
    double fracChng = Math::Abs((newDen - oldDen) / oldDen);

    JEGA_LOGGING_IF_ON(
        string incdec(newDen < oldDen ? "increased" : "decreased");
        )

    JEGAIFLOG_CF_II(fracChng != 0.0, GetLogger(), lverbose(), this,
        ostream_entry(lverbose(), GetName() +
            ": \"best fitness\" density " + incdec + " by "
            ) << Math::Round(fracChng*100.0, GetNumDP()) << "%."
        )

    JEGAIFLOG_CF_II(fracChng == 0.0, GetLogger(), lverbose(), this,
        text_entry(lverbose(), GetName() +
            ": \"best fitness\" density did not change this generation.")
        )

    return fracChng;
}

double
MOGAConverger::GetFractionDominated(
    const JEGA::Utilities::DesignOFSortSet& curr
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    // if the previous set has zero size, we cannot compute.
    if(_prevSet.empty()) return 0.0;

    // go through in and count how many are dominated by designs in "by".
    // The return will be whatever percentage of in.size() count is.
    eddy::utilities::uint64_t numDom = 0;

    const DesignOFSortSet::const_iterator e(_prevSet.end());
    for(DesignOFSortSet::const_iterator it(_prevSet.begin()); it!=e; ++it)
        numDom += MultiObjectiveStatistician::IsDominatedByAtLeast1(**it, curr)
                  ? 1 : 0;

    double fracDom = static_cast<double>(numDom)/_prevSet.size();

    JEGAIFLOG_CF_II(fracDom != 0.0, GetLogger(), lverbose(), this,
        ostream_entry(lverbose(), GetName() + ": ")
            << Math::Round(fracDom*100.0, GetNumDP())
            << "% of the previous \"best fitness\" population is dominated by "
               "designs in the current \"best fitness\" population."
        )

    JEGAIFLOG_CF_II(fracDom == 0.0, GetLogger(), lverbose(), this,
        text_entry(lverbose(), GetName() + ": No \"best fitness\" designs "
            "from the previous population are dominated by \"best fitness\" "
            "designs in the current population.")
        )

    return fracDom;
}


void
MOGAConverger::UpdatePreviousSet(
    const DesignOFSortSet& newSet
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // we have to go through and destroy all the designs in the
    // current set because we created them.  They do not have to go
    // to the target because they are duplicates of mainstream Designs.
    _prevSet.flush();

    // now create new designs that are duplicates of those in the new set
    // and put them into our previous set.
    const DesignOFSortSet::const_iterator e(newSet.end());

    for(DesignOFSortSet::const_iterator it(newSet.begin()); it!=e; ++it)
        _prevSet.insert(GetAlgorithm().GetNewDesign(*(*it)));
}

void
MOGAConverger::UpdateOverallExtremes(
    const DoubleExtremes& newExtremes
    )
{
    EDDY_FUNC_DEBUGSCOPE
    this->_prevExtremes = newExtremes;
}

DesignOFSortSet
MOGAConverger::GetBest(
    const DesignOFSortSet& of,
    const FitnessRecord& fitnesses
    )
{
    EDDY_FUNC_DEBUGSCOPE

    DesignOFSortSet ret;
    double bestFit = eddy::utilities::numeric_limits<double>::smallest();

    for(DesignOFSortSet::const_iterator it(of.begin()); it!=of.end(); ++it)
    {
        double currFit = fitnesses.GetFitness(**it);
        if(currFit == -DBL_MAX) continue;

        if(currFit > bestFit)
        {
            ret.clear();
            ret.insert(*it);
            bestFit = currFit;
        }
        else if(currFit == bestFit) ret.insert(*it);
    }

    return ret;
}

/*
================================================================================
Subclass Overridable Methods
================================================================================
*/


string
MOGAConverger::GetName(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return MOGAConverger::Name();
}

string
MOGAConverger::GetDescription(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return MOGAConverger::Description();
}

GeneticAlgorithmOperator*
MOGAConverger::Clone(
    GeneticAlgorithm& algorithm
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return new MOGAConverger(*this, algorithm);
}

double
MOGAConverger::GetMetricValue(
    const DesignGroup& group,
    const FitnessRecord& fitnesses
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // We will track our various metrics and return the one that
    // is the "best" (most improvement).  We only care about our
    // best designs as defined by the fitnesses so start by extracting
    // those.  Ideally, they are the non-dominated so we will call them
    // the pareto here.
    DesignOFSortSet pareto(GetBest(group.GetOFSortContainer(), fitnesses));

    JEGALOG_II(GetLogger(), lverbose(), this,
        ostream_entry(lverbose(), GetName() + ": Operating on ")
            << pareto.size() << " \"best fitness\" designs."
        )

    // Start by getting our newest pareto extremes.  We don't have to
    // use the multi objective statistician because we know all our
    // designs are non dominated.
    DoubleExtremes newExt =
        DesignStatistician::GetObjectiveFunctionExtremes(pareto);

    double maxFrac = 0.0;

    // The first thing to check is that there are previous stuff
    // (that this is not the first generation).  If it is, then
    // we set our previous stuff and return 0.0 as our best progress.
    if(GetAlgorithm().GetGenerationNumber() > 1)
    {
        // Start with our maximum change in range along any Pareto dimension.
        double maxExp = GetMaxRangeChange(newExt);

        // Now get our change in density.
        double denInc = GetDensityChange(pareto, newExt);

        // Finally, get the percentage of Designs in the old group that
        // are dominated by designs in the new group.
        double fracDom = GetFractionDominated(pareto);

        // The return will be the max of all these metrics.
        maxFrac = Math::Max(Math::Max(maxExp, denInc), fracDom);
    }

    // Since we have them now, take our new extremes as our old ones in prep
    // for the next generation.
    UpdateOverallExtremes(newExt);

    // record the current as the previous set
    UpdatePreviousSet(pareto);

    return maxFrac;
}

bool
MOGAConverger::CheckConvergence(
    const DesignGroup& group,
    const FitnessRecord& fitnesses
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // make sure the sets of group are synchronized.
    JEGA_LOGGING_IF_ON(bool synced = ) group.SynchronizeOFAndDVContainers();

    JEGAIFLOG_CF_II(!synced, GetLogger(), lquiet(), this,
        text_entry(lquiet(), GetName() + ": Group synchronization failed.  "
                   "Proceeding with group in current state.")
        )

	return MetricTrackerConvergerBase::CheckConvergence(group, fitnesses);

    //// Look for stopping criteria using base class converger.
    //if(MaxGenEvalConverger::CheckConvergence(group, fitnesses))
    //    return true;

    //// now get our max percent change and put it in our tracker.
    //double maxPct = GetMetricValue(group, fitnesses);
    //GetMetricTracker().Push(maxPct);

    //JEGALOG_II(GetLogger(), lverbose(), this,
    //    ostream_entry(lverbose(),
    //        GetName() + ": Pushed max percentage value of ")
    //        << Math::Round(maxPct*100.0, GetNumDP())
    //        << " onto the convergence stack."
    //    )

    //// If we have not yet stored the required number of metric
    //// values, we certainly cannot be done.
    //if(!GetMetricTracker().IsFull()) return false;

    //// now we figure out if we are below our threshold or not.
    //MetricTracker::LocPair maxEPct(GetMetricTracker().MaxValue());

    //JEGALOG_II(GetLogger(), lverbose(), this,
    //    ostream_entry(lverbose(), GetName() +
    //        ": Max existing percentage metric is ")
    //        << Math::Round(maxEPct.second*100.0, GetNumDP())
    //        << "% at location " << maxEPct.first << " of "
    //        << GetMetricTracker().GetStackMaxDepth() << " (cutoff is "
    //        << Math::Round(GetPercentChange()*100.0, GetNumDP()) << "%)."
    //    )

    //SetConverged(maxEPct.second <= GetPercentChange());

    //// return true if converged and false otherwise.
    //return GetConverged();
}






/*
================================================================================
Private Methods
================================================================================
*/
double
MOGAConverger::ComputeVolume(
    const DoubleExtremes& extremes
    )
{
    EDDY_FUNC_DEBUGSCOPE

    DoubleExtremes::size_type size = extremes.size();
    double vol = 1.0;
    for(DoubleExtremes::size_type i=0; i<size; ++i)
    {
        double r = extremes.get_range(i);
        if(r != 0.0) vol *= r;
    }

    return vol;
}








/*
================================================================================
Structors
================================================================================
*/


MOGAConverger::MOGAConverger(
    GeneticAlgorithm& algorithm
    ) :
        MetricTrackerConvergerBase(algorithm),
        _prevSet(),
        _prevExtremes(algorithm.GetDesignTarget().GetNOF(), DBL_MAX, -DBL_MAX)
{
    EDDY_FUNC_DEBUGSCOPE
}

MOGAConverger::MOGAConverger(
    const MOGAConverger& copy
    ) :
        MetricTrackerConvergerBase(copy),
        _prevSet(copy._prevSet),
        _prevExtremes(copy._prevExtremes)
{
    EDDY_FUNC_DEBUGSCOPE
}

MOGAConverger::MOGAConverger(
    const MOGAConverger& copy,
    GeneticAlgorithm& algorithm
    ) :
        MetricTrackerConvergerBase(copy, algorithm),
        _prevSet(copy._prevSet),
        _prevExtremes(copy._prevExtremes)
{
    EDDY_FUNC_DEBUGSCOPE
}

MOGAConverger::~MOGAConverger(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    _prevSet.flush();
}






/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Algorithms
} // namespace JEGA
