// $Id: APPSPACK_Citizen_TGP.cpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_TGP.cpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_TGP.cpp
  \brief Implements APPSPACK::Citizen_TGP

  \todo Josh Comment parameters.
*/

#include "APPSPACK_Citizen_TGP.hpp"
#include "APPSPACK_Combiner_Generic.hpp"
#include "APPSPACK_Executor_Oracle.hpp"
#include "APPSPACK_Print.hpp"

int APPSPACK::Citizen::TGP::staticCount = 0;
int APPSPACK::Citizen::TGP::staticEvalCount = 0;

APPSPACK::Citizen::TGP::TGP(Parameter::List& params_in,
                            const Constraints::Linear& constraints_in,
                            Combiner::Generic& combiner_in, int rank_in,
                            const string name_in) :
  params(params_in.sublist(name_in)),
  name(name_in + "(TGP)"),
  combiner(combiner_in),
  constraints(constraints_in),
  pointCount(0),
  executableName(params.getParameter("Executable Name", "a.out")),
  candidateFile(params.getParameter("Candidate File", "XX.txt")),
  defaultStep(params.getParameter("Default Step", .25)),
  precision(params.getParameter("Output File Precision", 14)),
  nCandidates(params.getParameter("Number Of Candidates", 100)),
  nTrialPoints(params.getParameter("Trial Points", 1)),
  evalsFileName(params.getParameter("Evaluations File", "EvalFile.txt")),
  appendEvalsFile(params.getParameter("Append Evalualation File", false)),
  initialSampleSize(params.getParameter("Initial Sample", 100)),
  minEvaluations(params.getParameter("Minimum Evaluations", 0)),
  useNoiseSupport(params.getParameter("Noise Support", false)),
  contractionFactor(params.getParameter("Contraction Factor", .025)),
  useGlobal(params.getParameter("Use Global Search", true)),
  globalTolerance(params.getParameter("Global Tolerance", .05)),
  tgpCacheTolerance(params.getParameter("TGP Cache Tolerance", 1e-2)),
  tgpRunning(false),
  initialSampleComplete(false),
  workerRank(rank_in)
{ 
}

APPSPACK::Citizen::TGP::~TGP()
{
}

void APPSPACK::Citizen::TGP::preprocess()
{ 
  // Tell worker to initialize.
  APPSPACK::GCI::initSend();
  workerFileName="Worker_"+evalsFileName;
  APPSPACK::GCI::pack(workerFileName);
  APPSPACK::GCI::pack(tgpCacheTolerance);
  APPSPACK::GCI::send(APPSPACK::Executor::Oracle::Init, workerRank);

  // Determine lower and upper bounds in R-notation.
  //! R-notation bounds: "c(l1,12,...,ln),c(u1,u2,...,un)".
  const Vector& bLower=constraints.getLower();
  const Vector& bUpper=constraints.getUpper();

  // Write using string streams.
  ostringstream ss;

  // Process upper and lower bounds.
  ss << "\"c(" << bLower[0];
  for (int i=1; i<bLower.size(); i++)
    ss << "," << bLower[i];
  ss << "),c(" << bUpper[0];
  for (int i=1; i<bUpper.size(); i++)
    ss << "," <<bUpper[i];
  ss << ")\"";

  // Load in initial LHS points.  
  tgpCommandPrefix = executableName + " " + workerFileName + " " + ss.str();

  // Add noise argument.
  if (useNoiseSupport == true)
    tgpCommandPrefix += " 1 ";
  else 
    tgpCommandPrefix += " 0 ";

  if (appendEvalsFile)    
    evalsOutStream.open(evalsFileName.c_str(), ios::app);
  else
    evalsOutStream.open(evalsFileName.c_str());

  if (initialSampleSize == 0)
    return;

  // No read in initial sample of LHS points.
  createStrings(false, initialSampleSize);
  cout << "Calling: " << tgpCommand << endl;
  system(tgpCommand.c_str());
  
  // Grab all points.
  readOutputFile(initialLHS, initialSampleSize);

}

void APPSPACK::Citizen::TGP::postprocess()
{
  // Send a termination command to oracle worker
  APPSPACK::GCI::initSend();
  APPSPACK::GCI::pack(1);
  APPSPACK::GCI::send(APPSPACK::Executor::Oracle::Terminate, workerRank);

  evalsOutStream.close();
}

void APPSPACK::Citizen::TGP::exchange(const APPSPACK::ConveyorList& R,
                                      APPSPACK::ConveyorList& Wnew, 
                                      const map<string, vector<int> >& ownermap)
{
  const vector<int>& citizen_tags = ownermap.find(name)->second;
  writeNewEvals(R, ownermap);
  pointCount += R.size();  

  if (initialLHS.getNrows() > 0)
  {
    // Add points in initialSampleSize and delete to signify they have been added.
    addTrialPoints(Wnew, initialLHS);
    initialLHS.clear();
    return;
  }
  if (citizen_tags.size() == 0)
  {
    // All points initialLHS points have been evaluated.  
    initialSampleComplete=true;
  }
  
  if (!initialSampleComplete || pointCount < minEvaluations)
  {
    return;
  }

  if (!tgpRunning && (staticEvalCount != pointCount))
  {
    staticEvalCount = pointCount;
    createStrings(true, nCandidates);
    GCI::initSend();
    GCI::pack(tgpCommand);
    packNewEvals();
    GCI::send(APPSPACK::Executor::Oracle::Generate, workerRank);  
    tgpRunning=true;
  }

  if (GCI::probe(APPSPACK::Executor::Oracle::Generate))
  {
    int msgtag, rank;
    string exit_status;
    APPSPACK::GCI::recv(APPSPACK::Executor::Oracle::Generate, workerRank);
    APPSPACK::GCI::bufinfo(msgtag, rank);
    APPSPACK::GCI::unpack(exit_status);
    tgpRunning=false;

    // Get trial points.
    Matrix TP;
    readOutputFile(TP, nTrialPoints);

    // Now add trials in TP to Wnew;
    addTrialPoints(Wnew, TP);

    cout << "TGP is submitting points: " << endl;
    Wnew.print("Wnew");
  }

}

void APPSPACK::Citizen::TGP::vote(const ConveyorList& W, 
                                  const map<string, vector<int> >& ownermap,
                                  vector<int>& tagOrder)
{
  tagOrder.clear();

  // We have submitted queueOrder.size() points and want them at the front
  // with the points first in the queueOrder having greatest priority.
  
  // First need to strip off points that have already been evaluated.
  vector<int> wtags;
  W.getTagList(wtags);
  vector<int> itags;
  sort(wtags.begin(), wtags.end());
  sort(queueOrder.begin(), queueOrder.end());

  std::set_intersection(wtags.begin(),wtags.end(), queueOrder.begin(), 
			queueOrder.end(), std::inserter(itags, itags.begin()));

  for (int i=0; i<itags.size(); i++)
    tagOrder.push_back(itags[i]);
}

APPSPACK::Citizen::State APPSPACK::Citizen::TGP::getState()
{
  return APPSPACK::Citizen::StopRequest;
}

const string& APPSPACK::Citizen::TGP::getName() const
{
  return name;
}

bool APPSPACK::Citizen::TGP::worker()
{
  // Blocking receive for the next message
  int msgtag, junk, taskid;
  APPSPACK::GCI::recv();
  APPSPACK::GCI::bufinfo(msgtag, junk);
  
  // Check for termination.
  if (msgtag == APPSPACK::Executor::Oracle::Terminate)
  {
    delete tgpCache;
    return false;
  }
  else if (msgtag == APPSPACK::Executor::Oracle::Init)
  { 
    APPSPACK::GCI::unpack(workerFileName);
    APPSPACK::GCI::unpack(tgpCacheTolerance);
    evalsOutStream.open(workerFileName.c_str());
    APPSPACK::Cache::Point::scaling=constraints.getScaling();
    params.setParameter("Cache Comparison Tolerance", tgpCacheTolerance);
    params.setParameter("Cache Output File", "CachedTest.txt");
    tgpCache = new APPSPACK::Cache::Manager(params, APPSPACK::Cache::Point::scaling);
    return true;
  }

  // Run oracle command.
  resetEvalRecord();
  APPSPACK::GCI::unpack(tgpCommand); 
  APPSPACK::GCI::unpack(F);
  APPSPACK::GCI::unpack(X);
  APPSPACK::GCI::unpack(Xtags);
  APPSPACK::GCI::unpack(Xstep);
  APPSPACK::GCI::unpack(ownerName);
  
  // Write out X and F.
  for (int k=0; k<F.size(); k++)
  {
    const Vector& x=X[k];
    const Vector& f=F[k];
    Vector fcached;
    if (!tgpCache->isCached(x,fcached))
    {
      tgpCache->insert(x,f);
      int tag=Xtags[k];
      evalsOutStream << "f=["; 
      for (int i = 0; i < f.size(); i ++) 
        evalsOutStream << APPSPACK::Print::formatDouble(f[i], precision) << " ";
      evalsOutStream << "] ";    
      // Output x.
      evalsOutStream << "x=["; 
      for (int i = 0; i < x.size(); i ++) 
        evalsOutStream << APPSPACK::Print::formatDouble(x[i], precision) << " ";
      evalsOutStream << "] ";
      // Output tag.
      evalsOutStream << "tag=" <<  tag;
      evalsOutStream << " name=" << ownerName[k];
      evalsOutStream << endl;
    }
  }
  evalsOutStream.flush();

  cout << "tgpCommand=" << tgpCommand << endl;
  system(tgpCommand.c_str());
  string exit_status="TGP call complete";
  cout << exit_status << endl;
  
  APPSPACK::GCI::initSend();
  APPSPACK::GCI::pack(exit_status);
  APPSPACK::GCI::send(APPSPACK::Executor::Oracle::Generate, 0);
  
  return true;
}

void APPSPACK::Citizen::TGP::createStrings(bool useTGP, int nPoints)
{
  fileTag = IntToString(staticCount++);
  string nPointsString = IntToString(nPoints);

  // LHS Sampling or TGP.
  if (useTGP)
    tgpCommand = tgpCommandPrefix + " " + nPointsString + " 1 ";
  else 
    tgpCommand = tgpCommandPrefix+ " " + nPointsString + " 0 ";

  // Global or local search arguments. (g=1 or g=2).
  if (useGlobal)
    tgpCommand = tgpCommand + " 2 ";
  else 
    tgpCommand = tgpCommand + " 1 ";

  // Determine neighbor bounds in R-notation.
  //! R-notation bounds: "c(l1,12,...,ln),c(u1,u2,...,un)".
  const Vector& bLower=constraints.getLower();
  const Vector& bUpper=constraints.getUpper();

  // Write using string streams.
  ostringstream ss;
  
  if (bestX.size() == 0)
  {
    // Process upper and lower bounds.
    ss << "\"c(" << bLower[0];
    for (int i=1; i<bLower.size(); i++)
      ss << "," << bLower[i];
    ss << "),c(" << bUpper[0];
    for (int i=1; i<bUpper.size(); i++)
      ss << "," <<bUpper[i];
    ss << ")\"";
  }
  else
  {
    // Have best point.
    ss << "\"c(" << bestX[0] - min(contractionFactor*(bUpper[0] - bLower[0]), bestX[0] - bLower[0]);
    for (int i=1; i<bLower.size(); i++)
      ss << "," << bestX[i] - min(contractionFactor*(bUpper[i] - bLower[i]), bestX[i] - bLower[i]);

    ss << "),c(" << bestX[0] + min(contractionFactor*(bUpper[0] - bLower[0]), bUpper[0] - bestX[0]);
    for (int i=1; i<bUpper.size(); i++)
      ss << "," << bestX[i] + min(contractionFactor*(bUpper[i] - bLower[i]), bUpper[i] - bestX[i]);
    ss << ")\"";
  }
  
  tgpCommand = tgpCommand + " " + ss.str() + " " + fileTag;
}

void APPSPACK::Citizen::TGP::packNewEvals()
{
  // Pack
  GCI::pack(F);
  GCI::pack(X);
  GCI::pack(Xtags);
  GCI::pack(Xstep);
  GCI::pack(ownerName);

  resetEvalRecord();
}

void APPSPACK::Citizen::TGP::resetEvalRecord()
{
  // Clean up.
  F.clear();
  X.clear();
  Xtags.clear();
  ownerName.clear();
  Xstep.resize(0);
}

void APPSPACK::Citizen::TGP::writeNewEvals(const APPSPACK::ConveyorList& R,
                                           const map<string, vector<int> >& ownermap)
{
  // Output results to TGP cache file.
  const vector<int>& citizen_tags = ownermap.find(name)->second;
  list<APPSPACK::Point*>::const_iterator itr;
  for (itr=R.begin(); itr!=R.end(); itr++)
  {
    int tag = (*itr)->getTag();
    Vector f = (*itr)->getVecF();
    Vector x = (*itr)->getX();
    double step = (*itr)->getStep();
    Point::State state = (*itr)->getState();
    
    if (step <  globalTolerance)
      useGlobal = false;
    
    if ( (state == Point::EvaluatedInsufficientDecrease) ||
         ( state == Point::Evaluated) )
    {    
      if (bestF.size() == 0)
      {
        // F,X not yet set.
        bestX=x;
        bestF=f;
      }
      else if (combiner(f) < combiner(bestF))
      {
        bestX=x;
        bestF=f;
        cout << "TGP new best point = " << x << ", " << f << endl;
      }
      
      F.push_back(f);
      X.push_back(x);
      Xtags.push_back(tag);
      Xstep.push_back(step);
      map<string, vector<int> >::const_iterator oit;
      for (oit=ownermap.begin(); oit !=ownermap.end(); oit++)
        if (find(oit->second.begin(), oit->second.end(), tag) != oit->second.end())
          ownerName.push_back(oit->first);
      if (ownerName.size() != F.size())
      {
        cerr << "APPSPACK::Citizen::TGP::writeNewEvals size mismatch" << endl;
        throw "APPSPACK Error";
      }
         
      // Output f.
      evalsOutStream << "f=["; 
      for (int i = 0; i < f.size(); i ++) 
        evalsOutStream << APPSPACK::Print::formatDouble(f[i], precision) << " ";
      evalsOutStream << "] ";
      
      // Output x.
      evalsOutStream << "x=["; 
      for (int i = 0; i < x.size(); i ++) 
        evalsOutStream << APPSPACK::Print::formatDouble(x[i], precision) << " ";
      evalsOutStream << "] ";
      
      // Output tag.
      evalsOutStream << "tag=" <<  tag;
      if (find(citizen_tags.begin(), citizen_tags.end(), tag) != citizen_tags.end())
        evalsOutStream << " name=" << name;
      evalsOutStream << endl;
    } // End if evaluation is cached.
  }
}

void APPSPACK::Citizen::TGP::readOutputFile(APPSPACK::Matrix& TP, int numPoints)
{
  TP.clear();

  ifstream fin;
  string name = candidateFile + "." + fileTag;
  fin.open(name.c_str());
  if (!fin) 
  {
    cerr << "\nWarning - cannot open candiate file \"" << candidateFile << "\"." << endl;
    fin.close();
    return;
  }

  // TP are trial points to pass back.
  Vector xi;
  int cnt=0;
  string line;
  while (getline(fin, line) && (cnt < numPoints))
  {
    StringToVector(line, xi);
    TP.addRow(xi);
    cnt++;
  }  
}

void APPSPACK::Citizen::TGP::addTrialPoints(ConveyorList& Wnew, const Matrix& TP)
{
  // Set up remaining empty variables to create a parentless point.
  Vector f;
  string msg="(" + name + ")";
  
  queueOrder.resize(0);
  for (int i=0; i<TP.getNrows(); i++)
  {
    const Vector& x=TP.getRow(i);
    Point* p = new Point(x, f, defaultStep, 0.0, combiner, msg);    
    Wnew.push(p);
    queueOrder.push_back(p->getTag());
  }
}

string APPSPACK::Citizen::TGP::IntToString(int n) const
{
  ostringstream ss;
  ss << n;
  return ss.str();
}

void APPSPACK::Citizen::TGP::StringToVector(const string& item, 
					    APPSPACK::Vector& x) const
{
  x.resize(0);
  istringstream ss(item);
  
  double xi;
  while (ss >> xi)
    x.push_back(xi);
}

