/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        MergedVariables
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "MergedVariables.H"
#include "ProblemDescDB.H"
#include "data_io.h"
#include "data_util.h"

static const char rcsId[]="@(#) $Id";


namespace Dakota {

/** In this class, a merged data approach is used in which continuous
    and discrete arrays are combined into a single continuous array
    (integrality is relaxed; the converse of truncating reals is not
    currently supported but could be in the future if needed).
    Iterators/strategies which use this class include:
    BranchBndOptimizer.  Extract fundamental variable types and labels
    and merge continuous and discrete domains to create aggregate
    arrays and views.  */
MergedVariables::
MergedVariables(const ProblemDescDB& problem_db, const pair<short,short>& view):
  Variables(BaseConstructor(), problem_db, view)
{
  // Create merged arrays
  RealDenseVector mdv, msv;
  aggregate_merged(
    problem_db.get_rdv("variables.continuous_design.initial_point"),
    problem_db.get_idv("variables.discrete_design.initial_point"), mdv);
  aggregate_merged(
    problem_db.get_rdv("variables.continuous_state.initial_state"),
    problem_db.get_idv("variables.discrete_state.initial_state"), msv);

  // Aggregate into all arrays
  aggregate_data(mdv, problem_db.get_rdv("variables.uncertain.initial_point"),
    msv, allContinuousVars);
  aggregate_data(problem_db.get_dsa("variables.continuous_design.labels"),
    problem_db.get_dsa("variables.discrete_design.labels"),
    problem_db.get_dsa("variables.uncertain.labels"),
    problem_db.get_dsa("variables.continuous_state.labels"),
    problem_db.get_dsa("variables.discrete_state.labels"),
    allContinuousLabels);

  initialize_all_continuous_var_types_ids(true);
  //initialize_all_discrete_var_types_ids();

  // Construct active/inactive views of all arrays
  build_views();

#ifdef REFCOUNT_DEBUG
  Cout << "Letter instantiated: variablesView active = " << variablesView.first
       << " inactive = " << variablesView.second << endl;
#endif
}


void MergedVariables::reshape_rep(const Sizet2DArray& vars_comps)
{
  size_t num_acv = vars_comps[0][0] + vars_comps[1][0] + vars_comps[2][0] +
    vars_comps[3][0];

  allContinuousVars.resize(num_acv);
  allContinuousLabels.resize(boost::extents[num_acv]);

  initialize_all_continuous_var_types_ids(true);
  //initialize_all_discrete_var_types_ids();

  // Construct active/inactive views of all arrays
  build_views();
}


void MergedVariables::build_active_views()
{
  // Initialize continuousVarTypes/discreteVarTypes and continuousVarIds/
  // mergedDiscreteIds.  Don't bleed over any logic about supported view
  // combinations; rather, keep this class general and encapsulated.
  size_t num_cdv = variablesComponents[0][1],
    num_ddv = variablesComponents[0][2], num_mdv = num_cdv + num_ddv,
    num_csv = variablesComponents[3][1], num_dsv = variablesComponents[3][2],
    num_msv = num_csv + num_dsv,         num_auv = variablesComponents[1][0],
    num_euv = variablesComponents[2][0], num_uv  = num_auv + num_euv;

  // Initialize active views
  //cvStart = dvStart = numCV = numDV = 0;
  switch (variablesView.first) {
  case MERGED_DISTINCT_DESIGN: {
    numCV = num_mdv;
    // initialize mergedDiscreteIds
    mergedDiscreteIds.reshape(num_ddv);
    size_t offset = num_cdv + 1;
    for (size_t i=0; i<num_ddv; i++)
      mergedDiscreteIds[i] = i + offset;
    break;
  }
  case MERGED_DISTINCT_ALEATORY_UNCERTAIN:
    cvStart = num_mdv;          numCV = num_auv; break;
  case MERGED_DISTINCT_EPISTEMIC_UNCERTAIN:
    cvStart = num_mdv+num_auv;  numCV = num_euv; break;
  case MERGED_DISTINCT_UNCERTAIN:
    cvStart = num_mdv;          numCV = num_uv;  break;
  case MERGED_DISTINCT_STATE: {
    cvStart = num_mdv + num_uv; numCV = num_msv;
    // initialize mergedDiscreteIds
    mergedDiscreteIds.reshape(num_dsv);
    size_t offset = num_mdv + num_uv + num_csv + 1;
    for (size_t i=0; i<num_dsv; i++)
      mergedDiscreteIds[i] = i + offset;
    break;
  }
  }
  if (numCV)
    continuousVars
      = RealDenseVector(Teuchos::View, &allContinuousVars[cvStart], numCV);
}


void MergedVariables::build_inactive_views()
{
  // Initialize continuousVarTypes/discreteVarTypes and continuousVarIds/
  // mergedDiscreteIds.  Don't bleed over any logic about supported view
  // combinations; rather, keep this class general and encapsulated.
  size_t num_cdv = variablesComponents[0][1],
    num_ddv = variablesComponents[0][2], num_auv = variablesComponents[1][0],
    num_euv = variablesComponents[2][0], num_csv = variablesComponents[3][1],
    num_dsv = variablesComponents[3][2];

  // Initialize inactive views
  //icvStart = idvStart = numICV = numIDV = 0;
  switch (variablesView.second) {
  case MERGED_DISTINCT_DESIGN:
    numICV   = num_cdv + num_ddv;                                       break;
  case MERGED_DISTINCT_ALEATORY_UNCERTAIN:
    icvStart = num_cdv + num_ddv;           numICV = num_auv;           break;
  case MERGED_DISTINCT_EPISTEMIC_UNCERTAIN:
    icvStart = num_cdv + num_ddv + num_auv; numICV = num_euv;           break;
  case MERGED_DISTINCT_UNCERTAIN:
    icvStart = num_cdv + num_ddv;           numICV = num_auv + num_euv; break;
  case MERGED_DISTINCT_STATE:
    icvStart = num_cdv + num_ddv + num_auv + num_euv;
    numICV   = num_csv + num_dsv;                                       break;
  }
  if (numICV)
    inactiveContinuousVars
      = RealDenseVector(Teuchos::View, &allContinuousVars[icvStart], numICV);
}


// Reordering is required in all read/write cases that will be visible to the
// user since all derived vars classes should use the same CDV/DDV/UV/CSV/DSV
// ordering for clarity.  Neutral file I/O, binary streams, and packed buffers
// do not need to reorder (so long as read/write are consistent) since this data
// is not intended for public consumption.
void MergedVariables::read(istream& s)
{ read_data(s, allContinuousVars, allContinuousLabels); }


void MergedVariables::write(ostream& s) const
{ write_data(s, allContinuousVars, allContinuousLabels); }


void MergedVariables::write_aprepro(ostream& s) const
{ write_data_aprepro(s, allContinuousVars, allContinuousLabels); }


void MergedVariables::read_annotated(istream& s)
{
  // ASCII version for neutral file I/O.
  if (variablesComponents.empty()) {
    variablesComponents.reshape(4);
    variablesComponents[0].reshape(3); // no sum
    variablesComponents[1].reshape(1); // sum only
    variablesComponents[2].reshape(1); // sum only
    variablesComponents[3].reshape(3); // no sum
  }
  s >> variablesComponents[0][1] >> variablesComponents[0][2]
    >> variablesComponents[1][0] >> variablesComponents[2][0]
    >> variablesComponents[3][1] >> variablesComponents[3][2];
  read_data_annotated(s, allContinuousVars, allContinuousLabels);
  build_views();
}


void MergedVariables::write_annotated(ostream& s) const
{
  // ASCII version for neutral file I/O.
  s << variablesComponents[0][1] << ' ' << variablesComponents[0][2] << ' '
    << variablesComponents[1][0] << ' ' << variablesComponents[2][0] << ' '
    << variablesComponents[3][1] << ' ' << variablesComponents[3][2] << ' ';
  write_data_annotated(s, allContinuousVars, allContinuousLabels);
}


void MergedVariables::write_tabular(ostream& s) const
{
  // Tabular format ordering is cdv:uv:csv:ddv:dsv
  size_t num_cdv    = variablesComponents[0][1],
         num_ddv    = variablesComponents[0][2], num_dv = num_cdv + num_ddv,
         num_uv_csv = variablesComponents[1][0] + variablesComponents[2][0] +
                      variablesComponents[3][1],
         num_dsv    = variablesComponents[3][2];
  write_data_partial_tabular(s, 0, num_cdv, allContinuousVars);
  write_data_partial_tabular(s, num_dv, num_uv_csv, allContinuousVars);
  write_data_partial_tabular(s, num_cdv, num_ddv, allContinuousVars);
  write_data_partial_tabular(s, num_dv+num_uv_csv+num_ddv, num_dsv,
			     allContinuousVars);
}


void MergedVariables::read(BiStream& s)
{
  // Binary version.
  if (variablesComponents.empty()) {
    variablesComponents.reshape(4);
    variablesComponents[0].reshape(3); // no sum
    variablesComponents[1].reshape(1); // sum only
    variablesComponents[2].reshape(1); // sum only
    variablesComponents[3].reshape(3); // no sum
  }
  s >> variablesComponents[0][1] >> variablesComponents[0][2]
    >> variablesComponents[1][0] >> variablesComponents[2][0]
    >> variablesComponents[3][1] >> variablesComponents[3][2];
  read_data(s, allContinuousVars, allContinuousLabels);
  build_views();
}


void MergedVariables::write(BoStream& s) const
{
  // Binary version.
  s << variablesComponents[0][1] << variablesComponents[0][2]
    << variablesComponents[1][0] << variablesComponents[2][0]
    << variablesComponents[3][1] << variablesComponents[3][2];
  write_data(s, allContinuousVars, allContinuousLabels);
}


void MergedVariables::read(MPIUnpackBuffer& s)
{
  // MPI buffer version.
  if (variablesComponents.empty()) {
    variablesComponents.reshape(4);
    variablesComponents[0].reshape(3); // no sum
    variablesComponents[1].reshape(1); // sum only
    variablesComponents[2].reshape(1); // sum only
    variablesComponents[3].reshape(3); // no sum
  }
  s >> variablesComponents[0][1] >> variablesComponents[0][2]
    >> variablesComponents[1][0] >> variablesComponents[2][0]
    >> variablesComponents[3][1] >> variablesComponents[3][2];
  read_data(s, allContinuousVars, allContinuousLabels);
  build_views();
}


void MergedVariables::write(MPIPackBuffer& s) const
{
  // MPI buffer version.
  s << variablesComponents[0][1] << variablesComponents[0][2]
    << variablesComponents[1][0] << variablesComponents[2][0]
    << variablesComponents[3][1] << variablesComponents[3][2];
  write_data(s, allContinuousVars, allContinuousLabels);
}

} // namespace Dakota
