/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        DataVariables
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "DataVariables.H"
#include "data_io.h"


namespace Dakota {

// Default constructor:
DataVariablesRep::DataVariablesRep():
  numContinuousDesVars(0), numDiscreteDesVars(0), numNormalUncVars(0),
  numLognormalUncVars(0), numUniformUncVars(0), numLoguniformUncVars(0),
  numTriangularUncVars(0), numExponentialUncVars(0), numBetaUncVars(0),
  numGammaUncVars(0), numGumbelUncVars(0), numFrechetUncVars(0),
  numWeibullUncVars(0), numHistogramUncVars(0), numIntervalUncVars(0),
  numContinuousStateVars(0), numDiscreteStateVars(0),
  histogramUncSkylineFlag(false), referenceCount(1)
{ }


void DataVariablesRep::write(MPIPackBuffer& s) const
{
  s << idVariables          << numContinuousDesVars   << numDiscreteDesVars
    << numNormalUncVars     << numLognormalUncVars    << numUniformUncVars
    << numLoguniformUncVars << numTriangularUncVars   << numExponentialUncVars
    << numBetaUncVars       << numGammaUncVars        << numGumbelUncVars
    << numFrechetUncVars    << numWeibullUncVars      << numHistogramUncVars
    << numIntervalUncVars   << numContinuousStateVars << numDiscreteStateVars;

  // Design arrays
  s << continuousDesignVars << continuousDesignLowerBnds
    << continuousDesignUpperBnds << continuousDesignScaleTypes
    << continuousDesignScales << discreteDesignVars << discreteDesignLowerBnds
    << discreteDesignUpperBnds << continuousDesignLabels
    << discreteDesignLabels;

  // Uncertain arrays
  s << normalUncMeans << normalUncStdDevs << normalUncLowerBnds
    << normalUncUpperBnds << lognormalUncMeans << lognormalUncStdDevs
    << lognormalUncErrFacts << lognormalUncLowerBnds << lognormalUncUpperBnds
    << uniformUncLowerBnds << uniformUncUpperBnds << loguniformUncLowerBnds
    << loguniformUncUpperBnds << triangularUncModes << triangularUncLowerBnds
    << triangularUncUpperBnds << exponentialUncBetas << betaUncAlphas
    << betaUncBetas << betaUncLowerBnds << betaUncUpperBnds << gammaUncAlphas
    << gammaUncBetas << gumbelUncAlphas << gumbelUncBetas << frechetUncAlphas
    << frechetUncBetas << weibullUncAlphas << weibullUncBetas
    << histogramUncSkylineFlag << histogramUncBinPairs << histogramUncPointPairs
    << intervalUncBasicProbs << intervalUncBounds << uncertainCorrelations
    << uncertainVars << uncertainLowerBnds << uncertainUpperBnds
    << uncertainLabels;

  // State arrays
  s << continuousStateVars << continuousStateLowerBnds
    << continuousStateUpperBnds << discreteStateVars
    << discreteStateLowerBnds << discreteStateUpperBnds
    << continuousStateLabels << discreteStateLabels;
}


void DataVariablesRep::read(MPIUnpackBuffer& s)
{
  s >> idVariables          >> numContinuousDesVars   >> numDiscreteDesVars
    >> numNormalUncVars     >> numLognormalUncVars    >> numUniformUncVars
    >> numLoguniformUncVars >> numTriangularUncVars   >> numExponentialUncVars
    >> numBetaUncVars       >> numGammaUncVars        >> numGumbelUncVars
    >> numFrechetUncVars    >> numWeibullUncVars      >> numHistogramUncVars
    >> numIntervalUncVars   >> numContinuousStateVars >> numDiscreteStateVars;

  // Design arrays
  s >> continuousDesignVars >> continuousDesignLowerBnds
    >> continuousDesignUpperBnds >> continuousDesignScaleTypes
    >> continuousDesignScales >> discreteDesignVars >> discreteDesignLowerBnds
    >> discreteDesignUpperBnds >> continuousDesignLabels
    >> discreteDesignLabels;

  // Uncertain arrays
  s >> normalUncMeans >> normalUncStdDevs >> normalUncLowerBnds
    >> normalUncUpperBnds >> lognormalUncMeans >> lognormalUncStdDevs
    >> lognormalUncErrFacts >> lognormalUncLowerBnds >> lognormalUncUpperBnds
    >> uniformUncLowerBnds >> uniformUncUpperBnds >> loguniformUncLowerBnds
    >> loguniformUncUpperBnds >> triangularUncModes >> triangularUncLowerBnds
    >> triangularUncUpperBnds >> exponentialUncBetas >> betaUncAlphas
    >> betaUncBetas >> betaUncLowerBnds >> betaUncUpperBnds >> gammaUncAlphas
    >> gammaUncBetas >> gumbelUncAlphas >> gumbelUncBetas >> frechetUncAlphas
    >> frechetUncBetas >> weibullUncAlphas >> weibullUncBetas
    >> histogramUncSkylineFlag >> histogramUncBinPairs >> histogramUncPointPairs
    >> intervalUncBasicProbs >> intervalUncBounds >> uncertainCorrelations
    >> uncertainVars >> uncertainLowerBnds >> uncertainUpperBnds
    >> uncertainLabels;

  // State arrays
  s >> continuousStateVars >> continuousStateLowerBnds
    >> continuousStateUpperBnds >> discreteStateVars
    >> discreteStateLowerBnds >> discreteStateUpperBnds
    >> continuousStateLabels >> discreteStateLabels;
}


void DataVariablesRep::write(ostream& s) const
{
  s << idVariables          << numContinuousDesVars   << numDiscreteDesVars
    << numNormalUncVars     << numLognormalUncVars    << numUniformUncVars
    << numLoguniformUncVars << numTriangularUncVars   << numExponentialUncVars
    << numBetaUncVars       << numGammaUncVars        << numGumbelUncVars
    << numFrechetUncVars    << numWeibullUncVars      << numHistogramUncVars
    << numIntervalUncVars   << numContinuousStateVars << numDiscreteStateVars;

  // Design arrays
  s << continuousDesignVars << continuousDesignLowerBnds
    << continuousDesignUpperBnds << continuousDesignScaleTypes
    << continuousDesignScales << discreteDesignVars << discreteDesignLowerBnds
    << discreteDesignUpperBnds << continuousDesignLabels
    << discreteDesignLabels;

  // Uncertain arrays
  s << normalUncMeans << normalUncStdDevs << normalUncLowerBnds
    << normalUncUpperBnds << lognormalUncMeans << lognormalUncStdDevs
    << lognormalUncErrFacts << lognormalUncLowerBnds << lognormalUncUpperBnds
    << uniformUncLowerBnds << uniformUncUpperBnds << loguniformUncLowerBnds
    << loguniformUncUpperBnds << triangularUncModes << triangularUncLowerBnds
    << triangularUncUpperBnds << exponentialUncBetas << betaUncAlphas
    << betaUncBetas << betaUncLowerBnds << betaUncUpperBnds << gammaUncAlphas
    << gammaUncBetas << gumbelUncAlphas << gumbelUncBetas << frechetUncAlphas
    << frechetUncBetas << weibullUncAlphas << weibullUncBetas
    << histogramUncSkylineFlag << histogramUncBinPairs << histogramUncPointPairs
    << intervalUncBasicProbs << intervalUncBounds << uncertainCorrelations
    << uncertainVars << uncertainLowerBnds << uncertainUpperBnds
    << uncertainLabels;

  // State arrays
  s << continuousStateVars << continuousStateLowerBnds
    << continuousStateUpperBnds << discreteStateVars
    << discreteStateLowerBnds << discreteStateUpperBnds
    << continuousStateLabels << discreteStateLabels;
}


DataVariables::DataVariables(): dataVarsRep(new DataVariablesRep())
{
#ifdef REFCOUNT_DEBUG
  Cout << "DataVariables::DataVariables(), dataVarsRep referenceCount = "
       << dataVarsRep->referenceCount << endl;
#endif
}


DataVariables::DataVariables(const DataVariables& data_vars)
{
  // Increment new (no old to decrement)
  dataVarsRep = data_vars.dataVarsRep;
  if (dataVarsRep) // Check for an assignment of NULL
    dataVarsRep->referenceCount++;

#ifdef REFCOUNT_DEBUG
  Cout << "DataVariables::DataVariables(DataVariables&)" << endl;
  if (dataVarsRep)
    Cout << "dataVarsRep referenceCount = " << dataVarsRep->referenceCount
	 << endl;
#endif
}


DataVariables DataVariables::operator=(const DataVariables& data_vars)
{
  if (dataVarsRep != data_vars.dataVarsRep) { // normal case: old != new
    // Decrement old
    if (dataVarsRep) // Check for NULL
      if ( --dataVarsRep->referenceCount == 0 ) 
	delete dataVarsRep;
    // Assign and increment new
    dataVarsRep = data_vars.dataVarsRep;
    if (dataVarsRep) // Check for NULL
      dataVarsRep->referenceCount++;
  }
  // else if assigning same rep, then do nothing since referenceCount
  // should already be correct

#ifdef REFCOUNT_DEBUG
  Cout << "DataVariables::operator=(DataVariables&)" << endl;
  if (dataVarsRep)
    Cout << "dataVarsRep referenceCount = " << dataVarsRep->referenceCount
	 << endl;
#endif

  return *this;
}


DataVariables::~DataVariables()
{
  if (dataVarsRep) { // Check for NULL
    --dataVarsRep->referenceCount; // decrement
#ifdef REFCOUNT_DEBUG
    Cout << "dataVarsRep referenceCount decremented to "
         << dataVarsRep->referenceCount << endl;
#endif
    if (dataVarsRep->referenceCount == 0) {
#ifdef REFCOUNT_DEBUG
      Cout << "deleting dataVarsRep" << endl;
#endif
      delete dataVarsRep;
    }
  }
}

} // namespace Dakota
