/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        Minimizer
//- Description:  Base class for the optimizer and least squares branches
//-               of the iterator hierarchy.
//- Owner:        Mike Eldred
//- Version: $Id: DakotaMinimizer.H 5780 2009-03-26 22:14:19Z mseldre $

#ifndef DAKOTA_MINIMIZER_H
#define DAKOTA_MINIMIZER_H

#include "DakotaIterator.H"
#include "DakotaResponse.H"


namespace Dakota {


/// Base class for the optimizer and least squares branches of the
/// iterator hierarchy.

/** The Minimizer class provides common data and functionality
    for Optimizer and LeastSq. */

class Minimizer: public Iterator
{
  //
  //- Heading: Friends
  //

  /// the SOLBase class is not derived the iterator hierarchy but still needs
  /// access to iterator hierarchy data (to avoid attribute replication)
  friend class SOLBase;
  /// the SNLLBase class is not derived the iterator hierarchy but still needs
  /// access to iterator hierarchy data (to avoid attribute replication)
  friend class SNLLBase;

public:

  //
  //- Heading: Virtual member function redefinitions
  //

  const Variables& variables_results() const;
  const Response&  response_results()  const;

  const VariablesArray& variables_array_results() const;
  const ResponseArray&  response_array_results()  const;

  void response_results_active_set(const ActiveSet& set);

protected:

  //
  //- Heading: Constructors and destructor
  //

  Minimizer();             ///< default constructor
  Minimizer(Model& model); ///< standard constructor

  /// alternate constructor for "on the fly" instantiations
  Minimizer(NoDBBaseConstructor, Model& model);
  /// alternate constructor for "on the fly" instantiations
  Minimizer(NoDBBaseConstructor, size_t num_lin_ineq, size_t num_lin_eq,
	    size_t num_nln_ineq, size_t num_nln_eq);

  ~Minimizer();            ///< destructor

  //
  //- Heading: Virtual member function redefinitions
  //

  void derived_pre_run();
  void derived_post_run();

  //
  //- Heading: Convenience/Helper functions
  //

  //  scaling initialization helper functions:

  /// initialize scaling types, multipliers, and offsets; perform error
  /// checking
  void initialize_scaling();
  /// general helper function for initializing scaling types and factors on a 
  /// vector of variables, functions, constraints, etc.
  void compute_scaling(int object_type, int auto_type, int num_vars,
		       RealDenseVector& lbs, RealDenseVector& ubs,
		       RealDenseVector& targets,
		       const StringArray& scale_strings, 
		       const RealVector& scales,
		       IntVector& scale_types, RealVector& scale_mults,
		       RealVector& scale_offsets);
  /// automatically compute a single scaling factor -- bounds case
  bool compute_scale_factor(const Real lower_bound, const Real upper_bound,
			    Real *multiplier, Real *offset);
  /// automatically compute a single scaling factor -- target case
  bool compute_scale_factor(const Real target, Real *multiplier);

  //  conversion helper functions between native/user and scaled/iterator:

  /// variables conversion map for RecastModel used in scaling: transform
  /// variables from scaled to native (user) space
  static void variables_recast(const Variables& scaled_vars, 
			       Variables& native_vars);
  /// secondary response conversion map for RecastModel used in scaling:
  /// transform constraints (fns, grads, Hessians) from native (user) to 
  // iterator space
  static void secondary_resp_recast(const Variables& native_vars,
				    const Variables& scaled_vars,
				    const Response& native_response,
				    Response& scaled_response);
  /// determine if response transformation is needed due to variable
  /// transformations
  bool need_resp_trans_byvars(const ShortArray& asv, int start_index, 
			      int num_resp);
  /// general RealVector mapping from native to scaled variables vectors: 
  RealDenseVector modify_n2s(const RealDenseVector& native_vars,
			     const IntVector& scale_types,
			     const RealVector& multipliers,
			     const RealVector& offsets) const;
  /// general RealVector mapping from scaled to native variables (and values) 
  RealDenseVector modify_s2n(const RealDenseVector& scaled_vars,
			     const IntVector& scale_types,
			     const RealVector& multipliers,
			     const RealVector& offsets) const;
  /// map reponses from native to scaled variable space
  void response_modify_n2s(const Variables& scaled_vars,
			   const Response& native_response,
			   Response& scaled_response,
			   int native_offset, int recast_offset,
			   int num_responses) const;
  /// map responses from scaled to native variable space
  void response_modify_s2n(const Variables& native_vars,
			   const Response& scaled_response,
			   Response& native_response,
			   int scaled_offset, int native_offset,
			   int num_responses) const;
  /// general linear coefficients mapping from native to scaled space 
  RealDenseMatrix lin_coeffs_modify_n2s(const RealDenseMatrix& native_coeffs, 
    const RealVector& cv_multipliers, const RealVector& lin_multipliers) const;

  /// print scaling information for a particular response type in tabular form
  void print_scaling(const String& info, const IntVector& scale_types,
		     const RealVector& scale_mults,
		     const RealVector& scale_offsets, 
		     const StringArray& labels);

  //
  //- Heading: Data
  //

  Real constraintTol; ///< optimizer/least squares constraint tolerance

  /// cutoff value for inequality constraint and continuous variable bounds
  Real bigRealBoundSize;
  /// cutoff value for discrete variable bounds
  int bigIntBoundSize;

  /// number of nonlinear inequality constraints
  size_t numNonlinearIneqConstraints;
  /// number of nonlinear equality constraints
  size_t numNonlinearEqConstraints;

  /// number of linear inequality constraints
  size_t numLinearIneqConstraints;
  /// number of linear equality constraints
  size_t numLinearEqConstraints;

  /// total number of nonlinear constraints
  int numNonlinearConstraints;
  /// total number of linear constraints
  int numLinearConstraints;
  /// total number of linear and nonlinear constraints
  int numConstraints;

  /// convenience flag for denoting the presence of user-specified bound
  /// constraints.  Used for method selection and error checking.
  bool boundConstraintFlag;

  bool speculativeFlag; ///< flag for speculative gradient evaluations

  /// number of objective functions or least squares terms in the user's model
  size_t numUserPrimaryFns;
  /// number of objective functions or least squares terms in iterator's view
  size_t numIterPrimaryFns;

  // scaling data follow 
  bool       scaleFlag;              ///< flag for overall scaling status
  bool       varsScaleFlag;          ///< flag for variables scaling
  bool       primaryRespScaleFlag;   ///< flag for primary response scaling
  bool       secondaryRespScaleFlag; ///< flag for secondary response scaling

  IntVector  cvScaleTypes;           ///< scale flags for continuous vars.
  RealVector cvScaleMultipliers;     ///< scales for continuous variables
  RealVector cvScaleOffsets;         ///< offsets for continuous variables

  IntVector  responseScaleTypes;         ///< scale flags for all responses
  RealVector responseScaleMultipliers;   ///< scales for all responses
  RealVector responseScaleOffsets;       ///< offsets for all responses (zero
                                         ///< for functions, not for nonlin con)

  IntVector  linearIneqScaleTypes;       ///< scale flags for linear ineq
  RealVector linearIneqScaleMultipliers; ///< scales for linear ineq constrs.
  RealVector linearIneqScaleOffsets;     ///< offsets for linear ineq constrs.

  IntVector  linearEqScaleTypes;         ///< scale flags for linear eq.
  RealVector linearEqScaleMultipliers;   ///< scales for linear constraints
  RealVector linearEqScaleOffsets;       ///< offsets for linear constraints

  /// pointer to Minimizer used in static member functions
  static Minimizer* minimizerInstance;
  /// pointer containing previous value of minimizerInstance
  Minimizer* prevMinInstance;

  /// convenience flag for gradType == numerical && methodSource == vendor
  bool vendorNumericalGradFlag;

  Variables  bestVariables; ///< best variables found in minimization
  Response   bestResponse;  ///< best response found in minimization

  /// collection of all best solution variables.
  VariablesArray bestVariablesArray;
  /// collection of all best solution responses.
  ResponseArray bestResponseArray;


private:

  //
  //- Heading: Data
  //
};


inline Minimizer::Minimizer(): scaleFlag(false)
{ }


inline Minimizer::~Minimizer() { }


inline const Variables& Minimizer::variables_results() const
{ return bestVariables; }


inline const Response& Minimizer::response_results() const
{ return bestResponse; }


inline const VariablesArray& Minimizer::variables_array_results() const
{
  // If the variables array is empty, we will assume that the minimizer that
  // was run does not support multiple solutions.  We will therefore implement
  // the default behavior of putting the single solution into the array and
  // returning it.  Note that we are already at the letter level once we are
  // in this method.
  //if(bestVariablesArray.empty())
    // Ugly but must cast away const or make method non-const which is worse.
    //const_cast<VariablesArray&>(bestVariablesArray).push_back(
    //  this->variables_results());
  return bestVariablesArray;
}


inline const ResponseArray& Minimizer::response_array_results() const
{
  // If the response array is empty, we will asssume that the
  // minimizer that was run does not support multiple solutions.  We
  // will therefore implement the default behavior of putting the
  // single solution into the array and returning it.  Note that we
  // are already at the letter level once we are in this method.
  //if(bestResponseArray.empty())
    // Ugly but must cast away const or make method non-const which is worse.
    //const_cast<ResponseArray&>(bestResponseArray).push_back(
    //  this->response_results());
  return bestResponseArray;
}


inline void Minimizer::response_results_active_set(const ActiveSet& set)
{ bestResponse.active_set(set); }

} // namespace Dakota

#endif
